/*
 * Copyright (C) 2005 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*  modified and restricted to data handling and buffer allocation
 *  binder and ipc support removed
 *  transfer of complex objects removed
 */


#define LOG_TAG "Parcel"
//#define LOG_NDEBUG 0


#include "config.h"
#include <cstddef>
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <sys/mman.h>
#include <string.h>

#include "ril_i.h"

#include "Parcel.h"
#include "utils_log.h"
#include "utils_misc.h"

#ifndef INT32_MAX
#define INT32_MAX ((int32_t)(2147483647))
#endif

#define LOG_REFS(...)
//#define LOG_REFS(...) LOG(LOG_DEBUG, "Parcel", __VA_ARGS__)

#define UTF16_REPLACEMENT_CHAR 0xfffd

/* Clever trick from Dianne that returns 1-4 depending on leading bit sequence*/
#define UTF8_SEQ_LENGTH(ch) (((0xe5000000 >> ((ch >> 3) & 0x1e)) & 3) + 1)

/* note: macro expands to multiple lines */
#define UTF8_SHIFT_AND_MASK(unicode, byte)  \
            (unicode)<<=6; (unicode) |= (0x3f & (byte));

#define UNICODE_UPPER_LIMIT 0x10fffd

#ifdef __cplusplus
template<bool> struct CompileTimeAssert;
template<> struct CompileTimeAssert<true> {};
#define COMPILE_TIME_ASSERT(_exp) \
    template class CompileTimeAssert<  (_exp) >;

#define COMPILE_TIME_ASSERT_FUNCTION_SCOPE(_exp) \
    CompileTimeAssert< (_exp) >();

#endif

// ---------------------------------------------------------------------------

#define PAD_SIZE(s) (((s)+3)&~3)

int32_t Parcel::ParcelEnd = 0xff00aa55;

Parcel::Parcel()
{
    initState();
}

Parcel::~Parcel()
{
    freeDataNoInit();
}


const uint8_t* Parcel::data() const
{
    return mData;
}

size_t Parcel::dataSize() const
{
    return (mDataSize > mDataPos ? mDataSize : mDataPos);
}

size_t Parcel::dataAvail() const
{
    // TODO: decide what to do about the possibility that this can
    // report an available-data size that exceeds a Java int's max
    // positive value, causing havoc.  Fortunately this will only
    // happen if someone constructs a Parcel containing more than two
    // gigabytes of data, which on typical phone hardware is simply
    // not possible.
    return dataSize() - dataPosition();
}

size_t Parcel::dataPosition() const
{
    return mDataPos;
}

size_t Parcel::dataCapacity() const
{
    return mDataCapacity;
}

status_t Parcel::setDataRef(uint8_t* buffer, size_t len)
{
    if (mData)
        return NO_MEMORY;
    mData=buffer;
    mDataSize=len;
    mDataPos=0;
    mDataCapacity=len;
    return NO_ERROR;
}

status_t Parcel::unsetDataRef(uint8_t* buffer)
{
    if (mData!=buffer)
        return NO_MEMORY;
    mData=NULL;
    mDataSize=0;
    mDataPos=0;
    mDataCapacity=0;
    return NO_ERROR;
}

status_t Parcel::setDataSize(size_t size)
{
    status_t err;
    err = continueWrite(size);
    if (err == NO_ERROR) {
        mDataSize = size;
        ACLOGV(LOG_TAG,"Parcel::%s: Setting data size of %p to %d", __func__, this, mDataSize);
    }
    return err;
}

void Parcel::setDataPosition(size_t pos) const
{
    mDataPos = pos;
}

status_t Parcel::setDataCapacity(size_t size)
{
    if (size > mDataCapacity) return continueWrite(size);
    return NO_ERROR;
}

status_t Parcel::setData(const uint8_t* buffer, size_t len)
{
    status_t err = restartWrite(len);
    if (err == NO_ERROR) {
        memcpy(const_cast<uint8_t*>(data()), buffer, len);
        mDataSize = len;
    }
    return err;
}


status_t Parcel::appendFrom(const Parcel *parcel, size_t offset, size_t len)
{
    status_t err;
    const uint8_t *data = parcel->mData;
    int startPos = mDataPos;
    int firstIndex = -1, lastIndex = -2;

        if (len == 0) {
        return NO_ERROR;
    }

    // range checks against the source parcel size
    if ((offset > parcel->mDataSize)
            || (len > parcel->mDataSize)
            || (offset + len > parcel->mDataSize)) {
        return BAD_VALUE;
    }

    if ((mDataSize+len) > mDataCapacity) {
        // grow data
        err = growData(len);
        if (err != NO_ERROR) {
            return err;
        }
    }

    // append data
    memcpy(mData + mDataPos, data + offset, len);
    mDataPos += len;
    mDataSize += len;

    err = NO_ERROR;

    return err;
}

status_t Parcel::errorCheck() const
{
    return mError;
}

void Parcel::setError(status_t err)
{
    mError = err;
}

status_t Parcel::finishWrite(size_t len)
{
    //printf("Finish write of %d\n", len);
    mDataPos += len;
    ACLOGV(LOG_TAG,"Parcel::%s: Setting data pos of %p to %d", __func__, his, mDataPos);
    if (mDataPos > mDataSize) {
        mDataSize = mDataPos;
        ACLOGV(LOG_TAG,"Parcel::%s: Setting data size of %p to %d", __func__, this, mDataSize);
    }
    //printf("New pos=%d, size=%d\n", mDataPos, mDataSize);
    return NO_ERROR;
}

status_t Parcel::writeUnpadded(const void* data, size_t len)
{

    void* const d = writeInplaceUnpadded(len);

    if (d) {
        ACLOGV(LOG_TAG,"Parcel::%s: to %p; len=%d",__func__, d, len);
        memcpy(d, data, len);
        return NO_ERROR;
    }
    return mError;
}

status_t Parcel::write(const void* data, size_t len)
{
    void* const d = writeInplace(len);
    if (d) {
        memcpy(d, data, len);
        return NO_ERROR;
    }
    return mError;
}

void* Parcel::writeInplace(size_t len)
{
    const size_t padded = PAD_SIZE(len);

    // sanity check for integer overflow
    if (mDataPos+padded < mDataPos) {
        return NULL;
    }

    if ((mDataPos+padded) <= mDataCapacity) {
restart_write:
        //printf("Writing %ld bytes, padded to %ld\n", len, padded);
        uint8_t* const data = mData+mDataPos;

        // Need to pad at end?
        if (padded != len) {
#if BYTE_ORDER == BIG_ENDIAN
            static const uint32_t mask[4] = {
                0x00000000, 0xffffff00, 0xffff0000, 0xff000000
            };
#endif
#if BYTE_ORDER == LITTLE_ENDIAN
            static const uint32_t mask[4] = {
                0x00000000, 0x00ffffff, 0x0000ffff, 0x000000ff
            };
#endif
            //printf("Applying pad mask: %p to %p\n", (void*)mask[padded-len],
            //    *reinterpret_cast<void**>(data+padded-4));
            *reinterpret_cast<uint32_t*>(data+padded-4) &= mask[padded-len];
        }

        finishWrite(padded);
        return data;
    }

    status_t err = growData(padded);
    if (err == NO_ERROR) goto restart_write;
    return NULL;
}

void* Parcel::writeInplaceUnpadded(size_t len)
{
    const size_t unpadded = len;

    // sanity check for integer overflow
    if (mDataPos+unpadded < mDataPos) {
        return NULL;
    }

    if ((mDataPos+unpadded) <= mDataCapacity) {
restart_write:
        ACLOGV(LOG_TAG,"Parcel::%s: Writing %ld bytes, unpadded to %ld", __func__, len, unpadded);
        uint8_t* const data = mData+mDataPos;

        // Need to pad at end?
        if (unpadded != len) {
#if BYTE_ORDER == BIG_ENDIAN
            static const uint32_t mask[4] = {
                0x00000000, 0xffffff00, 0xffff0000, 0xff000000
            };
#endif
#if BYTE_ORDER == LITTLE_ENDIAN
            static const uint32_t mask[4] = {
                0x00000000, 0x00ffffff, 0x0000ffff, 0x000000ff
            };
#endif
            ACLOGV(LOG_TAG, "Parcel::%s: Applying pad mask: %p to %p", __func__,(void*)mask[unpadded-len],
                *reinterpret_cast<void**>(data+unpadded-4));
            *reinterpret_cast<uint32_t*>(data+unpadded-4) &= mask[unpadded-len];
        }

        finishWrite(unpadded);
        return data;
    }

    status_t err = growData(unpadded);
    if (err == NO_ERROR) goto restart_write;
    return NULL;
}



status_t Parcel::writeInt32(int32_t val)
{
    return writeAligned(val);
}

status_t Parcel::writeInt64(int64_t val)
{
    return writeAligned(val);
}

status_t Parcel::writeFloat(float val)
{
    return writeAligned(val);
}

status_t Parcel::writeDouble(double val)
{
    return writeAligned(val);
}

status_t Parcel::writeIntPtr(intptr_t val)
{
    return writeAligned(val);
}

status_t Parcel::writeCString(const char* str)
{
    return write(str, strlen(str)+1);
}

status_t Parcel::writeString16(const char16_t* str, size_t len)
{
    if (str == NULL) return writeInt32(-1);

    status_t err = writeInt32(len);
    //ALOGD(LOG_TAG,"Parcel::writeString16: len = %d",len);
    if (err == NO_ERROR) {
        len *= sizeof(char16_t);
        uint8_t* data = (uint8_t*)writeInplace(len+sizeof(char16_t));
        if (data) {
            memcpy(data, str, len);
            *reinterpret_cast<char16_t*>(data+len) = 0;
            //ALOGD(LOG_TAG,"Parcel::writeString16: copy done");
            return NO_ERROR;
        }
        err = mError;
    }
    return err;
}

void Parcel::remove(size_t start, size_t amt)
{
    ACLOGE(LOG_TAG,"Parcel::%s: not yet implemented!",__func__);
}

status_t Parcel::read(void* outData, size_t len) const
{
    if ((mDataPos+PAD_SIZE(len)) >= mDataPos && (mDataPos+PAD_SIZE(len)) <= mDataSize) {
        memcpy(outData, mData+mDataPos, len);
        mDataPos += PAD_SIZE(len);
        ACLOGV(LOG_TAG,"Parcel::%s: Setting data pos of %p to %d",__func__, this, mDataPos);
        return NO_ERROR;
    }
    return NOT_ENOUGH_DATA;
}

status_t Parcel::readUnpadded(void* outData, size_t len) const
{
    //note:  padding is not considered here !!
    if ((mDataPos+len) >= mDataPos && (mDataPos+len) <= mDataSize) {
        memcpy(outData, mData+mDataPos, len);
        mDataPos += len;
        ACLOGV(LOG_TAG,"Parcel::%s: Setting data pos of %p to %d",__func__, this, mDataPos);
        return NO_ERROR;
    }
    return NOT_ENOUGH_DATA;
}


const void* Parcel::readInplace(size_t len) const
{
    if ((mDataPos+PAD_SIZE(len)) >= mDataPos && (mDataPos+PAD_SIZE(len)) <= mDataSize) {
        const void* data = mData+mDataPos;
        mDataPos += PAD_SIZE(len);
        ACLOGV(LOG_TAG,"Parcel::readInplace: Setting data pos of %p to %d", __func__, this, mDataPos);
        return data;
    }
    return NULL;
}

template<class T>
status_t Parcel::readAligned(T *pArg) const {
    COMPILE_TIME_ASSERT_FUNCTION_SCOPE(PAD_SIZE(sizeof(T)) == sizeof(T));

    if ((mDataPos+sizeof(T)) <= mDataSize) {
        const void* data = mData+mDataPos;
        mDataPos += sizeof(T);
        *pArg =  *reinterpret_cast<const T*>(data);
        return NO_ERROR;
    } else {
        return NOT_ENOUGH_DATA;
    }
}

template<class T>
T Parcel::readAligned() const {
    T result;
    if (readAligned(&result) != NO_ERROR) {
        result = 0;
    }

    return result;
}

template<class T>
status_t Parcel::writeAligned(T val) {
    COMPILE_TIME_ASSERT_FUNCTION_SCOPE(PAD_SIZE(sizeof(T)) == sizeof(T));

    if ((mDataPos+sizeof(val)) <= mDataCapacity) {
restart_write:
        *reinterpret_cast<T*>(mData+mDataPos) = val;
        return finishWrite(sizeof(val));
    }

    status_t err = growData(sizeof(val));
    if (err == NO_ERROR) goto restart_write;
    return err;
}

status_t Parcel::readInt32(int32_t *pArg) const
{
    return readAligned(pArg);
}

int32_t Parcel::readInt32() const
{
    return readAligned<int32_t>();
}

status_t Parcel::readInt64(int64_t *pArg) const
{
    return readAligned(pArg);
}


int64_t Parcel::readInt64() const
{
    return readAligned<int64_t>();
}

status_t Parcel::readFloat(float *pArg) const
{
    return readAligned(pArg);
}


float Parcel::readFloat() const
{
    return readAligned<float>();
}

status_t Parcel::readDouble(double *pArg) const
{
    return readAligned(pArg);
}

double Parcel::readDouble() const
{
    return readAligned<double>();
}

status_t Parcel::readIntPtr(intptr_t *pArg) const
{
    return readAligned(pArg);
}

intptr_t Parcel::readIntPtr() const
{
    return readAligned<intptr_t>();
}

const char* Parcel::readCString() const
{
    const size_t avail = mDataSize-mDataPos;
    if (avail > 0) {
        const char* str = reinterpret_cast<const char*>(mData+mDataPos);
        // is the string's trailing NUL within the parcel's valid bounds?
        const char* eos = reinterpret_cast<const char*>(memchr(str, 0, avail));
        if (eos) {
            const size_t len = eos - str;
            mDataPos += PAD_SIZE(len+1);
            ACLOGV(LOG_TAG,"Parcel::%s: Setting data pos of %p to %d",__func__, this, mDataPos);
            return str;
        }
    }
    return NULL;
}


size_t Parcel::readString16(char16_t* buffer,size_t bufferSize) const
{
    int32_t size = readInt32();
    // watch for potential int overflow from size+1
    if (size >= 0 && bufferSize >= (size+1)*sizeof(char16_t)) {
        //make a raw copy of the string16 to the provided buffer
        const uint8_t *data = this->data();
        memcpy(buffer,data,(size+1)*sizeof(char16_t));
        *(buffer+ (size+1)*sizeof(char16_t))=0;
        const char16_t* str = (const char16_t*)readInplace((size+1)*sizeof(char16_t));
        //ALOGD(LOG_TAG,"Parcel::readString16: done, new pos = %d",dataPosition());
        return (size+1)*sizeof(char16_t);
    }
    //ALOGD(LOG_TAG,"Parcel::readString16: done, new pos = %d", dataPosition());

    return 0;
}

const char16_t* Parcel::readString16Inplace(size_t* outLen) const
{
    int32_t size = readInt32();
    //ALOGD(LOG_TAG,"Parcel::readString16Inplace: len = %d at %d",size, dataPosition());
    // watch for potential int overflow from size+1
    if (size >= 0 && size < INT32_MAX) {
        *outLen = size;
        const char16_t* str = (const char16_t*)readInplace((size+1)*sizeof(char16_t));
        if (str != NULL) {
            return str;
        }
    }
    *outLen = 0;
    return NULL;
}

void Parcel::freeData()
{
    freeDataNoInit();
    initState();
}

void Parcel::freeDataNoInit()
{
    if (mData) {
        //ALOGD(LOG_TAG,"Parcel[%p]::freeDataNoInit: free 0x%x", mData);
        FREE(mData);
        mData=NULL;
    }
}

status_t Parcel::growData(size_t len)
{
    size_t newSize = ((mDataSize+len)*3)/2;
    return (newSize <= mDataSize)
            ? (status_t) NO_MEMORY
            : continueWrite(newSize);
}

status_t Parcel::restartWrite(size_t desired)
{

    uint8_t* data = (uint8_t*)REALLOC(mData, desired);
    //ALOGD(LOG_TAG,"Parcel::restartWrite: realloc[%d] to 0x%x", desired, data );
    if (!data && desired > mDataCapacity) {
        ACLOGD(LOG_TAG,"Parcel::%s: realloc failed", __func__);
        mError = NO_MEMORY;
        return NO_MEMORY;
    }

    if (data) {
        mData = data;
        mDataCapacity = desired;
    }

    mDataSize = mDataPos = 0;
    ACLOGV(LOG_TAG,"Parcel::%s: Setting data size of %p to %d", __func__, this, mDataSize);
    ACLOGV(LOG_TAG,"Parcel::%S: Setting data pos of %p to %d",  __func__, this, mDataPos);

    return NO_ERROR;
}

status_t Parcel::continueWrite(size_t desired)
{
    if (mData) {

        // We own the data, so we can just do a realloc().
        if (desired > mDataCapacity) {
            uint8_t* data = (uint8_t*)REALLOC(mData, desired);
            //ALOGD(LOG_TAG,"Parcel[%p]::continueWrite: realloc[%d] to 0x%x", desired, data);
            if (data) {
                mData = data;
                mDataCapacity = desired;
            } else if (desired > mDataCapacity) {
                ACLOGD(LOG_TAG,"Parcel::%s: realloc failed", __func__);
                mError = NO_MEMORY;
                return NO_MEMORY;
            }
        } else {
            if (mDataSize > desired) {
                mDataSize = desired;
                ACLOGV(LOG_TAG,"Parcel::%s: Setting data size of %p to %d", __func__, this, mDataSize);
            }
            if (mDataPos > desired) {
                mDataPos = desired;
                ACLOGV(LOG_TAG,"Parcel::%s: Setting data pos of %p to %d", __func__, this, mDataPos);
            }
        }

    } else {
        // This is the first data.  Easy!
        uint8_t* data = (uint8_t*)MALLOC(desired);
        //ALOGD(LOG_TAG,"Parcel::continue realloc[%d] to 0x%x", desired, data);
        if (!data) {
            mError = NO_MEMORY;
            return NO_MEMORY;
        }

        if(!(mDataCapacity == 0 )) {
            ACLOGE(LOG_TAG,"Parcel::%s:: %d/%d", __func__, mDataCapacity, desired);
        }

        mData = data;
        mDataSize = mDataPos = 0;
        ACLOGV(LOG_TAG,"Parcel::%s: Setting data size of %p to %d", __func__, this, mDataSize);
        ACLOGV(LOG_TAG,"Parcel::%s: Setting data pos of %p to %d", __func__, this, mDataPos);
        mDataCapacity = desired;
    }

    return NO_ERROR;
}

void Parcel::initState()
{
    mError = NO_ERROR;
    mData = NULL;
    mDataSize = 0;
    mDataCapacity = 0;
    mDataPos = 0;
    ACLOGV(LOG_TAG,"Parcel::%s: Setting data size of %p to %d",__func__, this, mDataSize);
    ACLOGV(LOG_TAG,"Parcel::%s: Setting data pos of %p to %d", __func__, this, mDataPos);
}

char* Parcel::strdupReadString(Parcel &p)
{
    size_t stringlen;
    const char16_t *s16;

    s16 = p.readString16Inplace(&stringlen);
    return strndup16to8(s16, stringlen);
}

/**
 * Convert a UTF-16 string to UTF-8.
 *
 * Make sure you allocate "dest" with the result of strblen16to8(),
 * not just "strlen16()".
 */
char * Parcel::strndup16to8 (const char16_t* s, size_t n)
{
    char *ret;

    if (s == NULL) {
        return NULL;
    }

    ret = (char*)MALLOC(strnlen16to8(s, n) + 1);

    strncpy16to8 (ret, s, n);

    return ret;
}

void Parcel::writeStringToParcel(const char *s)
{
    char16_t *s16;
    size_t s16_len=0;
    s16 = strdup8to16(s, &s16_len);

    ACLOGV(LOG_TAG,"Parcel::%s: length of s16 string = %d",__func__, s16_len);
    writeString16(s16, s16_len);
    ACLOGV(LOG_TAG,"Parcel::%s: new position = %d",__func__, p.dataPosition());

    if (s16 !=NULL) {
        FREE(s16);
    }
}

/**
 * out_len is an out parameter (which may not be null) containing the
 * length of the UTF-16 string (which may contain embedded \0's)
 */

char16_t * Parcel::strdup8to16 (const char* s, size_t *out_len)
{
    char16_t *ret;
    size_t len;

    if (s == NULL) return NULL;

    len = strlen8to16(s);

    // no plus-one here. UTF-16 strings are not null terminated
    ret = (char16_t *) MALLOC (sizeof(char16_t) * len);

    return strcpy8to16 (ret, s, out_len);
}

/**
 * Like "strlen", but for strings encoded with Java's modified UTF-8.
 *
 * The value returned is the number of UTF-16 characters required
 * to represent this string.
 */
size_t Parcel::strlen8to16 (const char* utf8Str)
{
    size_t len = 0;
    int ic;
    int expected = 0;

    while ((ic = *utf8Str++) != '\0') {
        /* bytes that start 0? or 11 are lead bytes and count as characters.*/
        /* bytes that start 10 are extention bytes and are not counted */

        if ((ic & 0xc0) == 0x80) {
            /* count the 0x80 extention bytes. if we have more than
             * expected, then start counting them because strcpy8to16
             * will insert UTF16_REPLACEMENT_CHAR's
             */
            expected--;
            if (expected < 0) {
                len++;
            }
        } else {
            len++;
            expected = UTF8_SEQ_LENGTH(ic) - 1;

            /* this will result in a surrogate pair */
            if (expected == 3) {
                len++;
            }
        }
    }
    return len;
}

/**
 * out_len is an out parameter (which may not be null) containing the
 * length of the UTF-16 string (which may contain embedded \0's)
 */

char16_t * Parcel::strcpy8to16 (char16_t *utf16Str, const char*utf8Str,
                                       size_t *out_len)
{
    char16_t *dest = utf16Str;

    while (*utf8Str != '\0') {
        uint32_t ret;

        ret = getUtf32FromUtf8(&utf8Str);

        if (ret <= 0xffff) {
            *dest++ = (char16_t) ret;
        } else if (ret <= UNICODE_UPPER_LIMIT)  {
            /* Create surrogate pairs */
            /* See http://en.wikipedia.org/wiki/UTF-16/UCS-2#Method_for_code_points_in_Plane_1.2C_Plane_2 */

            *dest++ = 0xd800 | ((ret - 0x10000) >> 10);
            *dest++ = 0xdc00 | ((ret - 0x10000) &  0x3ff);
        } else {
            *dest++ = UTF16_REPLACEMENT_CHAR;
        }
    }

    *out_len = dest - utf16Str;

    return utf16Str;
}

/**
 * Given a UTF-16 string, compute the length of the corresponding UTF-8
 * string in bytes.
 */
size_t Parcel::strnlen16to8(const char16_t* utf16Str, size_t len)
{
   size_t utf8Len = 0;

   while (len--) {
       unsigned int uic = *utf16Str++;

       if (uic > 0x07ff)
           utf8Len += 3;
       else if (uic > 0x7f || uic == 0)
           utf8Len += 2;
       else
           utf8Len++;
   }
   return utf8Len;
}

/**
 * Convert a Java-Style UTF-16 string + length to a JNI-Style UTF-8 string.
 *
 * This basically means: embedded \0's in the UTF-16 string are encoded
 * as "0xc0 0x80"
 *
 * Make sure you allocate "utf8Str" with the result of strlen16to8() + 1,
 * not just "len".
 *
 * Please note, a terminated \0 is always added, so your result will always
 * be "strlen16to8() + 1" bytes long.
 */
char* Parcel::strncpy16to8(char* utf8Str, const char16_t* utf16Str, size_t len)
{
    char* utf8cur = utf8Str;

    while (len--) {
        unsigned int uic = *utf16Str++;

        if (uic > 0x07ff) {
            *utf8cur++ = (uic >> 12) | 0xe0;
            *utf8cur++ = ((uic >> 6) & 0x3f) | 0x80;
            *utf8cur++ = (uic & 0x3f) | 0x80;
        } else if (uic > 0x7f || uic == 0) {
            *utf8cur++ = (uic >> 6) | 0xc0;
            *utf8cur++ = (uic & 0x3f) | 0x80;
        } else {
            *utf8cur++ = uic;

            if (uic == 0) {
                break;
            }
        }
    }

   *utf8cur = '\0';

   return utf8Str;
}

/*
 * Retrieve the next UTF-32 character from a UTF-8 string.
 *
 * Stops at inner \0's
 *
 * Returns UTF16_REPLACEMENT_CHAR if an invalid sequence is encountered
 *
 * Advances "*pUtf8Ptr" to the start of the next character.
 */
uint32_t Parcel::getUtf32FromUtf8(const char** pUtf8Ptr)
{
    uint32_t ret;
    int seq_len;
    int i;

    /* Mask for leader byte for lengths 1, 2, 3, and 4 respectively*/
    static const char leaderMask[4] = {0xff, 0x1f, 0x0f, 0x07};

    /* Bytes that start with bits "10" are not leading characters. */
    if (((**pUtf8Ptr) & 0xc0) == 0x80) {
        (*pUtf8Ptr)++;
        return UTF16_REPLACEMENT_CHAR;
    }

    /* note we tolerate invalid leader 11111xxx here */
    seq_len = UTF8_SEQ_LENGTH(**pUtf8Ptr);

    ret = (**pUtf8Ptr) & leaderMask [seq_len - 1];

    if (**pUtf8Ptr == '\0') return ret;

    (*pUtf8Ptr)++;
    for (i = 1; i < seq_len ; i++, (*pUtf8Ptr)++) {
        if ((**pUtf8Ptr) == '\0') return UTF16_REPLACEMENT_CHAR;
        if (((**pUtf8Ptr) & 0xc0) != 0x80) return UTF16_REPLACEMENT_CHAR;

        UTF8_SHIFT_AND_MASK(ret, **pUtf8Ptr);
    }

    return ret;
}

unsigned char * Parcel::strdupReadByteArrayUnpadded(Parcel &p, int size)
{
    status_t status;

    ACLOGV(LOG_TAG,"Parcel::%s ..",__func__);
    unsigned char *dataArray = (unsigned char *)MALLOC(size);//new replaces

    while (1){
        if (dataArray==NULL){
            ACLOGW(LOG_TAG,"Parcel::%s: could not alloc dup buffer",__func__);
            break;
        }

        status=p.readUnpadded(dataArray,size);
        if (status!=NO_ERROR){
            ACLOGW(LOG_TAG,"Parcel::%s: parcel read failed",__func__);
            FREE(dataArray);
            dataArray=NULL;
        }
        break;
    }
    return dataArray;
}

unsigned char *Parcel::strdupReadByteArray(Parcel &p, int size)
{
    status_t status;
    ACLOGV(LOG_TAG,"Parcel::%s..",__func__);
    unsigned char *dataArray = (unsigned char *)MALLOC(size);//new replaces

    while (1){
        if (dataArray==NULL){
            ACLOGW(LOG_TAG,"Parcel::%s: could not alloc dup buffer",__func__);
            break;
        }

        status=p.read(dataArray,size);
        if (status!=NO_ERROR){
            ACLOGW(LOG_TAG,"Parcel::%s: parcel read failed",__func__);
            FREE(dataArray);
            dataArray=NULL;
        }
        break;
    }
    return dataArray;
}


