/*
**
** Copyright 2006, The Android Open Source Project
**
** Licensed under the Apache License, Version 2.0 (the "License");
** you may not use this file except in compliance with the License.
** You may obtain a copy of the License at
**
**     http://www.apache.org/licenses/LICENSE-2.0
**
** Unless required by applicable law or agreed to in writing, software
** distributed under the License is distributed on an "AS IS" BASIS,
** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
** See the License for the specific language governing permissions and
** limitations under the License.
**
**
*/

#include "config.h"
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <pthread.h>
#include <sys/un.h>
#include <string.h>
#include <stdint.h>
#include <netinet/in.h>
#include <sys/poll.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <arpa/inet.h>




#include "utils_misc.h"
#include "utils_log.h"
#include "ril_i.h"
#include "Parcel.h"
#include "RilProc.h"
#include "rilEvent.h"

#define LISTEN_BACKLOG 20
#define SOCKET_NAME_RIL "/var/rild"
#define ANDROID_SOCKET_NAMESPACE_FILESYSTEM 2

#define ANDROID_WAKE_LOCK_NAME "radio-interface"
#define MAX_PROCID 0xffff
#define MIN_PROCID 1

#define MAX_BUFFERID 0xffff
#define MIN_BUFFERID 1

#define EVENT_ALWAYS_ENABLED(id)    ((id==RIL_UNSOL_RESPONSE_RADIO_STATE_CHANGED)||(id==RIL_UNSOL_RIL_CONNECTED))


/* data structure to manage the process-specific data */
#define RIL_UNSOL_COMMANDS_SIZE    RIL_UNSOL_RESPONSE_TOP-RIL_UNSOL_RESPONSE_BASE+1

// match with constant in RIL.java
#define MAX_COMMAND_BYTES (8 * 1024)
#define MAX_CLIENTS 32

typedef struct sendBuffer
{
    int                         len;
    int                         count;
    unsigned char               *data;
    struct sendBuffer           *pNext;
    int                         bufferId;
}ril_sendBuffer_t;

typedef struct sendBufferListElement
{
    int                         procCount;
    int                         bufferId;
    unsigned char               *data;
    int                         len;
    struct sendBufferListElement *pNext;
}ril_sendBufferListElement_t;

typedef struct sendBufferList{
    ril_sendBufferListElement_t *pFirst;
    pthread_mutex_t              lock;
    int                          nextId;
}ril_sendBufferList_t;


typedef struct{
    int count;
    unsigned char header[4];
}ril_rcvDataHeader_t;

typedef struct{
    int count;
    int len;
    int    size;
    unsigned char *data;
}ril_rcvDataContents_t;

typedef struct ril_procData{
    ril_sendBuffer_t    *pSendList;
    ril_rcvDataHeader_t recDataHdr;
    ril_rcvDataContents_t recDataBody;
    int                 unsolResponseEnabled[RIL_UNSOL_COMMANDS_SIZE];
}ril_procData_t;

/* data structure to control the the data of all processes */
typedef struct {
    ril_procState_t     state;
    int                 fd;
    int                 procId;
    int                 invalCount;
    ril_procData_t      *pProc;
    pthread_mutex_t     lock;
}ril_procElement_t;

#define PROC_NUMBER 16
/* data structure to control the the data of all processes */
typedef struct {
    int nextId;
    ril_procElement_t   procs[PROC_NUMBER];
    pthread_mutex_t     lock;
    rilEvent_t          *pMonitor;
    struct timeval      monitorTime;
}ril_procList_t;

static ril_procList_t s_procList;

static ril_sendBufferList_t s_sendBuffers;

static rilReader_props_t s_socketProps;
namespace rilProc {
static int getPollout(ril_procList_t *pList,int *pPollOut);

static int addToProcList(ril_procList_t *pList, int fd,  ril_procData_t *procData);
static void lockProcList(const char *caller,ril_procList_t *pList);
static void unlockProcList(const char *caller,ril_procList_t *pList);

static int getNextProcId(ril_procList_t *pList, int isLocked);

// send buffer handling
static void lockBufferList(ril_sendBufferList_t *pList);
static void unlockBufferList(ril_sendBufferList_t *pList);

static void sendEventRaw (int unsolResponse, const void *data, size_t dataSize);
static int processResponseCallback(int fd);
static void releaseBuffer(int bufferId);
static int duplicateBuffer(int procId, int bufferId, int isLocked);
static int getNextBufferId(int isLocked);



static int getProcIdFromFd(ril_procList_t *pList,int fd);
static ril_procData_t *getProc(const char *caller, ril_procList_t *pList, int procId);

static void putProc(const char *caller, ril_procList_t *pList, ril_procData_t *pProc);
static void checkProcList(short flags, void *param);

static int getProcIdAt(ril_procList_t *pList, int idx);

static int createInetProc(int fd, struct sockaddr_in *peeraddr);

static int inetListenerCommand (int fd);
static int processCommandsCallback(int fd);
static void closeCommandCallback (int fd);

static struct pollfd fds[MAX_CLIENTS];
static int nfds=0;
static pthread_mutex_t fds_lock;

static ril_sendBuffer_t *takeSendBuffer(ril_procList_t *pList, int procId);
static void grantSendBuffer(ril_procList_t *pList, int procId);
static int addSendBuffer(ril_procList_t *pList, int procId, ril_sendBuffer_t *pBuffer);


//internal helper
static void clearPollOut(void);
static void setPollOut(int fdCommand);
static void addSocket(int fdCommand);
static void releaseSocket(int fdCommand);
static void invalProc(ril_procList_t *pList, int procId);

void *socketLoop(void *param);

};

void rilProc::clearPollOut(void)
{
    int i;

    ACLOGD(LOG_TAG,"rilProc::%s..",__FUNCTION__);

    pthread_mutex_lock(&fds_lock);
    for (i=0; i<MAX_CLIENTS; i++){
        fds[i].events &= ~POLLOUT;
    }
    pthread_mutex_unlock(&fds_lock);
}

void rilProc::setPollOut(int fdCommand)
{
    int i;

    ACLOGD(LOG_TAG,"rilProc::%s: socket %d ",__FUNCTION__,fdCommand);

    pthread_mutex_lock(&fds_lock);
    for (i=0; i<nfds; i++){
        if (fds[i].fd==fdCommand){
            fds[i].events |= POLLOUT;
            break;
        }
    }
    pthread_mutex_unlock(&fds_lock);
}



void rilProc::addSocket(int fdCommand)
{
    ACLOGN(LOG_TAG,"rilProc::%s: socket %d ",__FUNCTION__,fdCommand);

    pthread_mutex_lock(&fds_lock);
    fds[nfds].fd=fdCommand;
    fds[nfds].events=POLLIN;
    nfds++;
    pthread_mutex_unlock(&fds_lock);
}

static void rilProc::releaseSocket(int fdCommand)
{
    int idx,idy;

    ACLOGN(LOG_TAG,"rilProc::%s: socket %d ",__FUNCTION__,fdCommand);

    pthread_mutex_lock(&fds_lock);

    idy=-1;
    for (idx=0;idx<nfds;idx++){
        if (fds[idx].fd==fdCommand){
            idy=idx;
            break;
        }
    }
    if (idy>=0){
        for (idx=idy;idx<nfds-1;idx++){
            fds[idx]=fds[idx+1];
        }
        nfds--;
    }
    pthread_mutex_unlock(&fds_lock);

    ACLOGN(LOG_TAG,"rilProc::%s: nfds = %d ",__FUNCTION__,nfds);
}

void rilProc::invalProc(ril_procList_t *pList, int procId)
{
    do  {
        if (pList==NULL){
            break;
        }

        rilProc::lockProcList(__FUNCTION__,pList);
        for (int idx=0; idx < PROC_NUMBER; idx++){
            if ((pList->procs[idx].procId == procId)&&
                (pList->procs[idx].state == PROC_ACT)){
                pList->procs[idx].state = PROC_SHOULD_CLOSE;
                break;
            }
        }
        rilProc::unlockProcList(__FUNCTION__,pList);
    } while (0);
}

#define MAX_CLIENTS 32

/** \brief rilProc::socketLoop
 *
 *   processing loop of reader thread
 *
 *  \param[in] param thread input data
 *
 *  \return thread result data
 */
void *rilProc::socketLoop(void *param)
{
    int rc;
    int ret;
    int bufferLength=0;
    int fdInetListen=0;
    struct sockaddr_in name;
    int trials;
    int timeout;
    int doClose=0;
    int endServer=0;
    int idx;
    int procNum;
    int procId;
    int fdSendWaiting;
    long save_fd;
    int pollOutNum;

    int pollOutFds[MAX_CLIENTS];
    int closeSockets[MAX_CLIENTS];

    int fdOut[MAX_CLIENTS];
    int fdOutNumber;
    int fdIn[MAX_CLIENTS];
    int fdInNumber;

    int closeSocketNumber;
    int current_size=0, i, j;

    ACLOGN(LOG_TAG,"rilProc::%s: start reader loop..",__FUNCTION__);

    while(1) {

        ACLOGD(LOG_TAG, "rilProc::%s: inet support",__FUNCTION__);

        /* Create the socket. */
        fdInetListen = socket (PF_INET, SOCK_STREAM, 0);
        if (fdInetListen < 0){
            ACLOGE(LOG_TAG, "rilProc::%s: could not create inet socket",__FUNCTION__);
            break;
        }

        int on=1;
        ACLOGD(LOG_TAG,"rilProc::%s: setsockopt..",__FUNCTION__);
        if (setsockopt(fdInetListen, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) < 0){
           ACLOGE(LOG_TAG,"rilProc::%s: setsockopt(SO_REUSEADDR) failed",__FUNCTION__);
        }

        int keepalive=1;
        if (setsockopt(fdInetListen, SOL_SOCKET, SO_KEEPALIVE, &keepalive , sizeof(keepalive ))<0){
            ACLOGE(LOG_TAG,"rilProc::%s: setsockopt(SO_REUSEADDR) failed",__FUNCTION__);
        }

        /* Give the socket a name. */
        ACLOGD(LOG_TAG, "rilProc::%s: inet address: %s",__FUNCTION__,s_socketProps.inetName);
        name.sin_family = AF_INET;
        name.sin_port = htons (s_socketProps.inetPort);
        if (inet_aton(s_socketProps.inetName,&name.sin_addr)==0){
            ACLOGE(LOG_TAG, "rilProc::%s: could not create valid inet address",__FUNCTION__);
            close(fdInetListen);
            fdInetListen=0;
            break;
        }

        ACLOGD(LOG_TAG,"rilProc::%s: bind socket ..",__FUNCTION__);
        rc=bind(fdInetListen,(struct sockaddr *)&name,sizeof(name));
        if (rc<0){
            ACLOGE(LOG_TAG,"rilProc::%s: could not bind",__FUNCTION__);
            close(fdInetListen);
            fdInetListen=0;
            break;
        }

        /* set listen backlog */
        trials=20;
        timeout=2;
        ACLOGD(LOG_TAG,"rilProc::%s: set listen ..",__FUNCTION__);
        while (listen(fdInetListen, LISTEN_BACKLOG) != 0){
            ACLOGW(LOG_TAG, "rilProc::%s: inet listen failed; sleep..",__FUNCTION__);
            if (timeout <10) {
                timeout+=2;
            }
            sleep(timeout);
            trials--;
            if (trials<0) {
                ACLOGE(LOG_TAG, "rilProc::%s: listening on inet socket finally failed!!",__FUNCTION__);
                endServer=1; //shutdown process
                break;
            }
        }

        /* set socket to non-blocking */
        save_fd = fcntl(fdInetListen, F_GETFL );
        save_fd |= O_NONBLOCK;
        fcntl(fdInetListen, F_SETFL, save_fd );

        break;

    }//while

    /* initialize pollfd */

    if (fdInetListen>0){
        rilProc::addSocket(fdInetListen);
    } else {
        endServer=1;
    }

    /* Note: There is another timeout in ril-client/ril_client.cpp
       Historically timeouts set to 200ms, but this results in
       up to 200ms latency in delivery of messages which is an
       issue for GNSS dead-reckoning.  Experimentation appears
       to show no real increase in CPU usage if this value is
       reduced even to 20ms.  Reducing initially to 50ms so
       that if necessary some further reduction is possible
       without major rework of RIL. */
    timeout = 50; // Poll timeout in ms

    //loop forever
    while (endServer==0) {
        // initialize send polling (only if we have data to send)
        ACLOGD(LOG_TAG,"rilProc::%s: run socket loop .. ",__FUNCTION__);

        rilProc::clearPollOut();
        pollOutNum = rilProc::getPollout(&s_procList,pollOutFds);

        for (idx=0;idx<pollOutNum;idx++){
            rilProc::setPollOut(pollOutFds[idx]);
        }

        ACLOGD(LOG_TAG, "rilProc::%s: wait for %d sockets..",__FUNCTION__,nfds);
        rc=poll( fds, nfds, timeout);
        if (rc<0){
            ACLOGE(LOG_TAG,"rilProc::%s: poll failed; errno: %d", __FUNCTION__, errno);
            if (errno==EINTR){
                continue;
            }
            endServer=1;
            continue;
        }
        if (rc==0){
            ACLOGD(LOG_TAG,"rilProc::%s: timeout; do again",__FUNCTION__);
            continue;
        }

        /* check and process fired file descriptors*/
        current_size=nfds;
        for (i=0; i<current_size; i++){
            closeSockets[i]=-1;
        }

        fdInNumber=0;
        fdOutNumber=0;
        closeSocketNumber=0;

        for (i=0; i<current_size; i++){

            if (fds[i].revents==0){
                continue;
            }

            if (fds[i].revents & POLLHUP){
                if ((fdInetListen>0)&&(fds[i].fd == fdInetListen)){
                    ACLOGI(LOG_TAG,"rilProc::%s:  HUP event, ignore  .. ",__FUNCTION__);
                } else {
                    ACLOGN(LOG_TAG,"rilProc::%s:  HUP event, socket %d closed ",__FUNCTION__,fds[i].fd);
                    closeSockets[closeSocketNumber++]=fds[i].fd;
                }
                continue;
            }

            if (fds[i].revents & POLLERR){
                if ((fdInetListen>0)&&(fds[i].fd == fdInetListen)){
                    ACLOGI(LOG_TAG,"rilProc::%s:  ERR event, ignore  .. ",__FUNCTION__);
                } else {
                    ACLOGN(LOG_TAG,"rilProc::%s:  ERR event, socket %d closed ",__FUNCTION__,fds[i].fd);
                    closeSockets[closeSocketNumber++]=fds[i].fd;
                }
                continue;
            }

            if (fds[i].revents & POLLIN){
                fdIn[fdInNumber++]=fds[i].fd;
            }

            if (fds[i].revents & POLLOUT){
                fdOut[fdOutNumber++]=fds[i].fd;
            }
        }

        endServer=0;

        if (fdInNumber>0){
            ACLOGD(LOG_TAG,"%s: input from %d clients ",__FUNCTION__,fdInNumber);
        }

        for (i=0; i<fdInNumber; i++){
            // listener socket
            if ((fdInetListen>0)&&(fdIn[i] == fdInetListen)){
                 ACLOGI(LOG_TAG,"rilProc::%s[%d]: listener command .. ",__FUNCTION__,fdIn[i]);
                 endServer= rilProc::inetListenerCommand(fdInetListen);
                 ACLOGD(LOG_TAG,"rilProc::%s: listener command done; endServer = %d ",
                            __FUNCTION__,endServer);
            } else {

                //must be a standard socket: receive data from socket
                ACLOGI(LOG_TAG,"rilProc::%s[%d]: receive data from socket .. ",__FUNCTION__,fdIn[i]);
                doClose=rilProc::processCommandsCallback(fdIn[i]);
                if (doClose){
                    ACLOGN(LOG_TAG,"rilProc::%s: close socket %d",__FUNCTION__,fdIn[i]);
                    closeSockets[closeSocketNumber++]=fdIn[i];
                }
            }
        }

        if (fdOutNumber>0){
            ACLOGD(LOG_TAG,"rilProc::%s: output to %d clients ",__FUNCTION__,fdOutNumber);
        }

        for (i=0; i<fdOutNumber; i++){
             ACLOGI(LOG_TAG,"rilProc::%s[%d]: exec response callback  .. ",__FUNCTION__,fdOut[i]);
             doClose=rilProc::processResponseCallback(fdOut[i]);
             if (doClose){
                 ACLOGN(LOG_TAG,"rilProc::%s: close socket %d",__FUNCTION__,fdOut[i]);
                 closeSockets[closeSocketNumber]= fdOut[i];
            }
        }

        for (i=0; i<closeSocketNumber; i++){
            if (closeSockets[i] > 0){
                ACLOGN(LOG_TAG,"rilProc::%s: now really closing socket %d .. ",__FUNCTION__,closeSockets[i]);
                rilProc::closeCommandCallback(closeSockets[i]);
            }
        }

        ACLOGD(LOG_TAG,"rilProc::%s: check and process fired file descriptors done .. ",__FUNCTION__);

    } //while

    rilProc::checkProcList(0,0);
    return param;
}


int rilProc::getPollout(ril_procList_t *pList, int *pPollOut)
{
    int retNum=0;

    do {
        if (pList==NULL){
            break;
        }
        rilProc::lockProcList(__FUNCTION__,pList);
        for (int idx=0;idx<PROC_NUMBER;idx++){
            if (pList->procs[idx].procId<=0 ){
                continue;

            }
            if (pList->procs[idx].state != PROC_ACT){
                continue;
            }

            if ((pList->procs[idx].pProc!=NULL)&&
                (pList->procs[idx].pProc->pSendList!=NULL)){
                pPollOut[retNum] = pList->procs[idx].fd;
                retNum++;
            }
        }
        rilProc::unlockProcList(__FUNCTION__,pList);
    } while (0);

    return retNum;

}

void rilProc::closeCommandCallback (int fd)
{
    int procId;
    int idx,idy;
    ACLOGD(LOG_TAG,"rilProc::%s for fd %d",__FUNCTION__,fd);

    do{
        procId=  rilProc::getProcIdFromFd(&s_procList, fd);
        if (procId<=0){
            ACLOGN(LOG_TAG,"rilProc::%s[%d]: could not find procId",__FUNCTION__,fd);
            break;
        }
        rilProc::invalProc(&s_procList,procId);
        ACLOGI(LOG_TAG,"rilProc::%s: libril_onProcClosed(data)..",__FUNCTION__);
        libril_onProcClosed(procId);

    } while(0);

    rilProc::releaseSocket(fd);

}

int rilProc::inetListenerCommand (int fd)
{
//    int ret;
    struct sockaddr_in peeraddr;
    int fdCommand=0;
    int idx;
    int socklen=sizeof(peeraddr);
    int endServer=0;
    int procId;
    long save_fd;

    ACLOGD(LOG_TAG, "rilProc::%s for socket %d",__FUNCTION__,fd);

    do{
        fdCommand = accept(fd, (struct sockaddr *) &peeraddr, (socklen_t *)&socklen);
        if (fdCommand < 0 ) {
            ACLOGE(LOG_TAG,"rilProc::%s: Error on accept() errno:%d",__FUNCTION__, errno);
              if ((errno != EWOULDBLOCK) && (errno != EAGAIN)){
                 endServer=1;
            }
            break;
        }

        if (fdCommand == 0 ) {
            break;
        }

        /* set socket to non-blocking */
        save_fd = fcntl(fdCommand, F_GETFL );
        save_fd |= O_NONBLOCK;
        fcntl( fdCommand, F_SETFL, save_fd );

        procId = rilProc::createInetProc(fdCommand,&peeraddr);
        if (procId<=0){
            ACLOGE(LOG_TAG,"rilProc::%s: invalid procId, close socket : %d",__FUNCTION__, fdCommand);
            close(fdCommand);
            fdCommand=0;
            continue;
        }

        ACLOGN(LOG_TAG,"rilProc::%s: add  %d to fds ",__FUNCTION__, fdCommand);
        rilProc::addSocket(fdCommand);

        ACLOGN(LOG_TAG,"rilProc::%s: libril_onProcConnect for %d ",__FUNCTION__, procId);
        libril_onProcConnect(procId);
        ACLOGN(LOG_TAG,"rilProc::%s: libril_onProcConnect done",__FUNCTION__);
        break;

    } while(fdCommand > 0 );

    return endServer;
}

/** \brief rilProc::getProcIdAt
 *
 *   retrieve proc id of object with index idx within the default proc list
 *
 *  \param[in] pList list object
 *  \param[in] idx  list index
 *
 *  \return proc identifier
 *
 */
int rilProc::getProcIdAt(ril_procList_t *pList, int idx)
{
    int id=0;
    ril_procData_t *pCurrent=NULL;

    ACLOGI(LOG_TAG,"rilProc::%s: at %d",__FUNCTION__,idx);

    do {
        if (pList == NULL){
            break;
        }
        rilProc::lockProcList(__FUNCTION__,pList);
        if (pList->procs[idx].state == PROC_ACT){
            id=pList->procs[idx].procId;
        }
        rilProc::unlockProcList(__FUNCTION__,pList);

    } while(0);
    return id;
}

/** \brief rilProc::lockProcList
 *
 * lock proc list access
 *
 * \param[in] caller calling entity
 * \param[in] pList list object
 *
 */
void rilProc::lockProcList(const char * caller,ril_procList_t *pList)
{
    //ALOGD(LOG_TAG,"%s called by %s..",__FUNCTION__,caller);
    pthread_mutex_lock(&pList->lock);
};

/** \brief rilProc::unlockProcList
 *
 * unlock proc list access
 *
 * \param[in] caller calling entity
 * \param[in] pList list object
 *
 */
void rilProc::unlockProcList(const char *caller,ril_procList_t *pList)
{
    //ALOGD(LOG_TAG,"%s called by %s..",__FUNCTION__,caller);
    pthread_mutex_unlock(&pList->lock);
};

/** \brief rilProc::lockBufferList
 *
 * lock buffers list access
 *
 * \param[in] pList list object
 *
 */
void rilProc::lockBufferList(ril_sendBufferList_t *pList)
{
    //ALOGD(LOG_TAG,"%s",__FUNCTION__);
    pthread_mutex_lock(&pList->lock);
};

/** \brief rilProc::unlockBufferList
 *
 * unlock buffers list access
 *
 * \param[in] pList list object
 *
 */
void rilProc::unlockBufferList(ril_sendBufferList_t *pList)
{
    //ALOGD(LOG_TAG,"%s",__FUNCTION__);
    pthread_mutex_unlock(&pList->lock);
};

#define INVAL_LIMIT 3

/** \brief rilProc::checkProcList
 *
 * tests whether proc elements are stil valid
 *
 * \param[in] flags (dummy, related to ril_event defs not used here)
 * \param[in] param  user-defined param (dummy, related to ril_event defs not used here)
 *
 */
void rilProc::checkProcList(short flags, void *param)
{
    int len;
    int bufferId;
    ril_sendBuffer_t *pCurrent;
    ril_sendBuffer_t *pNext=NULL;
    int toCheck;
    int idx;


    ACLOGD(LOG_TAG,"rilProc::%s..",__FUNCTION__ );

    do {
        if (param==NULL){
            break;
        }
        ril_procList_t *pList=(ril_procList_t *)param;
        toCheck=0;

        do {
            bufferId=0; //holds the id of the send buffer to be cleared
            rilProc::lockProcList(__FUNCTION__,pList);   //get exclusive acces to proc list

            for (idx = toCheck; idx <PROC_NUMBER; idx++){

                ACLOGD(LOG_TAG,"rilProc::%s[%d] ..",__FUNCTION__, idx );
                // go through the complete list

                if (pList->procs[idx].state == PROC_SHOULD_CLOSE){ // first step: close socket if closing is requested
                    ACLOGI(LOG_TAG,"rilProc::%s[%d]: close socket..",__FUNCTION__ ,idx);
                    close(pList->procs[idx].fd);
                    pList->procs[idx].state = PROC_CLOSED;
                }

                if (pList->procs[idx].state != PROC_CLOSED){ // if not closed, no further action is required
                    ACLOGD(LOG_TAG,"rilProc::%s[%d]: entry ignored",__FUNCTION__ ,idx);
                    toCheck=idx+1; //make sure that you do restart checking with next index
                    continue;
                }

                if (pList->procs[idx].invalCount < INVAL_LIMIT){
                    ACLOGI(LOG_TAG,"rilProc::%s[%d]: wait for inval limit",__FUNCTION__ ,idx);
                    pList->procs[idx].invalCount++;  // delay cleanup of proc
                    toCheck=idx+1; //make sure that you do restart checking with next index
                    continue;
                }

                if (pthread_mutex_trylock(&pList->procs[idx].lock)!=0){ // try to get exclusive access to proc
                    ACLOGI(LOG_TAG,"rilProc::%s[%d]: could not access; retry later",__FUNCTION__ ,idx);
                    toCheck=idx+1; //make sure that you do restart checking with next index
                    continue;
                }

                if (pList->procs[idx].pProc==NULL){ // now we have exclusive access: cleanup receive resources
                    ACLOGD(LOG_TAG,"rilProc::%s[%d]: no proc data, continue",__FUNCTION__ ,idx);
                    toCheck=idx+1; //make sure that you do restart checking with next index
                    continue;
                }

                ACLOGI(LOG_TAG,"rilProc::%s[%d]: cleanup receive buffer..",__FUNCTION__ ,idx)
                if (pList->procs[idx].pProc->recDataBody.data!=NULL){
                    FREE(pList->procs[idx].pProc->recDataBody.data);
                    pList->procs[idx].pProc->recDataBody.data=NULL;
                }

                pCurrent=pList->procs[idx].pProc->pSendList;  // evaluate/detach first send buffer (if any)
                if (pCurrent!=NULL){
                    pNext=pCurrent->pNext;
                    bufferId=pCurrent->bufferId;
                    ACLOGI(LOG_TAG,"rilProc::%s[%d]: prepare cleanup of send buffer %d",__FUNCTION__ ,idx, bufferId);
                    FREE(pCurrent);
                    pList->procs[idx].pProc->pSendList=pNext;
                }

                if (bufferId <= 0 ){ //send list is empty; now i can remove the proc from system
                    libril_onProcClosed(pList->procs[idx].procId);
                    pList->procs[idx].procId=0;
                    pList->procs[idx].state=PROC_INIT;

                    FREE(pList->procs[idx].pProc);
                    pList->procs[idx].pProc=NULL;

                    ACLOGI(LOG_TAG,"rilProc::%s[%d]: cleanup complete",__FUNCTION__ ,idx);
                    toCheck=idx+1;    //make sure that you do restart checking with next index
                }

                pthread_mutex_unlock(&pList->procs[idx].lock);

                if (bufferId > 0){ // we have to clear a send buffer; this has to be done outside of proc mutexes
                    break;
                }
            } //for
            rilProc::unlockProcList(__FUNCTION__,pList);

            if (bufferId>0){
                ACLOGI(LOG_TAG,"rilProc::%s: do cleanup send buffer %d",__func__ , bufferId);
                rilProc::releaseBuffer(bufferId);
            }

        } while (bufferId>0);
    } while (0);

}

/** \brief rilProc::addToProcList
 *
 * add specified entry to the list of active processes
 *
 * \param[in] pList processing list
 * \param[in] fd file descriptor
 * \param[in] procData proc object to be added to list *
 *
 * \return result of operation
 */
int rilProc::addToProcList(ril_procList_t *pList, int fd,  ril_procData_t *procData)
{
    int result=0;

    ACLOGD(LOG_TAG,"rilProc::%s[%d] .. ",__FUNCTION__,fd);
    rilProc::lockProcList(__FUNCTION__,pList);

    for (int idx=0;idx <PROC_NUMBER ; idx++){
        if ((pList->procs[idx].procId<=0)&&
            (pList->procs[idx].pProc==NULL)&&
            (pList->procs[idx].state==PROC_INIT)){

            pList->procs[idx].fd=fd;
            result=rilProc::getNextProcId(pList,1);
            pList->procs[idx].procId=result;
            pList->procs[idx].invalCount=0;
            pList->procs[idx].pProc=procData;
            pList->procs[idx].state=PROC_ACT;
            ACLOGN(LOG_TAG,"rilProc::%s[%d]: proc %d added ",__FUNCTION__,fd, result);
            break;
        }
    }

    rilProc::unlockProcList(__FUNCTION__,pList);
    return result;
}

/** \brief rilProc::init
 *
 * initialize rilProc entities
 *
 *  \param[in] inetAddr IP address
 *  \param[in] inetPort TCP port
 *
 *  \return success of operation
 *
 */
int rilProc::init(const char *inetAddr, int inetPort)
{
    //loglevel=LOG_INFO;

    pthread_t tid;
    int rc;
    pthread_attr_t attr;

    nfds=0;
    memset (fds, 0, sizeof(fds));
    pthread_mutex_init(&fds_lock,NULL);

    pthread_mutex_init(&s_sendBuffers.lock,NULL);
    ACLOGD(LOG_TAG,"%s..",__FUNCTION__);
    s_sendBuffers.pFirst=NULL;
    s_sendBuffers.nextId=MIN_BUFFERID;

    pthread_mutex_init(&s_procList.lock,NULL);
    s_procList.nextId=MIN_PROCID;
    ACLOGD(LOG_TAG,"rilProc::%s..",__FUNCTION__);

    for (int idx=0;idx < PROC_NUMBER; idx++){
        s_procList.procs[idx].fd=0;
        s_procList.procs[idx].procId=0;
        s_procList.procs[idx].invalCount=0;
        s_procList.procs[idx].pProc=NULL;
        s_procList.procs[idx].state=PROC_INIT;
        pthread_mutex_init(&s_procList.procs[idx].lock,NULL);
    }

    /* add timer monitor*/
    s_procList.monitorTime.tv_sec=5;
    s_procList.monitorTime.tv_usec=0;

    pthread_attr_init (&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);

    //create reader thread
    s_socketProps.inetName=inetAddr;
    s_socketProps.inetPort=inetPort;

    rc=pthread_create(&tid, &attr, rilProc::socketLoop, NULL);

    if (rc==0) {
        s_procList.pMonitor= rilEvent_create();
        rilEvent_set(s_procList.pMonitor, true, rilProc::checkProcList, &s_procList);
        rilEvent_timerAdd(s_procList.pMonitor,&s_procList.monitorTime);

#ifdef MEM_CTL
        struct timeval      memCheckTime;
        memCheckTime.tv_sec=30;
        memCheckTime.tv_usec=0;

        rilEvent_t *pMemCheck=rilEvent_create();
        rilEvent_set(pMemCheck, true, libril_checkMem, NULL);
        rilEvent_timerAdd(pMemCheck,&memCheckTime);
#endif
    }

    return rc;
}

/** \brief rilProc::setState
 *
 *   set process state
 *
 *  \param[in] procId process identifier
 *  \param[in] state state to be set
 */
void rilProc::setState(int procId, ril_procState_t state)
{

    rilProc::lockProcList(__FUNCTION__,&s_procList);

    for (int idx=0;idx< PROC_NUMBER;idx++){
        if (s_procList.procs[idx].procId == procId){
            s_procList.procs[idx].state=state;
            break;
        }
    }

    rilProc::unlockProcList(__FUNCTION__,&s_procList);
}

/** \brief rilProc::getCommandFd
 *
 *   retrieve process specific socket file descripto
 *
 *  \param[in] procId process identifier
 *
 *  \return fieldescripto or 0(in error case)
 */

int rilProc::getCommandFd(int procId)
{
    int result=0;

    rilProc::lockProcList(__FUNCTION__,&s_procList);
    for (int idx=0;idx< PROC_NUMBER;idx++){
        if ((s_procList.procs[idx].procId==procId) &&
            (s_procList.procs[idx].state==PROC_ACT)){
            result=s_procList.procs[idx].fd;
            break;
        }
    }
    rilProc::unlockProcList(__FUNCTION__,&s_procList);

    return result;
}

/** \brief rilProc::sendResponseRaw
 *
 *   send data to appliaction process
 *
 *  \param[in] procId process identifier
 *  \param[in] data data buffer
 *  \param[in] dataSize  size of data buffer
 *  \param[in] isLocked  indicates if caller has locked the list access
 *
 *  \return result of send process
 */
int rilProc::sendResponseRaw (int procId, const void *data, size_t dataSize, int isLocked)
{
    int ret=-1;
    ril_sendBuffer_t *pCurrent;
    ril_sendBuffer_t *pBuffer=NULL;
    ril_sendBufferListElement_t *pNewListElement=NULL;
    ril_sendBufferListElement_t *pCurrListElement;

    ril_procData_t *proc;
    unsigned char *dataBuffer=NULL;
    uint32_t header;
    int listLen;
    int isAdded;

    ACLOGI(LOG_TAG, "rilProc::%s:  for proc %d",__FUNCTION__,procId);

    if (isLocked==0){
        rilProc::lockBufferList(&s_sendBuffers);
    }

    do{
        pBuffer= (ril_sendBuffer_t *)MALLOC(sizeof(ril_sendBuffer_t));
        if (pBuffer==0){
            ACLOGE(LOG_TAG,"rilProc::%s[%d]: could not create buffer object",__FUNCTION__,procId);
            break;
        }

        pNewListElement= (ril_sendBufferListElement_t *)MALLOC(sizeof(ril_sendBufferListElement_t));
        if (pNewListElement==0){
            ACLOGE(LOG_TAG,"rilProc::%s[%d]: could not create buffer listelement object",__FUNCTION__,procId);
            FREE(pBuffer);
            break;
        }

        dataBuffer =(unsigned char *)MALLOC(dataSize+sizeof(header));
        if (dataBuffer==0){
            ACLOGE(LOG_TAG,"rilProc::%s[%d]: could not create data buffer",__FUNCTION__,procId);
            FREE(pBuffer);
            FREE(pNewListElement);
            break;
        }

        header = htonl(dataSize);
        memcpy(dataBuffer,&header,sizeof(header));
        memcpy(dataBuffer+sizeof(header),data,dataSize);

        pBuffer->data=dataBuffer;
        pBuffer->len=dataSize+sizeof(header);
        pBuffer->count=0;
        pBuffer->bufferId=rilProc::getNextBufferId(1);
        pBuffer->pNext=NULL;

        /* add buffer to process send list */
        ACLOGI(LOG_TAG, "rilProc::%s[%d]:  add buffer %d  with %d byte contents to process send list..",__FUNCTION__,procId,pBuffer->bufferId,dataSize);
        isAdded= rilProc::addSendBuffer(&s_procList, procId,pBuffer);

        // add related list element to global buffer list */

        pNewListElement->pNext=NULL;
        pNewListElement->bufferId=pBuffer->bufferId;
        pNewListElement->procCount=isAdded;
        pNewListElement->data=pBuffer->data;
        pNewListElement->len=pBuffer->len;
        ret=pBuffer->bufferId;

        ACLOGD(LOG_TAG, "rilProc::%s[%d]:  add list element for buffer %d to global buffer list..",__FUNCTION__,procId,pBuffer->bufferId);
        pCurrListElement=s_sendBuffers.pFirst;
        if (pCurrListElement == NULL){
            s_sendBuffers.pFirst=pNewListElement;
            ACLOGD(LOG_TAG, "rilProc::%s[%d]:  set sendBuffers head to %d",__FUNCTION__,procId,pNewListElement->bufferId);
        } else {
            listLen=1;
            while(pCurrListElement->pNext!=NULL){
                pCurrListElement=pCurrListElement->pNext;
                listLen++;
            }
            pCurrListElement->pNext=pNewListElement;
            ACLOGD(LOG_TAG, "rilProc::%s[%d]:  attach %d to list end of sendBuffers(len: %d)",__FUNCTION__,procId,pNewListElement->bufferId,listLen);
        }

    } while(0);

    if (isLocked==0){
        rilProc::unlockBufferList(&s_sendBuffers);
    }

    return ret;
}

/** \brief rilProc::sendResponseRaw
 *
 *   send data to application process
 *
 *  \param[in] procId process identifier
 *  \param[in] p parcel data buffer
 *
 *  \return result of send process
 */
int rilProc::sendResponse (int procId, Parcel &p)
{
    return rilProc::sendResponseRaw(procId, p.data(), p.dataSize(),0);
}

/** \brief rilProc::processCommandsCallback
 *
 *   process command socket callback
 *
 *  \param[in] fd command socket file descriptor
 *
 *  \return result of operation
 *
 */
int rilProc::processCommandsCallback(int fd)
{
    int offset=0;
    int remaining=0;
    unsigned char dummy[4];
    int countRead;
    int doCloseProc=0;
    ril_procData_t *pProc=NULL;
    int procId;
    int headerRetrieved;

    ACLOGD(LOG_TAG,"rilProc::%s for fd %d",__FUNCTION__,fd);

    do {

        procId=rilProc::getProcIdFromFd(&s_procList,fd);
        if (procId <=0){
            ACLOGW(LOG_TAG,"rilProc::%s[%d]: could not find procId",__FUNCTION__,fd);
            break;
        }

        pProc=rilProc::getProc(__FUNCTION__,&s_procList,procId);
        if (pProc==NULL){
            ACLOGW(LOG_TAG,"rilProc::%s[%d]: could not find process",__FUNCTION__,fd);
            break;
        }

        ACLOGD(LOG_TAG,"rilProc::%s[%d]: header count: %d; body count: %d",
                __FUNCTION__,fd,pProc->recDataHdr.count,pProc->recDataBody.count);

        headerRetrieved=0;

        if (pProc->recDataHdr.count < sizeof(pProc->recDataHdr.header)){

            pProc->recDataBody.count=0;
            pProc->recDataBody.len=0;

            ACLOGI(LOG_TAG,"rilProc::%s[%d] proc %d has %d header data retrieved ",__FUNCTION__,fd,procId,pProc->recDataHdr.count);

            offset=pProc->recDataHdr.count;
            remaining= sizeof(pProc->recDataHdr.header) - pProc->recDataHdr.count;
            countRead = recv(fd,&(pProc->recDataHdr.header[offset]),remaining,0);
            ACLOGD(LOG_TAG,"rilProc::%s[%d] proc %d read %d header data; received: %d ",__FUNCTION__,fd, procId,remaining, countRead);

            headerRetrieved = 1;

            if (countRead<0){
                if (errno == EAGAIN){
                    ACLOGI(LOG_TAG,"rilProc::%s[%d]: EAGAIN: retry later",__FUNCTION__,fd);
                    break;
                }
                if (errno == EWOULDBLOCK){
                    ACLOGI(LOG_TAG,"rilProc::%s[%d]: EWOULDBLOCK: retry later",__FUNCTION__,fd);
                    break;
                }
                if (errno == EINTR){
                    ACLOGI(LOG_TAG,"rilProc::%s[%d]: EINTR: retry later",__FUNCTION__,fd);
                    break;
                }
                ACLOGE(LOG_TAG,"rilProc::%s[%d]: error on reading command socket errno: %d",__FUNCTION__,fd, errno);
                doCloseProc=1;
                break;
            }

            if (countRead == 0){
                ACLOGE(LOG_TAG,"rilProc::%s[%d]: socket closed by peer",__FUNCTION__,fd);
                doCloseProc=1;
                break;
            }

            pProc->recDataHdr.count += countRead;
               if (pProc->recDataHdr.count < sizeof(pProc->recDataHdr.header)){
                ACLOGI(LOG_TAG,"rilProc::%s[%d]: insufficient header data retrieved: %d",__FUNCTION__,fd, pProc->recDataHdr.count);
                break;
               }
               //convert  data length from network byte order to host byte order
               pProc->recDataBody.len=(((pProc->recDataHdr.header[0] & 0xff) << 24) |
                                    ((pProc->recDataHdr.header[1] & 0xff) << 16) |
                                    ((pProc->recDataHdr.header[2] & 0xff) << 8) |
                                     (pProc->recDataHdr.header[3] & 0xff));

               ACLOGD(LOG_TAG,"rilProc::%s[%d] proc %d has body length %d ",__FUNCTION__,fd,procId,pProc->recDataBody.len);
               pProc->recDataBody.count=0;
        }

        if (pProc->recDataBody.len == 0){
            ACLOGD(LOG_TAG,"rilProc::%s[%d]: body length not available; break",__FUNCTION__,fd);
            break;
        }

        if (pProc->recDataBody.count == 0) {
            //initialise data buffer
            if (pProc->recDataBody.len>pProc->recDataBody.size){

                // we must reallocate a bigger receive buffer: free existing buffer first
                ACLOGI(LOG_TAG,"rilProc::%s[%d] proc %d reallocate bigger body buffer ",__FUNCTION__,fd,procId);
                if (pProc->recDataBody.data!=NULL){
                    FREE(pProc->recDataBody.data);
                    pProc->recDataBody.data=NULL;
                    pProc->recDataBody.size=0;
                }

                // now allocate receive buffer; minimum size MAX_COMMAND_BYTES
                if (pProc->recDataBody.len > MAX_COMMAND_BYTES){
                    pProc->recDataBody.data=(unsigned char *)MALLOC(pProc->recDataBody.len);
                    if (pProc->recDataBody.data!=NULL){
                        pProc->recDataBody.size=pProc->recDataBody.len;
                    }
                } else {
                    pProc->recDataBody.data=(unsigned char *)MALLOC(MAX_COMMAND_BYTES);
                    if (pProc->recDataBody.data!=NULL){
                        pProc->recDataBody.size=MAX_COMMAND_BYTES;
                    }
                }
                ACLOGI(LOG_TAG,"rilProc::%s[%d] proc %d buffer with %d bytes allocated ",
                        __FUNCTION__,fd, procId,pProc->recDataBody.size);
            }
            // shrink down receive buffer size if possible
            if (pProc->recDataBody.len<MAX_COMMAND_BYTES){
                if (pProc->recDataBody.size>MAX_COMMAND_BYTES){
                    ACLOGD(LOG_TAG,"rilProc::%s[%d] proc %d reallocate smaller body buffer ",__FUNCTION__,fd,procId);
                    if (pProc->recDataBody.data!=NULL){
                        FREE(pProc->recDataBody.data);
                        pProc->recDataBody.data=NULL;
                        pProc->recDataBody.size=0;
                    }
                    pProc->recDataBody.data=(unsigned char *)MALLOC(MAX_COMMAND_BYTES);
                    if (pProc->recDataBody.data!=NULL){
                        pProc->recDataBody.size=MAX_COMMAND_BYTES;
                    }
                    ACLOGD(LOG_TAG,"rilProc::%s[%d] proc %d rec buffer with %d bytes allocated ",
                            __FUNCTION__,fd, procId,pProc->recDataBody.size);
                }
            }
        }

        if (headerRetrieved){
            ACLOGD(LOG_TAG,"rilProc::%s[%d]: have read header data; break",__FUNCTION__,fd);
            break;
        }

        if (pProc->recDataBody.count<pProc->recDataBody.len){
            ACLOGD(LOG_TAG,"rilProc::%s[%d] proc %d retrieve body data from %d, total: %d ",
                    __FUNCTION__,fd, procId,pProc->recDataBody.count,pProc->recDataBody.len);
            offset=pProc->recDataBody.count;
            if (pProc->recDataBody.len>pProc->recDataBody.size){
                //could not allocate sufficient data buffer => dummy read
                ACLOGW(LOG_TAG,"rilProc::%s[%d] proc %d has not sufficient body; dummy read..",__FUNCTION__,fd,procId);
                remaining= sizeof(dummy);
                countRead = recv(fd,dummy,remaining,MSG_DONTWAIT);
            } else {
                remaining=pProc->recDataBody.len-pProc->recDataBody.count;
                countRead = recv(fd,pProc->recDataBody.data + offset ,remaining,0);
            }

            if (countRead<0){
                if (errno == EAGAIN){
                    ACLOGW(LOG_TAG,"rilProc::%s[%d]: EAGAIN: retry later",__FUNCTION__,fd);
                    break;
                }
                if (errno == EWOULDBLOCK){
                    ACLOGW(LOG_TAG,"rilProc::%s[%d]: EWOULDBLOCK: retry later",__FUNCTION__,fd);
                    break;
                }
                if (errno == EINTR){
                    ACLOGW(LOG_TAG,"rilProc::%s[%d]: EINTR: retry later",__FUNCTION__,fd);
                    break;
                }

                ACLOGE(LOG_TAG,"rilProc::%s[%d]: error on reading command socket errno: %d",__FUNCTION__, fd, errno);
                doCloseProc=1;
                break;
            }

            if (countRead == 0){
                ACLOGE(LOG_TAG,"rilProc::%s[%d]: socket closed by peer",__FUNCTION__,fd);
                doCloseProc=1;
                break;
            }

               pProc->recDataBody.count += countRead;
               ACLOGD(LOG_TAG,"rilProc::%s[%d] proc %d now retrieved body %d bytes ",__FUNCTION__,fd,procId,pProc->recDataHdr.count);
               if (pProc->recDataBody.count>=pProc->recDataBody.len){
                   ACLOGD(LOG_TAG,"rilProc::%s[%d] proc %d body retrieved",__FUNCTION__,fd,procId);
               }
        }

        // now evaluate data if valid data are available;
        if (pProc->recDataBody.len <= pProc->recDataBody.count){

            int dataLen=pProc->recDataBody.len;
            unsigned char *procData = pProc->recDataBody.data;

            ACLOGI(LOG_TAG,"rilProc::%s[%d] proc %d process request buffer(len:%d) ",__FUNCTION__,fd, procId, dataLen);
            pProc->recDataHdr.count=0;

            rilProc::putProc(__FUNCTION__,&s_procList,pProc);
            pProc=0;

            libril_processCommandBuffer(procData, dataLen, procId);
        }

    } while (0);

    if (pProc){
        rilProc::putProc(__FUNCTION__,&s_procList,pProc);
    }
    return doCloseProc;
}


/** \brief rilProc::createInetProc
 *
 *   create proc object for inet socket access
 *
 *  \param[in] fd command socket file descriptor
 *  \param[in] peeraddr  socket address
 *  \return process id (or 0)
 *
 */
 int rilProc::createInetProc(int fd, struct sockaddr_in *peeraddr)
{
    ril_procData_t *pProc=NULL;
    int result;
    int idx;

    ACLOGI(LOG_TAG, "rilProc::%s[%d] .. ",__FUNCTION__,fd);

    do {
        pProc = (ril_procData_t*) MALLOC(sizeof(ril_procData_t));
        if (pProc==NULL){
            result=-1;
            break;
        }

        pProc->recDataHdr.count=0;
        pProc->recDataBody.count=0;
        pProc->recDataBody.len=0;
        pProc->recDataBody.size=0;
        pProc->recDataBody.data=NULL;
        pProc->pSendList=NULL;

        for (idx=0;idx<RIL_UNSOL_COMMANDS_SIZE;idx++) {
            pProc->unsolResponseEnabled[idx]=0;
        }
           pProc->unsolResponseEnabled[RIL_UNSOL_RIL_CONNECTED-RIL_UNSOL_RESPONSE_BASE]=1;

        result=rilProc::addToProcList(&s_procList, fd, pProc);
        if (result <0){
            FREE(pProc);
            ACLOGE(LOG_TAG, "rilProc::%s: could not add to proc list",__FUNCTION__);
        } else {
             ACLOGD(LOG_TAG, "rilProc::%s: proc %d added to proc list",__FUNCTION__,result);
        }


    } while(0);

    ACLOGD(LOG_TAG, "rilProc::%s: done",__FUNCTION__);
    return result;
}


/** \brief rilProc::getProc
 *
 *   get access to proc object
 *
 *  \param[in] caller identifies calling instance
 *  \param[in] pList list of processes
 *  \param[in] procId process identifier
 *
 *  \return process object (or NULL)
 */
ril_procData_t *rilProc::getProc(const char *caller, ril_procList_t *pList, int procId)
{
    ril_procData_t *pFound=NULL;
    //int loglevel=LOG_INFO;

    ACLOGI(LOG_TAG,"rilProc::%s: called by %s for proc %d",__FUNCTION__,caller, procId);

    do {
        if (pList == NULL){
            break;
        }
        if (procId <= 0){
            break;
        }

        for (int idx=0;idx < PROC_NUMBER; idx ++){
            if ((pList->procs[idx].procId == procId)&&
                (pList->procs[idx].state==PROC_ACT)){
                pthread_mutex_lock(&pList->procs[idx].lock);
                if (pList->procs[idx].pProc){
                    pFound=pList->procs[idx].pProc;
                } else {
                    pthread_mutex_unlock(&pList->procs[idx].lock);
                }
                break;
            }
        }
    } while (0);
    return pFound;
}

/** \brief rilProc::grantSendBuffer
 *
 *   release send buffer
 *
 *  \param[in] pList list of processes
 *  \param[in] procId process identifier
 *
 */
void rilProc::grantSendBuffer(ril_procList_t *pList, int procId)
{
    ril_procData_t *pProc=NULL;
    ril_sendBuffer_t *pCurrent;
    ril_sendBuffer_t *pNext=NULL;

    //int loglevel = LOG_INFO;
    int bufferId=0;

    do {
        pProc=rilProc::getProc(__FUNCTION__,pList,procId);
        if (pProc==NULL){
            break;
        }

        pCurrent=pProc->pSendList;
        if(pCurrent){
            pNext=pCurrent->pNext;
            if (pCurrent->count>=pCurrent->len){
                bufferId=pCurrent->bufferId;
                FREE(pCurrent);
                pProc->pSendList=pNext;
            }
        }
    } while (0);

    if (pProc){
        rilProc::putProc(__FUNCTION__,pList,pProc);
    }

    if (bufferId>0){
        rilProc::releaseBuffer(bufferId);
    }
}

/** \brief rilProc::addSendBuffer
 *
 *   add buffer to  send buffer list
 *
 *  \param[in] pList list of processes
 *  \param[in] procId process identifier
 *  \param[in] pBuffer buffer pointer
 *
 *  \return success of operation
 */
int rilProc::addSendBuffer(ril_procList_t *pList, int procId, ril_sendBuffer_t *pBuffer)
{
    int result=0;
    ril_procData_t *pProc=NULL;
    ril_sendBuffer_t *pCurrent;

    //int loglevel = LOG_INFO;

    do{
        pProc=rilProc::getProc(__FUNCTION__,pList,procId);
        if (pProc==NULL){
            break;
        }

        if (pBuffer==NULL){
            break;
        }

        result=1;
        pBuffer->pNext=NULL;

        if (pProc->pSendList==NULL){
            pProc->pSendList=pBuffer;
            break;
        }

        pCurrent=pProc->pSendList;
        while (pCurrent->pNext){
            pCurrent = pCurrent->pNext;
        }
        pCurrent->pNext=pBuffer;
    } while (0);

    if (pProc){
        rilProc::putProc(__FUNCTION__,pList,pProc);
    }

    return result;
}

/** \brief rilProc::takeSendBuffer
 *
 *   take  buffer from  send buffer list
 *
 *  \param[in] pList list of processes
 *  \param[in] procId process identifier
 *
 *  \return buffer pointer (or NULL)
 */

ril_sendBuffer_t *rilProc::takeSendBuffer(ril_procList_t *pList, int procId)
{

    ril_sendBuffer_t *pSendBuffer=NULL;
    ril_sendBuffer_t *pCurrent;
    ril_sendBuffer_t *pNext=NULL;

    ril_procData_t *pProc=NULL;
    //int loglevel=LOG_INFO;
    int bufferId=0;

    do {
        pProc=rilProc::getProc(__FUNCTION__,pList,procId);
        if (pProc==NULL){
            break;
        }

        bufferId=0;
        pCurrent=pProc->pSendList;

        if (pCurrent){
            pNext=pCurrent->pNext;
            if (pCurrent->count>=pCurrent->len){
                bufferId=pCurrent->bufferId;
                FREE(pCurrent);
                pProc->pSendList=pNext;
            } else {
                pSendBuffer=pCurrent;
            }
        } else {
            break;
        }

        rilProc::putProc(__FUNCTION__,pList,pProc);
        pProc=NULL;

        if (bufferId>0){
            rilProc::releaseBuffer(bufferId);
        }

    } while (0);

    if (pProc){
        rilProc::putProc(__FUNCTION__,pList,pProc);
    }
    return pSendBuffer;

}

/** \brief rilProc::getProcIdFromFd
 *
 *   get proc identifier related to file descriptor of communication socket
 *
 *  \param[in] pList list of processes
 *  \param[in] fd file (socket) descriptor
 *
 *  \return process object (or NULL)
 *
 */
int rilProc::getProcIdFromFd(ril_procList_t *pList, int fd)
{
    int result=0;
    ACLOGI(LOG_TAG,"%s: for fd %d",__FUNCTION__,fd);
    do {
        if (pList == NULL){
            break;
        }
        if (fd <= 0){
            break;
        }

        for (int idx=0;idx < PROC_NUMBER; idx ++){
            if ((pList->procs[idx].fd == fd) &&
                (pList->procs[idx].procId > 0) &&
                (pList->procs[idx].state==PROC_ACT)){
                result=pList->procs[idx].procId;
                break;
            }
        }
    } while (0);
    return result;
}

/** \brief rilProc::putProc
 *
 *   release access to proc object
 *
 *  \param[in] caller calling instance
 *  \param[in] pList list of processese
 *  \param[in] pProc process object
 *
 */
void rilProc::putProc(const char *caller,ril_procList_t *pList, ril_procData_t *pProc)
{
    //int loglevel=LOG_INFO;
    ACLOGI(LOG_TAG,"%s for function %s",__FUNCTION__,caller);

    do {
        if (pList==NULL){
            break;
        }
        for (int idx=0; idx<PROC_NUMBER; idx++){
            if (pList->procs[idx].state!=PROC_ACT){
                continue;
            }
            if (pList->procs[idx].procId<=0){
                continue;
            }
            if (pList->procs[idx].pProc != pProc){
                continue;
            }
            pthread_mutex_unlock(&pList->procs[idx].lock);
            break;
        }

    } while(0);

}

/** \brief rilProc::getNextProcId
 *
 *   return unambiguous process identifier
 *
 *  \param[in] pList list of processese
 *  \param[in] isLocked indicates if caller haslocked list access
 *
 *  \return identifier
 *
 */
int rilProc::getNextProcId(ril_procList_t *pList, int isLocked)
{
    int result=-1;
    ril_procData_t *pCurrent=NULL;

    int findNext;

    do {
        if (pList==NULL){
            break;
        }
        if (isLocked==0) {
            rilProc::lockProcList(__FUNCTION__,pList);
        }

        result=pList->nextId++;
        if (pList->nextId > MAX_PROCID) {
            pList->nextId = MIN_PROCID;
        }

        do{
            findNext=0;
            for (int idx=0; idx<PROC_NUMBER; idx++){
                if ((pList->procs[idx].procId!=0)&&
                    (pList->procs[idx].procId==pList->nextId)){
                    pList->nextId++;
                    if (pList->nextId > MAX_PROCID) {
                        pList->nextId = MIN_PROCID;
                    }
                    findNext=1;
                    break;
                }
            }
        } while (findNext);

        if (isLocked==0) {
            rilProc::unlockProcList(__FUNCTION__,pList);
        }
    } while (0);

    return result;
}

/** \brief rilProc::getNextBufferId
 *
 *   return unambiguous buffer identifier
 *
 *  \param[in] isLocked indicates if caller haslocked list access
 *
 *  \return identifier
 *
 */
int rilProc::getNextBufferId(int isLocked)
{
    int result;

    ril_sendBufferListElement_t  *pCurrent;

    if (isLocked==0){
        rilProc::lockBufferList(&s_sendBuffers);
    }

    result=s_sendBuffers.nextId++;
    if (s_sendBuffers.nextId >MAX_BUFFERID) {
        s_sendBuffers.nextId = MIN_BUFFERID;
    }

    pCurrent=s_sendBuffers.pFirst;
    while (pCurrent){
        if (pCurrent->bufferId==s_sendBuffers.nextId){
            s_sendBuffers.nextId++;
            if (s_sendBuffers.nextId > MAX_BUFFERID) {
                s_sendBuffers.nextId = MIN_BUFFERID;
            }
            pCurrent=s_sendBuffers.pFirst;
        } else {
            pCurrent = pCurrent->pNext;
        }
    }

    if (isLocked==0){
        rilProc::unlockBufferList(&s_sendBuffers);
    }
    return result;
}

/** \brief rilProc::sendEvent
 *
 *   send data to application process
 *
 *  \param[in] unsolResponse event id
 *  \param[in] p  data buffer
 *
 */
void rilProc::sendEvent (int unsolResponse,Parcel &p)
{
    rilProc::sendEventRaw(unsolResponse,p.data(),p.dataSize());
}

/** \brief rilProc::sendEventRaw
 *
 *   send data to application process
 *
 *  \param[in] unsolResponse event id
 *  \param[in] data data buffer
 *  \param[in] dataSize  size of data buffer
 *
 */
void rilProc::sendEventRaw (int unsolResponse, const void *data, size_t dataSize)
{
    int ret=-1;
    int procId;
    int bufferId=0;

    rilProc::lockBufferList(&s_sendBuffers);
    ACLOGD(LOG_TAG,"%s[%d]: send event to all procs..",__FUNCTION__,unsolResponse);

    for (int idx=0; idx < PROC_NUMBER; idx++){

        procId=rilProc::getProcIdAt(&s_procList,idx);
        if (procId<=0){
            continue;
        }
        ACLOGD(LOG_TAG,"%s[%d]: proc Id %d found for index %d",
                __FUNCTION__, unsolResponse, procId, idx);


        //attach to proc list
        if (bufferId==0){
            bufferId=rilProc::sendResponseRaw(procId,data,dataSize,1);
            ACLOGI(LOG_TAG,"%s[%d]: add buffer %d for proc %d",
                        __FUNCTION__, unsolResponse, bufferId, procId);
            continue;
        }

        if (rilProc::duplicateBuffer(procId,bufferId,1)<0){
            ACLOGI(LOG_TAG,"%s[%d]: could not duplicate buffer %d for proc %d",
                        __FUNCTION__,unsolResponse,bufferId,procId);

            bufferId=rilProc::sendResponseRaw(procId,data,dataSize,1);
            ACLOGI(LOG_TAG,"%s[%d]: additional buffer %d for proc %d",
                        __FUNCTION__, unsolResponse, bufferId, procId);
        } else {
            ACLOGI(LOG_TAG,"%s[%d]: buffer with id %d duplicated ",
                        __FUNCTION__, unsolResponse, bufferId);
        }

    }

    rilProc::unlockBufferList(&s_sendBuffers);
}

/** \brief rilProc::duplicateBuffer
 *
 *   duplicate existing send buffer reference and attach a
 *   ril_sendBuffer object to the procList related to procId
 *
 *  \param[in] procId proc object identifier
 *  \param[in] bufferId data object identifier
 *  \param[in] isLocked  indicates if accees lois done by caller
 *
 *  \return success(=0) or error (<0)
 */
int rilProc::duplicateBuffer(int procId, int bufferId, int isLocked)
{
    ril_sendBuffer_t *pCurrent=NULL;
    ril_sendBuffer_t *pBuffer=NULL;
    ril_procData_t *pProc;
    int ret=-1;
    ACLOGD(LOG_TAG,"%s .. for buffer %d",__FUNCTION__,bufferId);
    //int loglevel=LOG_INFO;

    if (isLocked==0){
        rilProc::lockBufferList(&s_sendBuffers);
    }

    pProc=rilProc::getProc(__FUNCTION__,&s_procList,procId);
    do {
        ACLOGD(LOG_TAG,"%s[%d]: buffer identifier %d ",__FUNCTION__,procId,bufferId);
        if (pProc == NULL){
            ACLOGE(LOG_TAG,"%s: could not access proc for id %d",__FUNCTION__,procId);
            break;
        }

        pBuffer= (ril_sendBuffer_t *)MALLOC(sizeof(ril_sendBuffer_t));
        if (pBuffer == 0){
            ACLOGE(LOG_TAG,"%s[%d]: could not create buffer object for buffer %d",__FUNCTION__,procId, bufferId);
            break;
        }

        ret=-1;
        ril_sendBufferListElement_t *pCurrListElement=s_sendBuffers.pFirst;
        while(pCurrListElement){
            if (pCurrListElement->bufferId!=bufferId){
                pCurrListElement=pCurrListElement->pNext;
                continue;
            }

            ACLOGD(LOG_TAG,"%s[%d]: buffer %d found in buffer list",__FUNCTION__,procId,bufferId);
            pBuffer->count=0;
            pBuffer->bufferId=bufferId;
            pBuffer->data=pCurrListElement->data;
            pBuffer->len=pCurrListElement->len;
            pBuffer->pNext=NULL;
            pCurrListElement->procCount++;
            ret=0;
            break;
        }

        if (ret<0){
            ACLOGE(LOG_TAG,"%s[%d]: could not duplicate buffer object",__FUNCTION__,procId);
            FREE(pBuffer);
            break;
        }

        pBuffer->pNext=NULL;
        if (pProc->pSendList==NULL){
            pProc->pSendList=pBuffer;
        } else{
            pCurrent=pProc->pSendList;
            while(pCurrent->pNext!=NULL){
                pCurrent=pCurrent->pNext;
            }
            pCurrent->pNext=pBuffer;
        }
    } while(0);

    if (pProc){
        rilProc::putProc(__FUNCTION__,&s_procList,pProc);
    }

    if (isLocked==0){
        rilProc::unlockBufferList(&s_sendBuffers);
    }

    return ret;
}

/** \brief rilProc::processResponseCallback
 *
 *   process response socket callback
 *
 *  \param[in] fd command socket file descriptor
 *
 *  \return result of operation
 */
int rilProc::processResponseCallback(int fd)
{
    int ret;
    int writeOffset=0;
    int remaining=0;
    int written;
    int doCloseProc=0;
    ril_sendBuffer_t *pCurrent;
    unsigned char *toWrite;
    ACLOGD(LOG_TAG,"%s for fd %d",__FUNCTION__,fd);
    int procId;
    int errorNumber;
    int bufferId;
    //int loglevel=LOG_INFO;

    do {
        procId=rilProc::getProcIdFromFd(&s_procList,fd);
        if (procId <= 0){
            ACLOGW(LOG_TAG,"%s[%d]: could not find process id",__FUNCTION__,fd);
            break;
        }

        ACLOGI(LOG_TAG, "%s: get first send buffer for %d",__FUNCTION__,procId);
        pCurrent=rilProc::takeSendBuffer(&s_procList,procId);
        if (pCurrent==NULL){
            ACLOGD(LOG_TAG,"%s[%d]: nothing to do",__FUNCTION__,fd);
            break;
        }
        bufferId=pCurrent->bufferId;

        ACLOGI(LOG_TAG,"%s[%d]: buffer %d with len %d, count: %d",
                __FUNCTION__, fd, bufferId, pCurrent->len, pCurrent->count);
        remaining=pCurrent->len - pCurrent->count;
        writeOffset=pCurrent->count;
        toWrite=(unsigned char *)(pCurrent->data);

        written = send(fd, toWrite + writeOffset, remaining, 0);
        if (written >= 0) {
            pCurrent->count +=written;
            errorNumber=0;
        } else {
            errorNumber=errno;
        }
        rilProc::grantSendBuffer(&s_procList,procId);

        if (errorNumber !=0) {
            if ((errorNumber==EPIPE)||(errorNumber==ENOTSOCK)||(errorNumber==EBADF)||
                (errorNumber==EDESTADDRREQ)||(errorNumber==ENOTCONN)){
                ACLOGE (LOG_TAG,"%s[%d]: broken socket",__FUNCTION__, fd);
                doCloseProc=-1;
            } else if ((errorNumber == EAGAIN)||(errorNumber == EWOULDBLOCK)){
                ACLOGD(LOG_TAG,"%s[%d]: send would block for buffer %d; repeat",
                        __FUNCTION__, fd, bufferId);
                break;
            }
        }
    } while (0);

    ACLOGD(LOG_TAG,"%s[%d]: done; result = %d",__FUNCTION__, fd, doCloseProc);

    return doCloseProc;
}

/** \brief rilProc::releaseBuffer
 *
 *  release data buffer
 *
 *  \param[in] bufferId unambiguous identifier of buffer
 *
 */
void rilProc::releaseBuffer(int bufferId)
{
    int ret=-1;

    ril_sendBufferListElement_t *pCurrent=NULL;
    ril_sendBufferListElement_t *pPrev=NULL;

    ACLOGI(LOG_TAG,"%s: release buffer %d",__FUNCTION__,bufferId);

    rilProc::lockBufferList(&s_sendBuffers);

    ACLOGD(LOG_TAG,"%s: go through the sendBuffers list ..",__FUNCTION__);

    pCurrent=s_sendBuffers.pFirst;

    while(pCurrent){

        if (pCurrent->bufferId!=bufferId){
            ACLOGD(LOG_TAG,"%s: different buffer id, continue",__FUNCTION__);
            pPrev=pCurrent;
            pCurrent=pCurrent->pNext;
            continue;
        }

        ACLOGD(LOG_TAG,"%s: buffer found",__FUNCTION__);
        pCurrent->procCount--;
        if (pCurrent->procCount<=0){

            ACLOGD(LOG_TAG,"%s: buffer to be removed %p ",__FUNCTION__,pCurrent);
            if (pPrev){
                pPrev->pNext=pCurrent->pNext;
            } else {
                s_sendBuffers.pFirst=pCurrent->pNext;
            }

            if (pCurrent->data){
                FREE(pCurrent->data);
            }

            FREE(pCurrent);
            pCurrent=NULL;
        }
        break;
    }

    ACLOGD(LOG_TAG,"%s: release buffer %d done ",__FUNCTION__,bufferId);

    rilProc::unlockBufferList(&s_sendBuffers);
}

