/* //device/libs/telephony/ril.cpp
**
** Copyright 2006, The Android Open Source Project
**
** Licensed under the Apache License, Version 2.0 (the "License",__func__);
** you may not use this file except in compliance with the License.
** You may obtain a copy of the License at
**
**     http://www.apache.org/licenses/LICENSE-2.0
**
** Unless required by applicable law or agreed to in writing, software
** distributed under the License is distributed on an "AS IS" BASIS,
** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
** See the License for the specific language governing permissions and
** limitations under the License.
*/

#include "config.h"

#include <cstdint>
#include <climits>
#define MAX_CHAR16_BUFFER 64
#include <sys/syslog.h>
#include <stdint.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <errno.h>
#include <assert.h>
#include <ctype.h>
#include <sys/un.h>
#include <assert.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <glob.h>
#include <pthread.h>
#include <signal.h>
#include <pwd.h>
#include "ril_i.h"
#include "ril.h"
#include "utils_log.h"
#include "ril_msim.h"

#include "RilProc.h"
#include "Parcel.h"

#include "utils_misc.h"

#include "rilEvent.h"

void __attribute__ ((constructor)) lib_init(void);
void __attribute__ ((destructor)) lib_fin(void);


// Basically: memset buffers that the client library
// shouldn't be using anymore in an attempt to find
// memory usage issues sooner.
#define MEMSET_FREED 1

#define NUM_ELEMS(a)     (sizeof (a) / sizeof (a)[0])
#define MIN(a,b) ((a)<(b) ? (a) : (b))

/* Constants for response types */
#define RESPONSE_SOLICITED 0
#define RESPONSE_UNSOLICITED 1
#define RESPONSE_SOLICITED_ACK 2
#define RESPONSE_SOLICITED_ACK_EXP 3

/* Negative values for private RIL errno's */
#define RIL_ERRNO_INVALID_RESPONSE -1
#define RIL_ERRNO_NOT_SUPPORTED    -2


typedef struct {
    int requestNumber;
    RIL_Errno (*dispatchFunction) (Parcel &p, struct RequestInfo *pRI);
    int(*responseFunction) (Parcel &p, void *response, size_t responselen);
} CommandInfo;

typedef struct {
    int requestNumber;
    int (*responseFunction) (Parcel &p, void *response, size_t responselen);
} UnsolResponseInfo;

typedef struct RequestInfo {
    int32_t             token;   //this is not RIL_Token
    CommandInfo         *pCI;
    int                 procId;
    struct RequestInfo  *p_next;
    char                cancelled;
    char                local;   // responses to local commands do not go back to command process
    int                 wasAckSent;
} RequestInfo;

typedef struct UserCallbackInfo {
    RIL_TimedCallback         p_callback;
    void                     *userParam;
    rilEvent_t                 *pEvent;
    struct UserCallbackInfo *p_next;
} UserCallbackInfo;

static void ril_UnsolicitedProcResponse(int unsolResponse, void *data,
                                size_t datalen,int procId);


extern "C" void
RIL_onRequestComplete(RIL_Token t, RIL_Errno e,void *response, size_t responselen);

extern "C" void
RIL_onRequestAck(RIL_Token t);


mem_header_t *mem_hdr;


/*******************************************************************/

RIL_RadioFunctions s_callbacks = {0, NULL, NULL, NULL, NULL, NULL};
static int s_registerCalled = 0;

static pthread_t s_tid_dispatch;

static const struct timeval TIMEVAL_WAKE_TIMEOUT = {1,0};

static pthread_mutex_t s_pendingRequestsMutex = PTHREAD_MUTEX_INITIALIZER;
static struct RequestInfo  *s_pendingRequests=NULL;

static UserCallbackInfo *s_last_wake_timeout_info = NULL;

static void *s_lastNITZTimeData = NULL;
static size_t s_lastNITZTimeDataSize;

static int s_readerTid=0;

static int ril_majorVersion;
static int ril_minorVersion;

//int loglevel=LOG_NOTICE;

/*******************************************************************/
namespace radio {

    // dispatch functions without input data (void)
    static RIL_Errno getCurrentCallsRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno HangupWaitingOrBackgroundRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno HangupForegroundResumeBackgroundRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno switchWaitingOrHoldingAndActiveRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno conferenceRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno rejectCallRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getLastCallFailCauseRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getSignalStrengthRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getDataRadioTechnologyRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getVoiceRegistrationStateRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getDataRegistrationStateRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getOperatorRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno cancelPendingUssdRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getClirRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getImeiRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getImeiSvRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno acceptCallRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getNetworkSelectionModeRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno setNetworkSelectionModeAutomaticRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getAvailableNetworksRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno stopDtmfRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getBasebandVersionRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getMuteRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno explicitCallTransferRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getPreferredNetworkTypeRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getNeighboringCidsRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getTTYModeRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getGsmBroadcastConfigRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getDeviceIdentityRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno exitEmergencyCallbackModeRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getSmscAddressRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno reportStkServiceIsRunningRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getClipRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getDataCallListRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getAvailableBandModesRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getIccCardStatusRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getCellInfoListRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno getImsRegistrationStateRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno getHardwareConfigRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno stopLceServiceRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno pullLceDataRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno getModemActivityInfoRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno getAllowedCarriersRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno stopNetworkScanRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno stopKeepaliveRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno requestShutdownRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno getRadioCapabilityRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno setCustomXmlPayloadRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno setEcallTimerRestartRequest (Parcel& p, RequestInfo *pRI);

    // dispatch functions with numerical (ints) input data
    static RIL_Errno hangupConnectionRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getCallWaitingRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno setCallWaitingRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno setClirRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno separateConnectionRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno acknowledgeLastIncomingGsmSmsRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno setMuteRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno sendDeviceStateRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno setSuppServiceNotificationsRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno deleteSmsOnSimRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno setBandModeRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno setPreferredNetworkTypeRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno setRadioPowerRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno handleStkCallSetupRequestFromSimRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno setLocationUpdatesRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno setTTYModeRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno setGsmBroadcastActivationRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno reportSmsMemoryStatusRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno iccCloseLogicalChannelRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno setCellInfoListRateRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno nvResetConfigRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno setDataAllowedRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno startLceServiceRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno setIndicationFilterRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno setSimCardPowerRequest (Parcel& p, RequestInfo *pRI);

    // dispatch functions with single string input data
    static RIL_Errno sendDtmfRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno sendUssdRequest (Parcel& p, RequestInfo *pRI);
    static RIL_Errno setNetworkSelectionModeManualRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno startDtmfRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno sendEnvelopeRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno sendTerminalResponseToSimRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno setSmscAddressRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno requestIsimAuthenticationRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno sendEnvelopeWithStatusRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno iccOpenLogicalChannelRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getIncrementalNetworkScanRequest(Parcel& p, RequestInfo *pRI);

    // dispatch functions with multiple strings (strings array) data
    static RIL_Errno supplyIccPinForAppRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno supplyIccPukForAppRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno supplyIccPin2ForAppRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno supplyIccPuk2ForAppRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno changeIccPinForAppRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno changeIccPin2ForAppRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno supplyNetworkDepersonalizationRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getIMSIForAppRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno deactivateDataCallRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getFacilityLockForAppRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno setFacilityLockForAppRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno setBarringPasswordRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno oemHookStringsRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno acknowledgeIncomingGsmSmsWithPduReqquest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno sendSmsRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno sendSMSExpectMoreRequest(Parcel& p, RequestInfo *pRI);

    static RIL_Errno iccIOForAppRequest (Parcel &p, RequestInfo *pRI);

    // dedicated dispatch functions
    // RIL_Dial unput data
    static RIL_Errno dialRequest (Parcel& p, RequestInfo *pRI);
    // RIL_GSM_BroadcastSmsConfigInfo
    static RIL_Errno setGsmBroadcastConfigRequest(Parcel &p, RequestInfo *pRI);
    // data 'blob' ( array aof usnigned char)
    static RIL_Errno oemHookRawRequest(Parcel& p, RequestInfo *pRI);
    // strings array as input but addtional input checks
    static RIL_Errno setupDataCallRequest (Parcel& p, RequestInfo *pRI);
    // RIL_SIM_APDU input data
    static RIL_Errno iccTransmitApduBasicChannelRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno iccTransmitApduLogicalChannelRequest(Parcel& p, RequestInfo *pRI);
    // RIL_CallForwardInfo input data
    static RIL_Errno setCallForwardRequest(Parcel& p, RequestInfo *pRI);
    static RIL_Errno getCallForwardStatusRequest(Parcel& p, RequestInfo *pRI);
    // RIL_SMS_WriteArgs input data
    static RIL_Errno writeSmsToSimRequest (Parcel &p, RequestInfo *pRI);
    // strings input from socket converted to RIL_InitialAttachApn
    static RIL_Errno setInitialAttachApnRequest (Parcel& p, RequestInfo *pRI);
    //RIL_IMS_SMS_Message as input data
    static RIL_Errno sendImsSmsRequest (Parcel& p, RequestInfo *pRI);
    // ints input converted to RIL_NV_ReadItem
    static RIL_Errno nvReadItemRequest (Parcel& p, RequestInfo *pRI);
    // ints + string input converted to RIL_NV_WriteItem
    static RIL_Errno nvWriteItemRequest (Parcel& p, RequestInfo *pRI);
    // ints input converted to RIL_SelectUiccSub
    static RIL_Errno setUiccSubscriptionRequest (Parcel& p, RequestInfo *pRI);
    //int + 2 strings converted to RIL_SimAuthentication
    static RIL_Errno requestIccSimAuthenticationRequest (Parcel& p, RequestInfo *pRI);
    //ints and byte array data converted to RIL_RadioCapability
    static RIL_Errno setRadioCapabilityRequest (Parcel& p, RequestInfo *pRI);
    //ints and byte array data converted to RIL_KeepaliveRequest
    static RIL_Errno startKeepaliveRequest (Parcel& p, RequestInfo *pRI);
    //ints and byte array data converted to RIL_NetworkScanRequest
    static RIL_Errno startNetworkScanRequest (Parcel& p, RequestInfo *pRI);
    //ints,strings and byte array data converted to RIL_CarrierInfoForImsiEncryption
    static RIL_Errno setCarrierInfoForImsiEncryptionRequest (Parcel& p, RequestInfo *pRI);
    //ints,strings and byte array data converted to RIL_CarrierRestrictions
    static RIL_Errno setAllowedCarriersRequest (Parcel& p, RequestInfo *pRI);
    //ints,strings  data converted to RIL_DataProfileInfo / RIL_DataProfileInfo_v15
    static RIL_Errno setDataProfileRequest (Parcel& p, RequestInfo *pRI);

    static RIL_Errno getVoiceRadioTechnologyRequest (Parcel& p, RequestInfo *pRI);


    /* handling of RIL indication data */

    // response  functions without data (void)
    static int dialResponse(Parcel &p, void *response, size_t responselen);
    static int hangupConnectionResponse(Parcel &p, void *response, size_t responselen);
    static int hangupWaitingOrBackgroundResponse(Parcel &p, void *response, size_t responselen);
    static int hangupForegroundResumeBackgroundResponse(Parcel &p, void *response, size_t responselen);
    static int switchWaitingOrHoldingAndActiveResponse(Parcel &p, void *response, size_t responselen);
    static int conferenceResponse(Parcel &p, void *response, size_t responselen);
    static int rejectCallResponse(Parcel &p, void *response, size_t responselen);
    static int sendDtmfResponse(Parcel &p, void *response, size_t responselen);
    static int sendUssdResponse(Parcel &p, void *response, size_t responselen);
    static int setClirResponse(Parcel &p, void *response, size_t responselen);
    static int setCallForwardResponse(Parcel &p, void *response, size_t responselen);
    static int setCallWaitingResponse(Parcel &p, void *response, size_t responselen);
    static int acknowledgeLastIncomingGsmSmsResponse(Parcel &p, void *response, size_t responselen);
    static int acceptCallResponse(Parcel &p, void *response, size_t responselen);
    static int deactivateDataCallResponse(Parcel &p, void *response, size_t responselen);
    static int startDtmfResponse(Parcel &p, void *response, size_t responselen);
    static int stopDtmfResponse(Parcel &p, void *response, size_t responselen);
    static int separateConnectionResponse(Parcel &p, void *response, size_t responselen);
    static int setMuteResponse(Parcel &p, void *response, size_t responselen);
    static int setSuppServiceNotificationsResponse(Parcel &p, void *response, size_t responselen);
    static int deleteSmsOnSimResponse(Parcel &p, void *response, size_t responselen);
    static int setBandModeResponse(Parcel &p, void *response, size_t responselen);
    static int sendTerminalResponseToSimResponse(Parcel &p, void *response, size_t responselen);
    static int handleStkCallSetupRequestFromSimResponse(Parcel &p, void *response, size_t responselen);
    static int setPreferredNetworkTypeResponse(Parcel &p, void *response, size_t responselen);
    static int setLocationUpdatesResponse(Parcel &p, void *response, size_t responselen);
    static int setTTYModeResponse(Parcel &p, void *response, size_t responselen);
    static int setGsmBroadcastConfigResponse(Parcel &p, void *response, size_t responselen);
    static int setGsmBroadcastActivationResponse(Parcel &p, void *response, size_t responselen);
    static int exitEmergencyCallbackModeResponse(Parcel &p, void *response, size_t responselen);
    static int setSmscAddressResponse(Parcel &p, void *response, size_t responselen);
    static int reportStkServiceIsRunningResponse(Parcel &p, void *response, size_t responselen);
    static int acknowledgeIncomingGsmSmsWithPduResponse(Parcel &p, void *response, size_t responselen);
    static int iccCloseLogicalChannelResponse(Parcel &p, void *response, size_t responselen);
    static int reportSmsMemoryStatusResponse(Parcel &p, void *response, size_t responselen);
    static int explicitCallTransferResponse(Parcel &p, void *response, size_t responselen);
    static int setBarringPasswordResponse(Parcel &p, void *response, size_t responselen);
    static int setNetworkSelectionModeAutomaticResponse(Parcel &p, void *response, size_t responselen);
    static int setNetworkSelectionModeManualResponse(Parcel &p, void *response, size_t responselen);
    static int cancelPendingUssdResponse(Parcel &p, void *response, size_t responselen);
    static int setRadioPowerResponse(Parcel &p, void *response, size_t responselen);
    static int sendDeviceStateResponse(Parcel &p, void *response, size_t responselen);
    static int setInitialAttachApnResponse(Parcel &p, void *response, size_t responselen);
    static int setCellInfoListRateResponse(Parcel &p, void *response, size_t responselen);
    static int nvWriteItemResponse(Parcel &p, void *response, size_t responselen);
    static int nvResetConfigResponse(Parcel &p, void *response, size_t responselen);
    static int setUiccSubscriptionResponse(Parcel &p, void *response, size_t responselen);
    static int setDataAllowedResponse(Parcel &p, void *response, size_t responselen);
    static int setDataProfileResponse(Parcel &p, void *response, size_t responselen);
    static int requestShutdownResponse(Parcel &p, void *response, size_t responselen);
    static int setIndicationFilterResponse(Parcel &p, void *response, size_t responselen);
    static int setSimCardPowerResponse(Parcel &p, void *response, size_t responselen);
    static int setCarrierInfoForImsiEncryptionResponse(Parcel &p, void *response, size_t responselen);
    static int startNetworkScanResponse(Parcel &p, void *response, size_t responselen);
    static int stopNetworkScanResponse(Parcel &p, void *response, size_t responselen);
    static int stopKeepaliveResponse(Parcel &p, void *response, size_t responselen);
    static int setCustomXmlPayloadResponse(Parcel &p, void *response, size_t responselen);
    static int setEcallTimerRestartResponse(Parcel &p, void *response, size_t responselen);



    // responses with a  fix number of integer values
    static int getFacilityLockForAppResponse(Parcel &p, void *response, size_t responselen); // 1 int values
    static int getClipResponse(Parcel &p, void *response, size_t responselen); // 1 int values
    static int writeSmsToSimResponse(Parcel &p, void *response, size_t responselen); // 1 int values
    static int getPreferredNetworkTypeResponse(Parcel &p, void *response, size_t responselen); // 1 int values
    static int getTTYModeResponse(Parcel &p, void *response, size_t responselen); // 1 int values
    static int getVoiceRadioTechnologyResponse(Parcel &p, void *response, size_t responselen); // 1 int values
    static int getDataRadioTechnologyResponse(Parcel &p,void *response, size_t responselen);
    static int setAllowedCarriersResponse(Parcel &p, void *response, size_t responselen); // 1 int values
    static int getMuteResponse(Parcel &p, void *response, size_t responselen); // 1 int values
    static int getNetworkSelectionModeResponse(Parcel &p, void *response, size_t responselen); // 1 int values
    static int getClirResponse(Parcel &p, void *response, size_t responselen); //2 int value
    static int getCallWaitingResponse(Parcel &p, void *response, size_t responselen); //2 int values
    static int getImsRegistrationStateResponse(Parcel &p, void *response, size_t responselen); //1 value

    // responses with a variable number of integer values
    static int supplyIccPinForAppResponse(Parcel &p, void *response, size_t responselen); //0 or 1 int value
    static int supplyIccPukForAppResponse(Parcel &p, void *response, size_t responselen); //0 or 1 int value
    static int supplyIccPin2ForAppResponse(Parcel &p, void *response, size_t responselen); //0 or 1 int value
    static int supplyIccPuk2ForAppResponse(Parcel &p, void *response, size_t responselen); //0 or 1 int value
    static int changeIccPinForAppResponse(Parcel &p, void *response, size_t responselen); //0 or 1 int value
    static int changeIccPin2ForAppResponse(Parcel &p, void *response, size_t responselen); //0 or 1 int value
    static int supplyNetworkDepersonalizationResponse(Parcel &p, void *response, size_t responselen); //0 or 1 int value
    static int setFacilityLockForAppResponse(Parcel &p, void *response, size_t responselen); // o or 1 int values
    static int getLastCallFailCauseResponse(Parcel &p, void *response, size_t responselen); //1 or 2 int value; dedicated handling
    static int getAvailableBandModesResponse(Parcel &p, void *response, size_t responselen); //array: size + array values
    static int iccOpenLogicalChannelResponse(Parcel &p, void *response, size_t responselen); //array: size + array values



    // responses with strings array data
    static int getOperatorResponse(Parcel &p, void *response, size_t responselen);
    static int getAvailableNetworksResponse(Parcel &p, void *response, size_t responselen);
    static int oemHookStringsResponse(Parcel &p, void *response, size_t responselen);
    static int getDeviceIdentityResponse(Parcel &p, void *response, size_t responselen);
    static int getIMSIForAppResponse(Parcel &p, void *response, size_t responselen); //single string
    static int getImeiResponse(Parcel &p, void *response, size_t responselen); //single string
    static int getImeiSvResponse(Parcel &p, void *response, size_t responselen); //single string
    static int getBasebandVersionResponse(Parcel &p, void *response, size_t responselen); //single string
    static int sendEnvelopeResponse(Parcel &p, void *response, size_t responselen); //single string
    static int getSmscAddressResponse(Parcel &p, void *response, size_t responselen); //single string
    static int requestIsimAuthenticationResponse(Parcel &p, void *response, size_t responselen); //single string
    static int getIccCardStatusResponse(Parcel &p, void *response, size_t responselen);
    static int getSignalStrengthResponse(Parcel &p,void *response, size_t responselen);
    static int getCallForwardStatusResponse(Parcel &p, void *response, size_t responselen);
    static int nvReadItemResponse(Parcel &p, void *response, size_t responselen);//single string
    static int getIncrementalNetworkScanResponse(Parcel &p,void *response, size_t responselen);

    //responses with dedicated data
    static int responseIccIo(const char *caller, Parcel &p, void *response, size_t responselen);
    static int sendEnvelopeWithStatusResponse(Parcel &p, void *response, size_t responselen);
    static int iccTransmitApduBasicChannelResponse(Parcel &p, void *response, size_t responselen);
    static int iccIOForAppResponse(Parcel &p, void *response, size_t responselen);
    static int iccTransmitApduLogicalChannelResponse(Parcel &p, void *response, size_t responselen);
    static int requestIccSimAuthenticationResponse(Parcel &p, void *response, size_t responselen);
    static int getGsmBroadcastConfigResponse(Parcel &p, void *response, size_t responselen);
    static int getVoiceRegistrationStateResponse(Parcel &p,void *response, size_t responselen);
    static int getDataRegistrationStateResponse(Parcel &p,void *response, size_t responselen);
    static int getNeighboringCidsResponse(Parcel &p, void *response, size_t responselen);
    static int sendSmsResponse(Parcel &p, void *response, size_t responselen);
    static int sendSmsResponseExpectMore(Parcel &p, void *response, size_t responselen);
    static int setupDataCallResponse(Parcel &p, void *response, size_t responselen);
    static int getCurrentCallsResponse(Parcel &p, void *response, size_t responselen);
    static int getDataCallListResponse(Parcel &p, void *response, size_t responselen);
    static int sendImsSmsResponse(Parcel &p, void *response, size_t responselen); // const RIL_SMS_Response
    static int getHardwareConfigResponse(Parcel &p, void *response, size_t responselen);  // array of  RIL_HardwareConfig.
    static int startKeepaliveResponse(Parcel &p, void *response, size_t responselen); // RIL_KeepaliveStatus
    static int getModemActivityInfoResponse(Parcel &p, void *response, size_t responselen); // RIL_ActivityStatsInfo
    static int pullLceDataResponse(Parcel &p, void *response, size_t responselen); // RIL_LceDataInfo
    static int startLceServiceResponse(Parcel &p, void *response, size_t responselen); //  RIL_LceStatusInfo
    static int stopLceServiceResponse(Parcel &p, void *response, size_t responselen); //  RIL_LceStatusInfo
    static int getRadioCapabilityResponse(Parcel &p, void *response, size_t responselen); // RIL_RadioCapability
    static int setRadioCapabilityResponse(Parcel &p, void *response, size_t responselen); //RIL_RadioCapability
    static int getCellInfoListResponse(Parcel &p, void *response, size_t responselen); //array of  RIL_CellInfo_v12
    static int getAllowedCarriersResponse(Parcel &p, void *response, size_t responselen); //RIL_CarrierRestrictions


    /* handling of RIL indication data */

    // indication  functions without data (void)
    static int imsNetworkStateChangedInd(Parcel &p, void *response, size_t responselen);
    static int callStateChangedInd(Parcel &p, void *response, size_t responselen);
    static int radioStateChangedInd(Parcel &p, void *response, size_t responselen);
    static int networkStateChangedInd(Parcel &p, void *response, size_t responselen);
    static int simSmsStorageFullInd(Parcel &p, void *response, size_t responselen);
    static int stkSessionEndInd(Parcel &p, void *response, size_t responselen);
    static int enterEmergencyCallbackModeInd(Parcel &p, void *response, size_t responselen);
    static int exitEmergencyCallbackModeInd(Parcel &p, void *response, size_t responselen);
    static int resendIncallMuteInd(Parcel &p, void *response, size_t responselen);
    static int callRingInd(Parcel &p, void *response, size_t responselen);
    static int simStatusChangedInd(Parcel &p, void *response, size_t responselen);
    static int carrierInfoForImsiEncryptionInd(Parcel &p, void *response, size_t responselen);

    //  indications with a fixed number of integer values
    static int stkCallSetupInd(Parcel &p, void *response, size_t responselen);            // 1 int value
    static int restrictedStateChangedInd(Parcel &p, void *response, size_t responselen);  // 1 int value
    static int indicateRingbackToneInd(Parcel &p, void *response, size_t responselen);  // 1 int value
    static int rilConnectedInd(Parcel &p, void *response, size_t responselen);  // 1 int value
    static int voiceRadioTechChangedInd(Parcel &p, void *response, size_t responselen);  // 1 int value
    static int dataRadioTechChangedInd(Parcel &p, void *response, size_t responselen);  // 1 int value
    static int getDataRadioTechChangedInd(Parcel &p,void *response, size_t responselen);
    static int subscriptionStatusChangedInd(Parcel &p, void *response, size_t responselen);  // 1 int value
    static int srvccStateNotifyInd(Parcel &p, void *response, size_t responselen); //RIL_SrvccState = 1 num value
    static int newSmsOnSimInd(Parcel &p, void *response, size_t responselen); //index value

    // indications with a string data
    static int newSmsInd(Parcel &p, void *response, size_t responselen);
    static int newSmsStatusReportInd(Parcel &p, void *response, size_t responselen);
    static int nitzTimeReceivedInd(Parcel &p, void *response, size_t responselen);
    static int stkProactiveCommandInd(Parcel &p, void *response, size_t responselen);
    static int stkEventNotifyInd(Parcel &p, void *response, size_t responselen);
    static int stkCallControlAlphaNotifyInd(Parcel &p, void *response, size_t responselen);
    static int modemResetInd(Parcel &p, void *response, size_t responselen);

    //indications with strings array data
    static int onUssdInd(Parcel &p, void *response, size_t responselen);

    //indications with dedicated data
    static int simRefreshInd(Parcel &p, void *response, size_t responselen);  // RIL_SimRefreshResponse_v7
    static int suppSvcNotifyInd(Parcel &p, void *response, size_t responselen);  // RIL_SuppSvcNotification
    static int newBroadcastSmsInd(Parcel &p, void *response, size_t responselen);
    static int oemHookRawIndOrResp(Parcel &p, void *response, size_t responselen);
    static int dataCallListChangedInd(Parcel &p, void *response, size_t responselen); // list of RIL_Data_Call_Response_v11
    static int currentSignalStrengthInd(Parcel &p, void *response, size_t responselen); //Signal strength v11
    static int cellInfoListInd(Parcel &p, void *response, size_t responselen); // RIL_CellInfo_v12
    static int hardwareConfigChangedInd(Parcel &p, void *response, size_t responselen); // array of RIL_HardwareConfig
    static int radioCapabilityIndicationInd(Parcel &p, void *response, size_t responselen); //RIL_RadioCapability
    static int onSupplementaryServiceIndicationInd(Parcel &p, void *response, size_t responselen); //RIL_StkCcUnsolSsResponse
    static int lceDataInd(Parcel &p, void *response, size_t responselen);  //RIL_LceDataInfo
    static int pcoDataInd(Parcel &p, void *response, size_t responselen); // RIL_PCO_Data
    static int networkScanResultInd(Parcel &p, void *response, size_t responselen); // RIL_NetworkScanResult
    static int keepaliveStatusInd(Parcel &p, void *response, size_t responselen); // RIL_KeepAlive
    static int incrementalNetworkScanInd(Parcel &p, void *response, size_t responselen);
    static int imsSettingsMsdInd(Parcel &p, void *response, size_t responselen);
    static int voiceGetLatestMsdInd(Parcel &p, void *response, size_t responselen);

}; //namespace radio

// generic dispatch functions
static RIL_Errno dispatchVoidGeneric (Parcel &p, int reqNumber, RequestInfo *pRI);
static RIL_Errno dispatchStringsGeneric (Parcel &p, int reqNumber, RequestInfo *pRI);
static RIL_Errno dispatchSmsSendGeneric (Parcel &p, int reqNumber, RequestInfo *pRI);
static RIL_Errno dispatchApduGeneric (Parcel& p, int reqNumber, RequestInfo *pRI);
static RIL_Errno dispatchCallForwardGeneric(Parcel& p, int reqid,RequestInfo *pRI);
static RIL_Errno dispatchIntsGeneric(Parcel& p, int reqId, RequestInfo *pRI);
static RIL_Errno dispatchSingleStringGeneric(Parcel& p, int reqId, RequestInfo *pRI);

// generic response( or indication) functions
// reponse/indication contains afix number of integer values
static int genericIntFixedRespOrInd(const char * caller,int paramNumber, Parcel &p, void *response, size_t responselen);
static int genericIntVarRespOrInd(const char * caller,int paramNumber, Parcel &p, void *response, size_t responselen);
static int genericIntSingleOrEmptyRespOrInd(const char * caller, Parcel &p, void *response, size_t responselen);

static int genericStringsRespOrInd(const char * caller,int paramNumber, Parcel &p, void *response, size_t responselen);
static int genericStringsUpToRespOrInd(const char * caller,int paramNumber, Parcel &p, void *response, size_t responselen);
static int responseSendSmsGeneric (Parcel &p, int id, void *response, size_t responselen);
static int genericLceResponse(Parcel &p, int reqId, void *response, size_t responselen);
static int genericLceDataIndOrResp(Parcel &p, int id, void *response, size_t responselen);
static int genericCellInfoListIndOrResp(Parcel &p, int id, void *response, size_t responselen);


//helper functions (response processing)
static int convertResponseHexStringEntryToInt(char **response, int index, int numStrings);
static int convertResponseStringEntryToInt(char **response, int index, int numStrings) ;
static void fillCellIdentityFromVoiceRegStateResponseString(RIL_CellIdentity_v16 &cellIdentity,int numStrings, void* response);
static void fillCellIdentityFromDataRegStateResponseString(RIL_CellIdentity_v16 &cellIdentity,int numStrings, void* response);
static int sendCellIdentityResponse(Parcel &p,RIL_CellIdentity_v16 &rilCellIdentity);
static bool isServiceTypeCfQuery(RIL_SsServiceType serType, RIL_SsRequestType reqType);
static void freeSetDataProfileData(int numDataProfiles, RIL_DataProfileInfo **dataProfilePtrs);
static void freeSetDataProfileData_v15(int numDataProfiles, RIL_DataProfileInfo_v15 **dataProfilePtrs);
static int responseStrings(Parcel &p, void *response, size_t responselen);

static int responseRilSignalStrengthV11(Parcel &p, void *response, size_t responselen);
static int responseRilSignalStrengthV6(Parcel &p, void *response, size_t responselen);

static RIL_CellInfoType getCellInfoTypeRadioTechnology(char *rat);

static int checkAndDequeueRequestInfoAck(struct RequestInfo *pRI, bool isAck);

namespace libril {
static void sendErrorResponseFromRequestInfo( RequestInfo *pRI, RIL_Errno e);
static void sendErrorResponseFromInputData(int request, int32_t token, int procId, RIL_Errno e);
RequestInfo* createRequest(int procId, int request, int32_t token);
};


extern "C" const char * requestToString(int request);
extern "C" const char * failCauseToString(RIL_Errno);
extern "C" const char * callStateToString(RIL_CallState);
extern "C" const char * radioStateToString(RIL_RadioState);

const double NANOSECONDS_TO_MILLISECONDS = 1.0 / 1000000.0;

static const char *libril_version=RILVERSION;

void libril_checkMem(short flags, void *param)
{
    MEM_DUMP;
}

void lib_init()
{
    ALOGOPEN(LOG_TAG);
    mem_hdr=cpp_mem_init();
    sscanf(libril_version,"%d.%d",&ril_majorVersion,&ril_minorVersion);
}

int libril_getInterfaceVersion()
{
    return 100 * ril_majorVersion + ril_minorVersion;
}

void lib_fin()
{
    ALOGCLOSE();
};


/*
 * native public static long elapsedRealtime();
 */
int64_t elapsedRealtime(){
       int64_t when;
       struct timeval t;
       t.tv_sec = t.tv_usec = 0;
       gettimeofday(&t, NULL);

        when = (t.tv_sec)*1000000000LL + (t.tv_usec)*1000LL;
         return (int64_t) when*NANOSECONDS_TO_MILLISECONDS;
}

static UserCallbackInfo * internalRequestTimedCallback
    (RIL_TimedCallback callback, void *param,const struct timeval *relTime);

/** Index == requestNumber */
static CommandInfo s_commands[] = {
    #include "ril_commands.h"
};

static UnsolResponseInfo s_unsolResponses[] = {
    #include "ril_unsol_commands.h"
};

/* For older RILs that do not support new commands RIL_REQUEST_VOICE_RADIO_TECH and
   RIL_UNSOL_VOICE_RADIO_TECH_CHANGED messages, decode the voice radio tech from
   radio state message and store it. Every time there is a change in Radio State
   check to see if voice radio tech changes and notify telephony
 */
int voiceRadioTech = -1;


void libril::sendErrorResponseFromInputData(int request, int32_t token, int procId, RIL_Errno e)
{

    Parcel p;
    p.writeInt32 (RESPONSE_SOLICITED);
    p.writeInt32 (token);
    p.writeInt32 (e);

    if (rilProc::getCommandFd(procId) <=0) {
        ACLOGD (LOG_TAG,"%s: Command channel closed",__func__);
    }

    ACLOGI (LOG_TAG,"%s: sendResponse to proc %d",__func__,procId);
    int ret=rilProc::sendResponse(procId, p);
    if (ret<0){
        ACLOGE(LOG_TAG,"%s: client socket broken",__func__);
        rilProc::setState(procId,PROC_SHOULD_CLOSE);
    }
}

void libril::sendErrorResponseFromRequestInfo(RequestInfo *pRI, RIL_Errno e)
{
    do {
        if (pRI ==0) {
            ACLOGE (LOG_TAG,"%s: missing info",__func__);
            break;
        }
        if (!checkAndDequeueRequestInfoAck(pRI,false)) {
            ACLOGE (LOG_TAG,"%s: invalid RIL_Token",__func__);
            FREE(pRI);
            break;
        }

        Parcel p;
        p.writeInt32 (RESPONSE_SOLICITED);
        p.writeInt32 (pRI->token);
        p.writeInt32 (e);

        if (rilProc::getCommandFd(pRI->procId) <=0) {
            ACLOGD (LOG_TAG,"%s: Command channel closed",__func__);
        }

        ACLOGI (LOG_TAG,"%s: sendResponse to proc %d",__func__,pRI->procId);
        int ret=rilProc::sendResponse(pRI->procId, p);
        if (ret<0){
            ACLOGE(LOG_TAG,"%s: client socket broken",__func__);
            rilProc::setState(pRI->procId,PROC_SHOULD_CLOSE);
        }
        FREE(pRI);
    } while (0);
}

void   nullParcelReleaseFunction (const uint8_t* data, size_t dataSize,
                                    const size_t* objects, size_t objectsSize, void* cookie) {
    // do nothing -- the data reference lives longer than the Parcel object
}

/**
 * To be called from dispatch thread
 * Issue a single local request, ensuring that the response
 * is not sent back up to the command process
 */
static void
issueLocalRequest(int request, void *data, int len) {
    RequestInfo *pRI;
    int ret;

    ACLOGV(LOG_TAG,"issueLocalRequest %d", request);

    pRI = (RequestInfo *)calloc(1, sizeof(RequestInfo));

    pRI->local = 1;
    pRI->token = 0xffffffff;        // token is not used in this context
    pRI->pCI = &(s_commands[request]);

    ret = pthread_mutex_lock(&s_pendingRequestsMutex);
    assert (ret == 0);

    pRI->p_next = s_pendingRequests;
    s_pendingRequests = pRI;

    ret = pthread_mutex_unlock(&s_pendingRequestsMutex);
    assert (ret == 0);

    ACLOGV(LOG_TAG,"C[locl]> %s", requestToString(request));

    s_callbacks.onRequest(request, data, len, pRI);
}

int libril_processCommandBuffer(void *buffer, size_t buflen, int procId)  //NON SUPPORTED ISSUE HERE
{
    Parcel p;
    status_t status;
    int32_t request;
    int32_t token;
    RequestInfo *pRI;
    int ret;

    ACLOGD(LOG_TAG,"%s..",__func__);

    p.setData((uint8_t *) buffer, buflen);

    // status checked at end
    status = p.readInt32(&request);
    status = p.readInt32 (&token);  // unambiguous message identifier

    do {
        if (status != NO_ERROR) {
            ACLOGE(LOG_TAG,"%s[%d]: invalid request block",__func__,procId);
            break;
        }

        if (request < 1 || request >= (int32_t)NUM_ELEMS(s_commands)) {
            ACLOGE(LOG_TAG,"%s[%d]: unsupported request code %d token %d",__func__,procId, request, token);
            libril::sendErrorResponseFromInputData(request,token,procId,RIL_E_REQUEST_NOT_SUPPORTED);
            break;
        }

        ACLOGI(LOG_TAG,"%s[%d]: code %d token %d",__func__,procId, request, token);

        pRI = libril::createRequest(procId, request, token);
        if (!pRI){
            ACLOGW(LOG_TAG,"%s[%d]: could not create request",__func__,procId);
            break;
        }

        RIL_Errno result=pRI->pCI->dispatchFunction(p, pRI);
        if (result !=RIL_E_SUCCESS){
            libril::sendErrorResponseFromRequestInfo(pRI,result);
        }
    } while (0);

    ACLOGD(LOG_TAG,"%s[%d]: done",__func__,procId);
    return 0;
}

RequestInfo* libril::createRequest(int procId, int request, int32_t token)
{
    RequestInfo *pRI=0;
    int ret;

    do {
        pRI = (RequestInfo *)MALLOC(sizeof(RequestInfo));

        if (pRI == 0){
            ACLOGW(LOG_TAG,"%s[%d]: could not allocate pRI",__func__,procId);
            libril::sendErrorResponseFromInputData(request,token,procId,RIL_E_MISSING_RESOURCE);
            break;
        }

        pRI->local=0;
        pRI->cancelled=0;
        pRI->procId=procId;
        pRI->token = token;
        pRI->wasAckSent=0;
        pRI->pCI = &(s_commands[request]);
        ret = pthread_mutex_lock(&s_pendingRequestsMutex);
        assert (ret == 0);

        // add new request to the list of pending requests
        pRI->p_next = s_pendingRequests;
        s_pendingRequests = pRI;

        ret = pthread_mutex_unlock(&s_pendingRequestsMutex);
        assert (ret == 0);

    } while (0);
    return pRI;
}


static void invalidCommandBlock (RequestInfo *pRI)
{
    ACLOGE(LOG_TAG,"%s: invalid command block for token %d request %s",__func__,
                pRI->token, requestToString(pRI->pCI->requestNumber));
}

//dispatch functions with void data
RIL_Errno dispatchVoidGeneric (Parcel &p, int reqNumber, RequestInfo *pRI)
{
    s_callbacks.onRequest(reqNumber, NULL, 0, pRI);
    return RIL_E_SUCCESS;
}

RIL_Errno  radio::getCurrentCallsRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_GET_CURRENT_CALLS, pRI);
}

RIL_Errno radio::HangupWaitingOrBackgroundRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_HANGUP_WAITING_OR_BACKGROUND, pRI);
}

RIL_Errno radio::HangupForegroundResumeBackgroundRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_HANGUP_FOREGROUND_RESUME_BACKGROUND, pRI);
}

RIL_Errno radio::switchWaitingOrHoldingAndActiveRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_SWITCH_WAITING_OR_HOLDING_AND_ACTIVE, pRI);
}

RIL_Errno radio::conferenceRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_CONFERENCE, pRI);
}

RIL_Errno radio::rejectCallRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_UDUB, pRI);
}

RIL_Errno radio::getLastCallFailCauseRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_LAST_CALL_FAIL_CAUSE, pRI);
}

RIL_Errno radio::getSignalStrengthRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_SIGNAL_STRENGTH, pRI);
}

RIL_Errno radio::getDataRadioTechnologyRequest(Parcel& p, RequestInfo *pRI)
{
    ACLOGV(LOG_TAG,"%s..",__func__);
    return dispatchVoidGeneric (p, RIL_REQUEST_DATA_RADIO_TECH, pRI);
}
RIL_Errno radio::getIncrementalNetworkScanRequest(Parcel& p, RequestInfo *pRI)
{
    ACLOGV(LOG_TAG,"%s..",__func__);
    return dispatchVoidGeneric (p, RIL_REQUEST_INCREMENTAL_NETWORK_SCAN, pRI);
}

RIL_Errno radio::getVoiceRegistrationStateRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_VOICE_REGISTRATION_STATE, pRI);
}

RIL_Errno radio::getDataRegistrationStateRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_DATA_REGISTRATION_STATE, pRI);
}

RIL_Errno radio::getOperatorRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_OPERATOR, pRI);
}

RIL_Errno radio::cancelPendingUssdRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_CANCEL_USSD, pRI);
}

RIL_Errno radio::getClirRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_GET_CLIR, pRI);
}

RIL_Errno radio::getImeiRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_GET_IMEI, pRI);
}

RIL_Errno radio::getImeiSvRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_GET_IMEISV, pRI);
}

RIL_Errno radio::acceptCallRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_ANSWER, pRI);
}

RIL_Errno radio::getNetworkSelectionModeRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_QUERY_NETWORK_SELECTION_MODE, pRI);
}

RIL_Errno radio::setNetworkSelectionModeAutomaticRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_SET_NETWORK_SELECTION_AUTOMATIC, pRI);
}

RIL_Errno radio::getAvailableNetworksRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_QUERY_AVAILABLE_NETWORKS, pRI);
}

RIL_Errno radio::stopDtmfRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_DTMF_STOP, pRI);
}

RIL_Errno radio::getBasebandVersionRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_BASEBAND_VERSION, pRI);
}

RIL_Errno radio::getMuteRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_GET_MUTE, pRI);
}

RIL_Errno radio::explicitCallTransferRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_EXPLICIT_CALL_TRANSFER, pRI);
}

RIL_Errno radio::getPreferredNetworkTypeRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_GET_PREFERRED_NETWORK_TYPE, pRI);
}

RIL_Errno radio::getNeighboringCidsRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_GET_NEIGHBORING_CELL_IDS, pRI);
}

RIL_Errno radio::getTTYModeRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_QUERY_TTY_MODE, pRI);
}

RIL_Errno radio::getGsmBroadcastConfigRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_GSM_GET_BROADCAST_SMS_CONFIG, pRI);
}

RIL_Errno radio::getDeviceIdentityRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_DEVICE_IDENTITY, pRI);
}

RIL_Errno radio::exitEmergencyCallbackModeRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_EXIT_EMERGENCY_CALLBACK_MODE, pRI);
}

RIL_Errno radio::getSmscAddressRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_GET_SMSC_ADDRESS, pRI);
}

RIL_Errno radio::reportStkServiceIsRunningRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_REPORT_STK_SERVICE_IS_RUNNING, pRI);
}

RIL_Errno radio::getClipRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_QUERY_CLIP, pRI);
}

RIL_Errno radio::getDataCallListRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_DATA_CALL_LIST, pRI);
}

RIL_Errno radio::getAvailableBandModesRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_QUERY_AVAILABLE_BAND_MODE, pRI);
}

RIL_Errno radio::getIccCardStatusRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_GET_SIM_STATUS, pRI);
}

RIL_Errno radio::getCellInfoListRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_GET_CELL_INFO_LIST, pRI);
}

RIL_Errno radio::getHardwareConfigRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_GET_HARDWARE_CONFIG, pRI);
}

RIL_Errno radio::requestShutdownRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_SHUTDOWN, pRI);
}

RIL_Errno radio::getModemActivityInfoRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_GET_ACTIVITY_INFO, pRI);
}

RIL_Errno radio::getAllowedCarriersRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_GET_CARRIER_RESTRICTIONS, pRI);
}

RIL_Errno radio::stopNetworkScanRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_STOP_NETWORK_SCAN, pRI);
}

RIL_Errno radio::getImsRegistrationStateRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_IMS_REGISTRATION_STATE, pRI);
}

RIL_Errno radio::stopLceServiceRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_STOP_LCE, pRI);
}

RIL_Errno radio::pullLceDataRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_PULL_LCEDATA, pRI);
}

RIL_Errno radio::stopKeepaliveRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_STOP_KEEPALIVE, pRI);
}

RIL_Errno radio::getRadioCapabilityRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_GET_RADIO_CAPABILITY, pRI);
}

RIL_Errno radio::setCustomXmlPayloadRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_SET_CUSTOM_XML_PAYLOAD, pRI);
}

RIL_Errno radio::setEcallTimerRestartRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchVoidGeneric (p, RIL_REQUEST_ECALL_TIMER_RESTART, pRI);
}


RIL_Errno dispatchIntsGeneric(Parcel& p, int reqId, RequestInfo *pRI)
{
    int32_t count;
    status_t status;
    int *pInts=NULL;

    size_t dataLen=0;
    int invalid=1;
    RIL_Errno result=RIL_E_SUCCESS;

    do  {
        status = p.readInt32 (&count);
        if ((status != NO_ERROR) || (count == 0)) {
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        dataLen = sizeof(int) * count;
        if (dataLen>0){
            pInts = (int *)MALLOC(dataLen);
            if (pInts==NULL) {
                result=RIL_E_MISSING_RESOURCE;
                break;
            }
        }

        status=NO_ERROR;
        for (int i = 0 ; i < count ; i++) {
            int32_t t;
            status = p.readInt32(&t);
            if (status != NO_ERROR) {
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }
            pInts[i] = (int)t;
            ACLOGV(LOG_TAG,"%s[%d]: %d", __func__, i,pInts[i]);
        }

        if (status != NO_ERROR) {
            break;
        }

    } while (0);

    if (result != RIL_E_SUCCESS){
        invalidCommandBlock(pRI);
    } else {
        s_callbacks.onRequest(reqId, const_cast<int *>(pInts), dataLen, pRI);
    }

    if (pInts != NULL){
        FREE(pInts);
    }
    return result;
}

RIL_Errno radio::hangupConnectionRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_HANGUP,pRI);
}

RIL_Errno radio::getCallWaitingRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_QUERY_CALL_WAITING,pRI);
}

RIL_Errno radio::setCallWaitingRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_SET_CALL_WAITING,pRI);
}

RIL_Errno radio::setClirRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_SET_CLIR,pRI);
}

RIL_Errno radio::separateConnectionRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_SEPARATE_CONNECTION,pRI);
}

RIL_Errno radio::acknowledgeLastIncomingGsmSmsRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_SMS_ACKNOWLEDGE,pRI);
}

RIL_Errno radio::setMuteRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_SET_MUTE,pRI);
}

RIL_Errno radio::sendDeviceStateRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_SCREEN_STATE,pRI);
}

RIL_Errno radio::setSuppServiceNotificationsRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_SET_SUPP_SVC_NOTIFICATION,pRI);
}

RIL_Errno radio::deleteSmsOnSimRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_DELETE_SMS_ON_SIM,pRI);
}

RIL_Errno radio::setBandModeRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_SET_BAND_MODE,pRI);
}

RIL_Errno radio::setPreferredNetworkTypeRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_SET_PREFERRED_NETWORK_TYPE,pRI);
}

RIL_Errno radio::setRadioPowerRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_RADIO_POWER,pRI);
}

RIL_Errno radio::handleStkCallSetupRequestFromSimRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_STK_HANDLE_CALL_SETUP_REQUESTED_FROM_SIM,pRI);
}

RIL_Errno radio::setLocationUpdatesRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_SET_LOCATION_UPDATES,pRI);
}

RIL_Errno radio::setTTYModeRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_SET_TTY_MODE,pRI);
}

RIL_Errno radio::setGsmBroadcastActivationRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_GSM_SMS_BROADCAST_ACTIVATION,pRI);
}

RIL_Errno radio::reportSmsMemoryStatusRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_REPORT_SMS_MEMORY_STATUS,pRI);
}

RIL_Errno radio::iccCloseLogicalChannelRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_SIM_CLOSE_CHANNEL,pRI);
}

RIL_Errno radio::setCellInfoListRateRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_SET_UNSOL_CELL_INFO_LIST_RATE,pRI);
}

RIL_Errno radio::nvResetConfigRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_NV_RESET_CONFIG,pRI);
}

RIL_Errno radio::setDataAllowedRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_ALLOW_DATA,pRI);
}


RIL_Errno radio::setIndicationFilterRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_SET_UNSOLICITED_RESPONSE_FILTER,pRI);
}

RIL_Errno radio::setSimCardPowerRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchIntsGeneric(p,RIL_REQUEST_SET_SIM_CARD_POWER,pRI);
}

RIL_Errno dispatchSingleStringGeneric(Parcel& p, int reqId, RequestInfo *pRI)
{
    RIL_Errno result;
    char *string8 = Parcel::strdupReadString(p);
    if (string8 != NULL) {
        ACLOGW(LOG_TAG,"%s(%d): %s",  __func__, reqId, string8);
        s_callbacks.onRequest(reqId, string8, sizeof(char *), pRI);
        result=RIL_E_SUCCESS;
        FREE(string8);
    } else {
        result=RIL_E_INVALID_ARGUMENTS;
        ACLOGW(LOG_TAG,"%s(%d): NULL", __func__,reqId);
    }
    return result;
}

RIL_Errno radio::sendDtmfRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchSingleStringGeneric(p,RIL_REQUEST_DTMF, pRI);
}

RIL_Errno radio::sendUssdRequest (Parcel& p, RequestInfo *pRI)
{
    return dispatchSingleStringGeneric(p,RIL_REQUEST_SEND_USSD, pRI);
}

RIL_Errno radio::setNetworkSelectionModeManualRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchSingleStringGeneric(p,RIL_REQUEST_SET_NETWORK_SELECTION_MANUAL, pRI);
}

RIL_Errno radio::startDtmfRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchSingleStringGeneric(p,RIL_REQUEST_DTMF_START, pRI);
}

RIL_Errno radio::sendEnvelopeRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchSingleStringGeneric(p,RIL_REQUEST_STK_SEND_ENVELOPE_COMMAND, pRI);
}

RIL_Errno radio::sendTerminalResponseToSimRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchSingleStringGeneric(p,RIL_REQUEST_STK_SEND_TERMINAL_RESPONSE, pRI);
}

RIL_Errno radio::setSmscAddressRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchSingleStringGeneric(p,RIL_REQUEST_SET_SMSC_ADDRESS, pRI);
}

RIL_Errno radio::requestIsimAuthenticationRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchSingleStringGeneric(p,RIL_REQUEST_ISIM_AUTHENTICATION, pRI);
}

RIL_Errno radio::sendEnvelopeWithStatusRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchSingleStringGeneric(p,RIL_REQUEST_STK_SEND_ENVELOPE_WITH_STATUS, pRI);
}

RIL_Errno radio::startLceServiceRequest (Parcel& p, RequestInfo *pRI)
{
    int32_t dataNum;
    status_t status;
    RIL_LceStatusInfo lsi;
    RIL_Errno result=RIL_E_SUCCESS;

    ACLOGV(LOG_TAG,"%s..",__func__);
    do {
        status = p.readInt32 (&dataNum);

        if (status != NO_ERROR) {
            ACLOGW(LOG_TAG,"%s: could not read data number",  __func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (dataNum !=2 ) { //written via generic ints command with 2 ints values !()
            ACLOGW(LOG_TAG,"%s:  data number mismatch: 2/%d",  __func__,dataNum);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32 ((int *)&lsi.lce_status);

        if (status != NO_ERROR) {
            ACLOGW(LOG_TAG,"%s: could not read lce_status",  __func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32 ((int *)&lsi.actual_interval_ms);

        if (status != NO_ERROR) {
            ACLOGW(LOG_TAG,"%s: could not read actual_interval_ms",  __func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        result =RIL_E_SUCCESS;
    } while (0);

    if (result==RIL_E_SUCCESS) {
        s_callbacks.onRequest(pRI->pCI->requestNumber, &lsi, sizeof(lsi), pRI);
    } else {
        invalidCommandBlock(pRI);
    }
    return result;
}

RIL_Errno radio::iccOpenLogicalChannelRequest(Parcel& p, RequestInfo *pRI)
{
    int32_t dataNum;
    status_t status;
    size_t datalen;
    RIL_Errno result=RIL_E_SUCCESS;
    RIL_OpenChannelParams ocp = {};

    ACLOGV(LOG_TAG,"%s..",__func__);
    do {
        status = p.readInt32 (&dataNum);

        if (status != NO_ERROR) {
            ACLOGW(LOG_TAG,"%s: could not read data number",  __func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (dataNum !=1 ) {
            ACLOGW(LOG_TAG,"%s:  data number mismatch: 1/%d",  __func__,dataNum);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        ocp.aidPtr = Parcel::strdupReadString(p);
        status = p.readInt32 (&ocp.p2);
        if (status != NO_ERROR) {
            ACLOGW(LOG_TAG,"%s: could not read p2 ",  __func__);
            if (ocp.aidPtr){
                FREE(ocp.aidPtr);
                ocp.aidPtr=0;
            }
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        result =RIL_E_SUCCESS;
    } while (0);

    if (result==RIL_E_SUCCESS) {
        s_callbacks.onRequest(pRI->pCI->requestNumber, &ocp, sizeof(ocp), pRI);
    } else {
        invalidCommandBlock(pRI);
    }
    return result;
}


/** Callee expects const char ** */
RIL_Errno dispatchStringsGeneric (Parcel &p, int reqNumber, RequestInfo *pRI)
{
    int32_t countStrings;
    status_t status;
    size_t datalen;
    char **pStrings=NULL;
    RIL_Errno result=RIL_E_SUCCESS;

    ACLOGV(LOG_TAG,"%s..",__func__);
    do {
        status = p.readInt32 (&countStrings);

        if (status != NO_ERROR) {
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        if (((int)countStrings) == -1) {
            pStrings = NULL;
            datalen = 0;
        } else {
            if (countStrings == 0) {
                // just some non-null pointer
                pStrings = (char **)MALLOC(sizeof(char *));
                datalen = 0;
            } else {
                datalen = sizeof(char *) * countStrings;
                pStrings = (char **)MALLOC(datalen);
            }

            if (pStrings==NULL){
                result=RIL_E_MISSING_RESOURCE;
                break;
            }
        }

        ACLOGV(LOG_TAG,"%s(%d): number=%d ", __func__, pRI->pCI->requestNumber,countStrings);
        int i;
        for (i = 0 ; i < countStrings ; i++) {
            pStrings[i] =NULL;
        }

        for (i = 0 ; i < countStrings ; i++) {
            pStrings[i] = Parcel::strdupReadString(p);
            if (pStrings[i]!=NULL){
                ACLOGV(LOG_TAG,"%s[%d]: %s ", __func__, i,pStrings[i]);
            } else {
                ACLOGV(LOG_TAG,"%s[%d]: NULL ", __func__, i);
            }
        }

    } while (0);

    if (result==RIL_E_SUCCESS) {
        s_callbacks.onRequest(pRI->pCI->requestNumber, pStrings, datalen, pRI);
    } else {
        invalidCommandBlock(pRI);
    }

    if (pStrings != NULL) {
        for (int i = 0 ; i < countStrings ; i++) {
            if (pStrings[i]!=NULL) {
                FREE(pStrings[i]);
            }
        }
        FREE(pStrings);
    }
    return result;
}

RIL_Errno radio::supplyIccPinForAppRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchStringsGeneric(p,RIL_REQUEST_ENTER_SIM_PIN,pRI);
}

RIL_Errno radio::supplyIccPukForAppRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchStringsGeneric(p,RIL_REQUEST_ENTER_SIM_PUK,pRI);
}

RIL_Errno radio::supplyIccPin2ForAppRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchStringsGeneric(p,RIL_REQUEST_ENTER_SIM_PIN2,pRI);
}

RIL_Errno radio::supplyIccPuk2ForAppRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchStringsGeneric(p,RIL_REQUEST_ENTER_SIM_PUK2,pRI);
}

RIL_Errno radio::changeIccPinForAppRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchStringsGeneric(p,RIL_REQUEST_CHANGE_SIM_PIN,pRI);
}

RIL_Errno radio::changeIccPin2ForAppRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchStringsGeneric(p,RIL_REQUEST_CHANGE_SIM_PIN2,pRI);
}

RIL_Errno radio::supplyNetworkDepersonalizationRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchStringsGeneric(p,RIL_REQUEST_ENTER_NETWORK_DEPERSONALIZATION,pRI);
}

RIL_Errno radio::getIMSIForAppRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchStringsGeneric(p,RIL_REQUEST_GET_IMSI,pRI);
}

RIL_Errno radio::deactivateDataCallRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchStringsGeneric(p,RIL_REQUEST_DEACTIVATE_DATA_CALL,pRI);
}

RIL_Errno radio::getFacilityLockForAppRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchStringsGeneric(p,RIL_REQUEST_QUERY_FACILITY_LOCK,pRI);
}

RIL_Errno radio::setFacilityLockForAppRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchStringsGeneric(p,RIL_REQUEST_SET_FACILITY_LOCK,pRI);
}

RIL_Errno radio::setBarringPasswordRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchStringsGeneric(p,RIL_REQUEST_CHANGE_BARRING_PASSWORD,pRI);
}

RIL_Errno radio::oemHookStringsRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchStringsGeneric(p,RIL_REQUEST_OEM_HOOK_STRINGS,pRI);
}

RIL_Errno radio::acknowledgeIncomingGsmSmsWithPduReqquest(Parcel& p, RequestInfo *pRI)
{
    return dispatchStringsGeneric(p,RIL_REQUEST_ACKNOWLEDGE_INCOMING_GSM_SMS_WITH_PDU,pRI);
}

int readDataProfileWithVersion (Parcel& p, RIL_DataProfileInfo_v15&  profile, int version)
{
    int invalidData=-1;
    status_t status;

    do {
        //retrieve required data profile elements
        profile.apn=0;
        profile.protocol=0;
        profile.password=0;
        profile.user=0;
        profile.roamingProtocol=0;
        profile.mvnoType=0;
        profile.mvnoMatchData=0;

        status = p.readInt32(&profile.profileId);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read profileId",__func__);
            break;
        }
        profile.apn = Parcel::strdupReadString(p);
        profile.protocol = Parcel::strdupReadString(p);

        if (version >= 15){
            profile.roamingProtocol = Parcel::strdupReadString(p);
        }

        status = p.readInt32(&profile.authType);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read authType",__func__);
            break;
        }
        profile.user = Parcel::strdupReadString(p);
        profile.password = Parcel::strdupReadString(p);

        status = p.readInt32(&profile.type);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read type",__func__);
            break;
        }

        status = p.readInt32(&profile.maxConnsTime);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read maxConnsTime",__func__);
            break;
        }

        status = p.readInt32(&profile.maxConns);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read maxConns",__func__);
            break;
        }

        status = p.readInt32(&profile.waitTime);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read waitTime",__func__);
            break;
        }

        status = p.readInt32(&profile.enabled);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read enabled",__func__);
            break;
        }

        if (version >= 15){
            status = p.readInt32(&profile.supportedTypesBitmask);
            if (status != NO_ERROR){
                ACLOGW(LOG_TAG,"%s: could not read supportedTypesBitmask",__func__);
                break;
            }

            status = p.readInt32(&profile.bearerBitmask);
            if (status != NO_ERROR){
                ACLOGW(LOG_TAG,"%s: could not read bearerBitmask",__func__);
                break;
            }

            status = p.readInt32(&profile.mtu);
            if (status != NO_ERROR){
                ACLOGW(LOG_TAG,"%s: could not read mtu",__func__);
                break;
            }
            profile.mvnoType = Parcel::strdupReadString(p);
            profile.mvnoMatchData = Parcel::strdupReadString(p);
        }

        invalidData=0;
    } while (0);

    if (invalidData <0){
        if (profile.apn){ FREE(profile.apn);profile.apn=0;}
        if (profile.protocol){ FREE(profile.protocol);profile.protocol=0;}
        if (profile.user){ FREE(profile.user);profile.user=0;}
        if (profile.password){ FREE(profile.password);profile.password=0;}
        if (profile.roamingProtocol){ FREE(profile.roamingProtocol);profile.roamingProtocol=0;}
        if (profile.mvnoType){ FREE(profile.mvnoType);profile.mvnoType=0;}
        if (profile.mvnoMatchData){ FREE(profile.mvnoMatchData);profile.mvnoMatchData=0;}
    }
    return invalidData;
}

void freeSetDataProfileData(int numDataProfiles, RIL_DataProfileInfo **dataProfilePtrs)
{

    RIL_DataProfileInfo *dataProfile=0;

    for (int j = 0; j < numDataProfiles; j++) {
        dataProfile = dataProfilePtrs[j];
        if (dataProfile){
            if (dataProfile->apn){
                FREE(dataProfile->apn);
                (dataProfile->apn)=0;
            }
            if (dataProfile->protocol){
                FREE(dataProfile->protocol);
                (dataProfile->protocol)=0;
            }
            if (dataProfile->user){
                FREE(dataProfile->user);
                (dataProfile->user)=0;
            }
            if (dataProfile->user){
                FREE(dataProfile->password);
                (dataProfile->password)=0;
            }
            FREE(dataProfile);
            dataProfile=0;
        }
    }
    FREE(dataProfilePtrs);
}

void freeSetDataProfileData_v15(int numDataProfiles, RIL_DataProfileInfo_v15 **dataProfilePtrs)
{

    RIL_DataProfileInfo_v15 *dataProfile=0;

    for (int j = 0; j < numDataProfiles; j++) {
        dataProfile = dataProfilePtrs[j];
        if (dataProfile){
            if (dataProfile->apn){
                FREE(dataProfile->apn);
                (dataProfile->apn)=0;
            }
            if (dataProfile->protocol){
                FREE(dataProfile->protocol);
                (dataProfile->protocol)=0;
            }
            if (dataProfile->user){
                FREE(dataProfile->user);
                (dataProfile->user)=0;
            }
            if (dataProfile->user){
                FREE(dataProfile->password);
                (dataProfile->password)=0;
            }
            if (dataProfile->roamingProtocol){
                FREE(dataProfile->roamingProtocol);
                (dataProfile->roamingProtocol)=0;
            }
            if (dataProfile->mvnoMatchData){
                FREE(dataProfile->mvnoMatchData);
                (dataProfile->mvnoMatchData)=0;
            }

            FREE(dataProfile);
            dataProfile=0;
        }
    }
    FREE(dataProfilePtrs);
}


RIL_Errno  radio::setDataProfileRequest (Parcel& p, RequestInfo *pRI)
{
    status_t status;
    char *inputData;
    RIL_Errno result = RIL_E_SUCCESS;
    int num;
    RIL_DataProfileInfo **dataProfilePtrs=0;
    RIL_DataProfileInfo *dataProfiles=0;
    RIL_DataProfileInfo_v15  **dataProfilePtrs_v15=0;
    RIL_DataProfileInfo_v15 *dataProfiles_v15=0;

    do {
        status = p.readInt32(&num);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read num",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        if (s_callbacks.version <= 14) {
            dataProfiles = (RIL_DataProfileInfo *) MALLOC(num * sizeof(RIL_DataProfileInfo));
            if (dataProfiles == 0){
                ACLOGW(LOG_TAG,"%s: could not allocate profiles",__func__);
                result=RIL_E_MISSING_RESOURCE;
                break;
            }
            memset(dataProfiles,0,num * sizeof(RIL_DataProfileInfo));

            dataProfilePtrs = (RIL_DataProfileInfo **) MALLOC(num * sizeof(RIL_DataProfileInfo *));
            if (dataProfilePtrs == 0){
                ACLOGW(LOG_TAG,"%s: could not allocate profiles pointers",__func__);
                result=RIL_E_MISSING_RESOURCE;
                break;
            }
            memset(dataProfilePtrs,0,num * sizeof(RIL_DataProfileInfo *));
            for (size_t i = 0; i < num && result  == RIL_E_SUCCESS; i++) {
                dataProfilePtrs[i] = &dataProfiles[i];
                status = p.readInt32(&dataProfiles[i].profileId);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read profileId[%d]",__func__,i);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }

                dataProfiles[i].apn = Parcel::strdupReadString(p);
                dataProfiles[i].protocol = Parcel::strdupReadString(p);

                status = p.readInt32(&dataProfiles[i].authType);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read authType[%d]",__func__,i);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }

                dataProfiles[i].user = Parcel::strdupReadString(p);
                dataProfiles[i].password = Parcel::strdupReadString(p);

                status = p.readInt32(&dataProfiles[i].type);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read type[%d]",__func__,i);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }

                status = p.readInt32(&dataProfiles[i].maxConnsTime);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read maxConnsTime[%d]",__func__,i);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }

                status = p.readInt32(&dataProfiles[i].maxConns);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read maxConns[%d]",__func__,i);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }

                status = p.readInt32(&dataProfiles[i].waitTime);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read waitTime[%d]",__func__,i);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }

                status = p.readInt32(&dataProfiles[i].enabled);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read enabled[%d]",__func__,i);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }
            }
            if (result !=RIL_E_SUCCESS) {
                break;
            }

            s_callbacks.onRequest(RIL_REQUEST_SET_DATA_PROFILE, dataProfilePtrs, num * sizeof(RIL_DataProfileInfo *), pRI);
        } else {
            dataProfiles_v15 = (RIL_DataProfileInfo_v15 *) MALLOC(num * sizeof(RIL_DataProfileInfo_v15));
            if (dataProfiles_v15 == NULL){
                ACLOGW(LOG_TAG,"%s: could not allocate profiles",__func__);
                result=RIL_E_MISSING_RESOURCE;
                break;
            }
            memset(dataProfiles_v15,0,num * sizeof(RIL_DataProfileInfo_v15));

            dataProfilePtrs_v15 = (RIL_DataProfileInfo_v15 **) MALLOC(num * sizeof(RIL_DataProfileInfo_v15 *));
            if (dataProfilePtrs_v15 == NULL){
                ACLOGW(LOG_TAG,"%s: could not allocate profiles pointers",__func__);
                result=RIL_E_MISSING_RESOURCE;
                break;
            }
            memset(dataProfilePtrs_v15,0,num * sizeof(RIL_DataProfileInfo_v15*));

            for (size_t i = 0; i < num && result  == RIL_E_SUCCESS; i++) {
                dataProfilePtrs_v15[i] = &dataProfiles_v15[i];
                status = p.readInt32(&dataProfiles_v15[i].profileId);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read profileId[%d]",__func__,i);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }

                dataProfiles_v15[i].apn = Parcel::strdupReadString(p);
                dataProfiles_v15[i].protocol = Parcel::strdupReadString(p);
                dataProfiles_v15[i].roamingProtocol = Parcel::strdupReadString(p);

                status = p.readInt32(&dataProfiles_v15[i].authType);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read authType[%d]",__func__,i);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }

                dataProfiles_v15[i].user = Parcel::strdupReadString(p);
                dataProfiles_v15[i].password = Parcel::strdupReadString(p);

                status = p.readInt32(&dataProfiles_v15[i].type);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read type[%d]",__func__,i);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }

                status = p.readInt32(&dataProfiles_v15[i].maxConnsTime);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read maxConnsTime[%d]",__func__,i);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }

                status = p.readInt32(&dataProfiles_v15[i].maxConns);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read maxConns[%d]",__func__,i);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }

                status = p.readInt32(&dataProfiles_v15[i].waitTime);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read waitTime[%d]",__func__,i);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }

                status = p.readInt32(&dataProfiles_v15[i].enabled);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read enabled[%d]",__func__,i);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }

                status = p.readInt32(&dataProfiles_v15[i].supportedTypesBitmask);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read supportedTypesBitmask[%d]",__func__,i);
                    break;
                }

                status = p.readInt32(&dataProfiles_v15[i].bearerBitmask);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read bearerBitmask[%d]",__func__,i);
                    break;
                }

                status = p.readInt32(&dataProfiles_v15[i].mtu);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read mtu[%d]",__func__,i);
                    break;
                }
                dataProfiles_v15[i].mvnoType = Parcel::strdupReadString(p);
                dataProfiles_v15[i].mvnoMatchData = Parcel::strdupReadString(p);
            }
            if (result !=RIL_E_SUCCESS) {
                break;
            }

            s_callbacks.onRequest(RIL_REQUEST_SET_DATA_PROFILE, dataProfilePtrs_v15, num * sizeof(RIL_DataProfileInfo_v15 *), pRI);
        }
    } while (0);

    //cleanup;
    if (dataProfilePtrs){
        freeSetDataProfileData(num, dataProfilePtrs);
    }

    if (dataProfilePtrs_v15){
        freeSetDataProfileData_v15(num, dataProfilePtrs_v15);
    }


    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}

RIL_Errno radio::setInitialAttachApnRequest (Parcel& p, RequestInfo *pRI)
{
    status_t status;
    char *inputData;
    RIL_Errno result = RIL_E_SUCCESS;
    int dataNum;

    do {
        status = p.readInt32((int *)&dataNum);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read data number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (dataNum !=  1) {
            ACLOGW(LOG_TAG,"%s: data number mismatch: 1/%d",__func__,dataNum);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        //retrieve required data profile elements
        if (s_callbacks.version <= 14) {
            RIL_InitialAttachApn iaa={};
            iaa.apn= Parcel::strdupReadString(p);
            iaa.protocol=Parcel::strdupReadString(p);
            inputData=Parcel::strdupReadString(p);
            if (inputData){
                sscanf(inputData,"%d",&iaa.authtype);
                FREE(inputData);inputData=0;
            } else {
                iaa.authtype=0;
            }
            iaa.username=Parcel::strdupReadString(p);
            iaa.password=Parcel::strdupReadString(p);

            s_callbacks.onRequest(RIL_REQUEST_SET_INITIAL_ATTACH_APN, &iaa, sizeof(iaa), pRI);

            if (iaa.apn){ FREE(iaa.apn);iaa.apn=0;}
            if (iaa.protocol){ FREE(iaa.protocol);iaa.protocol=0;}
            if (iaa.username){ FREE(iaa.username);iaa.username=0;}
            if (iaa.password){ FREE(iaa.password);iaa.password=0;}

        } else {
            RIL_InitialAttachApn_v15 iaa = {};
            iaa.apn= Parcel::strdupReadString(p);
            iaa.protocol=Parcel::strdupReadString(p);
            iaa.roamingProtocol=Parcel::strdupReadString(p);
            inputData=Parcel::strdupReadString(p);
            if (inputData){
                sscanf(inputData,"%d",&iaa.authtype);
                FREE(inputData);inputData=0;
            } else {
                iaa.authtype=0;
            }
            iaa.username=Parcel::strdupReadString(p);
            iaa.password=Parcel::strdupReadString(p);
            inputData=Parcel::strdupReadString(p);
            if (inputData){
                sscanf(inputData,"%d",&iaa.supportedTypesBitmask);
                FREE(inputData);inputData=0;
            } else {
                iaa.supportedTypesBitmask=0;
            }
            if (inputData){
                sscanf(inputData,"%d",&iaa.bearerBitmask);
                FREE(inputData);inputData=0;
            } else {
                iaa.bearerBitmask=0;
            }
            if (inputData){
                sscanf(inputData,"%d",&iaa.modemCognitive);
                FREE(inputData);inputData=0;
            } else {
                iaa.modemCognitive=0;
            }
            if (inputData){
                sscanf(inputData,"%d",&iaa.mtu);
                FREE(inputData);inputData=0;
            } else {
                iaa.mtu=0;
            }
            iaa.mvnoType=Parcel::strdupReadString(p);
            iaa.mvnoMatchData=Parcel::strdupReadString(p);

            s_callbacks.onRequest(RIL_REQUEST_SET_INITIAL_ATTACH_APN, &iaa, sizeof(iaa), pRI);

            if (iaa.apn){ FREE(iaa.apn);iaa.apn=0;}
            if (iaa.protocol){ FREE(iaa.protocol);iaa.protocol=0;}
            if (iaa.username){ FREE(iaa.username);iaa.username=0;}
            if (iaa.password){ FREE(iaa.password);iaa.password=0;}
            if (iaa.roamingProtocol){ FREE(iaa.roamingProtocol);iaa.roamingProtocol=0;}
            if (iaa.mvnoType){ FREE(iaa.mvnoType);iaa.mvnoType=0;}
            if (iaa.mvnoMatchData){ FREE(iaa.mvnoMatchData);iaa.mvnoMatchData=0;}
        }
    } while (0);

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}

static RIL_Errno dispatchSendSmsGeneric (Parcel& p, const int requestId, RequestInfo *pRI)
{
    int32_t t;
    int32_t countStrings;
    status_t status;
    size_t datalen;
    char **pStrings = NULL;
    RIL_Errno result = RIL_E_SUCCESS;
    int idx;

    RIL_ByteArray smsByteArrays[2];
    RIL_ByteArray *p_smsc = & smsByteArrays[0];
    RIL_ByteArray *p_pdu = & smsByteArrays[1];


    ACLOGD(LOG_TAG, "%s,%d",__func__, __LINE__ );
    do {
        //number value added to be compliant to global format
        status = p.readInt32(&countStrings);
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not read data number",__func__);
            result = RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if ( countStrings != 2 ) {
            ACLOGW(LOG_TAG, "%s: number mismatch: 1/%d",__func__,countStrings);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        memset( p_smsc, 0, sizeof(RIL_ByteArray) );
        memset( p_pdu, 0, sizeof(RIL_ByteArray) );

        //retrieve smsc address first
        status = p.readInt32( & p_smsc->len );
        p_smsc->data = Parcel::strdupReadByteArray( p, p_smsc->len );

        // retrieve pdu second
        status = p.readInt32( & p_pdu->len );
        p_pdu->data = Parcel::strdupReadByteArray( p, p_pdu->len );

        s_callbacks.onRequest( requestId, smsByteArrays, 2 * sizeof( RIL_ByteArray ), pRI );

    } while (0);

    for( int idx=0; idx < 2; idx++ ) {
      if ( smsByteArrays[idx].data ) {
        FREE( smsByteArrays[idx].data );
      }
    }

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }

    return result;
}

RIL_Errno radio::sendSmsRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchSendSmsGeneric (p, RIL_REQUEST_SEND_SMS, pRI);
}

RIL_Errno radio::sendSMSExpectMoreRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchSendSmsGeneric (p, RIL_REQUEST_SEND_SMS_EXPECT_MORE, pRI);
}

// comment: not implemented yet on qmi site, 2023-03-01@peiker/wsc
RIL_Errno  radio::sendImsSmsRequest (Parcel& p, RequestInfo *pRI)
{
    RIL_IMS_SMS_Message rism = {};

    status_t status;
    char *inputData;
    char **pStrings=0;
    int countStrings = 2;
    int inputParam;
    int dataLen = sizeof(char *) * countStrings;
    RIL_Errno result=RIL_E_SUCCESS;
    int dataNum;

    do {
        status = p.readInt32((int *)&dataNum);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read data number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (dataNum !=  1) {
            ACLOGW(LOG_TAG,"%s: data number mismatch: 1/%d",__func__,dataNum);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&rism.tech);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read tech",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&rism.retry);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read retry",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&rism.messageRef);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read messageRef",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        pStrings = (char **)MALLOC(countStrings* sizeof(char *));
        if (pStrings == NULL) {
            ACLOGE(LOG_TAG,"%s : Memory allocation failed for request %s",__func__,requestToString(pRI->pCI->requestNumber));
            result=RIL_E_MISSING_RESOURCE;
            break;
        }
        memset(pStrings,0,countStrings* sizeof(char *));
        pStrings[0]=Parcel::strdupReadString(p);
        pStrings[1]=Parcel::strdupReadString(p);

        rism.message.gsmMessage=pStrings;

        s_callbacks.onRequest(RIL_REQUEST_IMS_SEND_SMS, &rism,  sizeof(RIL_RadioTechnologyFamily) +
                                            sizeof(uint8_t) + sizeof(int32_t) + dataLen, pRI);

    } while (0);

    if (pStrings){
        if (pStrings[0]) {FREE(pStrings[0]);};
        if (pStrings[1]) {FREE(pStrings[1]);};
        FREE(pStrings);
    }

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}

RIL_Errno  radio::nvReadItemRequest (Parcel& p, RequestInfo *pRI)
{
    status_t status;
    RIL_Errno result=RIL_E_SUCCESS;
    RIL_NV_ReadItem nvri = {};
    int dataNum;

    do{
        status = p.readInt32((int *)&dataNum);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read data number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (dataNum!=1){
            ACLOGW(LOG_TAG,"%s: data number mismatch: 1/%d",__func__,dataNum);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&nvri.itemID);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read itemID",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        s_callbacks.onRequest(RIL_REQUEST_NV_READ_ITEM, &nvri, sizeof(nvri), pRI);

    } while (0);

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;

}

RIL_Errno radio::nvWriteItemRequest (Parcel& p, RequestInfo *pRI)
{
    status_t status;
    RIL_Errno result=RIL_E_SUCCESS;
    RIL_NV_WriteItem nvwi = {};
    int dataNum;

    do{
        status = p.readInt32((int *)&dataNum);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read data number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (dataNum!=1){
            ACLOGW(LOG_TAG,"%s: data number mismatch: 1/%d",__func__,dataNum);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&nvwi.itemID);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read itemID",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        nvwi.value=Parcel::strdupReadString(p);

        s_callbacks.onRequest(RIL_REQUEST_NV_WRITE_ITEM, &nvwi, sizeof(nvwi), pRI);

    } while (0);

    if  (nvwi.value){
        FREE(nvwi.value);
    }

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}

RIL_Errno radio::setUiccSubscriptionRequest (Parcel& p, RequestInfo *pRI)
{
    status_t status;
    RIL_Errno result=RIL_E_SUCCESS;
    RIL_SelectUiccSub rilUiccSub = {};
    int dataNum;

    do{
        status = p.readInt32((int *)&dataNum);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read data number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (dataNum!=1){
            ACLOGW(LOG_TAG,"%s: data number mismatch: 1/%d",__func__,dataNum);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&rilUiccSub.slot);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read slot",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&rilUiccSub.app_index);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read app_index",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&rilUiccSub.sub_type);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read sub_type",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&rilUiccSub.act_status);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read sub_type",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        s_callbacks.onRequest(RIL_REQUEST_SET_UICC_SUBSCRIPTION, &rilUiccSub, sizeof(rilUiccSub), pRI);

    } while (0);

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}

RIL_Errno radio::requestIccSimAuthenticationRequest (Parcel& p, RequestInfo *pRI)
{
    status_t status;
    RIL_Errno result=RIL_E_SUCCESS;
    RIL_SimAuthentication pf = {};
    int dataNum;

    do{
        status = p.readInt32((int *)&dataNum);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read data number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (dataNum!=1){
            ACLOGW(LOG_TAG,"%s: data number mismatch: 1/%d",__func__,dataNum);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&pf.authContext);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read authContext",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        pf.authData=Parcel::strdupReadString(p);
        pf.aid=Parcel::strdupReadString(p);

        s_callbacks.onRequest(RIL_REQUEST_SIM_AUTHENTICATION, &pf, sizeof(pf), pRI);

    } while (0);

    if  (pf.authData){
        FREE(pf.authData);
    }
    if  (pf.aid){
        FREE(pf.aid);
    }

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}

RIL_Errno radio::setRadioCapabilityRequest (Parcel& p, RequestInfo *pRI)
{
    status_t status;
    RIL_Errno result=RIL_E_SUCCESS;
    RIL_RadioCapability rilRc = {};
    int dataNum;

    do{
        status = p.readInt32((int *)&dataNum);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read data number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (dataNum!=1){
            ACLOGW(LOG_TAG,"%s: data number mismatch: 1/%d",__func__,dataNum);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&rilRc.version);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read version",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&rilRc.session);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read session",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&rilRc.phase);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read phase",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&rilRc.rat);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read rat",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status=p.read(rilRc.logicalModemUuid, sizeof(rilRc.logicalModemUuid));
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read logicalModemUuid",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&rilRc.status);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read status",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        s_callbacks.onRequest(RIL_REQUEST_SET_RADIO_CAPABILITY, &rilRc, sizeof(rilRc), pRI);

    } while (0);

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}

RIL_Errno radio::setAllowedCarriersRequest (Parcel& p, RequestInfo *pRI)
{
    status_t status;
    RIL_Errno result=RIL_E_SUCCESS;
    RIL_CarrierRestrictions cr = {};
    RIL_Carrier *allowedCarriers = 0;
    RIL_Carrier *excludedCarriers = 0;
    int dataNum;

    do{
        status = p.readInt32((int *)&dataNum);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read data number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (dataNum!=1){
            ACLOGW(LOG_TAG,"%s: data number mismatch: 1/%d",__func__,dataNum);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&cr.len_allowed_carriers);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read len_allowed_carriers",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&cr.len_excluded_carriers);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read len_excluded_carriers",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (cr.len_allowed_carriers >0) {
            allowedCarriers = (RIL_Carrier *) MALLOC(cr.len_allowed_carriers * sizeof(RIL_Carrier));
            if (allowedCarriers == 0) {
                ALOGE(LOG_TAG,"%s: Memory allocation failed for request %s",__func__,requestToString(pRI->pCI->requestNumber));
                result=RIL_E_MISSING_RESOURCE;
                break;
            }
            memset(allowedCarriers ,0, cr.len_allowed_carriers * sizeof(RIL_Carrier));
        }
        cr.allowed_carriers = allowedCarriers;

        if (cr.len_excluded_carriers >0) {
            excludedCarriers = (RIL_Carrier *) MALLOC(cr.len_excluded_carriers * sizeof(RIL_Carrier));
            if (excludedCarriers == 0){
                ALOGE(LOG_TAG,"%s: Memory allocation failed for request %s",__func__,requestToString(pRI->pCI->requestNumber));
                result=RIL_E_MISSING_RESOURCE;
                break;
            }
            memset(excludedCarriers ,0, cr.len_excluded_carriers * sizeof(RIL_Carrier));
        }
        cr.excluded_carriers = excludedCarriers;

        for (int i = 0; i < cr.len_allowed_carriers; i++) {
            allowedCarriers[i].mcc = Parcel::strdupReadString(p);
            allowedCarriers[i].mnc = Parcel::strdupReadString(p);
            status = p.readInt32((int *)&allowedCarriers[i].match_type);
            if (status != NO_ERROR){
                ACLOGW(LOG_TAG,"%s: could not read allowedCarriers[%d] match_type",__func__,i);
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }
            allowedCarriers[i].match_data = Parcel::strdupReadString(p);
        }
        if (result !=RIL_E_SUCCESS){
            break;
        }

        for (int i = 0; i < cr.len_excluded_carriers; i++) {
            excludedCarriers[i].mcc = Parcel::strdupReadString(p);
            excludedCarriers[i].mnc = Parcel::strdupReadString(p);
            status = p.readInt32((int *)&excludedCarriers[i].match_type);
            if (status != NO_ERROR){
                ACLOGW(LOG_TAG,"%s: could not read excludedCarriers[%d] match_type",__func__,i);
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }
            excludedCarriers[i].match_data = Parcel::strdupReadString(p);
        }
        if (result !=RIL_E_SUCCESS){
            break;
        }

        s_callbacks.onRequest(RIL_REQUEST_SET_CARRIER_RESTRICTIONS, &cr, sizeof(RIL_CarrierRestrictions), pRI);

    } while (0);

    //resource cleanup
    if (allowedCarriers){
        for (int i = 0; i < cr.len_allowed_carriers; i++) {
            if (allowedCarriers[i].mcc){
                FREE(allowedCarriers[i].mcc);
                allowedCarriers[i].mcc=0;
            }
            if (allowedCarriers[i].mnc){
                FREE(allowedCarriers[i].mnc);
                allowedCarriers[i].mnc=0;
            }
            if (allowedCarriers[i].match_data){
                FREE(allowedCarriers[i].match_data);
                allowedCarriers[i].match_data=0;
            }
        }
        FREE(allowedCarriers);
    }

    if (excludedCarriers){
        for (int i = 0; i < cr.len_excluded_carriers; i++) {
            if (excludedCarriers[i].mcc){
                FREE(excludedCarriers[i].mcc);
                excludedCarriers[i].mcc=0;
            }
            if (excludedCarriers[i].mnc){
                FREE(excludedCarriers[i].mnc);
                excludedCarriers[i].mnc=0;
            }
            if (excludedCarriers[i].match_data){
                FREE(excludedCarriers[i].match_data);
                excludedCarriers[i].match_data=0;
            }
        }
        FREE(excludedCarriers);
    }

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}

RIL_Errno radio::startKeepaliveRequest (Parcel& p, RequestInfo *pRI)
{
    status_t status;
    RIL_Errno result=RIL_E_SUCCESS;
    RIL_KeepaliveRequest kaReq = {};
   int dataNum;

    do{
        status = p.readInt32((int *)&dataNum);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read data number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (dataNum!=1){
            ACLOGW(LOG_TAG,"%s: data number mismatch: 1/%d",__func__,dataNum);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&kaReq.type);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read type",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        if (kaReq.type == NATT_IPV4){
            status = p.read(&kaReq.sourceAddress,4);
        } else if (kaReq.type == NATT_IPV6){
            status = p.read(&kaReq.sourceAddress,16);
        } else {
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read source address",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&kaReq.sourcePort);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read source port",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        if (kaReq.type == NATT_IPV4){
            status = p.read(&kaReq.destinationAddress,4);
        } else if (kaReq.type == NATT_IPV6){
            status = p.read(&kaReq.destinationAddress,16);
        } else {
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read destination address",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        status = p.readInt32((int *)&kaReq.destinationPort);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read destination port",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&kaReq.maxKeepaliveIntervalMillis);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read maxKeepaliveIntervalMillis",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&kaReq.cid);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read cid",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        s_callbacks.onRequest(RIL_REQUEST_START_KEEPALIVE, &kaReq, sizeof(RIL_KeepaliveRequest), pRI);

    } while (0);


    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}

RIL_Errno radio::startNetworkScanRequest (Parcel& p, RequestInfo *pRI)
{
    status_t status;
    RIL_Errno result=RIL_E_SUCCESS;
    RIL_NetworkScanRequest scanReq = {};
    int dataNum;

    do{
        status = p.readInt32((int *)&dataNum);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read data number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (dataNum!=1){
            ACLOGW(LOG_TAG,"%s: data number mismatch: 1/%d",__func__,dataNum);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&scanReq.type);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read type",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&scanReq.interval);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read interval",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32((int *)&scanReq.specifiers_length);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read specifiers_length",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        for (int i=0;i<scanReq.specifiers_length; i++){
            status = p.readInt32((int *)&scanReq.specifiers[i].radio_access_network);
            if (status != NO_ERROR){
                ACLOGW(LOG_TAG,"%s: could not read radio_access_network[%d]",__func__,i);
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }
            status = p.readInt32((int *)&scanReq.specifiers[i].bands_length);
            if (status != NO_ERROR){
                ACLOGW(LOG_TAG,"%s: could not read band_length[%d]",__func__,i);
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }
            for (int j=0;j<scanReq.specifiers[i].bands_length; j++){
                if (scanReq.specifiers[i].radio_access_network == GERAN ){
                    status = p.readInt32((int *)&scanReq.specifiers[i].bands.geran_bands[j]);
                } else if (scanReq.specifiers[i].radio_access_network == UTRAN ) {
                    status = p.readInt32((int *)&scanReq.specifiers[i].bands.utran_bands[j]);
                } else if (scanReq.specifiers[i].radio_access_network == EUTRAN ) {
                    status = p.readInt32((int *)&scanReq.specifiers[i].bands.eutran_bands[j]);
                } else {
                    //TODO 5G support
                    result=RIL_E_INVALID_ARGUMENTS;
                    ACLOGW(LOG_TAG,"%s: unsupported rat type[%d]",__func__,i);
                    break;
                }
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read band_length[%d]",__func__,i);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }
            }
            if (status != NO_ERROR){
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }

            status = p.readInt32((int *)&scanReq.specifiers[i].channels_length);
            if (status != NO_ERROR){
                ACLOGW(LOG_TAG,"%s: could not read channels_length[%d]",__func__,i);
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }
            for (int j=0;j<scanReq.specifiers[i].channels_length && status == NO_ERROR ; j++){
                status = p.readInt32((int *)&scanReq.specifiers[i].channels[j]);
                if (status != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not read channels[%d]",__func__,j);
                    result=RIL_E_INVALID_ARGUMENTS;
                    break;
                }
            }
            if (status != NO_ERROR){
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }

        }
        s_callbacks.onRequest(RIL_REQUEST_START_NETWORK_SCAN, &scanReq, sizeof(scanReq), pRI);

    } while (0);


    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}

RIL_Errno radio::setCarrierInfoForImsiEncryptionRequest (Parcel& p, RequestInfo *pRI)
{
    status_t status;
    RIL_Errno result=RIL_E_SUCCESS;
    RIL_CarrierInfoForImsiEncryption imsiEncryption = {};
    int dataNum;

    do{
        status = p.readInt32((int *)&dataNum);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read data number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (dataNum!=1){
            ACLOGW(LOG_TAG,"%s: data number mismatch: 1/%d",__func__,dataNum);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        imsiEncryption.mcc=Parcel::strdupReadString(p);
        imsiEncryption.mnc=Parcel::strdupReadString(p);
        status = p.readInt32((int *)&imsiEncryption.carrierKeyLength);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read carrierKeyLength",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        imsiEncryption.carrierKey=(uint8_t*) MALLOC(imsiEncryption.carrierKeyLength);
        if (imsiEncryption.carrierKey==0){
            ACLOGW(LOG_TAG,"%s: could not allocate",__func__);
            result=RIL_E_MISSING_RESOURCE;
            break;
        }

        status = p.read((uint8_t *)imsiEncryption.carrierKey,imsiEncryption.carrierKeyLength);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read carrierKey",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        imsiEncryption.keyIdentifier = Parcel::strdupReadString(p);
        //note keyIdentifier == 0 might be a valid value

        status = p.readInt64((int64_t *)&imsiEncryption.expirationTime);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read expirationTime",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        s_callbacks.onRequest(RIL_REQUEST_SET_CARRIER_INFO_IMSI_ENCRYPTION, &imsiEncryption, sizeof(imsiEncryption), pRI);

    } while (0);

    if (imsiEncryption.mcc){
        FREE(imsiEncryption.mcc);
    }
    if (imsiEncryption.mnc){
        FREE(imsiEncryption.mnc);
    }
    if (imsiEncryption.carrierKey){
        FREE(imsiEncryption.carrierKey);
    }
    if (imsiEncryption.keyIdentifier){
        FREE(imsiEncryption.keyIdentifier);
    }

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;

}


/**
 * Callee expects const RIL_SMS_WriteArgs *
 * Payload is:
 *   int32_t status
 *   String pdu
 */
RIL_Errno radio::writeSmsToSimRequest (Parcel &p, RequestInfo *pRI)
{
    RIL_SMS_WriteArgs stWriteArgs;
    int32_t t;
    status_t status;
    RIL_Errno result=RIL_E_SUCCESS;
    int idx;

    memset (&stWriteArgs, 0, sizeof(stWriteArgs));

    int dataNum;

    do {
        //number value added to be compliant to global format
        status = p.readInt32(&dataNum);
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not read data number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (dataNum != 1){
            ACLOGW(LOG_TAG, "%s: number mismatch: 1/%d",__func__,dataNum);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32(&stWriteArgs.iStatus);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read status",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        //retrieve smsc data
        status = p.readInt32(&stWriteArgs.iSmscLen);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read status",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if(stWriteArgs.iSmscLen == 0){
            stWriteArgs.pSmscNum= 0;
        } else{
            stWriteArgs.pSmscNum = (unsigned char *)Parcel::strdupReadByteArray(p,stWriteArgs.iSmscLen);
        }

        // retrieve pdu data
        status = p.readInt32(&stWriteArgs.iPduLen);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read status",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if(stWriteArgs.iPduLen == 0){
            stWriteArgs.pPduData = 0;
        } else{
            stWriteArgs.pPduData = (unsigned char *)Parcel::strdupReadByteArray( p, stWriteArgs.iPduLen);
        }
        s_callbacks.onRequest(RIL_REQUEST_WRITE_SMS_TO_SIM, &stWriteArgs, sizeof(stWriteArgs), pRI);

    } while (0);

    if (stWriteArgs.pPduData) {
        FREE ((void *)stWriteArgs.pPduData);
    }
    if (stWriteArgs.pSmscNum){
        FREE((void *)stWriteArgs.pSmscNum);
    }

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}



/**
 * Callee expects const RIL_Dial *
 * Payload is:
 *   String address
 *   int32_t clir
 */
RIL_Errno radio::dialRequest (Parcel &p, RequestInfo *pRI)
{
    RIL_Dial dial;
    RIL_UUS_Info uusInfo;
    int32_t sizeOfDial;
    int32_t t;
    int32_t uusPresent;
    status_t status;
    RIL_Errno result=RIL_E_SUCCESS;

    ACLOGD(LOG_TAG, "%s..",__func__);
    memset (&dial, 0, sizeof(dial));
    int dataNum;

    do {
        //number value added to be compliant to global format
        status = p.readInt32(&dataNum);
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not read data number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (dataNum != 1){
            ACLOGW(LOG_TAG, "%s: number mismatch: 1/%d",__func__,dataNum);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        dial.address = Parcel::strdupReadString(p);
        if (dial.address==NULL){
            ACLOGW(LOG_TAG, "%s: could not get dial number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32(&t);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not get clir value",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        dial.clir = (int)t;

        status = p.readInt32(&t);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not get call type",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        dial.type = t;

        status = p.readInt32(&t);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not get ecall category",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        dial.ecall_cat = (int)t & ECALL_CAT_MASK;

        if (s_callbacks.version < 3) {
            // Remove when partners upgrade to version 3
            uusPresent = 0;
            sizeOfDial = sizeof(dial) - sizeof(RIL_UUS_Info *);
        } else {
            status = p.readInt32(&uusPresent);

            if (status != NO_ERROR) {
                ACLOGW(LOG_TAG, "%s: could not get uus present indication",__func__);
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }
        }

        if (uusPresent == 0) {
            dial.uusInfo = NULL;
        } else {
            int32_t len;
            memset(&uusInfo, 0, sizeof(RIL_UUS_Info));

            status = p.readInt32(&t);
            if (status !=NO_ERROR){
                ACLOGW(LOG_TAG, "%s: could not get uus type",__func__);
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }
            uusInfo.uusType = (RIL_UUS_Type) t;

            status = p.readInt32(&t);
            if (status !=NO_ERROR){
                ACLOGW(LOG_TAG, "%s: could not get uus dsc field",__func__);
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }
            uusInfo.uusDcs = (RIL_UUS_DCS) t;

            status = p.readInt32(&len);
            if (status != NO_ERROR) {
                ACLOGW(LOG_TAG, "%s: could not get uus data length",__func__);
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }

            // The java code writes -1 for null arrays
            if (((int) len) == -1) {
                uusInfo.uusData = NULL;
                len = 0;
            } else {
                uusInfo.uusData = (char*) p.readInplace(len);
            }

            uusInfo.uusLength = len;
            dial.uusInfo = &uusInfo;
        }

        /* log output (optional) */
        ACLOGD(LOG_TAG,"%s: number: %s",__func__,dial.address);
        ACLOGD(LOG_TAG,"%s: clir: %d",__func__,dial.clir);
        ACLOGD(LOG_TAG,"%s: type: %d",__func__,dial.type);

        if (dial.uusInfo ==NULL){
            ACLOGD(LOG_TAG,"%s: no uus info",__func__);
        } else {
            ACLOGD(LOG_TAG,"%s: uus type: %d",__func__,uusInfo.uusType);
            ACLOGD(LOG_TAG,"%s: uus dcs: %d",__func__,uusInfo.uusDcs);
            ACLOGD(LOG_TAG,"%s: uus data length: %d",__func__,uusInfo.uusLength);
            if (uusInfo.uusData==NULL){
                ACLOGD(LOG_TAG,"%s: uus data:NULL",__func__);
            } else {
                ACLOGD(LOG_TAG,"%s: uus data: %s",__func__,uusInfo.uusData);
            }
        }

        sizeOfDial = sizeof(dial);
        s_callbacks.onRequest(pRI->pCI->requestNumber, &dial, sizeOfDial, pRI);

    } while (0);

    if (dial.address !=NULL) {
        FREE (dial.address);
    }

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;

}

/**
 * Callee expects const RIL_SIM_IO *
 * Payload is:
 *   int32_t command
 *   int32_t fileid
 *   String path
 *   int32_t p1, p2, p3
 *   String data
 *   String pin2
 *   String aidPtr
 */
RIL_Errno radio::iccIOForAppRequest (Parcel &p, RequestInfo *pRI)
{
    union RIL_SIM_IO {
        RIL_SIM_IO_v6 v6;
        RIL_SIM_IO_v5 v5;
    } simIO;

    int num;
    int32_t t;
    int size;
    status_t status;
    RIL_Errno result=RIL_E_SUCCESS;

    memset (&simIO, 0, sizeof(simIO));

    do {
        //number value added to be compliant to global format
        status = p.readInt32(&num);
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not read data number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (num != 1){
            ACLOGW(LOG_TAG, "%s: number mismatch: 1/%d",__func__,num);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32(&t);
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not get command",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        simIO.v6.command = (int)t;

        status = p.readInt32(&t);
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not get file id",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        simIO.v6.fileid = (int)t;

        simIO.v6.path = Parcel::strdupReadString(p);
        if (simIO.v6.path ==NULL){
            ACLOGW(LOG_TAG, "%s: could not get file path",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32(&t);
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not read p1",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        simIO.v6.p1 = (int)t;

        status = p.readInt32(&t);
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not read p2",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        simIO.v6.p2 = (int)t;

        status = p.readInt32(&t);
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not read p3",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        simIO.v6.p3 = (int)t;

        simIO.v6.data = Parcel::strdupReadString(p);
        simIO.v6.pin2 = Parcel::strdupReadString(p);
        simIO.v6.aidPtr = Parcel::strdupReadString(p);

        ACLOGD(LOG_TAG, "%s: %d",__func__,simIO.v6.command);
        ACLOGD(LOG_TAG, "%s: file id: %d",__func__,simIO.v6.fileid);
        ACLOGD(LOG_TAG, "%s: path: %s",__func__,simIO.v6.path);
        ACLOGD(LOG_TAG, "%s: p1: %d",__func__,simIO.v6.p1);
        ACLOGD(LOG_TAG, "%s: p2: %d",__func__,simIO.v6.p2);
        ACLOGD(LOG_TAG, "%s: p3: %d",__func__,simIO.v6.p3);

        if (simIO.v6.data==NULL){
            ACLOGD(LOG_TAG, "%s: data: NULL",__func__);
        } else {
            ACLOGD(LOG_TAG, "%s: data: %s",__func__,simIO.v6.data);
        }

        if (simIO.v6.pin2==NULL){
            ACLOGD(LOG_TAG, "%s: pin2: NULL",__func__);
        } else {
            ACLOGD(LOG_TAG, "%s: pin2: %s",__func__,simIO.v6.pin2);
        }

        if (simIO.v6.aidPtr==NULL){
            ACLOGD(LOG_TAG, "%s: aidPtr: NULL",__func__);
        } else {
            ACLOGD(LOG_TAG, "%s: aidPtr: %s",__func__,simIO.v6.aidPtr);
        }

        size = (s_callbacks.version < 6) ? sizeof(simIO.v5) : sizeof(simIO.v6);
        s_callbacks.onRequest(RIL_REQUEST_SIM_IO, &simIO, size, pRI);

    } while (0);

    if (simIO.v6.path!=NULL) FREE (simIO.v6.path);
    if (simIO.v6.data!=NULL) FREE (simIO.v6.data);
    if (simIO.v6.pin2!=NULL) FREE (simIO.v6.pin2);
    if (simIO.v6.aidPtr!=NULL) FREE (simIO.v6.aidPtr);

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}

/**
 * Callee expects const RIL_SIM_APDU *
 * Payload is:
 *   int32_t sessionid
 *   int32_t cla
 *   int32_t instruction
 *   int32_t p1, p2, p3
 *   String data
 */
static RIL_Errno dispatchApduGeneric (Parcel &p, int reqId, RequestInfo *pRI)
{
    int32_t t;
    status_t status;
    RIL_SIM_APDU apdu;
    RIL_Errno result=RIL_E_SUCCESS;
    int num;

    ACLOGD(LOG_TAG,"%s..",__func__);
    memset (&apdu, 0, sizeof(RIL_SIM_APDU));

    // Note we only check status at the end. Any single failure leads to
    // subsequent reads filing.
    do {
        //number value added to be compliant to global format
        status = p.readInt32(&num);
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not read data number",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        if (num != 1){
            ACLOGW(LOG_TAG, "%s: number mismatch: 1/%d",__func__,num);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        status = p.readInt32(&t);
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not read sessionid",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        apdu.sessionid = (int)t;
        ACLOGD(LOG_TAG,"%s: session id = %d",__func__,apdu.sessionid);

        status = p.readInt32(&t);
        apdu.cla = (int)t;
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not read cla",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        ACLOGD(LOG_TAG,"%s: cla = %d",__func__,apdu.cla);

        status = p.readInt32(&t);
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not read instruction",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        apdu.instruction = (int)t;
        ACLOGD(LOG_TAG,"%s: instruction = %d",__func__,apdu.instruction);

        status = p.readInt32(&t);
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not read p1",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        apdu.p1 = (int)t;
        ACLOGD(LOG_TAG,"%s: p1 = %d",__func__,apdu.p1);

        status = p.readInt32(&t);
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not read p2",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        apdu.p2 = (int)t;
        ACLOGD(LOG_TAG,"%s: p2 = %d",__func__,apdu.p2);

        status = p.readInt32(&t);
        if (status !=NO_ERROR){
            ACLOGW(LOG_TAG, "%s: could not read p3",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        apdu.p3 = (int)t;
        ACLOGD(LOG_TAG,"%s: p3 = %d",__func__,apdu.p3);

        apdu.data = Parcel::strdupReadString(p);

        if (apdu.data !=NULL){
            ACLOGD(LOG_TAG,"%s: data = %s",__func__,apdu.data);
        } else {
            ACLOGD(LOG_TAG,"%s: data = NULL",__func__);
        }
        s_callbacks.onRequest(reqId, &apdu, sizeof(RIL_SIM_APDU), pRI);

    } while (0);

    if (apdu.data !=NULL){
        FREE(apdu.data);
    }

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}

RIL_Errno radio::iccTransmitApduBasicChannelRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchApduGeneric(p,RIL_REQUEST_SIM_TRANSMIT_APDU_BASIC,pRI);
}

RIL_Errno radio::iccTransmitApduLogicalChannelRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchApduGeneric(p,RIL_REQUEST_SIM_TRANSMIT_APDU_CHANNEL,pRI);
}


/**
 * Callee expects const RIL_CallForwardInfo *
 * Payload is:
 *  int32_t status/action
 *  int32_t reason
 *  int32_t serviceCode
 *  int32_t toa
 *  String number  (0 length -> null)
 *  int32_t timeSeconds
 */
RIL_Errno dispatchCallForwardGeneric(Parcel &p, int reqId, RequestInfo *pRI)
{
    RIL_CallForwardInfo cff;
    int32_t t;
    status_t status;
    RIL_Errno result=RIL_E_SUCCESS;

    memset (&cff, 0, sizeof(cff));

    do {

        status = p.readInt32(&t);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read status",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        cff.status = (int)t;

        status = p.readInt32(&t);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read reason",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        cff.reason = (int)t;

        status = p.readInt32(&t);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read service class",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        cff.serviceClass = (int)t;

        status = p.readInt32(&t);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read toa",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        cff.toa = (int)t;
        cff.number = Parcel::strdupReadString(p);

        status = p.readInt32(&t);
        if (status != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read seconds",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }
        cff.timeSeconds = (int)t;

        // special case: number 0-length fields is null

        if (cff.number != NULL && strlen (cff.number) == 0) {
            FREE (cff.number);
            cff.number = NULL;
        }

        ACLOGD(LOG_TAG,"%s: status: %d",__func__,cff.status);
        ACLOGD(LOG_TAG,"%s: serviceClass: %d",__func__,cff.serviceClass);
        ACLOGD(LOG_TAG,"%s: toa: %d",__func__,cff.toa);
        if (cff.number==NULL){
            ACLOGD(LOG_TAG,"%s: number: NULL",__func__);
        } else {
            ACLOGD(LOG_TAG,"%s: number: %s",__func__,cff.number);
        }
        ACLOGD(LOG_TAG,"%s: time: %d",__func__,cff.timeSeconds);

        s_callbacks.onRequest(reqId, &cff, sizeof(cff), pRI);

    } while (0);

    if (cff.number !=NULL)
        FREE (cff.number);

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}

RIL_Errno radio::setCallForwardRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchCallForwardGeneric(p,RIL_REQUEST_SET_CALL_FORWARD,pRI);
}

RIL_Errno radio::getCallForwardStatusRequest(Parcel& p, RequestInfo *pRI)
{
    return dispatchCallForwardGeneric(p,RIL_REQUEST_QUERY_CALL_FORWARD_STATUS,pRI);
}

RIL_Errno radio::oemHookRawRequest(Parcel &p, RequestInfo *pRI)
{
    int len;
    status_t status;
    const void *data;
    RIL_Errno result=RIL_E_SUCCESS;

    do {
        status = p.readInt32(&len);
        if (status != NO_ERROR) {
            ACLOGW(LOG_TAG,"%s: could not read length",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        // The java code writes -1 for null arrays
        if (((int)len) == -1) {
            data = NULL;
            len = 0;
        }

        data = p.readInplace(len);
        ACLOGD(LOG_TAG,"%s: request number: %d",__func__,pRI->pCI->requestNumber);

        unsigned char *responseData=(unsigned char *)data;

        if (loglevel > LOG_INFO) {
            for (int idx=0; idx<len;idx++){
                ACLOGV(LOG_TAG,"%s: data[%d] = %02x",__func__,idx,*(responseData+idx));
            }
        }
        s_callbacks.onRequest(RIL_REQUEST_OEM_HOOK_RAW, const_cast<void *>(data), len, pRI);
    } while (0);

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}

RIL_Errno radio::setGsmBroadcastConfigRequest(Parcel &p, RequestInfo *pRI)
{
    int32_t t;
    status_t status;
    int32_t num;
    RIL_Errno result=RIL_E_SUCCESS;

    do {
        status = p.readInt32(&num);
        if (status != NO_ERROR) {
            ACLOGW(LOG_TAG,"%s: could not read num",__func__);
            result=RIL_E_INVALID_ARGUMENTS;
            break;
        }

        RIL_GSM_BroadcastSmsConfigInfo gsmBci[num];
        RIL_GSM_BroadcastSmsConfigInfo *gsmBciPtrs[num];

        status=NO_ERROR;
        for (int i = 0 ; i < num  && result == RIL_E_SUCCESS; i++ ) {
            gsmBciPtrs[i] = &gsmBci[i];

            status = p.readInt32(&t);
            if (status != NO_ERROR) {
                ACLOGW(LOG_TAG,"%s: could not read fromServiceId[%d]",__func__,i);
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }
            gsmBci[i].fromServiceId = (int) t;

            status = p.readInt32(&t);
            if (status != NO_ERROR) {
                ACLOGW(LOG_TAG,"%s: could not read toServiceId[%d]",__func__,i);
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }
            gsmBci[i].toServiceId = (int) t;

            status = p.readInt32(&t);
            if (status != NO_ERROR) {
                ACLOGW(LOG_TAG,"%s: could not read fromCodeScheme[%d]",__func__,i);
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }
            gsmBci[i].fromCodeScheme = (int) t;

            status = p.readInt32(&t);
            if (status != NO_ERROR) {
                ACLOGW(LOG_TAG,"%s: could not read toCodeScheme[%d]",__func__,i);
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }
            gsmBci[i].toCodeScheme = (int) t;

            status = p.readInt32(&t);
            if (status != NO_ERROR) {
                ACLOGW(LOG_TAG,"%s: could not read selected[%d]",__func__,i);
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }
            gsmBci[i].selected = (uint8_t) t;
        }
        if ( result != RIL_E_SUCCESS) {
            break;
        }

        ACLOGD(LOG_TAG,"%s: number = %d",__func__,num);
        for (int i = 0 ; i < num ; i++ ) {
            ACLOGD(LOG_TAG,"%s: 0x%x",__func__,(int)gsmBciPtrs[i]);
            ACLOGD(LOG_TAG,"%s: fromServiceId[%d]: %d",__func__,i, gsmBci[i].fromServiceId);
            ACLOGD(LOG_TAG,"%s: toServiceId[%d]: %d",__func__,i, gsmBci[i].toServiceId);
            ACLOGD(LOG_TAG,"%s: fromCodeScheme[%d]: %d",__func__,i, gsmBci[i].fromCodeScheme);
            ACLOGD(LOG_TAG,"%s: toCodeScheme[%d]: %d",__func__,i, gsmBci[i].toCodeScheme);
            ACLOGD(LOG_TAG,"%s: selected[%d]: %d",__func__,i, gsmBci[i].selected);

        }

        s_callbacks.onRequest(RIL_REQUEST_GSM_SET_BROADCAST_SMS_CONFIG,
                              gsmBci,
                              num * sizeof(RIL_GSM_BroadcastSmsConfigInfo *),
                              pRI);
    } while (0);

    if (result != RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}

RIL_Errno radio::setupDataCallRequest(Parcel& p, RequestInfo *pRI)
{

    RIL_Errno result=RIL_E_SUCCESS;
    ACLOGD(LOG_TAG,"%s..",__func__);
    // The first bytes of the RIL parcel contain the request number and the
    // serial number - see libril_processCommandBuffer(). Copy them over too.
    int pos = p.dataPosition();

    ACLOGD(LOG_TAG,"%s: pos=%d",__func__,pos);

    int numParams = p.readInt32();

    ACLOGD(LOG_TAG,"%s: numParams =%d",__func__,numParams );

    do {
        if (s_callbacks.version < 4) {
            //not longer supported
            ACLOGW(LOG_TAG,"%s:version =%d is not supported",__func__,s_callbacks.version );
            result=RIL_E_REQUEST_NOT_SUPPORTED;
            break;
        }

        if (s_callbacks.version <= 14) {
            if (numParams <7) { //note: rmnet data interface name is optional !
                ACLOGW(LOG_TAG,"%s: too few params: %d",__func__,numParams );
                result=RIL_E_INVALID_ARGUMENTS;
                break;
            }
            p.setDataPosition(pos);
            result=dispatchStringsGeneric(p, RIL_REQUEST_SETUP_DATA_CALL, pRI);
        }
        //TODO
    } while (0);

    if (result !=RIL_E_SUCCESS){
       invalidCommandBlock(pRI);
    }
    return result;
}


// For backwards compatibility with RILs that dont support RIL_REQUEST_VOICE_RADIO_TECH.
// When all RILs handle this request, this function can be removed and
// the request can be sent directly to the RIL using dispatchVoid.
RIL_Errno radio::getVoiceRadioTechnologyRequest(Parcel& p, RequestInfo *pRI)
{
    RIL_Errno result=RIL_E_SUCCESS;
    RIL_RadioState state = s_callbacks.onStateRequest();

    do {
        if ((RADIO_STATE_UNAVAILABLE == state) || (RADIO_STATE_OFF == state)) {
            result=RIL_E_RADIO_NOT_AVAILABLE;
            break;
        }

        // RILs that support RADIO_STATE_ON should support this request.
        if (RADIO_STATE_ON == state) {
            s_callbacks.onRequest(RIL_REQUEST_VOICE_RADIO_TECH, NULL, 0, pRI);
        }

    } while (0);

    return result;
}

/** response is a char **, pointing to an array of char *'s
 *  The parcel will begin with the version
 */
static int responseStringsWithVersion(int version, Parcel &p, void *response, size_t responselen)
{
    p.writeInt32(version);
    return responseStrings(p, response, responselen);
}


/* delivered "data" is const char **
 * ((const char **)data)[0] points to a type code, which is
 *  one of these string values:
 *      "0"   USSD-Notify -- text in ((const char **)data)[1]
 *      "1"   USSD-Request -- text in ((const char **)data)[1]
 *      "2"   Session terminated by network
 *      "3"   other local client (eg, SIM Toolkit) has responded
 *      "4"   Operation not supported
 *      "5"   Network timeout
 *
 * The USSD session is assumed to persist if the type code is "1", otherwise
 * the current session (if any) is assumed to have terminated.
 *
 * ((const char **)data)[1] points to a message string if applicable, which
 * should always be in UTF-8.
 */
int radio::onUssdInd(Parcel &p, void *response, size_t responselen)
{
    return genericStringsUpToRespOrInd(__func__,2, p, response, responselen);
}

int radio::getOperatorResponse(Parcel &p, void *response, size_t responselen)
{
    return genericStringsRespOrInd(__func__,3, p, response, responselen);
}

int radio::oemHookStringsResponse(Parcel &p, void *response, size_t responselen)
{
    return genericStringsRespOrInd(__func__,-1, p, response, responselen);
}

int radio::getAvailableNetworksResponse(Parcel &p, void *response, size_t responselen)
{
    if (responselen % 4*sizeof(char *) != 0){
        return RIL_ERRNO_INVALID_RESPONSE;
    }
    return genericStringsRespOrInd(__func__,-1, p, response, responselen);
}

int radio::getDeviceIdentityResponse(Parcel &p, void *response, size_t responselen)
{
    return genericStringsRespOrInd(__func__,4, p, response, responselen);
}

int genericStringsUpToRespOrInd(const char * caller,int paramNumber, Parcel &p, void *response, size_t responselen)
{
    int numStrings;
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        ACLOGD(LOG_TAG,"%s..",caller);
        if (response == NULL && responselen != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response= NULL",__func__);
            break;
        }
        if (responselen % sizeof(char *) != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected multiple of %d",caller,
                (int)responselen, (int)sizeof(char *));
            break;
        }

        if (response == NULL) {
            p.writeInt32 (0);
            ret=0;
            ACLOGD(LOG_TAG,"%s: response =NULL",caller);
            break;
        }

        char **p_cur = (char **) response;
        numStrings = responselen / sizeof(char *); //should be one or two
        if ((paramNumber > 0) && (numStrings>paramNumber)){
            ACLOGW(LOG_TAG,"%s: invalid string number %d exceeded max=%d",caller, numStrings,paramNumber);
            break;
        }
        p.writeInt32 (numStrings);
        ACLOGD(LOG_TAG,"%s: number= %d",caller,numStrings);
        /* each string*/
        for (int i = 0 ; i < numStrings ; i++) {
            if (p_cur[i]==NULL){
                ACLOGD(LOG_TAG,"%s[%d]: NULL",caller,i);
            }else {
                ACLOGD(LOG_TAG,"%s[%d]: %s",caller,i,(char*)p_cur[i]);
            }
            p.writeStringToParcel( p_cur[i]);
        }
        ret=0;
    } while (0);

    return ret;
}

int genericStringsRespOrInd(const char * caller,int paramNumber, Parcel &p, void *response, size_t responselen)
{
    int numStrings;
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        ACLOGD(LOG_TAG,"%s..",caller);
        if (response == NULL && responselen != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response= NULL",__func__);
            break;
        }
        if (responselen % sizeof(char *) != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected multiple of %d",caller,
                (int)responselen, (int)sizeof(char *));
            break;
        }

        if (response == NULL) {
            p.writeInt32 (0);
            ret=0;
            ACLOGD(LOG_TAG,"%s: response =NULL",caller);
            break;
        }

        char **p_cur = (char **) response;
        numStrings = responselen / sizeof(char *);
        if ((numStrings!= paramNumber) && (paramNumber >0)){
            ACLOGW(LOG_TAG,"%s: invalid string number %d exceeded max=%d",caller, numStrings,paramNumber);
            break;
        }
        p.writeInt32 (numStrings);
        ACLOGD(LOG_TAG,"%s: number= %d",__func__,numStrings);

        /* each string*/
        for (int i = 0 ; i < numStrings ; i++) {
            if (p_cur[i]==NULL){
                ACLOGD(LOG_TAG,"%s[%d]: NULL",caller,i);
            }else {
                ACLOGD(LOG_TAG,"%s[%d]: %s",caller,i,(char*)p_cur[i]);
            }
            p.writeStringToParcel( p_cur[i]);
        }
        ret=0;
    } while (0);

    return ret;
}


/**
 * response is a char **, pointing to an array of char *'s
 */
int responseStrings(Parcel &p, void *response, size_t responselen)
{
    int numStrings;
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    while(1) {
        ACLOGD(LOG_TAG,"%s..",__func__);
        if (response == NULL && responselen != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response= NULL",__func__);
            break;
        }

        if (responselen % sizeof(char *) != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected multiple of %d",__func__,
                (int)responselen, (int)sizeof(char *));
            break;
        }

        if (response == NULL) {
            p.writeInt32 (0);
            ret=0;
            ACLOGD(LOG_TAG,"%s: response =NULL",__func__);
            break;
        }

        char **p_cur = (char **) response;

        numStrings = responselen / sizeof(char *);
        p.writeInt32 (numStrings);
        ACLOGD(LOG_TAG,"%s: number= %d",__func__,numStrings);

        /* each string*/
        for (int i = 0 ; i < numStrings ; i++) {
            if (p_cur[i]==NULL){
                ACLOGD(LOG_TAG,"%s[%d]: NULL",__func__,i);
            }else {
                ACLOGD(LOG_TAG,"%s[%d]: %s",__func__,i,(char*)p_cur[i]);
            }
            p.writeStringToParcel( p_cur[i]);
        }
        ret=0;
        break;
    }

    return ret;
}

RIL_CellInfoType getCellInfoTypeRadioTechnology(char *rat)
{
    RIL_CellInfoType result=RIL_CELL_INFO_TYPE_NONE;

    do {
        if (rat == NULL) {
            break;
        }
        int radioTech = atoi(rat);

        switch(radioTech) {
            case RADIO_TECH_GPRS:
            case RADIO_TECH_EDGE:
            case RADIO_TECH_GSM: {
                result=RIL_CELL_INFO_TYPE_GSM;
                break;
            }
            case RADIO_TECH_UMTS:
            case RADIO_TECH_HSDPA:
            case RADIO_TECH_HSUPA:
            case RADIO_TECH_HSPA:
            case RADIO_TECH_HSPAP: {
                result= RIL_CELL_INFO_TYPE_WCDMA;
                break;
            }
            case RADIO_TECH_IS95A:
            case RADIO_TECH_IS95B:
            case RADIO_TECH_1xRTT:
            case RADIO_TECH_EVDO_0:
            case RADIO_TECH_EVDO_A:
            case RADIO_TECH_EVDO_B:
            case RADIO_TECH_EHRPD: {
                result= RIL_CELL_INFO_TYPE_CDMA;
                break;
            }
            case RADIO_TECH_LTE:
            case RADIO_TECH_LTE_CA: {
                result= RIL_CELL_INFO_TYPE_LTE;
                break;
            }
            case RADIO_TECH_TD_SCDMA: {
                result= RIL_CELL_INFO_TYPE_TD_SCDMA;
            }
            case RADIO_TECH_NR: {
                result= RIL_CELL_INFO_TYPE_NR_5G;
            }
            default: {
                break;
            }
        }
    } while (0);
    return result;
}


int convertResponseHexStringEntryToInt(char **response, int index, int numStrings)
{
    const int hexBase = 16;
    if ((response != NULL) &&  (numStrings > index) && (response[index] != NULL)) {
        return strtol(response[index], NULL, hexBase);
    }
    return -1;
}

int convertResponseStringEntryToInt(char **response, int index, int numStrings) {
    if ((response != NULL) &&  (numStrings > index) && (response[index] != NULL)) {
        return atoi(response[index]);
    }
    return -1;
}


/* Fill Cell Identity info from Voice Registration State Response.
 * This fucntion is applicable only for RIL Version < 15.
 * Response is a  "char **".
 * First and Second entries are in hex string format
 * and rest are integers represented in ascii format. */
void fillCellIdentityFromDataRegStateResponseString(RIL_CellIdentity_v16 &rilCellIdentity,int numStrings, void* response)
{
    memset(&rilCellIdentity, -1, sizeof(RIL_CellIdentity_v16));

    char **pResponseStrings = (char **)response;
    rilCellIdentity.cellInfoType = getCellInfoTypeRadioTechnology(pResponseStrings[3]);

    switch(rilCellIdentity.cellInfoType) {
        case RIL_CELL_INFO_TYPE_GSM: {

            /* valid LAC are hexstrings in the range 0x0000 - 0xffff */
            rilCellIdentity.cellIdentityGsm.lac =
                    convertResponseHexStringEntryToInt(pResponseStrings, 1, numStrings);
            /* valid CID are hexstrings in the range 0x00000000 - 0xffffffff */
            rilCellIdentity.cellIdentityGsm.cid =
                    convertResponseHexStringEntryToInt(pResponseStrings, 2, numStrings);
            break;
        }
        case RIL_CELL_INFO_TYPE_WCDMA: {

            /* valid LAC are hexstrings in the range 0x0000 - 0xffff */
            rilCellIdentity.cellIdentityWcdma.lac =
                    convertResponseHexStringEntryToInt(pResponseStrings, 1, numStrings);
            /* valid CID are hexstrings in the range 0x00000000 - 0xffffffff */
            rilCellIdentity.cellIdentityWcdma.cid =
                    convertResponseHexStringEntryToInt(pResponseStrings, 2, numStrings);
            rilCellIdentity.cellIdentityWcdma.psc =
                    convertResponseStringEntryToInt(pResponseStrings, 14, numStrings);
            break;
        }
        case RIL_CELL_INFO_TYPE_TD_SCDMA:{

            /* valid LAC are hexstrings in the range 0x0000 - 0xffff */
            rilCellIdentity.cellIdentityTdscdma.lac =
                    convertResponseHexStringEntryToInt(pResponseStrings, 1, numStrings);
            /* valid CID are hexstrings in the range 0x00000000 - 0xffffffff */
            rilCellIdentity.cellIdentityTdscdma.cid =
                    convertResponseHexStringEntryToInt(pResponseStrings, 2, numStrings);
            break;
        }
        case RIL_CELL_INFO_TYPE_CDMA:{

            rilCellIdentity.cellIdentityCdma.basestationId =
                    convertResponseStringEntryToInt(pResponseStrings, 4, numStrings);
            /* Order of Lat. and Long. swapped between RIL and HIDL interface versions. */
            rilCellIdentity.cellIdentityCdma.latitude =
                    convertResponseStringEntryToInt(pResponseStrings, 5, numStrings);
            rilCellIdentity.cellIdentityCdma.longitude =
                    convertResponseStringEntryToInt(pResponseStrings, 6, numStrings);
            rilCellIdentity.cellIdentityCdma.systemId =
                    convertResponseStringEntryToInt(pResponseStrings, 8, numStrings);
            rilCellIdentity.cellIdentityCdma.networkId =
                    convertResponseStringEntryToInt(pResponseStrings, 9, numStrings);
            break;
        }
        case RIL_CELL_INFO_TYPE_LTE:{

            /* valid TAC are hexstrings in the range 0x0000 - 0xffff */
            rilCellIdentity.cellIdentityLte.tac =
                    convertResponseHexStringEntryToInt(pResponseStrings, 1, numStrings);
            /* valid CID are hexstrings in the range 0x00000000 - 0xffffffff */
            rilCellIdentity.cellIdentityLte.ci =
                    convertResponseHexStringEntryToInt(pResponseStrings, 2, numStrings);
            break;
        }
        case RIL_CELL_INFO_TYPE_NR_5G:
            //note: no tech values provided from string based  vendorlib info
            rilCellIdentity.cellIdentityNr5g.pci=INT_MAX;
            rilCellIdentity.cellIdentityNr5g.cid=-1;

            rilCellIdentity.cellIdentityNr5g.arfcn=0;
            if (numStrings >= 13) {
                rilCellIdentity.cellIdentityNr5g.mcc =
                        convertResponseStringEntryToInt(pResponseStrings, 11, numStrings);
                rilCellIdentity.cellIdentityNr5g.mnc =
                        convertResponseStringEntryToInt(pResponseStrings, 12, numStrings);
            } else {
                //these values are not known from strings array
                rilCellIdentity.cellIdentityNr5g.mcc=INT_MAX;
                rilCellIdentity.cellIdentityNr5g.mnc=INT_MAX;
            }
            break;

        default: {
            break;
        }
    }

}

void fillCellIdentityFromVoiceRegStateResponseString(RIL_CellIdentity_v16 &rilCellIdentity,int numStrings, void* response)
{
    memset(&rilCellIdentity, 0, sizeof(RIL_CellIdentity_v16));

    char **pResponseStrings = (char **)response;
    rilCellIdentity.cellInfoType = getCellInfoTypeRadioTechnology(pResponseStrings[3]);

    switch(rilCellIdentity.cellInfoType) {
        case RIL_CELL_INFO_TYPE_GSM:

            /* valid LAC are hexstrings in the range 0x0000 - 0xffff */
            rilCellIdentity.cellIdentityGsm.lac =
                    convertResponseHexStringEntryToInt(pResponseStrings, 1, numStrings);
            /* valid CID are hexstrings in the range 0x00000000 - 0xffffffff */
            rilCellIdentity.cellIdentityGsm.cid =
                    convertResponseHexStringEntryToInt(pResponseStrings, 2, numStrings);

            if (numStrings >= 13){
                rilCellIdentity.cellIdentityGsm.mcc =
                        convertResponseStringEntryToInt(pResponseStrings, 11, numStrings);
                rilCellIdentity.cellIdentityGsm.mnc =
                        convertResponseStringEntryToInt(pResponseStrings, 12, numStrings);
            } else {
                //these values are not known from strings array
                rilCellIdentity.cellIdentityGsm.mcc=INT_MAX;
                rilCellIdentity.cellIdentityGsm.mnc=INT_MAX;
            }

            //these values are not known from strings array
            rilCellIdentity.cellIdentityGsm.arfcn=0xFFFF;
            rilCellIdentity.cellIdentityGsm.bsic=0xFF;
            break;


        case RIL_CELL_INFO_TYPE_WCDMA:
            /* valid LAC are hexstrings in the range 0x0000 - 0xffff */
            rilCellIdentity.cellIdentityWcdma.lac =
                    convertResponseHexStringEntryToInt(pResponseStrings, 1, numStrings);
            /* valid CID are hexstrings in the range 0x00000000 - 0xffffffff */
            rilCellIdentity.cellIdentityWcdma.cid =
                    convertResponseHexStringEntryToInt(pResponseStrings, 2, numStrings);

            if (numStrings >= 13) {
                rilCellIdentity.cellIdentityWcdma.mcc =
                        convertResponseStringEntryToInt(pResponseStrings, 11, numStrings);
                rilCellIdentity.cellIdentityWcdma.mnc =
                        convertResponseStringEntryToInt(pResponseStrings, 12, numStrings);
            } else {
                //these values are not known from strings array
                rilCellIdentity.cellIdentityWcdma.mcc=INT_MAX;
                rilCellIdentity.cellIdentityWcdma.mnc=INT_MAX;
            }

            //these values are not known from strings array
            rilCellIdentity.cellIdentityWcdma.psc=0x1FF;
            rilCellIdentity.cellIdentityWcdma.uarfcn=0xFFFF;
            break;


        case RIL_CELL_INFO_TYPE_TD_SCDMA:

            /* valid LAC are hexstrings in the range 0x0000 - 0xffff */
            rilCellIdentity.cellIdentityTdscdma.lac =
                    convertResponseHexStringEntryToInt(pResponseStrings, 1, numStrings);
            /* valid CID are hexstrings in the range 0x00000000 - 0xffffffff */
            rilCellIdentity.cellIdentityTdscdma.cid =
                    convertResponseHexStringEntryToInt(pResponseStrings, 2, numStrings);

            if (numStrings >= 13) {
                rilCellIdentity.cellIdentityTdscdma.mcc =
                        convertResponseStringEntryToInt(pResponseStrings, 11, numStrings);
                rilCellIdentity.cellIdentityTdscdma.mnc =
                        convertResponseStringEntryToInt(pResponseStrings, 12, numStrings);
            } else {
                //these values are not known from strings array
                rilCellIdentity.cellIdentityTdscdma.mcc=INT_MAX;
                rilCellIdentity.cellIdentityTdscdma.mnc=INT_MAX;
            }
            break;

        case RIL_CELL_INFO_TYPE_LTE:

            /* valid TAC are hexstrings in the range 0x0000 - 0xffff */
            rilCellIdentity.cellIdentityLte.tac =
                    convertResponseStringEntryToInt(pResponseStrings, 6, numStrings);
            /* valid CID are hexstrings in the range 0x00000000 - 0xffffffff */
            rilCellIdentity.cellIdentityLte.ci =
                    convertResponseStringEntryToInt(pResponseStrings, 7, numStrings);

             rilCellIdentity.cellIdentityLte.pci =
                    convertResponseStringEntryToInt(pResponseStrings, 8, numStrings);

            if (numStrings >= 13) {
                rilCellIdentity.cellIdentityLte.mcc =
                        convertResponseStringEntryToInt(pResponseStrings, 11, numStrings);
                rilCellIdentity.cellIdentityLte.mnc =
                        convertResponseStringEntryToInt(pResponseStrings, 12, numStrings);
            } else {
                //these values are not known from strings array
                rilCellIdentity.cellIdentityLte.earfcn=0x3FFFF;
            }
            break;

        case RIL_CELL_INFO_TYPE_NR_5G:
            //note: no tech values provided from string based  vendorlib info

            rilCellIdentity.cellIdentityNr5g.pci=INT_MAX;
            rilCellIdentity.cellIdentityNr5g.cid=-1;
            rilCellIdentity.cellIdentityNr5g.tac=INT_MAX;
            rilCellIdentity.cellIdentityNr5g.arfcn=0;

            if (numStrings >= 13) {
                rilCellIdentity.cellIdentityNr5g.mcc =
                        convertResponseStringEntryToInt(pResponseStrings, 11, numStrings);
                rilCellIdentity.cellIdentityNr5g.mnc =
                        convertResponseStringEntryToInt(pResponseStrings, 12, numStrings);
            } else {
                //these values are not known from strings array
                rilCellIdentity.cellIdentityNr5g.mcc=INT_MAX;
                rilCellIdentity.cellIdentityNr5g.mnc=INT_MAX;
            }

            break;


        default:
            break;

    }
}

int sendCellIdentityResponse(Parcel &p,RIL_CellIdentity_v16 &rilCellIdentity)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        if (p.writeInt32( rilCellIdentity.cellInfoType) != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not write cell info type",__func__);
            break;
        };

        switch (rilCellIdentity.cellInfoType){
            case RIL_CELL_INFO_TYPE_GSM:
                //RIL_CellIdentityGsm_v12
                if(p.writeInt32( rilCellIdentity.cellIdentityGsm.mcc) != NO_ERROR){break;};
                if(p.writeInt32( rilCellIdentity.cellIdentityGsm.mnc) != NO_ERROR){break;};
                if(p.writeInt32( rilCellIdentity.cellIdentityGsm.lac) != NO_ERROR){break;};
                if(p.writeInt32( rilCellIdentity.cellIdentityGsm.cid) != NO_ERROR){break;};
                if(p.writeInt32( rilCellIdentity.cellIdentityGsm.arfcn) != NO_ERROR){break;};
                if(p.writeInt32( rilCellIdentity.cellIdentityGsm.bsic) != NO_ERROR){break;};
                ret=0;
                break;

            case RIL_CELL_INFO_TYPE_WCDMA:
                //RIL_CellIdentityWcdma_v12
                if(p.writeInt32( rilCellIdentity.cellIdentityWcdma.mcc) != NO_ERROR){break;};
                if(p.writeInt32( rilCellIdentity.cellIdentityWcdma.mnc) != NO_ERROR){break;};
                if(p.writeInt32( rilCellIdentity.cellIdentityWcdma.lac) != NO_ERROR){break;};
                if(p.writeInt32( rilCellIdentity.cellIdentityWcdma.cid) != NO_ERROR){break;};
                if(p.writeInt32( rilCellIdentity.cellIdentityWcdma.psc) != NO_ERROR){break;};
                if(p.writeInt32( rilCellIdentity.cellIdentityWcdma.uarfcn) != NO_ERROR){break;};
                ret=0;
                break;

           case RIL_CELL_INFO_TYPE_TD_SCDMA:
                //RIL_CellIdentityTdscdma
                if(p.writeInt32( rilCellIdentity.cellIdentityTdscdma.mcc) != NO_ERROR){break;};
                if(p.writeInt32( rilCellIdentity.cellIdentityTdscdma.mnc) != NO_ERROR){break;};
                if(p.writeInt32( rilCellIdentity.cellIdentityTdscdma.lac) != NO_ERROR){break;};
                if(p.writeInt32( rilCellIdentity.cellIdentityTdscdma.cid) != NO_ERROR){break;};
                if(p.writeInt32( rilCellIdentity.cellIdentityTdscdma.cpid) != NO_ERROR){break;};
                ret=0;
                break;

           case RIL_CELL_INFO_TYPE_LTE:
                //RIL_CellIdentityLte_v12
                if(p.writeInt32( rilCellIdentity.cellIdentityLte.mcc) != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not write lte mcc",__func__);
                    break;
                };
                if(p.writeInt32( rilCellIdentity.cellIdentityLte.mnc) != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not write lte mnc",__func__);
                    break;
                };
                if(p.writeInt32( rilCellIdentity.cellIdentityLte.ci) != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not write lte ci",__func__);
                    break;
                };
                if(p.writeInt32( rilCellIdentity.cellIdentityLte.pci) != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not write lte pci",__func__);
                    break;
                };
                if(p.writeInt32( rilCellIdentity.cellIdentityLte.tac) != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not write lte tac",__func__);
                    break;
                };
                if(p.writeInt32( rilCellIdentity.cellIdentityLte.earfcn) != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not write lte earfcn",__func__);
                    break;
                };
                ret=0;
                break;

          case RIL_CELL_INFO_TYPE_NR_5G:
                //RIL_CellIdentityNr5g
                if(p.writeInt32( rilCellIdentity.cellIdentityNr5g.mcc) != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not write 5g mcc",__func__);
                    break;
                };
                if(p.writeInt32( rilCellIdentity.cellIdentityNr5g.mnc) != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not write 5g mnc",__func__);
                    break;
                };
                if(p.writeInt32( rilCellIdentity.cellIdentityNr5g.pci) != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not write 5g pci",__func__);
                    break;
                };

                if(p.writeInt64( rilCellIdentity.cellIdentityNr5g.cid) != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not write 5g cid",__func__);
                    break;
                };
                if(p.writeInt32( rilCellIdentity.cellIdentityNr5g.tac) != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not write 5g tac",__func__);
                    break;
                };
                if(p.writeInt32( rilCellIdentity.cellIdentityNr5g.arfcn) != NO_ERROR){
                    ACLOGW(LOG_TAG,"%s: could not write 5g tac",__func__);
                    break;
                };
                ret=0;
                break;

            default:
                break;
        }
    } while (0);

    return ret;
}

int radio::getHardwareConfigResponse(Parcel &p, void *response, size_t responselen)
{
    int num = responselen / sizeof(RIL_HardwareConfig);
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    RIL_HardwareConfig *rilHardwareConfig=0;

    do {
        rilHardwareConfig = (RIL_HardwareConfig *) response;

        if(p.writeInt32(num) != NO_ERROR){break;}
        if (num == 0){
            ret=0;
        }
        for (int i = 0; i < num; i++) {
            if(p.writeInt32( rilHardwareConfig[i].type) != NO_ERROR){break;}
            if(p.write( rilHardwareConfig[i].uuid,MAX_UUID_LENGTH) != NO_ERROR){break;}
            if(p.writeInt32( rilHardwareConfig[i].state) != NO_ERROR){break;}

            if (rilHardwareConfig[i].type == RIL_HARDWARE_CONFIG_MODEM){
                if (p.writeInt32(rilHardwareConfig[i].cfg.modem.rilModel) != NO_ERROR){break;}
                if (p.writeInt32(rilHardwareConfig[i].cfg.modem.rat) != NO_ERROR){break;}
                if (p.writeInt32(rilHardwareConfig[i].cfg.modem.maxVoice) != NO_ERROR){break;}
                if (p.writeInt32(rilHardwareConfig[i].cfg.modem.maxData) != NO_ERROR){break;}
                if (p.writeInt32(rilHardwareConfig[i].cfg.modem.maxStandby) != NO_ERROR){break;}
            } else if (rilHardwareConfig[i].type == RIL_HARDWARE_CONFIG_SIM){
                if(p.write( rilHardwareConfig[i].cfg.sim.modemUuid,MAX_UUID_LENGTH) != NO_ERROR){break;}
            } else {
                break;
            }
            if (i==num-1){
                ret=0;
            }
        }
    } while (0);

    return 0;
}

int radio::getVoiceRegistrationStateResponse(Parcel &p, void *response, size_t responselen)
{
    int numStrings;
    int ret=RIL_ERRNO_INVALID_RESPONSE;
    numStrings = responselen / sizeof(char *);
    //int loglevel=LOG_INFO;


    do {
        ACLOGI(LOG_TAG,"%s..",__func__);

        if (response == NULL) {
            ACLOGW(LOG_TAG,"%s: invalid response = NULL",__func__);
            break;
        }

        p.writeInt32 (1); //single RIL_VoiceRegistrationStateResponse transferred

        if ((s_callbacks.version <= 14) && (responselen != sizeof(RIL_VoiceRegistrationStateResponse))) {

            if (numStrings != 15) {
                ACLOGW(LOG_TAG,"%s: invalid response length %d",__func__,numStrings);
                break;
            }

            char **p_cur = (char **) response;

            p.writeInt32( (p_cur[0]!=0)?std::atoi(p_cur[0]):RIL_UNKNOWN);   // registrationstate
            p.writeInt32((p_cur[3]!=0)?std::atoi(p_cur[3]):RADIO_TECH_UNKNOWN);    // rat
            p.writeInt32( (p_cur[7]!=0)?std::atoi(p_cur[7]):-1);   // css supported (CDMA only)
            p.writeInt32( (p_cur[10]!=0)?std::atoi(p_cur[10]):-1);  // roaming indicator (CDMA only)
            p.writeInt32( (p_cur[11]!=0)?std::atoi(p_cur[11]):-1);  // is in PRL (CDMA only)
            p.writeInt32( (p_cur[12]!=0)?std::atoi(p_cur[12]):-1);  // default roaming indicator (CDMA only)
            p.writeInt32( (p_cur[13]!=0)?std::atoi(p_cur[13]):0);  // reason for denial

            RIL_CellIdentity_v16 cellIdentity;
            fillCellIdentityFromVoiceRegStateResponseString(cellIdentity,numStrings, response);
            ACLOGI(LOG_TAG,"%s: cell info type = %d",__func__,cellIdentity.cellInfoType);

            ret=sendCellIdentityResponse(p,cellIdentity);

        } else {

            RIL_VoiceRegistrationStateResponse *voiceRegState =
                    (RIL_VoiceRegistrationStateResponse *)response;
            if (responselen != sizeof(RIL_VoiceRegistrationStateResponse)) {
                ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            } else {
                if(p.writeInt32( (RIL_RegState) voiceRegState->regState) != NO_ERROR){break;};
                if(p.writeInt32( voiceRegState->rat) != NO_ERROR){break;};
                if(p.writeInt32( voiceRegState->cssSupported) != NO_ERROR){break;};
                if(p.writeInt32( voiceRegState->roamingIndicator) != NO_ERROR){break;};
                if(p.writeInt32( voiceRegState->systemIsInPrl) != NO_ERROR){break;};
                if(p.writeInt32( voiceRegState->defaultRoamingIndicator) != NO_ERROR){break;};
                if(p.writeInt32( voiceRegState->reasonForDenial) != NO_ERROR){break;};

                ret=sendCellIdentityResponse(p,voiceRegState->cellIdentity);
            }
        }
    } while (0);

    ACLOGI(LOG_TAG,"%s: return = %d",__func__,ret);
    return ret;
}

int radio::getModemActivityInfoResponse(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        ACLOGD(LOG_TAG,"%s..",__func__);

        if (response == NULL) {
            ACLOGW(LOG_TAG,"%s: invalid response = NULL",__func__);
            break;
        }
        if (responselen != sizeof(RIL_ActivityStatsInfo)) {
            ACLOGW(LOG_TAG,"%s: invalid responselen: %d/%d",__func__,responselen,sizeof(RIL_ActivityStatsInfo));
            break;
        }

        p.writeInt32 (1); //single RIL_ActivityStatsInfo  transferred

        RIL_ActivityStatsInfo *resp = (RIL_ActivityStatsInfo *)response;
        if(p.writeInt32(resp->sleep_mode_time_ms) != NO_ERROR){break;};
        if(p.writeInt32(resp->idle_mode_time_ms) != NO_ERROR){break;};

        ret=0;
        for(int i = 0; i < RIL_NUM_TX_POWER_LEVELS; i++) {
            if(p.writeInt32(resp->tx_mode_time_ms[i]) != NO_ERROR){
                ret=RIL_ERRNO_INVALID_RESPONSE;
                break;
            }
        }
        if (ret !=0) {
            break;
        }

        if(p.writeInt32(resp->rx_mode_time_ms) != NO_ERROR){
            ret=RIL_ERRNO_INVALID_RESPONSE;
            break;
        }
    } while (0);
    return ret;
}


int genericLceResponse(Parcel &p, int reqId, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        ACLOGD(LOG_TAG,"%s..",__func__);

        if (response == NULL) {
            ACLOGW(LOG_TAG,"%s: invalid response = NULL",__func__);
            break;
        }
        if (responselen != sizeof(RIL_LceStatusInfo)) {
            ACLOGW(LOG_TAG,"%s: invalid responselen: %d/%d",__func__,responselen,sizeof(RIL_LceStatusInfo));
            break;
        }

        ACLOGD(LOG_TAG,"%s: request %d",__func__,reqId);

        p.writeInt32 (1); //single RIL_LceStatusInfo  transferred

        RIL_LceStatusInfo *resp = (RIL_LceStatusInfo *)response;
        if(p.writeInt32(resp->lce_status) != NO_ERROR){break;};
        if(p.writeInt32(resp->actual_interval_ms) != NO_ERROR){break;}
        ret=0;
    } while (0);
    return ret;

}
int radio::startLceServiceResponse(Parcel &p, void *response, size_t responselen)
{
    return genericLceResponse(p, RIL_REQUEST_START_LCE, response, responselen);
}

int radio::stopLceServiceResponse(Parcel &p, void *response, size_t responselen)
{
    return genericLceResponse(p, RIL_REQUEST_STOP_LCE, response, responselen);
}




int radio::getDataRegistrationStateResponse(Parcel &p,void *response, size_t responselen)
{
    int numStrings;
    int ret=RIL_ERRNO_INVALID_RESPONSE;
    numStrings = responselen / sizeof(char *);

    do {
        ACLOGD(LOG_TAG,"%s..",__func__);

        if (response == NULL) {
            ACLOGW(LOG_TAG,"%s: invalid response = NULL",__func__);
            break;
        }

        p.writeInt32 (1); //single RIL_DataRegistrationStateResponse transferred

        if ((s_callbacks.version <= 14) && (responselen != sizeof(RIL_DataRegistrationStateResponse))) {
            if ((numStrings != 6) && (numStrings != 11) && (numStrings != 13)) {
                ACLOGW(LOG_TAG,"%s: invalid response length %d",__func__,numStrings);
                break;
            }

            char **p_cur = (char **) response;

            p.writeInt32( (p_cur[0]!=0)?std::atoi(p_cur[0]):RIL_UNKNOWN);   // registrationstate
            p.writeInt32((p_cur[3]!=0)?std::atoi(p_cur[3]):RADIO_TECH_UNKNOWN);    // rat
            p.writeInt32( (p_cur[4]!=0)?std::atoi(p_cur[4]):0);  // reason for denial
            p.writeInt32( (p_cur[5]!=0)?std::atoi(p_cur[5]):0);  // maxDataCalls

            RIL_CellIdentity_v16 cellIdentity;
            fillCellIdentityFromDataRegStateResponseString(cellIdentity,numStrings, response);
            ret=sendCellIdentityResponse(p,cellIdentity);
        } else {
            RIL_DataRegistrationStateResponse *dataRegState =
                    (RIL_DataRegistrationStateResponse *)response;
            if (responselen != sizeof(RIL_DataRegistrationStateResponse)) {
                ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            } else {
                if(p.writeInt32( (RIL_RegState) dataRegState->regState) != NO_ERROR){break;};
                if(p.writeInt32( dataRegState->rat) != NO_ERROR){break;};
                if(p.writeInt32( dataRegState->reasonDataDenied ) != NO_ERROR){break;};
                if(p.writeInt32( dataRegState->maxDataCalls ) != NO_ERROR){break;};

                ret=sendCellIdentityResponse(p,dataRegState->cellIdentity);
            }
        }
    } while (0);

    return ret;
}


static int  genericSingleStringResponse(Parcel &p, int reqId, void *response, size_t responselen)
{
    /* one  string only */
    p.writeInt32(1);

    if ((response == NULL) || (responselen == 0)){
        ACLOGD(LOG_TAG,"%s[%d]: response = NULL",__func__,reqId);
    } else {
        ACLOGN(LOG_TAG,"%s[%d]: len = %d; write '%s'",__func__,reqId,responselen, (const char *)response);
    }
    p.writeStringToParcel((const char *)response);
    return 0;
}

int radio::stkCallControlAlphaNotifyInd(Parcel &p, void *response, size_t responselen)
{
    return genericSingleStringResponse(p, RIL_UNSOL_STK_CC_ALPHA_NOTIFY, response, responselen);
}

int radio::getIMSIForAppResponse(Parcel &p, void *response, size_t responselen)
{
    return genericSingleStringResponse(p, RIL_REQUEST_GET_IMSI, response, responselen);
}

int radio::getImeiResponse(Parcel &p, void *response, size_t responselen)
{
    return genericSingleStringResponse(p, RIL_REQUEST_GET_IMEI, response, responselen);
}

int radio::getImeiSvResponse(Parcel &p, void *response, size_t responselen)
{
    return genericSingleStringResponse(p, RIL_REQUEST_GET_IMEISV, response, responselen);
}

int radio::getBasebandVersionResponse(Parcel &p, void *response, size_t responselen)
{
    return genericSingleStringResponse(p, RIL_REQUEST_BASEBAND_VERSION, response, responselen);
}

int radio::sendEnvelopeResponse(Parcel &p, void *response, size_t responselen)
{
    return genericSingleStringResponse(p, RIL_REQUEST_STK_SEND_ENVELOPE_COMMAND, response, responselen);
}

int radio::getSmscAddressResponse(Parcel &p, void *response, size_t responselen)
{
    return genericSingleStringResponse(p, RIL_REQUEST_GET_SMSC_ADDRESS, response, responselen);
}

int radio::nvReadItemResponse(Parcel &p, void *response, size_t responselen)
{
    return genericSingleStringResponse(p, RIL_REQUEST_NV_READ_ITEM, response, responselen);
}

int radio::requestIsimAuthenticationResponse(Parcel &p, void *response, size_t responselen)
{
    return genericSingleStringResponse(p, RIL_REQUEST_ISIM_AUTHENTICATION, response, responselen);
}

 /* ANDROID12: "data" is now const char *
 * This is a pointer to a string containing the PDU of an SMS-DELIVER
 * as an ascii string of hex digits.
 * data will be process in the required manner
 * The PDU starts with the SMSC address per TS 27.005 (+CMT:)
 */
int radio::newSmsInd(Parcel &p, void *response, size_t responselen)
{
    ACLOGN(LOG_TAG,"%s: sms received; size: %d '",__func__,responselen);
    return genericSingleStringResponse(p, RIL_UNSOL_RESPONSE_NEW_SMS, response, responselen);
}

/* ANDROID12: "data" is now const char *
 * This is a pointer to a string containing the PDU of an SMS-DELIVER
 * as an ascii string of hex digits.
* data woill be process in the requirThe PDU starts with the SMSC address
 * per TS 27.005 (+CMT:)
 */
int radio::newSmsStatusReportInd(Parcel &p, void *response, size_t responselen)
{
    return genericSingleStringResponse(p, RIL_UNSOL_RESPONSE_NEW_SMS_STATUS_REPORT, response, responselen);
}

int radio::nitzTimeReceivedInd(Parcel &p, void *response, size_t responselen)
{
    return genericSingleStringResponse(p, RIL_UNSOL_NITZ_TIME_RECEIVED, response, responselen);
}

int radio::stkProactiveCommandInd(Parcel &p, void *response, size_t responselen)
{
    return genericSingleStringResponse(p, RIL_UNSOL_STK_PROACTIVE_COMMAND, response, responselen);
}

int radio::stkEventNotifyInd(Parcel &p, void *response, size_t responselen)
{
    return genericSingleStringResponse(p, RIL_UNSOL_STK_EVENT_NOTIFY, response, responselen);
}

int radio::modemResetInd(Parcel &p, void *response, size_t responselen)
{
    return genericSingleStringResponse(p, RIL_UNSOL_MODEM_RESTART, response, responselen);
}


int radio::dialResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::hangupConnectionResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}
int radio::hangupWaitingOrBackgroundResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::hangupForegroundResumeBackgroundResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}
int radio::switchWaitingOrHoldingAndActiveResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}
int radio::conferenceResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}
int radio::rejectCallResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}
int radio::sendDtmfResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}
int radio::sendUssdResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}
int radio::setClirResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}
int radio::setCallForwardResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setCallWaitingResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::acknowledgeLastIncomingGsmSmsResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::acceptCallResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::deactivateDataCallResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::startDtmfResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::stopDtmfResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::separateConnectionResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setMuteResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setSuppServiceNotificationsResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::deleteSmsOnSimResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setBandModeResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::sendTerminalResponseToSimResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::handleStkCallSetupRequestFromSimResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setPreferredNetworkTypeResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setLocationUpdatesResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setTTYModeResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setGsmBroadcastConfigResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setGsmBroadcastActivationResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::exitEmergencyCallbackModeResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setSmscAddressResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::reportStkServiceIsRunningResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::acknowledgeIncomingGsmSmsWithPduResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::iccCloseLogicalChannelResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::reportSmsMemoryStatusResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::explicitCallTransferResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setBarringPasswordResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setNetworkSelectionModeAutomaticResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setNetworkSelectionModeManualResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::cancelPendingUssdResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setRadioPowerResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::sendDeviceStateResponse(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setInitialAttachApnResponse(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::imsNetworkStateChangedInd(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setCellInfoListRateResponse(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::nvWriteItemResponse(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::nvResetConfigResponse(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setUiccSubscriptionResponse(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setDataAllowedResponse(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setDataProfileResponse(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::requestShutdownResponse(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setIndicationFilterResponse(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setSimCardPowerResponse(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setCarrierInfoForImsiEncryptionResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}
int radio::startNetworkScanResponse(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::stopNetworkScanResponse(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::stopKeepaliveResponse(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setCustomXmlPayloadResponse(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::setEcallTimerRestartResponse(Parcel &p, void *response, size_t responselen){
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}


int radio::networkStateChangedInd(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::radioStateChangedInd(Parcel &p, void *response, size_t responselen) {
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::callStateChangedInd(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::stkSessionEndInd(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::simSmsStorageFullInd(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}


int radio::enterEmergencyCallbackModeInd(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::resendIncallMuteInd(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::callRingInd(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::simStatusChangedInd(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::carrierInfoForImsiEncryptionInd(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

int radio::exitEmergencyCallbackModeInd(Parcel &p, void *response, size_t responselen)
{
    ACLOGD(LOG_TAG,"%s..",__func__);
    return 0;
}

// integer responses

int radio::supplyIccPinForAppResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntSingleOrEmptyRespOrInd(__func__,p,response,responselen);
}

int radio::supplyIccPukForAppResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntSingleOrEmptyRespOrInd(__func__,p,response,responselen);
}

int radio::supplyIccPin2ForAppResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntSingleOrEmptyRespOrInd(__func__,p,response,responselen);
}

int radio::supplyIccPuk2ForAppResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntSingleOrEmptyRespOrInd(__func__,p,response,responselen);
}

int radio::changeIccPinForAppResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntSingleOrEmptyRespOrInd(__func__,p,response,responselen);
}

int radio::changeIccPin2ForAppResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntSingleOrEmptyRespOrInd(__func__,p,response,responselen);
}

int radio::supplyNetworkDepersonalizationResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntSingleOrEmptyRespOrInd(__func__,p,response,responselen);
}

int radio::getLastCallFailCauseResponse(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;
    ACLOGV(LOG_TAG,"%s..",__func__);

    do {

        if (response == NULL) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            break;
        } else if (responselen == sizeof(int)) {  //get code only
            int *pInt = (int *) response;
            if (p.writeInt32(*pInt)!= NO_ERROR){break;}
            ret=0;
        } else if (responselen == sizeof(RIL_LastCallFailCauseInfo))  {
            p.writeInt32(1); // number of int value -- later number of RIL_LastFailCauseInfo
            RIL_LastCallFailCauseInfo *pCur= (RIL_LastCallFailCauseInfo *)response;
            if (p.writeInt32(pCur->cause_code) != NO_ERROR){break;}
            // Todo responder needs to be extended for p.writeStringToParcel(pCur->vendor_cause); handling
            ret=0;
        }
    } while (0);

    return ret;
}

int radio::getClirResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,2,p,response,responselen);
}

int radio::getCallWaitingResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,2,p,response,responselen);
}

int radio::getFacilityLockForAppResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::setFacilityLockForAppResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntSingleOrEmptyRespOrInd(__func__,p,response,responselen);
}

int radio::getClipResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::writeSmsToSimResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::getPreferredNetworkTypeResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::getTTYModeResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::getVoiceRadioTechnologyResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::setAllowedCarriersResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::stkCallSetupInd(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::restrictedStateChangedInd(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::indicateRingbackToneInd(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::rilConnectedInd(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::voiceRadioTechChangedInd(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::dataRadioTechChangedInd(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::incrementalNetworkScanInd(Parcel &p, void *response, size_t responselen)
{
    int numStrings;
    numStrings = responselen / sizeof(char *);
    return genericStringsRespOrInd(__func__,numStrings,p,response,responselen);
}

int radio::getDataRadioTechChangedInd(Parcel &p, void *response, size_t responselen)
{
    ACLOGV(LOG_TAG,"%s..",__func__);
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::subscriptionStatusChangedInd(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::srvccStateNotifyInd(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::newSmsOnSimInd(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::getMuteResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::getNetworkSelectionModeResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::getImsRegistrationStateResponse(Parcel &p, void *response, size_t responselen)
{
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::iccOpenLogicalChannelResponse(Parcel &p, void *response, size_t responselen)
{
    int paramNumber=responselen/sizeof(int);
    return genericIntVarRespOrInd(__func__,paramNumber,p,response,responselen);
}

int radio::getAvailableBandModesResponse(Parcel &p, void *response, size_t responselen)
{
    int paramNumber=responselen/sizeof(int);
    return genericIntVarRespOrInd(__func__,paramNumber,p,response,responselen);
}


int genericIntVarRespOrInd(const char * caller,int paramNumber, Parcel &p, void *response, size_t responselen)
{
    int numInts;
    int ret=RIL_ERRNO_INVALID_RESPONSE;
    ACLOGV(LOG_TAG,"%s..",__func__);

    do {

        if ((response == NULL) && (responselen != 0)) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",caller);
            break;
        }

        if (responselen % sizeof(int) != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected multiple of %d",caller,
                (int)responselen, (int)sizeof(int));
            break;
        }

        int *pInt = (int *) response;
        if (paramNumber <0) {
            numInts=*pInt;
            pInt++;
            numInts = responselen / sizeof(int *);
            if (numInts >= responselen/sizeof(int)){
                ACLOGW(LOG_TAG,"%s: missing values",caller);
                break;
            }
        } else{
            numInts=paramNumber;
        }

        if(p.writeInt32(numInts) != NO_ERROR){break;};
        ACLOGD(LOG_TAG,"%s: number= %d",caller,numInts);

        if (numInts == 0) {  //valid value
            ret=0;
            break;
        }

        for (int i = 0 ; i < numInts ; i++) {
            ACLOGD(LOG_TAG,"%s: value[%d]: %i",caller,i,pInt[i]);
            if(p.writeInt32(pInt[i]) != NO_ERROR){break;};
            if (numInts == i+1){
                ret=0;
            }
        }

    } while (0);
    return ret;
}

int genericIntFixedRespOrInd(const char * caller,int paramNumber, Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;
    ACLOGV(LOG_TAG,"%s..",__func__);

    do {

        if ((response == NULL) && (responselen != 0)) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",caller);
            break;
        }

        if (responselen % sizeof(int) != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected multiple of %d",caller,
                (int)responselen, (int)sizeof(int));
            break;
        }

        if (response==NULL) {
            ACLOGD(LOG_TAG,"%s: response = NULL",caller);
            break;
        }

        int *p_int = (int *) response;
        int numInts = responselen / sizeof(int);
        if (numInts != paramNumber){
            ACLOGW(LOG_TAG,"%s: invalid number %d; expected %d",caller,numInts,paramNumber);
            break;
        }

        if(p.writeInt32(numInts) != NO_ERROR){break;};
        if (numInts == 0) {  //valid value
            ret=0;
            break;
        }

        for (int i = 0 ; i < numInts ; i++) {
            ACLOGD(LOG_TAG,"%s: value[%d]: %i",caller,i,p_int[i]);
            if(p.writeInt32(p_int[i]) != NO_ERROR){break;};
            if (numInts == i+1){
                ret=0;
            }
        }

    } while (0);
    return ret;
}

int genericIntSingleOrEmptyRespOrInd(const char * caller, Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;
    ACLOGV(LOG_TAG,"%s..",__func__);

    do {

        if (responselen % sizeof(int) != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected multiple of %d",caller,
                (int)responselen, (int)sizeof(int));
            break;
        }

        int numInts = responselen / sizeof(int);
        if (numInts > 1){
            ACLOGW(LOG_TAG,"%s: invalid number %d; expected max. 1",caller,numInts);
            break;
        }
        if ((response == NULL) && (numInts != 0)) {
            ACLOGW(LOG_TAG,"%s: should have a non-zero response ",caller);
            break;
        }
        int *p_int = (int *) response;

        if(p.writeInt32(numInts) != NO_ERROR){break;};

        if (numInts > 0){
            if(p.writeInt32(p_int[0]) != NO_ERROR){break;};
        }
        ret=0;

    } while (0);
    return ret;
}



int radio::getCurrentCallsResponse(Parcel &p, void *response, size_t responselen)
{
    int num;

    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {

        if ((response == NULL) && (responselen != 0)) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            break;
        }

        if (responselen % sizeof (RIL_Call *) != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected multiple of %d",__func__,
                (int)responselen, (int)sizeof (RIL_Call *));
            break;
        }

        /* number of call info's */
        num = responselen / sizeof(RIL_Call *);
        ACLOGD(LOG_TAG,"%s: len = %d",__func__,num);
        if(p.writeInt32(num) != NO_ERROR ){break;};
        if (num == 0){
            ret=0;
            break;
        }

        for (int i = 0 ; i < num ; i++) {
            RIL_Call *p_cur = ((RIL_Call **) response)[i];

            ACLOGD(LOG_TAG,"%s: p_cur = %p",__func__,p_cur);

            /* each call info */
            p.writeInt32(p_cur->state);
            ACLOGD(LOG_TAG,"%s: state[%d]: %d",__func__,i,p_cur->state);
            p.writeInt32(p_cur->index);
            ACLOGD(LOG_TAG,"%s: index[%d]: %d",__func__,i,p_cur->index);
            p.writeInt32(p_cur->toa);
            ACLOGD(LOG_TAG,"%s: toa[%d]: %d",__func__,i,p_cur->toa);
            p.writeInt32(p_cur->isMpty);
            ACLOGD(LOG_TAG,"%s: isMpty[%d]: %d",__func__,i,p_cur->isMpty);
            p.writeInt32(p_cur->isMT);
            ACLOGD(LOG_TAG,"%s: isMT[%d]: %d",__func__,i,p_cur->isMT);
            p.writeInt32(p_cur->als);
            ACLOGD(LOG_TAG,"%s: als[%d]: %d",__func__,i,p_cur->als);
            p.writeInt32(p_cur->isVoice);
            ACLOGD(LOG_TAG,"%s: isVoice[%d]: %d",__func__,i,p_cur->isVoice);
            p.writeInt32(p_cur->isVoicePrivacy);
            ACLOGD(LOG_TAG,"%s: isVoicePrivacy[%d]: %d",__func__,i,p_cur->isVoicePrivacy);
            p.writeStringToParcel(p_cur->number);
            if (p_cur->number==NULL){
                ACLOGD(LOG_TAG,"%s: number[%d]: NULL",__func__,i);
            } else {
                ACLOGD(LOG_TAG,"%s: number[%d]: %s",__func__,i,p_cur->number);
            }

            p.writeInt32(p_cur->numberPresentation);
            ACLOGD(LOG_TAG,"%s: numberPresentation[%d]: %d",__func__,i,p_cur->numberPresentation);

            p.writeStringToParcel(p_cur->name);
            if (p_cur->name==NULL){
                ACLOGD(LOG_TAG,"%s: name[%d]: NULL",__func__,i);
            } else {
                ACLOGD(LOG_TAG,"%s: name[%d]: %s",__func__,i,p_cur->name);
            }

            p.writeInt32(p_cur->namePresentation);
            ACLOGD(LOG_TAG,"%s: namePresentation[%d]: %d",__func__,i,p_cur->namePresentation);

            if ((s_callbacks.version < 3) || (p_cur->uusInfo == 0) || (p_cur->uusInfo->uusData == 0)) {
                ACLOGN(LOG_TAG,"%s: uus info[%d]: NULL",__func__,i);
                p.writeInt32(0); /* UUS Information is absent */
            } else {
                ACLOGN(LOG_TAG,"%s: uus info[%d] available",__func__,i);
                RIL_UUS_Info *uusInfo = p_cur->uusInfo;
                p.writeInt32(1); /* UUS Information is present */
                p.writeInt32(uusInfo->uusType);
                ACLOGD(LOG_TAG,"%s: uus info type[%d]: %d",__func__,i,uusInfo->uusType);
                p.writeInt32(uusInfo->uusDcs);
                ACLOGD(LOG_TAG,"%s: uus info dcs[%d]: %d",__func__,i,uusInfo->uusDcs);
                p.writeInt32(uusInfo->uusLength);
                ACLOGD(LOG_TAG,"%s: uus info length[%d]: %d",__func__,i,uusInfo->uusLength);
                p.write(uusInfo->uusData, uusInfo->uusLength);
            }
        }
        ret=0;
    } while (0);

    return ret;
}


int radio::dataCallListChangedInd(Parcel &p, void *response, size_t responselen)
{
    int num;
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {

        if ((response == NULL) && (responselen != 0)) {
            ACLOGW(LOG_TAG,"%s: invalid response = NULL",__func__);
            break;
        }

        if ((responselen % sizeof(RIL_Data_Call_Response_v11) != 0) &&
            (responselen % sizeof(RIL_Data_Call_Response_v9) != 0) &&
            (responselen % sizeof(RIL_Data_Call_Response_v6) != 0) ) {
            ACLOGE(LOG_TAG,"%s: invalid response length %d ",__func__, (int)responselen );
            break;
        }

        if (s_callbacks.version < 5) {
            ACLOGW(LOG_TAG,"%s: unsupported version = %d",__func__,s_callbacks.version);
            break;
        }

        if (responselen % sizeof(RIL_Data_Call_Response_v11) == 0) {
            int num = responselen / sizeof(RIL_Data_Call_Response_v11);
            p.writeInt32(num);
            p.writeInt32(11); //insert version number
            RIL_Data_Call_Response_v11 *p_cur = (RIL_Data_Call_Response_v11 *) response;
            for (int i = 0; i < num; i++) {
                p.writeInt32((int)p_cur[i].status);
                p.writeInt32(p_cur[i].suggestedRetryTime);
                p.writeInt32(p_cur[i].cid);
                p.writeInt32(p_cur[i].active);
                p.writeStringToParcel(p_cur[i].type);
                p.writeStringToParcel(p_cur[i].ifname);
                p.writeStringToParcel(p_cur[i].addresses);
                p.writeStringToParcel(p_cur[i].dnses);
                p.writeStringToParcel(p_cur[i].gateways);
                p.writeStringToParcel(p_cur[i].pcscf);
                p.writeInt32((int)p_cur[i].mtu);
            }
            ret=0;
            break;
        }

        if (responselen % sizeof(RIL_Data_Call_Response_v9) == 0) {
            int num = responselen / sizeof(RIL_Data_Call_Response_v9);
            p.writeInt32(num);
            p.writeInt32(9); //insert version number
            RIL_Data_Call_Response_v9 *p_cur = (RIL_Data_Call_Response_v9 *) response;
            for (int i = 0; i < num; i++) {
                p.writeInt32((int)p_cur[i].status);
                p.writeInt32(p_cur[i].suggestedRetryTime);
                p.writeInt32(p_cur[i].cid);
                p.writeInt32(p_cur[i].active);
                p.writeStringToParcel(p_cur[i].type);
                p.writeStringToParcel(p_cur[i].ifname);
                p.writeStringToParcel(p_cur[i].addresses);
                p.writeStringToParcel(p_cur[i].dnses);
                p.writeStringToParcel(p_cur[i].gateways);
                p.writeStringToParcel(p_cur[i].pcscf);
                //p.writeInt32(-1);
            }
            ret=0;
            break;
        }

        if (responselen % sizeof(RIL_Data_Call_Response_v6) == 0) {
            int num = responselen / sizeof(RIL_Data_Call_Response_v6);
            p.writeInt32(num);
             p.writeInt32(6); //insert version number
            RIL_Data_Call_Response_v6 *p_cur = (RIL_Data_Call_Response_v6 *) response;
            for (int i = 0; i < num; i++) {
                p.writeInt32((int)p_cur[i].status);
                p.writeInt32(p_cur[i].suggestedRetryTime);
                p.writeInt32(p_cur[i].cid);
                p.writeInt32(p_cur[i].active);
                p.writeStringToParcel(p_cur[i].type);
                p.writeStringToParcel(p_cur[i].ifname);
                p.writeStringToParcel(p_cur[i].addresses);
                p.writeStringToParcel(p_cur[i].dnses);
                p.writeStringToParcel(p_cur[i].gateways);
                p.writeStringToParcel( "");
                p.writeInt32(-1);
            }
            ret=0;
            break;
        }

    } while (0);

    return ret;
}

int responseSendSmsGeneric (Parcel &p, int id, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do  {
        if (response == NULL) {
            ACLOGV(LOG_TAG,"%s: response = NULL",__func__);
            break;
        }

        if (responselen != sizeof (RIL_SMS_Response) ) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected %d",__func__,
                    (int)responselen, (int)sizeof (RIL_SMS_Response));
            break;
        }

        if (p.writeInt32(1) != NO_ERROR){break;} //number of basic data objects

        RIL_SMS_Response *p_cur = (RIL_SMS_Response *) response;

        if (p.writeInt32(p_cur->messageRef)!= NO_ERROR){break;}
        p.writeStringToParcel(p_cur->ackPDU);
        if (p.writeInt32(p_cur->errorCode)!= NO_ERROR){break;}

        ACLOGI(LOG_TAG,"%s: response on %d ",__func__,id);

        ACLOGD(LOG_TAG,"%s: messageRef=%d",__func__,p_cur->messageRef);

        if (p_cur->ackPDU!=NULL){
            ACLOGD(LOG_TAG,"%s: p_cur->ackPDU=%s",__func__,p_cur->ackPDU);
        } else {
            ACLOGD(LOG_TAG,"%s: p_cur->ackPDU=NULL",__func__);
        }
        ACLOGD(LOG_TAG,"%s: errorCode=%d",__func__,p_cur->errorCode);
        ret=0;
    } while (0);

    return ret;
}

int radio::sendSmsResponse(Parcel &p, void *response, size_t responselen)
{
    return responseSendSmsGeneric (p, RIL_REQUEST_SEND_SMS, response, responselen);
}

int radio::sendSmsResponseExpectMore(Parcel &p, void *response, size_t responselen)
{
    return responseSendSmsGeneric (p, RIL_REQUEST_SEND_SMS_EXPECT_MORE, response, responselen);
}

int radio::sendImsSmsResponse(Parcel &p, void *response, size_t responselen)
{
    return responseSendSmsGeneric (p, RIL_REQUEST_IMS_SEND_SMS, response, responselen);
}


int radio::getDataCallListResponse(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;
    //int loglevel=LOG_INFO;

    do  {

        if ((response == NULL) && (responselen != 0)) {
            ACLOGW(LOG_TAG,"%s: invalid response = NULL",__func__);
            break;
        }

        if ((responselen % sizeof(RIL_Data_Call_Response_v11) != 0) &&
            (responselen % sizeof(RIL_Data_Call_Response_v9) != 0) &&
            (responselen % sizeof(RIL_Data_Call_Response_v6) != 0))  {
            ACLOGE(LOG_TAG,"%s: invalid response length %d ",__func__, (int)responselen );
            break;
        }

        if (s_callbacks.version < 5) {
            ACLOGW(LOG_TAG,"%s: unsupported version = %d",__func__,s_callbacks.version);
            break;
        }

        if (responselen % sizeof(RIL_Data_Call_Response_v11) == 0) {
            int num = responselen / sizeof(RIL_Data_Call_Response_v11);
            p.writeInt32(num);
            p.writeInt32(11);

            ACLOGI(LOG_TAG,"%s: send call list v11 ..",__func__);
            RIL_Data_Call_Response_v11 *p_cur = (RIL_Data_Call_Response_v11 *) response;
            for (int i = 0; i < num; i++) {
                p.writeInt32((int)p_cur[i].status);
                p.writeInt32(p_cur[i].suggestedRetryTime);
                p.writeInt32(p_cur[i].cid);
                p.writeInt32(p_cur[i].active);
                p.writeStringToParcel(p_cur[i].type);
                p.writeStringToParcel(p_cur[i].ifname);
                p.writeStringToParcel(p_cur[i].addresses);
                p.writeStringToParcel(p_cur[i].dnses);
                p.writeStringToParcel(p_cur[i].gateways);
                p.writeStringToParcel(p_cur[i].pcscf);
                p.writeInt32((int)p_cur[i].mtu);
            }
            ret=0;
            break;
        }

        if (responselen % sizeof(RIL_Data_Call_Response_v9) == 0) {
            int num = responselen / sizeof(RIL_Data_Call_Response_v9);
            ACLOGI(LOG_TAG,"%s: send call list v9 ..",__func__);
            p.writeInt32(num);
            p.writeInt32(9);
            RIL_Data_Call_Response_v9 *p_cur = (RIL_Data_Call_Response_v9 *) response;
            for (int i = 0; i < num; i++) {
                p.writeInt32((int)p_cur[i].status);
                p.writeInt32(p_cur[i].suggestedRetryTime);
                p.writeInt32(p_cur[i].cid);
                p.writeInt32(p_cur[i].active);
                p.writeStringToParcel(p_cur[i].type);
                p.writeStringToParcel(p_cur[i].ifname);
                p.writeStringToParcel(p_cur[i].addresses);
                p.writeStringToParcel(p_cur[i].dnses);
                p.writeStringToParcel(p_cur[i].gateways);
                p.writeStringToParcel(p_cur[i].pcscf);
            }
            ret=0;
            break;
        }

        if (responselen % sizeof(RIL_Data_Call_Response_v6) == 0) {
            int num = responselen / sizeof(RIL_Data_Call_Response_v6);
            ACLOGI(LOG_TAG,"%s: send call list v6 ..",__func__);
            p.writeInt32(num);
            p.writeInt32(6);
            RIL_Data_Call_Response_v6 *p_cur = (RIL_Data_Call_Response_v6 *) response;
            for (int i = 0; i < num; i++) {
                p.writeInt32((int)p_cur[i].status);
                p.writeInt32(p_cur[i].suggestedRetryTime);
                p.writeInt32(p_cur[i].cid);
                p.writeInt32(p_cur[i].active);
                p.writeStringToParcel(p_cur[i].type);
                p.writeStringToParcel(p_cur[i].ifname);
                p.writeStringToParcel(p_cur[i].addresses);
                p.writeStringToParcel(p_cur[i].dnses);
                p.writeStringToParcel(p_cur[i].gateways);
            }
            ret=0;
            break;
        }

    } while (0);

    return ret;
}


int radio::setupDataCallResponse(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;
    do {

        if ((response == NULL) && (responselen !=0)){
            //note: an empty response/responselen=0 might be a valid result;
            // and provide a generic 'empty  response; see below
            ACLOGW(LOG_TAG,"%s: invalid response = NULL",__func__);
            break;
        }

        p.writeInt32(1); //number of basic data objects

        if (responselen == sizeof(RIL_Data_Call_Response_v11) ) {
            p.writeInt32(11); //version
            RIL_Data_Call_Response_v11 *pResponse= (RIL_Data_Call_Response_v11 *) response;
            p.writeInt32(pResponse->status);
            p.writeInt32(pResponse->suggestedRetryTime);
            p.writeInt32(pResponse->cid);
            p.writeInt32(pResponse->active);
            p.writeStringToParcel(pResponse->type);
            p.writeStringToParcel(pResponse->ifname);
            p.writeStringToParcel(pResponse->addresses);
            p.writeStringToParcel(pResponse->dnses);
            p.writeStringToParcel(pResponse->gateways);
            p.writeStringToParcel(pResponse->pcscf);
            p.writeInt32(pResponse->mtu);
            ret=0;
            break;
        }

        if (responselen == sizeof(RIL_Data_Call_Response_v9)) {
            p.writeInt32(9); //version
            RIL_Data_Call_Response_v9 *pResponse= (RIL_Data_Call_Response_v9 *) response;
            p.writeInt32(pResponse->status);
            p.writeInt32(pResponse->suggestedRetryTime);
            p.writeInt32(pResponse->cid);
            p.writeInt32(pResponse->active);
            p.writeStringToParcel(pResponse->type);
            p.writeStringToParcel(pResponse->ifname);
            p.writeStringToParcel(pResponse->addresses);
            p.writeStringToParcel(pResponse->dnses);
            p.writeStringToParcel(pResponse->gateways);
            p.writeStringToParcel(pResponse->pcscf);
            p.writeInt32(-1); //mtu not specified
            ret=0;
            break;
        }

        if (responselen == sizeof(RIL_Data_Call_Response_v6)) {
            p.writeInt32(6); //version
            RIL_Data_Call_Response_v6 *pResponse= (RIL_Data_Call_Response_v6 *) response;
            p.writeInt32(pResponse->status);
            p.writeInt32(pResponse->suggestedRetryTime);
            p.writeInt32(pResponse->cid);
            p.writeInt32(pResponse->active);
            p.writeStringToParcel(pResponse->type);
            p.writeStringToParcel(pResponse->ifname);
            p.writeStringToParcel(pResponse->addresses);
            p.writeStringToParcel(pResponse->dnses);
            p.writeStringToParcel(pResponse->gateways);
            p.writeStringToParcel("");//pscf not specified
            p.writeInt32(-1); //mtu not specified
            ret=0;
            break;
        }

        if (response != NULL) {
            ACLOGW(LOG_TAG,"%s: invalid (generic) response",__func__);
            break;
        }

        p.writeInt32(5); //minimum version
        p.writeInt32(PDP_FAIL_ERROR_UNSPECIFIED);
        p.writeInt32(0); //status
        p.writeInt32(-1); //suggestedRetryTime
        p.writeInt32(-1); //cid
        p.writeStringToParcel("");//type
        p.writeStringToParcel("");//ifname
        p.writeStringToParcel("");//addresses
        p.writeStringToParcel("");//dnses
        p.writeStringToParcel("");//gateways
        p.writeStringToParcel("");//pscf
        ret=0;
    } while (0);

    return ret;
}

int radio::newBroadcastSmsInd(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {

        if ((response == NULL) && (responselen != 0)) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL with responselen != 0",__func__);
            break;
        }

        p.writeInt32(1); //number of basic data objects

        // The java code reads -1 size as null byte array
        if (response == NULL) {
            ACLOGD(LOG_TAG,"%s: length = -1",__func__);
            p.writeInt32(-1);
        } else {
            p.writeInt32(responselen);
            ACLOGD(LOG_TAG,"%s: length = %d",__func__,responselen);
            p.write(response, responselen);
            unsigned char *responseData=(unsigned char *)response;
            for (int idx=0; idx<responselen;idx++){
                ACLOGN(LOG_TAG,"%s: data[%d] = 0x%c",__func__,idx,*(responseData+idx));
            }
        }
        ret=0;

    } while (0);

    return ret;
}

int radio::oemHookRawIndOrResp(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {

        if ((response == NULL) && (responselen != 0)) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL with responselen != 0",__func__);
            break;
        }

        if (p.writeInt32(1) != NO_ERROR) {break;} //number of basic data objects

        // The java code reads -1 size as null byte array
        if (response == NULL) {
            ACLOGD(LOG_TAG,"%s: length = -1",__func__);
            if (p.writeInt32(-1)!= NO_ERROR) {break;}
        } else {
            if (p.writeInt32(responselen) != NO_ERROR) {break;}
            ACLOGD(LOG_TAG,"%s: length = %d",__func__,responselen);
            if (p.write(response, responselen) != NO_ERROR) {break;}

            unsigned char *responseData=(unsigned char *)response;
            for (int idx=0; idx<responselen;idx++){
                ACLOGI(LOG_TAG,"%s: data[%d] = 0x%c",__func__,idx,*(responseData+idx));
            }
        }

        ret=0;

    } while (0);

    return ret;
}

int radio::sendEnvelopeWithStatusResponse(Parcel &p, void *response, size_t responselen)
{
    return radio::responseIccIo(__func__, p, response, responselen);
}

int radio::iccTransmitApduBasicChannelResponse(Parcel &p, void *response, size_t responselen)
{
    return radio::responseIccIo(__func__, p, response, responselen);
}

int radio::iccIOForAppResponse(Parcel &p, void *response, size_t responselen)
{
    return radio::responseIccIo(__func__, p, response, responselen);
}

int radio::iccTransmitApduLogicalChannelResponse(Parcel &p, void *response, size_t responselen)
{
    return radio::responseIccIo(__func__, p, response, responselen);
}

int radio::requestIccSimAuthenticationResponse(Parcel &p, void *response, size_t responselen)
{
    return radio::responseIccIo(__func__, p, response, responselen);
}

int radio::responseIccIo(const char *caller, Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        if (response == NULL){
            ACLOGW(LOG_TAG,"%s: missing response",caller);
            break;
        }

        if (responselen != sizeof (RIL_SIM_IO_Response) ) {
            ACLOGW(LOG_TAG,"%s: invalid response length was %d expected %d",caller,
                    (int)responselen, (int)sizeof (RIL_SIM_IO_Response));
            break;
        }

        if (p.writeInt32(1) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",caller);
            break;
        }

        RIL_SIM_IO_Response *p_cur = (RIL_SIM_IO_Response *) response;
        if (p.writeInt32(p_cur->sw1) != NO_ERROR) {
            ACLOGW(LOG_TAG,"%s: could not write sw1",caller);
            break;
        }

        if (p.writeInt32(p_cur->sw2)!= NO_ERROR) {
            ACLOGW(LOG_TAG,"%s: could not write sw2",caller);
            break;
        }

        p.writeStringToParcel(p_cur->simResponse);

        ACLOGD(LOG_TAG,"%s: sw1= %d",caller,p_cur->sw1);
        ACLOGD(LOG_TAG,"%s: sw2= %d",caller,p_cur->sw2);
        if (p_cur->simResponse==NULL){
            ACLOGD(LOG_TAG,"%s: response = NULL",caller);
        } else {
            ACLOGD(LOG_TAG,"%s: response = %s",caller,p_cur->simResponse);
        }
        ret=0;
    } while (0);
    return ret;
}

int radio::startKeepaliveResponse(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do  {

        if (response == NULL){
            ACLOGW(LOG_TAG,"%s: missing response",__func__);
            break;
        }

        if (responselen != sizeof (RIL_KeepaliveStatus ) ) {
            ACLOGW(LOG_TAG,"%s: invalid response length was %d expected %d",__func__,
                    (int)responselen, (int)sizeof (RIL_KeepaliveStatus ));
            break;
        }

        if (p.writeInt32(1) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }

        RIL_KeepaliveStatus *p_cur = (RIL_KeepaliveStatus *) response;
        if (p.writeInt32(p_cur->sessionHandle) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->code)!= NO_ERROR) {break;}
        ret=0;

    } while (0);

    return ret;
}

int radio::getRadioCapabilityResponse(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do  {

        if (response == NULL){
            ACLOGW(LOG_TAG,"%s: missing response",__func__);
            break;
        }

        if (responselen != sizeof (RIL_RadioCapability  ) ) {
            ACLOGW(LOG_TAG,"%s: invalid response length was %d expected %d",__func__,
                    (int)responselen, (int)sizeof (RIL_RadioCapability  ));
            break;
        }

        if (p.writeInt32(1) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }

        RIL_RadioCapability  *p_cur = (RIL_RadioCapability *) response;
        if (p.writeInt32(p_cur->version) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->session)!= NO_ERROR) {break;}
        if (p.writeInt32(p_cur->phase)!= NO_ERROR) {break;}
        if (p.writeInt32(p_cur->rat)!= NO_ERROR) {break;}
        if (p.write(p_cur->logicalModemUuid,MAX_UUID_LENGTH)!= NO_ERROR) {break;}
        if (p.writeInt32(p_cur->status)!= NO_ERROR) {break;}
        ret=0;

    } while (0);

    return ret;
}

int radio::getAllowedCarriersResponse(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do  {

        if (response == NULL){
            ACLOGW(LOG_TAG,"%s: missing response",__func__);
            break;
        }

        if (responselen != sizeof (RIL_CarrierRestrictions  ) ) {
            ACLOGW(LOG_TAG,"%s: invalid response length was %d expected %d",__func__,
                    (int)responselen, (int)sizeof (RIL_CarrierRestrictions  ));
            break;
        }

        if (p.writeInt32(1) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }

        RIL_CarrierRestrictions   *p_cur = (RIL_CarrierRestrictions  *) response;

        if(p.writeInt32(p_cur->len_allowed_carriers) != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not write len_allowed_carriers",__func__);
            break;
        }

        if ( p.writeInt32(p_cur->len_excluded_carriers) != NO_ERROR){
            ACLOGW(LOG_TAG,"%s: could not read len_excluded_carriers",__func__);
            break;
        }

        for (int i = 0; i < p_cur->len_allowed_carriers; i++) {
            p.writeStringToParcel(p_cur->allowed_carriers[i].mcc);
            p.writeStringToParcel(p_cur->allowed_carriers[i].mnc);
            p.writeInt32(p_cur->allowed_carriers[i].match_type);
            p.writeStringToParcel(p_cur->allowed_carriers[i].match_data );
        }

        for (int i = 0; i < p_cur->len_excluded_carriers; i++) {
            p.writeStringToParcel(p_cur->excluded_carriers[i].mcc);
            p.writeStringToParcel(p_cur->excluded_carriers[i].mnc);
            p.writeInt32(p_cur->excluded_carriers[i].match_type);
            p.writeStringToParcel(p_cur->excluded_carriers[i].match_data );
        }
        ret=0;

    } while (0);

    return ret;
}


int radio::setRadioCapabilityResponse(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do  {
        if (response == NULL) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            break;
        }

        if (responselen != sizeof (RIL_RadioCapability  ) ) {
            ACLOGW(LOG_TAG,"%s: invalid response length was %d expected %d",__func__,
                    (int)responselen, (int)sizeof (RIL_RadioCapability  ));
            break;
        }

        if (p.writeInt32(1) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }

        RIL_RadioCapability  *p_cur = (RIL_RadioCapability *) response;
        if (p.writeInt32(p_cur->version) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->session)!= NO_ERROR) {break;}
        if (p.writeInt32(p_cur->phase)!= NO_ERROR) {break;}
        if (p.writeInt32(p_cur->rat)!= NO_ERROR) {break;}
        int size = strlen(p_cur->logicalModemUuid)+1;
        if (p.write(p_cur->logicalModemUuid,MAX_UUID_LENGTH)!= NO_ERROR) {break;}
        if (p.writeInt32(p_cur->status)!= NO_ERROR) {break;}
        ret=0;

    } while (0);

    return ret;
}



int radio::getCallForwardStatusResponse(Parcel &p, void *response, size_t responselen)
{
    int num;
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {

        if ((response == NULL) && (responselen != 0)) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            break;
        }

        if (responselen % sizeof(RIL_CallForwardInfo *) != 0) {
            ACLOGW(LOG_TAG,"%s:invalid response length %d expected multiple of %d",__func__,
                    (int)responselen, (int)sizeof(RIL_CallForwardInfo *));
            break;
        }

        /* number of call info's */
        num = responselen / sizeof(RIL_CallForwardInfo *); // num=0 is a valid value
        if (p.writeInt32(num)!= NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }
        ACLOGD(LOG_TAG,"%s: element number= %d",__func__,num);

        for (int i = 0 ; i < num ; i++) {
            RIL_CallForwardInfo *p_cur = ((RIL_CallForwardInfo **) response)[i];

            p.writeInt32(p_cur->status);
            p.writeInt32(p_cur->reason);
            p.writeInt32(p_cur->serviceClass);
            p.writeInt32(p_cur->toa);
            p.writeStringToParcel(p_cur->number);
            p.writeInt32(p_cur->timeSeconds);

            ACLOGD(LOG_TAG,"%s: status[%d]: %d",__func__,i,p_cur->status);
            ACLOGD(LOG_TAG,"%s: reason[%d]: %d",__func__,i,p_cur->reason);
            ACLOGD(LOG_TAG,"%s: serviceClass[%d]: %d",__func__,i,p_cur->serviceClass);
            ACLOGD(LOG_TAG,"%s: toa[%d]: %d",__func__,i,p_cur->toa);
            if (p_cur->number==NULL){
                ACLOGD(LOG_TAG,"%s: number = NULL",__func__);
            } else {
                ACLOGD(LOG_TAG,"%s: number = %s",__func__,p_cur->number);
            }
            ACLOGD(LOG_TAG,"%s: timeSeconds[%d]: %d",__func__,i,p_cur->timeSeconds);
        }
        ret=0;
    } while (0);
    return ret;
}

int radio::suppSvcNotifyInd(Parcel &p, void *response, size_t responselen)
{

    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do  {
        if (response == NULL) {
            ACLOGW(LOG_TAG,"%s: response = NULL",__func__);
            break;
        }

        if (responselen != sizeof(RIL_SuppSvcNotification)) {
            ACLOGW(LOG_TAG,"%s: invalid response length was %d expected %d",__func__,
                    (int)responselen, (int)sizeof (RIL_SuppSvcNotification));
            break;
        }

        if (p.writeInt32(1) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }

        RIL_SuppSvcNotification *p_cur = (RIL_SuppSvcNotification *) response;
        if (p.writeInt32(p_cur->notificationType)!= NO_ERROR) {break;}
        if (p.writeInt32(p_cur->code)!= NO_ERROR) {break;}
        if (p.writeInt32(p_cur->index)!= NO_ERROR) {break;}
        if (p.writeInt32(p_cur->type)!= NO_ERROR) {break;}
        p.writeStringToParcel(p_cur->number);

        ACLOGD(LOG_TAG,"%s: notificationType: %d",__func__,p_cur->notificationType);
        ACLOGD(LOG_TAG,"%s: code: %d",__func__,p_cur->code);
        ACLOGD(LOG_TAG,"%s: index: %d",__func__,p_cur->index);
        ACLOGD(LOG_TAG,"%s: type: %d",__func__,p_cur->type);
        if (p_cur->number==NULL){
            ACLOGD(LOG_TAG,"%s: number = NULL",__func__);
        } else {
            ACLOGD(LOG_TAG,"%s: number = %s",__func__,p_cur->number);
        }

        ret=0;

    } while (0);

    return ret;
}

int radio::getNeighboringCidsResponse(Parcel &p, void *response, size_t responselen)
{
    int num;
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        if ((response == NULL) && (responselen != 0)) {
            ACLOGW(LOG_TAG,"%s: response = NULL",__func__);
            break;
        }

        if (responselen % sizeof (RIL_NeighboringCell *) != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected multiple of %d",__func__,
                (int)responselen, (int)sizeof (RIL_NeighboringCell *));
            break;
        }

        /* number of records */
        num = responselen / sizeof(RIL_NeighboringCell *);
        if ( p.writeInt32(num)!= NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }
        ACLOGD(LOG_TAG,"%s: number of cells: %d",__func__,num);
        if (num == 0){
            ret=0;
            break;
        }

        for (int i = 0 ; i < num ; i++) {
            RIL_NeighboringCell *p_cur = ((RIL_NeighboringCell **) response)[i];

            ACLOGD(LOG_TAG,"%s: cell[%d]: rssi= %d, cid= %s",__func__,i,p_cur->rssi,p_cur->cid);

            p.writeInt32(p_cur->rssi);
            p.writeStringToParcel( p_cur->cid);
        }
        ret=0;

    } while (0);
    return ret;
}

int radio::currentSignalStrengthInd(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;
    do {

        if (response == NULL) {
            ACLOGW(LOG_TAG,"%s: response = NULL",__func__);
            break;
        }

        if (p.writeInt32(1) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }

        if (responselen < sizeof (RIL_SignalStrength_v11)) {

            if (p.writeInt32(10) != NO_ERROR) {
                //number of basic data objects
                ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
                break;
            }

            RIL_SignalStrength_v10 *p_cur = ((RIL_SignalStrength_v10 *) response);

            if (p.writeInt32(p_cur->GW_SignalStrength.signalStrength) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->GW_SignalStrength.bitErrorRate) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->CDMA_SignalStrength.dbm) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->CDMA_SignalStrength.ecio) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->EVDO_SignalStrength.dbm) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->EVDO_SignalStrength.ecio) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->EVDO_SignalStrength.signalNoiseRatio) != NO_ERROR) {break;}

            if (p.writeInt32(p_cur->LTE_SignalStrength.signalStrength) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.rsrp) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.rsrq) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.rssnr) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.cqi) != NO_ERROR) {break;}

            if (p.writeInt32(p_cur->TD_SCDMA_SignalStrength.rscp) != NO_ERROR) {break;}

            ACLOGD(LOG_TAG,"%s: GW signalStrength: %d",__func__,p_cur->GW_SignalStrength.signalStrength);
            ACLOGD(LOG_TAG,"%s: GW error rate: %d",__func__,p_cur->GW_SignalStrength.bitErrorRate);
            ACLOGD(LOG_TAG,"%s: LTE signalStrength: %d",__func__,p_cur->LTE_SignalStrength.signalStrength);
            ACLOGD(LOG_TAG,"%s: LTE rsrp: %d",__func__,p_cur->LTE_SignalStrength.rsrp);
            ACLOGD(LOG_TAG,"%s: LTE rsrq: %d",__func__,p_cur->LTE_SignalStrength.rsrq);
            ACLOGD(LOG_TAG,"%s: LTE rssnr: %d",__func__,p_cur->LTE_SignalStrength.rssnr);
            ACLOGD(LOG_TAG,"%s: LTE cqi: %d",__func__,p_cur->LTE_SignalStrength.cqi);

        } else {

            if (p.writeInt32(11) != NO_ERROR) {
                //number of basic data objects
                ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
                break;
            }

            RIL_SignalStrength_v11 *p_cur = ((RIL_SignalStrength_v11 *) response);
            if (p.writeInt32(p_cur->GW_SignalStrength.signalStrength) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->GW_SignalStrength.bitErrorRate) != NO_ERROR) {break;}

            if (p.writeInt32(p_cur->CDMA_SignalStrength.dbm) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->CDMA_SignalStrength.ecio) != NO_ERROR) {break;}

            if (p.writeInt32(p_cur->EVDO_SignalStrength.dbm) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->EVDO_SignalStrength.ecio) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->EVDO_SignalStrength.signalNoiseRatio) != NO_ERROR) {break;}

            if (p.writeInt32(p_cur->LTE_SignalStrength.signalStrength) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.rsrp) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.rsrq) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.rssnr) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.cqi) != NO_ERROR) {break;}

            if (p.writeInt32(p_cur->TD_SCDMA_SignalStrength.rscp) != NO_ERROR) {break;}

            if (p.writeInt32(p_cur->NR5G_SignalStrength.rsrp) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->NR5G_SignalStrength.rsrq) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->NR5G_SignalStrength.sinr) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->NR5G_SignalStrength.csiRsrp) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->NR5G_SignalStrength.csiRsrq) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->NR5G_SignalStrength.csiSinr) != NO_ERROR){break;}

            ACLOGD(LOG_TAG,"%s: GW signalStrength: %d",__func__,p_cur->GW_SignalStrength.signalStrength);
            ACLOGD(LOG_TAG,"%s: GW error rate: %d",__func__,p_cur->GW_SignalStrength.bitErrorRate);
            ACLOGD(LOG_TAG,"%s: LTE signalStrength: %d",__func__,p_cur->LTE_SignalStrength.signalStrength);
            ACLOGD(LOG_TAG,"%s: LTE rsrp: %d",__func__,p_cur->LTE_SignalStrength.rsrp);
            ACLOGD(LOG_TAG,"%s: LTE rsrq: %d",__func__,p_cur->LTE_SignalStrength.rsrq);
            ACLOGD(LOG_TAG,"%s: LTE rssnr: %d",__func__,p_cur->LTE_SignalStrength.rssnr);
            ACLOGD(LOG_TAG,"%s: LTE cqi: %d",__func__,p_cur->LTE_SignalStrength.cqi);
            ACLOGD(LOG_TAG,"%s: NR5G rsrp: %d",__func__,p_cur->NR5G_SignalStrength.rsrp);
            ACLOGD(LOG_TAG,"%s: NR5G rsrq: %d",__func__,p_cur->NR5G_SignalStrength.rsrq);
            ACLOGD(LOG_TAG,"%s: NR5G sinr: %d",__func__,p_cur->NR5G_SignalStrength.sinr);
            ACLOGD(LOG_TAG,"%s: NR5G csiRsrp: %d",__func__,p_cur->NR5G_SignalStrength.csiRsrp);
            ACLOGD(LOG_TAG,"%s: NR5G csiRsrq: %d",__func__,p_cur->NR5G_SignalStrength.csiRsrq);
            ACLOGD(LOG_TAG,"%s: NR5G csiSinr: %d",__func__,p_cur->NR5G_SignalStrength.csiSinr);

        }
        ret=0;
    } while (0);

    return ret;
}

int genericCellInfoListIndOrResp(Parcel &p, int id, void *response, size_t responselen)
{
    int num;
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        if ((response == NULL) && (responselen != 0)) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            break;
        }

        if (responselen % sizeof(RIL_CellInfo_v12) != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected multiple of  %d",__func__,
                (int)responselen, (int)sizeof(RIL_CellInfo_v12));
            break;
        }
        num=responselen / sizeof(RIL_CellInfo_v12);

        ACLOGI(LOG_TAG,"%s: write request %d",__func__,id);

        if (p.writeInt32(num) != NO_ERROR) {break;}
        if (num == 0){
            ret=0;
            break;
        }

        RIL_CellInfo_v12 *p_cur= (RIL_CellInfo_v12 *)(response);
        for (int i =0; i<num; i++){

            if (p.writeInt32(p_cur[i].cellInfoType) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur[i].registered) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur[i].timeStampType) != NO_ERROR) {break;}
            if (p.writeInt64(p_cur[i].timeStamp) != NO_ERROR) {break;}

            if (p_cur[i].cellInfoType == RIL_CELL_INFO_TYPE_NONE) {
                //nothing to do
            }

            if (p_cur[i].cellInfoType == RIL_CELL_INFO_TYPE_GSM) {
                if (p.writeInt32(p_cur[i].CellInfo.gsm.cellIdentityGsm.mcc) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.gsm.cellIdentityGsm.mnc) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.gsm.cellIdentityGsm.lac) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.gsm.cellIdentityGsm.cid) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.gsm.signalStrengthGsm.signalStrength) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.gsm.signalStrengthGsm.bitErrorRate) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.gsm.signalStrengthGsm.timingAdvance) != NO_ERROR) {break;}
            }

            if (p_cur[i].cellInfoType == RIL_CELL_INFO_TYPE_CDMA) {
                if (p.writeInt32(p_cur[i].CellInfo.cdma.cellIdentityCdma.networkId) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.cdma.cellIdentityCdma.systemId) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.cdma.cellIdentityCdma.basestationId) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.cdma.cellIdentityCdma.longitude) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.cdma.cellIdentityCdma.latitude) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.cdma.signalStrengthCdma.dbm) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.cdma.signalStrengthCdma.ecio) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.cdma.signalStrengthEvdo.dbm) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.cdma.signalStrengthEvdo.ecio) != NO_ERROR) {break;}
            }

            if (p_cur[i].cellInfoType == RIL_CELL_INFO_TYPE_LTE) {
                if (p.writeInt32(p_cur[i].CellInfo.lte.cellIdentityLte.mcc) != NO_ERROR) {break;}             // RIL_CellIdentityLte_v12
                if (p.writeInt32(p_cur[i].CellInfo.lte.cellIdentityLte.mnc) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.lte.cellIdentityLte.ci) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.lte.cellIdentityLte.pci) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.lte.cellIdentityLte.tac) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.lte.cellIdentityLte.earfcn) != NO_ERROR) {break;}

                if (p.writeInt32(p_cur[i].CellInfo.lte.signalStrengthLte.signalStrength) != NO_ERROR) {break;} //RIL_LTE_SignalStrength_v8
                if (p.writeInt32(p_cur[i].CellInfo.lte.signalStrengthLte.rsrp) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.lte.signalStrengthLte.rsrq) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.lte.signalStrengthLte.rssnr) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.lte.signalStrengthLte.cqi) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.lte.signalStrengthLte.timingAdvance) != NO_ERROR) {break;}
            }

            if (p_cur[i].cellInfoType == RIL_CELL_INFO_TYPE_WCDMA) {
                if (p.writeInt32(p_cur[i].CellInfo.wcdma.cellIdentityWcdma.mcc) != NO_ERROR) {break;}              //RIL_CellIdentityWcdma_v12
                if (p.writeInt32(p_cur[i].CellInfo.wcdma.cellIdentityWcdma.mnc) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.wcdma.cellIdentityWcdma.lac) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.wcdma.cellIdentityWcdma.cid) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.wcdma.cellIdentityWcdma.uarfcn) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.wcdma.signalStrengthWcdma.signalStrength) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.wcdma.signalStrengthWcdma.bitErrorRate) != NO_ERROR) {break;}
            }

            if (p_cur[i].cellInfoType == RIL_CELL_INFO_TYPE_TD_SCDMA) {
                if (p.writeInt32(p_cur[i].CellInfo.tdscdma.cellIdentityTdscdma.mcc) != NO_ERROR) {break;}      //RIL_CellIdentityTdscdma
                if (p.writeInt32(p_cur[i].CellInfo.tdscdma.cellIdentityTdscdma.mnc) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.tdscdma.cellIdentityTdscdma.lac) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.tdscdma.cellIdentityTdscdma.cid) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.tdscdma.cellIdentityTdscdma.cpid) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].CellInfo.tdscdma.signalStrengthTdscdma.rscp) != NO_ERROR) {break;}  // RIL_TD_SCDMA_SignalStrength
            }

            if (num == i+1) {
                ret=0;
            }
        } //for
    } while (0);

    return ret;
}

int radio::cellInfoListInd(Parcel &p, void *response, size_t responselen)
{
    return genericCellInfoListIndOrResp(p, RIL_UNSOL_CELL_INFO_LIST, response, responselen);
}

int radio::getCellInfoListResponse(Parcel &p, void *response, size_t responselen)
{
    return genericCellInfoListIndOrResp(p, RIL_REQUEST_GET_CELL_INFO_LIST, response, responselen);
}

int radio::hardwareConfigChangedInd(Parcel &p, void *response, size_t responselen)
{
    int num;
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        if ((response == NULL) && (responselen != 0)) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            break;
        }

        if (responselen % sizeof(RIL_HardwareConfig) != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected multiple of  %d",__func__,
                (int)responselen, (int)sizeof(RIL_HardwareConfig));
            break;
        }
        num=responselen / sizeof(RIL_HardwareConfig);

        if (p.writeInt32(num) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }

        RIL_HardwareConfig *p_cur= (RIL_HardwareConfig *)(response);
        for (int i =0; i<num; i++){

            if (p.writeInt32(p_cur[i].type) != NO_ERROR) {break;}
            if (p.write(p_cur[i].uuid,MAX_UUID_LENGTH) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur[i].state) != NO_ERROR) {break;}

            if (p_cur[i].type == RIL_HARDWARE_CONFIG_MODEM){
                if (p.writeInt32(p_cur[i].cfg.modem.rilModel) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].cfg.modem.rat) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].cfg.modem.maxVoice) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].cfg.modem.maxData) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur[i].cfg.modem.maxStandby) != NO_ERROR) {break;}
            }

            if (p_cur[i].type == RIL_HARDWARE_CONFIG_SIM){
                if (p.write(p_cur[i].cfg.sim.modemUuid,MAX_UUID_LENGTH) != NO_ERROR) {break;}
            }

            if (num==i+1){
                ret=0;
            }
        }
    } while (0);
    return ret;
}

int radio::radioCapabilityIndicationInd(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        if (response == NULL) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            break;
        }

        if (responselen != sizeof(RIL_RadioCapability)) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected  %d",__func__,
                (int)responselen, (int)sizeof(RIL_RadioCapability));
                break;
        }

        if (p.writeInt32(1) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }

        RIL_RadioCapability *p_cur= (RIL_RadioCapability *)(response);

        if (p.writeInt32(p_cur->version) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->session) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->phase) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->rat) != NO_ERROR) {break;}
        if (p.write(p_cur->logicalModemUuid,MAX_UUID_LENGTH) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->status) != NO_ERROR) {break;}

        ret=0;
    } while (0);
    return ret;
}

bool isServiceTypeCfQuery(RIL_SsServiceType serType, RIL_SsRequestType reqType)
{
    if ((reqType == SS_INTERROGATION) &&
        (serType == SS_CFU ||
         serType == SS_CF_BUSY ||
         serType == SS_CF_NO_REPLY ||
         serType == SS_CF_NOT_REACHABLE ||
         serType == SS_CF_ALL ||
         serType == SS_CF_ALL_CONDITIONAL)) {
        return true;
    }
    return false;
}

int radio::onSupplementaryServiceIndicationInd(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;
    int num;

    do {
        if ((response == NULL) && (responselen != 0)) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            break;
        }

        if (responselen % sizeof(RIL_StkCcUnsolSsResponse) != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected  %d",__func__,
                (int)responselen, (int)sizeof(RIL_StkCcUnsolSsResponse));
            break;
        }

        num=responselen/sizeof(RIL_StkCcUnsolSsResponse);

        if (p.writeInt32(num) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }

        RIL_StkCcUnsolSsResponse *p_cur= (RIL_StkCcUnsolSsResponse *)(response);

        if (p.writeInt32(p_cur->serviceType) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->requestType) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->teleserviceType) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->serviceClass) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->result) != NO_ERROR) {break;}

        if (isServiceTypeCfQuery(p_cur->serviceType, p_cur->requestType)) {
            if (p.writeInt32(p_cur->cfData.numValidIndexes) != NO_ERROR) {break;}
            for (int i = 0; i < p_cur->cfData.numValidIndexes; i++) {
                RIL_CallForwardInfo *cf = &(p_cur->cfData.cfInfo[i]);
                if (p.writeInt32(cf->status) != NO_ERROR) {break;}
                if (p.writeInt32(cf->reason) != NO_ERROR) {break;}
                if (p.writeInt32(cf->serviceClass) != NO_ERROR) {break;}
                if (p.writeInt32(cf->toa) != NO_ERROR) {break;}
                p.writeStringToParcel(cf->number);
                if (p.writeInt32(cf->timeSeconds) != NO_ERROR) {break;}
                if (num == (i+1)){
                    ret =0;
                }
            }
        } else {
            for (int i = 0; i < SS_INFO_MAX; i++) {
                if (p.writeInt32(p_cur->ssInfo[i])!= NO_ERROR) {break;}
                if (num==i+1){
                    ret = 0;
                }
            }
        }
    } while (0);

    return ret;
}

int genericLceDataIndOrResp(Parcel &p, int id, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        if (response == NULL) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            break;
        }

        if (responselen != sizeof(RIL_LceDataInfo) != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected  %d",__func__,
                (int)responselen, (int)sizeof(RIL_LceDataInfo));
                break;
        }

        if (p.writeInt32(1) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }

        ACLOGD(LOG_TAG,"%s: request %d",__func__,id);

        RIL_LceDataInfo *p_cur= (RIL_LceDataInfo *)(response);
        if (p.writeInt32(p_cur->last_hop_capacity_kbps) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->confidence_level) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->lce_suspended) != NO_ERROR) {break;}
        ret=0;

    } while (0);
    return ret;

}  //RIL_LceDataInfo

int radio::lceDataInd(Parcel &p, void *response, size_t responselen)
{
    return genericLceDataIndOrResp(p,RIL_UNSOL_LCEDATA_RECV,response,responselen);
}

int radio::pullLceDataResponse(Parcel &p, void *response, size_t responselen)
{
    return genericLceDataIndOrResp(p,RIL_REQUEST_PULL_LCEDATA,response,responselen);
}

int radio::pcoDataInd(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        if (response == NULL) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            break;
        }

        if (responselen != sizeof(RIL_PCO_Data) != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected  %d",__func__,
                (int)responselen, (int)sizeof(RIL_PCO_Data));
            break;
        }

        if (p.writeInt32(1) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }

        RIL_PCO_Data *p_cur= (RIL_PCO_Data *)(response);
        if (p.writeInt32(p_cur->cid) != NO_ERROR) {break;}
        p.writeStringToParcel(p_cur->bearer_proto);
        if (p.writeInt32(p_cur->pco_id) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->contents_length) != NO_ERROR) {break;}
        if (p.write(p_cur->contents,p_cur->contents_length) != NO_ERROR) {break;}
        ret=0;
    } while (0);
    return ret;

} // RIL_PCO_Data

int radio::networkScanResultInd(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        if (response == NULL) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            break;
        }

        if (p.writeInt32(1) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }
        if ((responselen) != sizeof(RIL_NetworkScanResult)){
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected  %d",__func__,
                (int)responselen, (int)sizeof(RIL_NetworkScanResult));
            break;
        }

        RIL_NetworkScanResult *p_cur= (RIL_NetworkScanResult *)(response);
        if (p.writeInt32(p_cur->status) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->network_infos_length) != NO_ERROR) {break;}

        if (p_cur->network_infos_length==0){
            ret=0;
        }

        for (int i=0;i<p_cur->network_infos_length;i++){

            if (p.writeInt32(p_cur->network_infos[i].cellInfoType) != NO_ERROR) {break;}


            if (p.writeInt32(p_cur->network_infos[i].registered) != NO_ERROR) {break;}
            if (p.writeInt32(p_cur->network_infos[i].timeStampType) != NO_ERROR) {break;}
            if (p.writeInt64(p_cur->network_infos[i].timeStamp) != NO_ERROR) {break;}

            if (p_cur->network_infos[i].cellInfoType == RIL_CELL_INFO_TYPE_NONE) {
                //nothing to do
            }

            if (p_cur->network_infos[i].cellInfoType == RIL_CELL_INFO_TYPE_GSM) {
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.gsm.cellIdentityGsm.mcc) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.gsm.cellIdentityGsm.mnc) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.gsm.cellIdentityGsm.lac) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.gsm.cellIdentityGsm.cid) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.gsm.signalStrengthGsm.signalStrength) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.gsm.signalStrengthGsm.bitErrorRate) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.gsm.signalStrengthGsm.timingAdvance) != NO_ERROR) {break;}
            }

            if (p_cur->network_infos[i].cellInfoType == RIL_CELL_INFO_TYPE_CDMA) {
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.cdma.cellIdentityCdma.networkId) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.cdma.cellIdentityCdma.systemId) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.cdma.cellIdentityCdma.basestationId) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.cdma.cellIdentityCdma.longitude) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.cdma.cellIdentityCdma.latitude) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.cdma.signalStrengthCdma.dbm) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.cdma.signalStrengthCdma.ecio) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.cdma.signalStrengthEvdo.dbm) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.cdma.signalStrengthEvdo.ecio) != NO_ERROR) {break;}
            }

            if (p_cur->network_infos[i].cellInfoType == RIL_CELL_INFO_TYPE_LTE) {
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.lte.cellIdentityLte.mcc) != NO_ERROR) {break;}             // RIL_CellIdentityLte_v12
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.lte.cellIdentityLte.mnc) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.lte.cellIdentityLte.ci) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.lte.cellIdentityLte.pci) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.lte.cellIdentityLte.tac) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.lte.cellIdentityLte.earfcn) != NO_ERROR) {break;}

                if (p.writeInt32(p_cur->network_infos[i].CellInfo.lte.signalStrengthLte.signalStrength) != NO_ERROR) {break;} //RIL_LTE_SignalStrength_v8
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.lte.signalStrengthLte.rsrp) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.lte.signalStrengthLte.rsrq) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.lte.signalStrengthLte.rssnr) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.lte.signalStrengthLte.cqi) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.lte.signalStrengthLte.timingAdvance) != NO_ERROR) {break;}
            }

            if (p_cur->network_infos[i].cellInfoType == RIL_CELL_INFO_TYPE_WCDMA) {
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.wcdma.cellIdentityWcdma.mcc) != NO_ERROR) {break;}              //RIL_CellIdentityWcdma_v12
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.wcdma.cellIdentityWcdma.mnc) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.wcdma.cellIdentityWcdma.lac) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.wcdma.cellIdentityWcdma.cid) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.wcdma.cellIdentityWcdma.uarfcn) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.wcdma.signalStrengthWcdma.signalStrength) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.wcdma.signalStrengthWcdma.bitErrorRate) != NO_ERROR) {break;}
            }

            if (p_cur->network_infos[i].cellInfoType == RIL_CELL_INFO_TYPE_TD_SCDMA) {
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.tdscdma.cellIdentityTdscdma.mcc) != NO_ERROR) {break;}      //RIL_CellIdentityTdscdma
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.tdscdma.cellIdentityTdscdma.mnc) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.tdscdma.cellIdentityTdscdma.lac) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.tdscdma.cellIdentityTdscdma.cid) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.tdscdma.cellIdentityTdscdma.cpid) != NO_ERROR) {break;}
                if (p.writeInt32(p_cur->network_infos[i].CellInfo.tdscdma.signalStrengthTdscdma.rscp) != NO_ERROR) {break;}  // RIL_TD_SCDMA_SignalStrength
            }

            if (i+1 == p_cur->network_infos_length){
                ret=0; //all values processed
            }
        } //for

        if (ret == 0){
            if (p.writeInt32(p_cur->error) != NO_ERROR) {break;}
            ret=0;
        }
    } while (0);
    return ret;
}


int radio::keepaliveStatusInd(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        if (response == NULL) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            break;
        }

        if (responselen != sizeof(RIL_KeepaliveStatus) != 0) {
            ACLOGW(LOG_TAG,"%s: invalid response length %d expected  %d",__func__,
                (int)responselen, (int)sizeof(RIL_KeepaliveStatus));
                break;
        }

        if (p.writeInt32(1) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }

        RIL_KeepaliveStatus *p_cur= (RIL_KeepaliveStatus *)(response);
        if (p.writeInt32(p_cur->sessionHandle) != NO_ERROR) {break;}
        if (p.writeInt32(p_cur->code) != NO_ERROR) {break;}
        ret=0;

    } while (0);
    return ret;
}


int radio::imsSettingsMsdInd(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;
    ACLOGN( LOG_TAG,"%s,%d..", __func__, __LINE__ );

    do {
        if (responselen == sizeof(int)) {  //get transaction id
            int *pInt = (int *)response;
            if (p.writeInt32(*pInt) != NO_ERROR)
            {
                ACLOGE( LOG_TAG,"%s,%d: marshaling error occured!", __func__, __LINE__ );
                break;
            }
            ret=0;
        } else {
            ACLOGE(LOG_TAG,"%s,%d: invalid response: NULL!", __func__, __LINE__ );
            break;
        }
    } while (0);

    return ret;
}

int radio::voiceGetLatestMsdInd(Parcel &p, void *response, size_t responselen)
{
    ACLOGN( LOG_TAG,"%s,%d..", __func__, __LINE__ );

    return 0;
}


int responseRilSignalStrengthV11(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {

        if (response==NULL) {
            ACLOGW(LOG_TAG,"%s: response = NULL",__func__);
            break;
        }

        if (p.writeInt32(1) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }

        if (responselen < sizeof (RIL_SignalStrength_v11)) {

            if (p.writeInt32(10) != NO_ERROR) {
                //number of basic data objects
                ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
                break;
            }

            RIL_SignalStrength_v10 *p_cur = ((RIL_SignalStrength_v10 *) response);

            if (p.writeInt32(p_cur->GW_SignalStrength.signalStrength) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->GW_SignalStrength.bitErrorRate) != NO_ERROR){break;}

            if (p.writeInt32(p_cur->CDMA_SignalStrength.dbm) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->CDMA_SignalStrength.ecio) != NO_ERROR){break;}

            if (p.writeInt32(p_cur->EVDO_SignalStrength.dbm) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->EVDO_SignalStrength.ecio) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->EVDO_SignalStrength.signalNoiseRatio) != NO_ERROR){break;}

            if (p.writeInt32(p_cur->LTE_SignalStrength.signalStrength) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.rsrp) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.rsrq) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.rssnr) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.cqi) != NO_ERROR){break;}

            ACLOGD(LOG_TAG,"%s: GW signalStrength: %d",__func__,p_cur->GW_SignalStrength.signalStrength);
            ACLOGD(LOG_TAG,"%s: GW error rate: %d",__func__,p_cur->GW_SignalStrength.bitErrorRate);
            ACLOGD(LOG_TAG,"%s: LTE signalStrength: %d",__func__,p_cur->LTE_SignalStrength.signalStrength);
            ACLOGD(LOG_TAG,"%s: LTE rsrp: %d",__func__,p_cur->LTE_SignalStrength.rsrp);
            ACLOGD(LOG_TAG,"%s: LTE rsrq: %d",__func__,p_cur->LTE_SignalStrength.rsrq);
            ACLOGD(LOG_TAG,"%s: LTE rssnr: %d",__func__,p_cur->LTE_SignalStrength.rssnr);
            ACLOGD(LOG_TAG,"%s: LTE cqi: %d",__func__,p_cur->LTE_SignalStrength.cqi);
            ret=0;

        } else {

            if (p.writeInt32(11) != NO_ERROR) {
                //number of basic data objects
                ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
                break;
            }

            RIL_SignalStrength_v11 *p_cur = ((RIL_SignalStrength_v11 *) response);
            if (p.writeInt32(p_cur->GW_SignalStrength.signalStrength) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->GW_SignalStrength.bitErrorRate) != NO_ERROR){break;}

            if (p.writeInt32(p_cur->CDMA_SignalStrength.dbm) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->CDMA_SignalStrength.ecio) != NO_ERROR){break;}

            if (p.writeInt32(p_cur->EVDO_SignalStrength.dbm) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->EVDO_SignalStrength.ecio) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->EVDO_SignalStrength.signalNoiseRatio) != NO_ERROR){break;}

            if (p.writeInt32(p_cur->LTE_SignalStrength.signalStrength) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.rsrp) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.rsrq) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.rssnr) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->LTE_SignalStrength.cqi) != NO_ERROR){break;}

            if (p.writeInt32(p_cur->TD_SCDMA_SignalStrength.rscp) != NO_ERROR){break;}

            if (p.writeInt32(p_cur->NR5G_SignalStrength.rsrp) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->NR5G_SignalStrength.rsrq) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->NR5G_SignalStrength.sinr) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->NR5G_SignalStrength.csiRsrp) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->NR5G_SignalStrength.csiRsrq) != NO_ERROR){break;}
            if (p.writeInt32(p_cur->NR5G_SignalStrength.csiSinr) != NO_ERROR){break;}

            ACLOGD(LOG_TAG,"%s: GW signalStrength: %d",__func__,p_cur->GW_SignalStrength.signalStrength);
            ACLOGD(LOG_TAG,"%s: GW error rate: %d",__func__,p_cur->GW_SignalStrength.bitErrorRate);
            ACLOGD(LOG_TAG,"%s: LTE signalStrength: %d",__func__,p_cur->LTE_SignalStrength.signalStrength);
            ACLOGD(LOG_TAG,"%s: LTE rsrp: %d",__func__,p_cur->LTE_SignalStrength.rsrp);
            ACLOGD(LOG_TAG,"%s: LTE rsrq: %d",__func__,p_cur->LTE_SignalStrength.rsrq);
            ACLOGD(LOG_TAG,"%s: LTE rssnr: %d",__func__,p_cur->LTE_SignalStrength.rssnr);
            ACLOGD(LOG_TAG,"%s: LTE cqi: %d",__func__,p_cur->LTE_SignalStrength.cqi);
            ACLOGD(LOG_TAG,"%s: NR5G rsrp: %d",__func__,p_cur->NR5G_SignalStrength.rsrp);
            ACLOGD(LOG_TAG,"%s: NR5G rsrq: %d",__func__,p_cur->NR5G_SignalStrength.rsrq);
            ACLOGD(LOG_TAG,"%s: NR5G sinr: %d",__func__,p_cur->NR5G_SignalStrength.sinr);
            ACLOGD(LOG_TAG,"%s: NR5G csiRsrp: %d",__func__,p_cur->NR5G_SignalStrength.csiRsrp);
            ACLOGD(LOG_TAG,"%s: NR5G csiRsrq: %d",__func__,p_cur->NR5G_SignalStrength.csiRsrq);
            ACLOGD(LOG_TAG,"%s: NR5G csiSinr: %d",__func__,p_cur->NR5G_SignalStrength.csiSinr);
            ret=0;
        }

    } while (0);
    return ret;
}

int responseRilSignalStrengthV6(Parcel &p, void *response, size_t responselen)
{
    RIL_SignalStrength_v6 *p_cur = ((RIL_SignalStrength_v6 *) response);
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        if (response==NULL) {
            ACLOGW(LOG_TAG,"%s: response = NULL",__func__);
            break;
        }

        if (p.writeInt32(1) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }

        if (p.writeInt32(p_cur->GW_SignalStrength.signalStrength) != NO_ERROR){break;}
        if (p.writeInt32(p_cur->GW_SignalStrength.bitErrorRate) != NO_ERROR){break;}
        if (p.writeInt32(p_cur->CDMA_SignalStrength.dbm) != NO_ERROR){break;}
        if (p.writeInt32(p_cur->CDMA_SignalStrength.ecio) != NO_ERROR){break;}
        if (p.writeInt32(p_cur->EVDO_SignalStrength.dbm) != NO_ERROR){break;}
        if (p.writeInt32(p_cur->EVDO_SignalStrength.ecio) != NO_ERROR){break;}
        if (p.writeInt32(p_cur->EVDO_SignalStrength.signalNoiseRatio) != NO_ERROR){break;}

        if (p.writeInt32(p_cur->LTE_SignalStrength.signalStrength)  != NO_ERROR){break;}
        if (p.writeInt32(p_cur->LTE_SignalStrength.rsrp) != NO_ERROR){break;}
        if (p.writeInt32(p_cur->LTE_SignalStrength.rsrq) != NO_ERROR){break;}
        if (p.writeInt32(p_cur->LTE_SignalStrength.rssnr) != NO_ERROR){break;}
        if (p.writeInt32(p_cur->LTE_SignalStrength.cqi) != NO_ERROR){break;}

        ACLOGD(LOG_TAG,"%s: GW signalStrength: %d",__func__,p_cur->GW_SignalStrength.signalStrength);
        ACLOGD(LOG_TAG,"%s: GW error rate: %d",__func__,p_cur->GW_SignalStrength.bitErrorRate);
        ACLOGD(LOG_TAG,"%s: LTE signalStrength: %d",__func__,p_cur->LTE_SignalStrength.signalStrength);
        ACLOGD(LOG_TAG,"%s: LTE rsrp: %d",__func__,p_cur->LTE_SignalStrength.rsrp);
        ACLOGD(LOG_TAG,"%s: LTE rsrq: %d",__func__,p_cur->LTE_SignalStrength.rsrq);
        ACLOGD(LOG_TAG,"%s: LTE rssnr: %d",__func__,p_cur->LTE_SignalStrength.rssnr);
        ACLOGD(LOG_TAG,"%s: LTE cqi: %d",__func__,p_cur->LTE_SignalStrength.cqi);
        ret=0;
    } while (0);
    return ret;
}

int radio::getSignalStrengthResponse(Parcel &p,
                    void *response, size_t responselen)
{

    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {

        if (response==NULL) {
            ACLOGW(LOG_TAG,"%s: response = NULL",__func__);
            break;
        }

        if (responselen < sizeof (RIL_SignalStrength_v5)) {
            ACLOGE(LOG_TAG,"%s: invalid response length: %d",__func__,responselen);
            break;
        }

        if (responselen == sizeof (RIL_SignalStrength_v6)) {
            ret= responseRilSignalStrengthV6(p, response, responselen);
            break;
	    }

        if (responselen > sizeof (RIL_SignalStrength_v6)) {
            ret=responseRilSignalStrengthV11(p, response, responselen);
            break;
        }

        ret=0;

    } while (0);

     return ret;
}

int radio::getDataRadioTechnologyResponse(Parcel &p,void *response, size_t responselen)
{
    ACLOGV(LOG_TAG,"%s..",__func__);
    return genericIntFixedRespOrInd(__func__,1,p,response,responselen);
}

int radio::getIncrementalNetworkScanResponse(Parcel &p,void *response, size_t responselen)
{
    ACLOGV(LOG_TAG,"%s..",__func__);
    return genericStringsRespOrInd(__func__,1,p,response,responselen);
}

int radio::simRefreshInd(Parcel &p, void *response, size_t responselen)
{

    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do {
        if (response == NULL) {
            ACLOGE(LOG_TAG,"%s: invalid response: NULL",__func__);
            break;
        }

        if (p.writeInt32(1) != NO_ERROR) {
            //number of basic data objects
            ACLOGW(LOG_TAG,"%s: could not write object number",__func__);
            break;
        }

        if (s_callbacks.version >= 7) {
            RIL_SimRefreshResponse_v7 *p_cur = ((RIL_SimRefreshResponse_v7 *) response);
            p.writeInt32(p_cur->result);
            p.writeInt32(p_cur->ef_id);
            p.writeStringToParcel(p_cur->aid);

            ACLOGD(LOG_TAG,"%s: result: %d",__func__,p_cur->result);
            ACLOGD(LOG_TAG,"%s: ef_id: %d",__func__,p_cur->ef_id);
            ACLOGD(LOG_TAG,"%s: aid: %s",__func__,p_cur->aid);

        } else {
            int *p_cur = ((int *) response);
            p.writeInt32(p_cur[0]);
            p.writeInt32(p_cur[1]);
            p.writeStringToParcel(NULL);

            ACLOGD(LOG_TAG,"%s: [0]: %d",__func__,p_cur[0]);
            ACLOGD(LOG_TAG,"%s: [1]: %d",__func__,p_cur[1]);

        }
        ret=0;
    } while (0);

    return ret;

}

static void sendSimStatusAppInfo(Parcel &p, int num_apps, RIL_AppStatus appStatus[])
{

        ACLOGV(LOG_TAG,"%s: num_apps: %d",__func__,(int)num_apps);
        p.writeInt32(num_apps);

        for (int i = 0; i < num_apps; i++) {
            ACLOGV(LOG_TAG,"%s: apps[%d] app_type: %d",__func__,i,(int)appStatus[i].app_type);
            ACLOGV(LOG_TAG,"%s: apps[%d] app_state: %d",__func__,i,(int)appStatus[i].app_state);
            ACLOGV(LOG_TAG,"%s: apps[%d] perso_substate: %d",__func__,i,(int)appStatus[i].perso_substate);
            if (appStatus[i].aid_ptr !=NULL){
                ACLOGV(LOG_TAG,"%s: apps[%d] aid_ptr: %d",__func__,i,(int)appStatus[i].aid_ptr);
            } else{
                ACLOGV(LOG_TAG,"%s: apps[%d] aid_ptr: NULL",__func__,i);
            }
            if (appStatus[i].app_label_ptr !=NULL){
                ACLOGV(LOG_TAG,"%s: apps[%d] app_label_ptr: %d",__func__,i,(int)appStatus[i].app_label_ptr);
            } else{
                ACLOGV(LOG_TAG,"%s: apps[%d] app_label_ptr: NULL",__func__,i);
            }
            ACLOGV(LOG_TAG,"%s: apps[%d] pin1_replaced: %d",__func__,i,(int)appStatus[i].pin1_replaced);
            ACLOGV(LOG_TAG,"%s: apps[%d] pin1: %d",__func__,i,(int)appStatus[i].pin1);
            ACLOGV(LOG_TAG,"%s: apps[%d] pin2: %d",__func__,i,(int)appStatus[i].pin2);

            p.writeInt32(appStatus[i].app_type);
            p.writeInt32(appStatus[i].app_state);
            p.writeInt32(appStatus[i].perso_substate);
            p.writeStringToParcel((const char*)(appStatus[i].aid_ptr));
            p.writeStringToParcel((const char*)
                                          (appStatus[i].app_label_ptr));
            p.writeInt32(appStatus[i].pin1_replaced);
            p.writeInt32(appStatus[i].pin1);
            p.writeInt32(appStatus[i].pin2);
       }
}

int radio::getIccCardStatusResponse(Parcel &p, void *response, size_t responselen)
{
    int i;
    int ret=RIL_ERRNO_INVALID_RESPONSE;

    do  {

        if (response == NULL) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            break;
        }

        p.writeInt32(1);  //single data struct transferred

        //note: support for RIL_CardStatus_v5 removed; only v6 supported
        RIL_CardStatus_v6 *p_cur = ((RIL_CardStatus_v6 *) response);
        if ((response == NULL) ||
            (responselen != sizeof(RIL_CardStatus_v6)) ||
            (p_cur->gsm_umts_subscription_app_index >= p_cur->num_applications) ||
            (p_cur->cdma_subscription_app_index >= p_cur->num_applications) ||
            (p_cur->ims_subscription_app_index >= p_cur->num_applications)) {
            ACLOGW(LOG_TAG,"%s: invalid response",__func__);;
            break;
        }

        ACLOGD(LOG_TAG,"%s: response  card state: %d",__func__,(int)p_cur->card_state);
        ACLOGD(LOG_TAG,"%s: universal pin state: %d",__func__,(int)p_cur->universal_pin_state);
        ACLOGD(LOG_TAG,"%s: gsm_umts_subscription_app_index: %d",__func__,(int)p_cur->gsm_umts_subscription_app_index);
        ACLOGD(LOG_TAG,"%s: cdma_subscription_app_index: %d",__func__,(int)p_cur->cdma_subscription_app_index);
        ACLOGD(LOG_TAG,"%s: ims_subscription_app_index: %d",__func__,(int)p_cur->ims_subscription_app_index);

        p.writeInt32(p_cur->card_state);
        p.writeInt32(p_cur->universal_pin_state);
        p.writeInt32(p_cur->gsm_umts_subscription_app_index);
        p.writeInt32(p_cur->cdma_subscription_app_index);
        p.writeInt32(p_cur->ims_subscription_app_index);

        sendSimStatusAppInfo(p, p_cur->num_applications, p_cur->applications);

        ret=0;
    } while (0);
    return ret;
}

int radio::getGsmBroadcastConfigResponse(Parcel &p, void *response, size_t responselen)
{
    int ret=RIL_ERRNO_INVALID_RESPONSE;
    do{

        if (((response == NULL) && (responselen != 0))||
            (responselen % sizeof(RIL_GSM_BroadcastSmsConfigInfo *) != 0)) {
            ACLOGW(LOG_TAG,"%s: invalid response: NULL",__func__);
            break;
        }

        int num = responselen / sizeof(RIL_GSM_BroadcastSmsConfigInfo *);
        if (p.writeInt32(num) !=NO_ERROR){break;}

        ACLOGD(LOG_TAG,"%s: elements= %d",__func__,num);

        RIL_GSM_BroadcastSmsConfigInfo **p_cur =
                    (RIL_GSM_BroadcastSmsConfigInfo **) response;

        for (int i = 0; i < num; i++) {
            if (p.writeInt32(p_cur[i]->fromServiceId) != NO_ERROR){break;}
            if (p.writeInt32(p_cur[i]->toServiceId) != NO_ERROR){break;}
            if (p.writeInt32(p_cur[i]->fromCodeScheme) != NO_ERROR){break;}
            if (p.writeInt32(p_cur[i]->toCodeScheme) != NO_ERROR){break;}
            if (p.writeInt32(p_cur[i]->selected) != NO_ERROR){break;}

            ACLOGD(LOG_TAG,"%s: fromServiceId[%d]: %d",__func__,i,p_cur[i]->fromServiceId);
            ACLOGD(LOG_TAG,"%s: toServiceId[%d]: %d",__func__,i,p_cur[i]->toServiceId);
            ACLOGD(LOG_TAG,"%s: fromCodeScheme[%d]: %d",__func__,i,p_cur[i]->fromCodeScheme);
            ACLOGD(LOG_TAG,"%s: toCodeScheme[%d]: %d",__func__,i,p_cur[i]->toCodeScheme);
            ACLOGD(LOG_TAG,"%s: selected[%d]: %d",__func__,i,p_cur[i]->selected);

            if (num == i+1){
               ret=0;
            }

        }

        if (num == 0){
            ret=0;
        }
    } while (0);

    return ret;
}


void libril_onProcClosed(int procId) {

    int ret;
    RequestInfo *p_cur;

    /* mark pending requests as "cancelled" so we dont report responses */
    ret = pthread_mutex_lock(&s_pendingRequestsMutex);
    ACLOGI(LOG_TAG,"%s: mark pending as cancelled(ret:%i)",__func__,ret);
    assert (ret == 0);

    p_cur = s_pendingRequests;

    for (p_cur = s_pendingRequests; p_cur != NULL; p_cur  = p_cur->p_next) {
        if (p_cur->procId==procId){
            p_cur->cancelled = 1;
        }
    }

    ret = pthread_mutex_unlock(&s_pendingRequestsMutex);
    ACLOGI(LOG_TAG,"%s(ret:%i): done",__func__,ret);
    assert (ret == 0);
}

static void userTimerCallback (short flags, void *param)
{
    UserCallbackInfo *p_info;

    p_info = (UserCallbackInfo *)param;
    p_info->p_callback(p_info->userParam);

    // FIXME generalize this...there should be a cancel mechanism
    if (s_last_wake_timeout_info != NULL && s_last_wake_timeout_info == p_info) {
        s_last_wake_timeout_info = NULL;
    }
    FREE(p_info);
}

// Used for testing purpose only.
extern "C" void RIL_setcallbacks (const RIL_RadioFunctions *callbacks)
{
    memcpy(&s_callbacks, callbacks, sizeof (RIL_RadioFunctions));
}

extern "C" int RIL_startEventLoop(char *inetAddr, int port)
{
    int ret;

    //start socket handling thread
    do {
        ret=rilEvent_init();
        if (ret<0){
            break;
        }

        ret=rilProc::init(inetAddr,port);
    } while(0);

    return ret;
}


extern "C" int
RIL_register (const RIL_RadioFunctions *callbacks, char *inetAddr, int inetPort)
{
    int flags;
    int result=-1;

    do {
        for (int i = 0; i < (int)NUM_ELEMS(s_commands); i++) {
           if (i != s_commands[i].requestNumber){
                ACLOGE(LOG_TAG,"%s:  request mismatch : %d/%d",__func__,
                i, s_commands[i].requestNumber);
                return result;
            }
        }

        for (int i = 0; i < (int)NUM_ELEMS(s_unsolResponses); i++) {
            ACLOGN(LOG_TAG,"%s: test %d ..",__func__, i);
            if (i + RIL_UNSOL_RESPONSE_BASE != s_unsolResponses[i].requestNumber) {
                ACLOGE(LOG_TAG,"%s:  unsol response mismatch : %d/%d",__func__,
                        i + RIL_UNSOL_RESPONSE_BASE, s_unsolResponses[i].requestNumber);
                return result;
            }
        }

        if (callbacks == NULL) {
            ACLOGE(LOG_TAG,"%s: RIL_RadioFunctions * null",__func__);
            break;
        }

        if (callbacks->version < RIL_VERSION_MIN) {
            ACLOGE(LOG_TAG,"%s: version %d is too old, min version is %d",__func__,
                callbacks->version, RIL_VERSION_MIN);
            break;
        }

        if (callbacks->version > RIL_VERSION) {
            ACLOGE(LOG_TAG,"%s: version %d is too new, max version is %d",__func__,
                callbacks->version, RIL_VERSION);
            break;
        }

        ACLOGN(LOG_TAG,"%s: RIL version %d",__func__, callbacks->version);


        if (s_registerCalled > 0) {
            ACLOGE(LOG_TAG,"%s: called more than once - subsequent call ignored",__func__);
            break;
        }

        s_registerCalled = 1;
        memcpy(&s_callbacks, callbacks, sizeof (RIL_RadioFunctions));

        result=0;
    } while (0);

    return result;
}

// Check and remove RequestInfo if its a response and not just ack sent back
int checkAndDequeueRequestInfoAck(struct RequestInfo *pRI, bool isAck)
{
    int ret = 0;

    if (pRI == NULL) {
        return 0;
    }

    pthread_mutex_lock(&s_pendingRequestsMutex);

    for(RequestInfo **ppCur = &s_pendingRequests; *ppCur != NULL; ppCur = &((*ppCur)->p_next)) {
        if (pRI == *ppCur) {
            ret=1;
            if (isAck) {
                if (pRI->wasAckSent == 1){
                    ACLOGD(LOG_TAG,"%s: Ack was already sent for %s",
                    __func__,requestToString(pRI->pCI->requestNumber));
                    ret=0;
                } else {
                   pRI->wasAckSent = 1;
                }
            } else {
                *ppCur = (*ppCur)->p_next;
            }
            break;
        }
    }

    pthread_mutex_unlock(&s_pendingRequestsMutex);

    return ret;
}


extern "C" void
RIL_onRequestAck(RIL_Token t)
{

    RequestInfo *pRI;
    pRI = (RequestInfo *)t;
    int ret;

    if (!checkAndDequeueRequestInfoAck(pRI, true)) {
        ACLOGE (LOG_TAG,"%s: invalid RIL_Token", __func__);
        return;
    }

    if (pRI->cancelled == 0){
        Parcel p;
        p.writeInt32( RESPONSE_SOLICITED_ACK);
        p.writeInt32(pRI->token);
        ret=rilProc::sendResponse(pRI->procId, p);
        if (ret<0){
            ACLOGE(LOG_TAG,"%s: client socket broken",__func__);
            rilProc::setState(pRI->procId,PROC_SHOULD_CLOSE);
        }
    }
}

extern "C" void
RIL_onRequestComplete(RIL_Token t, RIL_Errno e, void *response, size_t responselen)
{
    RequestInfo *pRI;
    int ret;
    size_t errorOffset;

    ALOGI(LOG_TAG,"%s: entered RIL token: 0x%x (errno: %d)",__func__,(int)t, (int)e);

    pRI = (RequestInfo *)t;

    if (!checkAndDequeueRequestInfoAck(pRI,false)) {
        ACLOGE (LOG_TAG,"%s: invalid RIL_Token",__func__);
        return;
    }
    if (pRI->local > 0) {
        ACLOGD(LOG_TAG,"%s: local request",__func__);
        // Locally issued command...void only!
        // response does not go back up the command socket
        ACLOGV(LOG_TAG,"%s:  %s",__func__, requestToString(pRI->pCI->requestNumber));

        goto done;
    }

    ACLOGV(LOG_TAG,"%s: token: 0x%x, proc: 0x%x ",__func__, pRI->token,pRI->pProc);
    ACLOGV(LOG_TAG,"%s: request: %s",__func__, requestToString(pRI->pCI->requestNumber));

    if (pRI->cancelled == 0) {
        ACLOGI(LOG_TAG,"%s: build response",__func__);
        Parcel p;

        if (s_callbacks.version >= 13 && pRI->wasAckSent == 1) {
            // If ack was already sent, then this call is an asynchronous response. So we need to
            // send id indicating that we expect an ack from RIL.java as we acquire wakelock here.
            p.writeInt32 (RESPONSE_SOLICITED_ACK_EXP);
        } else {
            p.writeInt32 (RESPONSE_SOLICITED);
        }
        p.writeInt32 (pRI->token);
        errorOffset = p.dataPosition();

        ACLOGD(LOG_TAG,"%s: request: %s; e: %d",__func__,
                        requestToString(pRI->pCI->requestNumber),e);
        p.writeInt32 (e);
        ret=0;
        // there is a response payload, no matter success or not.
        ret = pRI->pCI->responseFunction(p, response, responselen);

        ACLOGD(LOG_TAG,"%s: response function done: ret: %d",__func__,ret);
        if (ret==0){
            ret = p.writeInt32 (Parcel::ParcelEnd);
        }

        /* if an error occurred, rewind and mark it */
        if ((ret != 0) && (e == RIL_E_SUCCESS)) {
            e=RIL_E_INTERNAL_ERR;
            ACLOGW(LOG_TAG,"%s: ret error after response function: %d",__func__,ret);
            p.setDataPosition(errorOffset);
            p.writeInt32 (e);
        }

        if (e != RIL_E_SUCCESS) {
            ACLOGE(LOG_TAG,"%s: %s fails by %s",__func__,
                   requestToString(pRI->pCI->requestNumber), failCauseToString(e));
        }

        if (rilProc::getCommandFd(pRI->procId) <=0) {
            ACLOGW(LOG_TAG,"%s: Command channel already closed",__func__);
        }

        ACLOGI(LOG_TAG,"%s: sendResponse to proc %d",__func__,pRI->procId);
        ret=rilProc::sendResponse(pRI->procId, p);
        if (ret<0){
            ACLOGE(LOG_TAG,"%s: client socket broken",__func__);
            rilProc::setState(pRI->procId,PROC_SHOULD_CLOSE);
        }

    } else {
        ACLOGW(LOG_TAG,"%s: request cancelled",__func__);
    }

done:
    ACLOGI(LOG_TAG,"%s: done",__func__);
    FREE(pRI);
}


static void
grabPartialWakeLock()
{
  //for backward compatibility to android definitions
}

static void
releaseWakeLock()
{
    //for backward compatibility to android definitions
}


/**
 * Timer callback to put us back to sleep before the default timeout
 */
static void wakeTimeoutCallback (void *param) {
    // We're using "param != NULL" as a cancellation mechanism
    if (param == NULL) {
        //ACLOGD(LOG_TAG,"%s: releasing wake lock",__func__);

        releaseWakeLock();
    } else {
        //ACLOGD(LOG_TAG,"%s: releasing wake lock CANCELLED",__func__);
    }
}


extern "C" void RIL_onUnsolicitedResponse(int unsolResponse, void *data, size_t datalen)
{
    ACLOGV(LOG_TAG,"%s ..", __func__);
    ril_UnsolicitedProcResponse(unsolResponse,data,datalen,0);
}

static void ril_UnsolicitedProcResponse(int unsolResponse, void *data,
                                size_t datalen,int procId)
{
    int unsolResponseIndex;
    int idx;
    int ret;
    int64_t timeReceived = 0;
    bool shouldScheduleTimeout = false;

    ACLOGD(LOG_TAG,"%s[%d]: proc id = %d, data len= %d",
            __func__, unsolResponse, procId, datalen);

    if (s_registerCalled == 0){
        // Ignore RIL_onUnsolicitedResponse before RIL_register
        ACLOGW(LOG_TAG,"%s: called before RIL_register; ignore ", __func__);
        return;
    }

    unsolResponseIndex = unsolResponse - RIL_UNSOL_RESPONSE_BASE;
    if ((unsolResponseIndex < 0)
        || (unsolResponseIndex >= (int32_t)NUM_ELEMS(s_unsolResponses))) {
        ACLOGE(LOG_TAG,"%s: unsupported unsolicited response code %d",
                __func__, unsolResponse);
        return;
    }

    ACLOGD(LOG_TAG,"%s: response index %d", __func__, unsolResponseIndex);

    // Mark the time this was received, doing this
    // after grabing the wakelock incase getting
    // the elapsedRealTime might cause us to goto
    // sleep.
    if (unsolResponse == RIL_UNSOL_NITZ_TIME_RECEIVED) {
        timeReceived = elapsedRealtime();
    }

    ACLOGI(LOG_TAG,"%s: %s", __func__, requestToString(unsolResponse));

    Parcel p;

    p.writeInt32 (RESPONSE_UNSOLICITED);
    p.writeInt32 (unsolResponse);

    ret = s_unsolResponses[unsolResponseIndex].responseFunction(p, data, datalen);
    if (ret != 0) {
        ACLOGW(LOG_TAG,"%s: could not handle response", __func__);
        // Problem with the response. Don't continue;
        goto error_exit;
    }

    // some things get more payload
    switch(unsolResponse) {
        case RIL_UNSOL_RESPONSE_RADIO_STATE_CHANGED:
            p.writeInt32(s_callbacks.onStateRequest());
            ACLOGW(LOG_TAG,"%s: radio state: %s", __func__,
                    radioStateToString(s_callbacks.onStateRequest()));
            break;


        case RIL_UNSOL_NITZ_TIME_RECEIVED:
            // Store the time that this was received so the
            // handler of this message can account for
            // the time it takes to arrive and process. In
            // particular the system has been known to sleep
            // before this message can be processed.
            p.writeInt64(timeReceived);
            break;
    }

    ACLOGV(LOG_TAG,"%s: now send the response to the procs..", __func__);

    if (procId>0) {
        ACLOGI(LOG_TAG,"%s: %s for dedicated process %d",
                __func__, requestToString(unsolResponse), procId);
        rilProc::sendResponse(procId,p);
    } else {
        ACLOGI(LOG_TAG,"%s: %s[%d] for all processes", __func__,
                requestToString(unsolResponse),unsolResponse);
        rilProc::sendEvent(unsolResponse,p);
    }

    if ( unsolResponse == RIL_UNSOL_NITZ_TIME_RECEIVED) {

        // Unfortunately, NITZ time is not poll/update like everything
        // else in the system. So, if the upstream client isn't connected,
        // keep a copy of the last NITZ response (with receive time noted
        // above) around so we can deliver it when it is connected

        if (s_lastNITZTimeData != NULL) {
            FREE (s_lastNITZTimeData);
            s_lastNITZTimeData = NULL;
        }

        s_lastNITZTimeData = MALLOC(p.dataSize());
        s_lastNITZTimeDataSize = p.dataSize();
        memcpy(s_lastNITZTimeData, p.data(), p.dataSize());
    }
    // For now, we automatically go back to sleep after TIMEVAL_WAKE_TIMEOUT
    // FIXME The java code should handshake here to release wake lock

    if (shouldScheduleTimeout) {
        // Cancel the previous request
        if (s_last_wake_timeout_info != NULL) {
            s_last_wake_timeout_info->userParam = (void *)1;
        }

        s_last_wake_timeout_info
            = internalRequestTimedCallback(wakeTimeoutCallback, NULL,
                                            &TIMEVAL_WAKE_TIMEOUT);
    }

    // Normal exit
    return;

error_exit:
    if (shouldScheduleTimeout) {
        releaseWakeLock();
    }
}



/** FIXME generalize this if you track UserCAllbackInfo, clear it
    when the callback occurs
*/
static UserCallbackInfo *
internalRequestTimedCallback (RIL_TimedCallback callback, void *param,
                                const struct timeval *relativeTime)
{
    struct timeval myRelativeTime;
    UserCallbackInfo *p_info;

    ACLOGV(LOG_TAG,"%s..",__func__);
    do {
        p_info = (UserCallbackInfo *) MALLOC (sizeof(UserCallbackInfo));
        if (p_info == NULL){
            ACLOGW(LOG_TAG,"%s: could noc alloc user info",__func__);
            break;
        }

        p_info->p_callback = callback;
        p_info->userParam = param;

        if (relativeTime == NULL) {
            /* treat null parameter as a 0 relative time */
            memset (&myRelativeTime, 0, sizeof(myRelativeTime));
        } else {
            /* FIXME I think event_add's tv param is really const anyway */
            memcpy (&myRelativeTime, relativeTime, sizeof(myRelativeTime));
        }

        p_info->pEvent=rilEvent_create();
        if(p_info->pEvent!=NULL){
            rilEvent_set(p_info->pEvent, false, userTimerCallback, p_info);
            rilEvent_timerAdd(p_info->pEvent, &myRelativeTime);
            rilEvent_trigger();
        } else {
            ACLOGW(LOG_TAG,"%s: could not create event object",__func__);
        }

    } while (0);

    return p_info;

}


extern "C" void RIL_requestTimedCallback (RIL_TimedCallback callback, void *param,
                                            const struct timeval *relativeTime)
{
    internalRequestTimedCallback (callback, param, relativeTime);
}

const char *failCauseToString(RIL_Errno e)
{
    switch(e) {
        case RIL_E_SUCCESS: return "E_SUCCESS";
        case RIL_E_RADIO_NOT_AVAILABLE: return "E_RAIDO_NOT_AVAILABLE";
        case RIL_E_GENERIC_FAILURE: return "E_GENERIC_FAILURE";
        case RIL_E_PASSWORD_INCORRECT: return "E_PASSWORD_INCORRECT";
        case RIL_E_SIM_PIN2: return "E_SIM_PIN2";
        case RIL_E_SIM_PUK2: return "E_SIM_PUK2";
        case RIL_E_REQUEST_NOT_SUPPORTED: return "E_REQUEST_NOT_SUPPORTED";
        case RIL_E_CANCELLED: return "E_CANCELLED";
        case RIL_E_OP_NOT_ALLOWED_DURING_VOICE_CALL: return "E_OP_NOT_ALLOWED_DURING_VOICE_CALL";
        case RIL_E_OP_NOT_ALLOWED_BEFORE_REG_TO_NW: return "E_OP_NOT_ALLOWED_BEFORE_REG_TO_NW";
        case RIL_E_SMS_SEND_FAIL_RETRY: return "E_SMS_SEND_FAIL_RETRY";
        case RIL_E_SIM_ABSENT:return "E_SIM_ABSENT";
        case RIL_E_ILLEGAL_SIM_OR_ME:return "E_ILLEGAL_SIM_OR_ME";
        case RIL_E_MISSING_RESOURCE: return "RIL_E_MISSING_RESOURCE";
        case RIL_E_NO_SUCH_ELEMENT: return "RIL_E_NO_SUCH_ELEMENT";
        case RIL_E_DIAL_MODIFIED_TO_USSD: return "RIL_E_DIAL_MODIFIED_TO_USSD";
        case RIL_E_DIAL_MODIFIED_TO_SS: return "RIL_E_DIAL_MODIFIED_TO_SS";
        case RIL_E_DIAL_MODIFIED_TO_DIAL: return "RIL_E_DIAL_MODIFIED_TO_DIAL";
        case RIL_E_USSD_MODIFIED_TO_DIAL: return "RIL_E_USSD_MODIFIED_TO_DIAL";
        case RIL_E_USSD_MODIFIED_TO_SS: return "RIL_E_USSD_MODIFIED_TO_SS";
        case RIL_E_USSD_MODIFIED_TO_USSD: return "RIL_E_USSD_MODIFIED_TO_USSD";
        case RIL_E_SS_MODIFIED_TO_DIAL: return "RIL_E_SS_MODIFIED_TO_DIAL";
        case RIL_E_SS_MODIFIED_TO_USSD: return "RIL_E_SS_MODIFIED_TO_USSD";
        case RIL_E_SUBSCRIPTION_NOT_SUPPORTED: return "RIL_E_SUBSCRIPTION_NOT_SUPPORTED";
        case RIL_E_SS_MODIFIED_TO_SS: return "RIL_E_SS_MODIFIED_TO_SS";
        case RIL_E_LCE_NOT_SUPPORTED: return "RIL_E_LCE_NOT_SUPPORTED";
        case RIL_E_NO_MEMORY: return "RIL_E_NO_MEMORY";
        case RIL_E_INTERNAL_ERR: return "RIL_E_INTERNAL_ERR";
        case RIL_E_SYSTEM_ERR: return "RIL_E_SYSTEM_ERR";
        case RIL_E_MODEM_ERR: return "RIL_E_MODEM_ERR";
        case RIL_E_INVALID_STATE: return "RIL_E_INVALID_STATE";
        case RIL_E_NO_RESOURCES: return "RIL_E_NO_RESOURCES";
        case RIL_E_SIM_ERR: return "RIL_E_SIM_ERR";
        case RIL_E_INVALID_ARGUMENTS: return "RIL_E_INVALID_ARGUMENTS";
        case RIL_E_INVALID_SIM_STATE: return "RIL_E_INVALID_SIM_STATE";
        case RIL_E_INVALID_MODEM_STATE: return "RIL_E_INVALID_MODEM_STATE";
        case RIL_E_INVALID_CALL_ID: return "RIL_E_INVALID_CALL_ID";
        case RIL_E_NO_SMS_TO_ACK: return "RIL_E_NO_SMS_TO_ACK";
        case RIL_E_NETWORK_ERR: return "RIL_E_NETWORK_ERR";
        case RIL_E_REQUEST_RATE_LIMITED: return "RIL_E_REQUEST_RATE_LIMITED";
        case RIL_E_SIM_BUSY: return "RIL_E_SIM_BUSY";
        case RIL_E_SIM_FULL: return "RIL_E_SIM_FULL";
        case RIL_E_NETWORK_REJECT: return "RIL_E_NETWORK_REJECT";
        case RIL_E_OPERATION_NOT_ALLOWED: return "RIL_E_OPERATION_NOT_ALLOWED";
        case RIL_E_EMPTY_RECORD: return "RIL_E_EMPTY_RECORD";
        case RIL_E_INVALID_SMS_FORMAT: return "RIL_E_INVALID_SMS_FORMAT";
        case RIL_E_ENCODING_ERR: return "RIL_E_ENCODING_ERR";
        case RIL_E_INVALID_SMSC_ADDRESS: return "RIL_E_INVALID_SMSC_ADDRESS";
        case RIL_E_NO_SUCH_ENTRY: return "RIL_E_NO_SUCH_ENTRY";
        case RIL_E_NETWORK_NOT_READY: return "RIL_E_NETWORK_NOT_READY";
        case RIL_E_NOT_PROVISIONED: return "RIL_E_NOT_PROVISIONED";
        case RIL_E_NO_SUBSCRIPTION: return "RIL_E_NO_SUBSCRIPTION";
        case RIL_E_NO_NETWORK_FOUND: return "RIL_E_NO_NETWORK_FOUND";
        case RIL_E_DEVICE_IN_USE: return "RIL_E_DEVICE_IN_USE";
        case RIL_E_ABORTED: return "RIL_E_ABORTED";
        case RIL_E_INVALID_RESPONSE: return "RIL_E_INVALID_RESPONSE";

        default: return "<unknown error>";
    }
}

const char *radioStateToString(RIL_RadioState s)
{
    switch(s) {
        case RADIO_STATE_OFF: return "RADIO_OFF";
        case RADIO_STATE_UNAVAILABLE: return "RADIO_UNAVAILABLE";
        case RADIO_STATE_ON:return"RADIO_ON";
        default: return "<unknown state>";
    }
}

const char *callStateToString(RIL_CallState s)
{
    switch(s) {
        case RIL_CALL_ACTIVE : return "ACTIVE";
        case RIL_CALL_HOLDING: return "HOLDING";
        case RIL_CALL_DIALING: return "DIALING";
        case RIL_CALL_ALERTING: return "ALERTING";
        case RIL_CALL_INCOMING: return "INCOMING";
        case RIL_CALL_WAITING: return "WAITING";
        default: return "<unknown state>";
    }
}

const char *requestToString(int request)
{
    /*
     cat libs/telephony/ril_commands.h \
     | egrep "^ *{RIL_" \
     | sed -re 's/\{RIL_([^,]+),[^,]+,([^}]+).+/case RIL_\1: return "\1";/'


     cat libs/telephony/ril_unsol_commands.h \
     | egrep "^ *{RIL_" \
     | sed -re 's/\{RIL_([^,]+),([^}]+).+/case RIL_\1: return "\1";/'

    */
    switch(request) {
        case RIL_REQUEST_GET_SIM_STATUS: return "GET_SIM_STATUS";
        case RIL_REQUEST_ENTER_SIM_PIN: return "ENTER_SIM_PIN";
        case RIL_REQUEST_ENTER_SIM_PUK: return "ENTER_SIM_PUK";
        case RIL_REQUEST_ENTER_SIM_PIN2: return "ENTER_SIM_PIN2";
        case RIL_REQUEST_ENTER_SIM_PUK2: return "ENTER_SIM_PUK2";
        case RIL_REQUEST_CHANGE_SIM_PIN: return "CHANGE_SIM_PIN";
        case RIL_REQUEST_CHANGE_SIM_PIN2: return "CHANGE_SIM_PIN2";
        case RIL_REQUEST_ENTER_NETWORK_DEPERSONALIZATION: return "ENTER_NETWORK_DEPERSONALIZATION";
        case RIL_REQUEST_GET_CURRENT_CALLS: return "GET_CURRENT_CALLS";
        case RIL_REQUEST_DIAL: return "DIAL";
        case RIL_REQUEST_GET_IMSI: return "GET_IMSI";
        case RIL_REQUEST_HANGUP: return "HANGUP";
        case RIL_REQUEST_HANGUP_WAITING_OR_BACKGROUND: return "HANGUP_WAITING_OR_BACKGROUND";
        case RIL_REQUEST_HANGUP_FOREGROUND_RESUME_BACKGROUND: return "HANGUP_FOREGROUND_RESUME_BACKGROUND";
        case RIL_REQUEST_SWITCH_WAITING_OR_HOLDING_AND_ACTIVE: return "SWITCH_WAITING_OR_HOLDING_AND_ACTIVE";
        case RIL_REQUEST_CONFERENCE: return "CONFERENCE";
        case RIL_REQUEST_UDUB: return "UDUB";
        case RIL_REQUEST_LAST_CALL_FAIL_CAUSE: return "LAST_CALL_FAIL_CAUSE";
        case RIL_REQUEST_SIGNAL_STRENGTH: return "SIGNAL_STRENGTH";
        case RIL_REQUEST_VOICE_REGISTRATION_STATE: return "VOICE_REGISTRATION_STATE";
        case RIL_REQUEST_DATA_REGISTRATION_STATE: return "DATA_REGISTRATION_STATE";
        case RIL_REQUEST_OPERATOR: return "OPERATOR";
        case RIL_REQUEST_RADIO_POWER: return "RADIO_POWER";
        case RIL_REQUEST_DTMF: return "DTMF";
        case RIL_REQUEST_SEND_SMS: return "SEND_SMS";
        case RIL_REQUEST_SEND_SMS_EXPECT_MORE: return "SEND_SMS_EXPECT_MORE";
        case RIL_REQUEST_SETUP_DATA_CALL: return "SETUP_DATA_CALL";
        case RIL_REQUEST_SIM_IO: return "SIM_IO";
        case RIL_REQUEST_SEND_USSD: return "SEND_USSD";
        case RIL_REQUEST_CANCEL_USSD: return "CANCEL_USSD";
        case RIL_REQUEST_GET_CLIR: return "GET_CLIR";
        case RIL_REQUEST_SET_CLIR: return "SET_CLIR";
        case RIL_REQUEST_QUERY_CALL_FORWARD_STATUS: return "QUERY_CALL_FORWARD_STATUS";
        case RIL_REQUEST_SET_CALL_FORWARD: return "SET_CALL_FORWARD";
        case RIL_REQUEST_QUERY_CALL_WAITING: return "QUERY_CALL_WAITING";
        case RIL_REQUEST_SET_CALL_WAITING: return "SET_CALL_WAITING";
        case RIL_REQUEST_SMS_ACKNOWLEDGE: return "SMS_ACKNOWLEDGE";
        case RIL_REQUEST_GET_IMEI: return "GET_IMEI";
        case RIL_REQUEST_GET_IMEISV: return "GET_IMEISV";
        case RIL_REQUEST_ANSWER: return "ANSWER";
        case RIL_REQUEST_DEACTIVATE_DATA_CALL: return "DEACTIVATE_DATA_CALL";
        case RIL_REQUEST_QUERY_FACILITY_LOCK: return "QUERY_FACILITY_LOCK";
        case RIL_REQUEST_SET_FACILITY_LOCK: return "SET_FACILITY_LOCK";
        case RIL_REQUEST_CHANGE_BARRING_PASSWORD: return "CHANGE_BARRING_PASSWORD";
        case RIL_REQUEST_QUERY_NETWORK_SELECTION_MODE: return "QUERY_NETWORK_SELECTION_MODE";
        case RIL_REQUEST_SET_NETWORK_SELECTION_AUTOMATIC: return "SET_NETWORK_SELECTION_AUTOMATIC";
        case RIL_REQUEST_SET_NETWORK_SELECTION_MANUAL: return "SET_NETWORK_SELECTION_MANUAL";
        case RIL_REQUEST_QUERY_AVAILABLE_NETWORKS : return "QUERY_AVAILABLE_NETWORKS ";
        case RIL_REQUEST_DTMF_START: return "DTMF_START";
        case RIL_REQUEST_DTMF_STOP: return "DTMF_STOP";
        case RIL_REQUEST_BASEBAND_VERSION: return "BASEBAND_VERSION";
        case RIL_REQUEST_SEPARATE_CONNECTION: return "SEPARATE_CONNECTION";
        case RIL_REQUEST_SET_PREFERRED_NETWORK_TYPE: return "SET_PREFERRED_NETWORK_TYPE";
        case RIL_REQUEST_GET_PREFERRED_NETWORK_TYPE: return "GET_PREFERRED_NETWORK_TYPE";
        case RIL_REQUEST_GET_NEIGHBORING_CELL_IDS: return "GET_NEIGHBORING_CELL_IDS";
        case RIL_REQUEST_SET_MUTE: return "SET_MUTE";
        case RIL_REQUEST_GET_MUTE: return "GET_MUTE";
        case RIL_REQUEST_QUERY_CLIP: return "QUERY_CLIP";
        case RIL_REQUEST_LAST_DATA_CALL_FAIL_CAUSE: return "LAST_DATA_CALL_FAIL_CAUSE";
        case RIL_REQUEST_DATA_CALL_LIST: return "DATA_CALL_LIST";
        case RIL_REQUEST_RESET_RADIO: return "RESET_RADIO";
        case RIL_REQUEST_OEM_HOOK_RAW: return "OEM_HOOK_RAW";
        case RIL_REQUEST_OEM_HOOK_STRINGS: return "OEM_HOOK_STRINGS";
        case RIL_REQUEST_SET_BAND_MODE: return "SET_BAND_MODE";
        case RIL_REQUEST_QUERY_AVAILABLE_BAND_MODE: return "QUERY_AVAILABLE_BAND_MODE";
        case RIL_REQUEST_STK_GET_PROFILE: return "STK_GET_PROFILE";
        case RIL_REQUEST_STK_SET_PROFILE: return "STK_SET_PROFILE";
        case RIL_REQUEST_STK_SEND_ENVELOPE_COMMAND: return "STK_SEND_ENVELOPE_COMMAND";
        case RIL_REQUEST_STK_SEND_TERMINAL_RESPONSE: return "STK_SEND_TERMINAL_RESPONSE";
        case RIL_REQUEST_STK_HANDLE_CALL_SETUP_REQUESTED_FROM_SIM: return "STK_HANDLE_CALL_SETUP_REQUESTED_FROM_SIM";
        case RIL_REQUEST_SCREEN_STATE: return "SCREEN_STATE";
        case RIL_REQUEST_SET_SUPP_SVC_NOTIFICATION: return "SET_SUPP_SVC_NOTIFICATION";
        case RIL_REQUEST_WRITE_SMS_TO_SIM: return "WRITE_SMS_TO_SIM";
        case RIL_REQUEST_DELETE_SMS_ON_SIM: return "DELETE_SMS_ON_SIM";
        case RIL_REQUEST_EXPLICIT_CALL_TRANSFER: return "EXPLICIT_CALL_TRANSFER";
        case RIL_REQUEST_SET_LOCATION_UPDATES: return "SET_LOCATION_UPDATES";
        case RIL_REQUEST_CDMA_SET_SUBSCRIPTION_SOURCE: return"CDMA_SET_SUBSCRIPTION_SOURCE";
        case RIL_REQUEST_CDMA_SET_ROAMING_PREFERENCE: return"CDMA_SET_ROAMING_PREFERENCE";
        case RIL_REQUEST_CDMA_QUERY_ROAMING_PREFERENCE: return"CDMA_QUERY_ROAMING_PREFERENCE";
        case RIL_REQUEST_SET_TTY_MODE: return"SET_TTY_MODE";
        case RIL_REQUEST_QUERY_TTY_MODE: return"QUERY_TTY_MODE";
        case RIL_REQUEST_CDMA_SET_PREFERRED_VOICE_PRIVACY_MODE: return"CDMA_SET_PREFERRED_VOICE_PRIVACY_MODE";
        case RIL_REQUEST_CDMA_QUERY_PREFERRED_VOICE_PRIVACY_MODE: return"CDMA_QUERY_PREFERRED_VOICE_PRIVACY_MODE";
        case RIL_REQUEST_CDMA_FLASH: return"CDMA_FLASH";
        case RIL_REQUEST_CDMA_BURST_DTMF: return"CDMA_BURST_DTMF";
        case RIL_REQUEST_CDMA_SEND_SMS: return"CDMA_SEND_SMS";
        case RIL_REQUEST_CDMA_SMS_ACKNOWLEDGE: return"CDMA_SMS_ACKNOWLEDGE";
        case RIL_REQUEST_GSM_GET_BROADCAST_SMS_CONFIG: return"GSM_GET_BROADCAST_SMS_CONFIG";
        case RIL_REQUEST_GSM_SET_BROADCAST_SMS_CONFIG: return"GSM_SET_BROADCAST_SMS_CONFIG";
        case RIL_REQUEST_GSM_SMS_BROADCAST_ACTIVATION: return "GSM_SMS_BROADCAST_ACTIVATION";
        case RIL_REQUEST_CDMA_GET_BROADCAST_SMS_CONFIG: return "CDMA_GET_BROADCAST_SMS_CONFIG";
        case RIL_REQUEST_CDMA_SET_BROADCAST_SMS_CONFIG: return "CDMA_SET_BROADCAST_SMS_CONFIG";
        case RIL_REQUEST_CDMA_SMS_BROADCAST_ACTIVATION: return "CDMA_SMS_BROADCAST_ACTIVATION";
        case RIL_REQUEST_CDMA_VALIDATE_AND_WRITE_AKEY: return"CDMA_VALIDATE_AND_WRITE_AKEY";
        case RIL_REQUEST_CDMA_SUBSCRIPTION: return"CDMA_SUBSCRIPTION";
        case RIL_REQUEST_CDMA_WRITE_SMS_TO_RUIM: return "CDMA_WRITE_SMS_TO_RUIM";
        case RIL_REQUEST_CDMA_DELETE_SMS_ON_RUIM: return "CDMA_DELETE_SMS_ON_RUIM";
        case RIL_REQUEST_DEVICE_IDENTITY: return "DEVICE_IDENTITY";
        case RIL_REQUEST_EXIT_EMERGENCY_CALLBACK_MODE: return "EXIT_EMERGENCY_CALLBACK_MODE";
        case RIL_REQUEST_GET_SMSC_ADDRESS: return "GET_SMSC_ADDRESS";
        case RIL_REQUEST_SET_SMSC_ADDRESS: return "SET_SMSC_ADDRESS";
        case RIL_REQUEST_REPORT_SMS_MEMORY_STATUS: return "REPORT_SMS_MEMORY_STATUS";
        case RIL_REQUEST_REPORT_STK_SERVICE_IS_RUNNING: return "REPORT_STK_SERVICE_IS_RUNNING";
        case RIL_REQUEST_CDMA_GET_SUBSCRIPTION_SOURCE: return "CDMA_GET_SUBSCRIPTION_SOURCE";
        case RIL_REQUEST_ISIM_AUTHENTICATION: return "ISIM_AUTHENTICATION";
        case RIL_REQUEST_ACKNOWLEDGE_INCOMING_GSM_SMS_WITH_PDU: return "ACKNOWLEDGE_INCOMING_GSM_SMS_WITH_PDU";
        case RIL_REQUEST_STK_SEND_ENVELOPE_WITH_STATUS: return "STK_SEND_ENVELOPE_WITH_STATUS";
        case RIL_REQUEST_VOICE_RADIO_TECH: return "VOICE_RADIO_TECH";
        case RIL_REQUEST_SIM_TRANSMIT_APDU_BASIC: return "SIM_TRANSMIT_APDU_BASIC";
        case RIL_REQUEST_SIM_OPEN_CHANNEL: return "SIM_OPEN_CHANNEL";
        case RIL_REQUEST_SIM_CLOSE_CHANNEL: return "SIM_CLOSE_CHANNEL";
        case RIL_REQUEST_SIM_TRANSMIT_APDU_CHANNEL: return "SIM_TRANSMIT_APDU_CHANNEL";
        case RIL_REQUEST_INCREMENTAL_NETWORK_SCAN: return "INCREMENTAL_NETWORK_SCAN";
        case RIL_UNSOL_RESPONSE_RADIO_STATE_CHANGED: return "UNSOL_RESPONSE_RADIO_STATE_CHANGED";
        case RIL_UNSOL_RESPONSE_CALL_STATE_CHANGED: return "UNSOL_RESPONSE_CALL_STATE_CHANGED";
        case RIL_UNSOL_RESPONSE_VOICE_NETWORK_STATE_CHANGED: return "UNSOL_RESPONSE_VOICE_NETWORK_STATE_CHANGED";
        case RIL_UNSOL_RESPONSE_NEW_SMS: return "UNSOL_RESPONSE_NEW_SMS";
        case RIL_UNSOL_RESPONSE_NEW_SMS_STATUS_REPORT: return "UNSOL_RESPONSE_NEW_SMS_STATUS_REPORT";
        case RIL_UNSOL_RESPONSE_NEW_SMS_ON_SIM: return "UNSOL_RESPONSE_NEW_SMS_ON_SIM";
        case RIL_UNSOL_ON_USSD: return "UNSOL_ON_USSD";
        case RIL_UNSOL_ON_USSD_REQUEST: return "UNSOL_ON_USSD_REQUEST(obsolete)";
        case RIL_UNSOL_NITZ_TIME_RECEIVED: return "UNSOL_NITZ_TIME_RECEIVED";
        case RIL_UNSOL_SIGNAL_STRENGTH: return "UNSOL_SIGNAL_STRENGTH";
        case RIL_UNSOL_STK_SESSION_END: return "UNSOL_STK_SESSION_END";
        case RIL_UNSOL_STK_PROACTIVE_COMMAND: return "UNSOL_STK_PROACTIVE_COMMAND";
        case RIL_UNSOL_STK_EVENT_NOTIFY: return "UNSOL_STK_EVENT_NOTIFY";
        case RIL_UNSOL_STK_CALL_SETUP: return "UNSOL_STK_CALL_SETUP";
        case RIL_UNSOL_SIM_SMS_STORAGE_FULL: return "UNSOL_SIM_SMS_STORAGE_FUL";
        case RIL_UNSOL_SIM_REFRESH: return "UNSOL_SIM_REFRESH";
        case RIL_UNSOL_DATA_CALL_LIST_CHANGED: return "UNSOL_DATA_CALL_LIST_CHANGED";
        case RIL_UNSOL_SUPP_SVC_NOTIFICATION: return "UNSOL_SUPP_SVC_NOTIFICATION";
        case RIL_UNSOL_CALL_RING: return "UNSOL_CALL_RING";
        case RIL_UNSOL_RESPONSE_SIM_STATUS_CHANGED: return "UNSOL_RESPONSE_SIM_STATUS_CHANGED";
        case RIL_UNSOL_RESPONSE_CDMA_NEW_SMS: return "UNSOL_NEW_CDMA_SMS";
        case RIL_UNSOL_RESPONSE_NEW_BROADCAST_SMS: return "UNSOL_NEW_BROADCAST_SMS";
        case RIL_UNSOL_CDMA_RUIM_SMS_STORAGE_FULL: return "UNSOL_CDMA_RUIM_SMS_STORAGE_FULL";
        case RIL_UNSOL_RESTRICTED_STATE_CHANGED: return "UNSOL_RESTRICTED_STATE_CHANGED";
        case RIL_UNSOL_ENTER_EMERGENCY_CALLBACK_MODE: return "UNSOL_ENTER_EMERGENCY_CALLBACK_MODE";
        case RIL_UNSOL_CDMA_CALL_WAITING: return "UNSOL_CDMA_CALL_WAITING";
        case RIL_UNSOL_CDMA_OTA_PROVISION_STATUS: return "UNSOL_CDMA_OTA_PROVISION_STATUS";
        case RIL_UNSOL_CDMA_INFO_REC: return "UNSOL_CDMA_INFO_REC";
        case RIL_UNSOL_OEM_HOOK_RAW: return "UNSOL_OEM_HOOK_RAW";
        case RIL_UNSOL_RINGBACK_TONE: return "UNSOL_RINGBACK_TONE";
        case RIL_UNSOL_RESEND_INCALL_MUTE: return "UNSOL_RESEND_INCALL_MUTE";
        case RIL_UNSOL_CDMA_SUBSCRIPTION_SOURCE_CHANGED: return "UNSOL_CDMA_SUBSCRIPTION_SOURCE_CHANGED";
        case RIL_UNSOL_CDMA_PRL_CHANGED: return "UNSOL_CDMA_PRL_CHANGED";
        case RIL_UNSOL_EXIT_EMERGENCY_CALLBACK_MODE: return "UNSOL_EXIT_EMERGENCY_CALLBACK_MODE";
        case RIL_UNSOL_RIL_CONNECTED: return "UNSOL_RIL_CONNECTED";
        case RIL_UNSOL_VOICE_RADIO_TECH_CHANGED: return "UNSOL_VOICE_RADIO_TECH_CHANGED";
        case RIL_UNSOL_DATA_RADIO_TECH_CHANGED: return "UNSOL_DATA_RADIO_TECH_CHANGED";
        case RIL_UNSOL_INCREMENTAL_NETWORK_SCAN: return "UNSOL_INCREMENTAL_NETWORK_SCAN";
        case RIL_UNSOL_IMS_SETTINGS_MSD_IND: return "RIL_UNSOL_IMS_SETTINGS_MSD_IND";
        case RIL_UNSOL_GET_LATEST_MSD_IND: return "RIL_UNSOL_GET_LATEST_MSD_IND";
        default: return "<unknown request>";
    }
}


void libril_onProcConnect(int procId) {

    // Inform we are connected and the ril version
    int rilVer = s_callbacks.version;
    int ret;

    do {
        if (procId<=0){
            break;
        }

        ACLOGI(LOG_TAG,"%s: RIL_UNSOL_RIL_CONNECTED: %d for proc %d ",__func__, rilVer, procId);
        ril_UnsolicitedProcResponse(RIL_UNSOL_RIL_CONNECTED,
                                            &rilVer, sizeof(rilVer),procId);
        // implicit radio state changed
        ACLOGI(LOG_TAG,"%s: RIL_UNSOL_RESPONSE_RADIO_STATE_CHANGED",__func__);
        ril_UnsolicitedProcResponse(RIL_UNSOL_RESPONSE_RADIO_STATE_CHANGED, NULL, 0, procId);


        // Send last NITZ time data, in case it was missed
        if (s_lastNITZTimeData != NULL) {
            ACLOGI(LOG_TAG,"%s: sendResponseRaw(NITZ)",__func__);
            ret=rilProc::sendResponseRaw(procId, s_lastNITZTimeData, s_lastNITZTimeDataSize,0);

            FREE(s_lastNITZTimeData);
            s_lastNITZTimeData = NULL;
        }

        // Get version string
        if (s_callbacks.getVersion != NULL) {
            const char *version;
            version = s_callbacks.getVersion();
            ACLOGI(LOG_TAG,"%s: RIL Daemon version: %s",__func__, version);
        } else {
            ACLOGI(LOG_TAG,"%s: RIL Daemon version: unavailable",__func__);
        }
    } while (0);
}
