/* //device/libs/telephony/ril_event.cpp
**
** Copyright 2008, The Android Open Source Project
**
** Licensed under the Apache License, Version 2.0 (the "License");
** you may not use this file except in compliance with the License.
** You may obtain a copy of the License at
**
**     http://www.apache.org/licenses/LICENSE-2.0
**
** Unless required by applicable law or agreed to in writing, software
** distributed under the License is distributed on an "AS IS" BASIS,
** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
** See the License for the specific language governing permissions and
** limitations under the License.
*/
#include "config.h"

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/cdefs.h>
#include <glob.h>
#include <time.h>
/*PEIKER modification: replace android utils include by corrrespondingcutils include */

#include "utils_log.h"
#include "utils_misc.h"
#include "ril_i.h"

/*PEIKER modification: replace android utils include by corrrespondingcutils include */
#include <string.h>
#include <sys/time.h>
#include <time.h>

#include <pthread.h>

#include "rilEvent.h"
static pthread_mutex_t listMutex;
#define MUTEX_ACQUIRE() pthread_mutex_lock(&listMutex)
#define MUTEX_RELEASE() pthread_mutex_unlock(&listMutex)
#define MUTEX_INIT() pthread_mutex_init(&listMutex, NULL)
#define MUTEX_DESTROY() pthread_mutex_destroy(&listMutex)

static pthread_mutex_t s_eventMutex;
static pthread_cond_t s_eventCond;

static void rilEvent_timerReAdd(rilEvent_t * ev);

typedef struct ril_event {
    struct ril_event    *next;
    struct ril_event    *prev;
    int                  index;
    bool                 persistent;
    bool                 isTimer;
    struct timeval       period;
    struct timeval       timeout;
    ril_event_cb         func;
    void                *param;
    int                  fired;
} rilEvent_t;


#ifndef timeradd
#define timeradd(tvp, uvp, vvp)                        \
    do {                                \
        (vvp)->tv_sec = (tvp)->tv_sec + (uvp)->tv_sec;        \
        (vvp)->tv_usec = (tvp)->tv_usec + (uvp)->tv_usec;       \
        if ((vvp)->tv_usec >= 1000000) {            \
            (vvp)->tv_sec++;                \
            (vvp)->tv_usec -= 1000000;            \
        }                            \
    } while (0)
#endif

#ifndef timercmp
#define timercmp(a, b, op)               \
        ((a)->tv_sec == (b)->tv_sec      \
        ? (a)->tv_usec op (b)->tv_usec   \
        : (a)->tv_sec op (b)->tv_sec)
#endif

#ifndef timersub
#define timersub(a, b, res)                           \
    do {                                              \
        (res)->tv_sec = (a)->tv_sec - (b)->tv_sec;    \
        (res)->tv_usec = (a)->tv_usec - (b)->tv_usec; \
        if ((res)->tv_usec < 0) {                     \
            (res)->tv_usec += 1000000;                \
            (res)->tv_sec -= 1;                       \
        }                                             \
    } while(0);
#endif


static rilEvent_t watch_table;
static rilEvent_t timer_list;
static rilEvent_t pending_list;
static int s_isTriggered=1;

static void *rilEvent_loop(void *param);

static void getNow(struct timeval * tv)
{
    struct timespec ts;
    clock_gettime(CLOCK_MONOTONIC, &ts);
    tv->tv_sec = ts.tv_sec;
    tv->tv_usec = ts.tv_nsec/1000;

}


static void rilEvent_initList(rilEvent_t *list)
{
    memset((void*)list,(int)0, sizeof(rilEvent_t));
    list->next = list;
    list->prev = list;
}

static void rilEvent_addToList(rilEvent_t *ev, rilEvent_t *list)
{
    ev->next = list;
    ev->prev = list->prev;
    ev->prev->next = ev;
    list->prev = ev;
}

static void rilEvent_removeFromList(rilEvent_t * ev)
{
    ev->next->prev = ev->prev;
    ev->prev->next = ev->next;
    ev->next = ev;
    ev->prev = ev;
}

static void rilEvent_processTimeouts()
{
    ACLOGV(LOG_TAG,"%s ..",__func__);

    MUTEX_ACQUIRE();
    struct timeval now;
    rilEvent_t * tev = timer_list.next;
    rilEvent_t * next;

    getNow(&now);
    // walk list, see if now >= ev->timeout for any events

    ACLOGD(LOG_TAG,"%s: Looking for timers <= %ds + %dus",__func__, (int)now.tv_sec, (int)now.tv_usec);

    while ((tev != &timer_list) && (timercmp(&now, &tev->timeout, >))) {
        // Timer expired
        ACLOGD(LOG_TAG,"%s: firing timer",__func__);
        next = tev->next;
        ACLOGD(LOG_TAG,"%s: remove from timer list %x",__func__,(int)tev);
        rilEvent_removeFromList(tev);
        ACLOGD(LOG_TAG,"%s: add to pending list %x",__func__,(int)tev);
        rilEvent_addToList(tev, &pending_list);
        tev = next;
    }
    MUTEX_RELEASE();
    ACLOGV(LOG_TAG,"%s done",__func__);
}

// Add event to watch list
void rilEvent_trigger()
{
    ACLOGD(LOG_TAG,"%s..",__func__);

    pthread_mutex_lock(&s_eventMutex );
    s_isTriggered=1;
    pthread_cond_signal(&s_eventCond );
    pthread_mutex_unlock(&s_eventMutex );
}

void rilEvent_fire(rilEvent_t *ev)
{
    ACLOGD(LOG_TAG,"%s..",__func__);

    MUTEX_ACQUIRE();
    if (ev){
        if (ev->isTimer==false){
            ev->fired=1;
        }
    }
    MUTEX_RELEASE();
}


static void rilEvent_processReadReadies()
{
    int n=0;
    rilEvent_t *next=NULL;

    MUTEX_ACQUIRE();

    rilEvent_t *tev = watch_table.next;

    while (tev != &watch_table){
        next = tev->next;
        if (tev->fired){
            rilEvent_removeFromList(tev);
            rilEvent_addToList(tev, &pending_list);
            n++;
        }
        tev = next;
    }

    MUTEX_RELEASE();

    ACLOGD(LOG_TAG,"%s: added: %d",__func__,n);
}

static void rilEvent_firePending()
{
    ACLOGV(LOG_TAG,"%s..",__func__);
    int n=0;

    struct ril_event * ev =NULL;

    while (1) {
        MUTEX_ACQUIRE();

        ev = pending_list.next;
        if (ev == &pending_list) {
            //ACLOGD(LOG_TAG,"%s: pending list is empty",__func__);
            MUTEX_RELEASE();
            break;
        }

        ACLOGV(LOG_TAG,"%s: remove from pending list : 0%x",__func__,(int)ev);
        rilEvent_removeFromList(ev);
        n++;

        MUTEX_RELEASE();

        ACLOGV(LOG_TAG,"%s: exec function..",__func__);
        ev->func(0, ev->param);
        ACLOGV(LOG_TAG,"%s: exec done",__func__);

        if (ev->persistent){
            ACLOGD(LOG_TAG,"%s: re-add to event list : 0%x",__func__,(int)ev);
            if (ev->isTimer){
                rilEvent_timerReAdd(ev);
            } else {
                rilEvent_addToList(ev, &watch_table);
            }
        } else {
            FREE(ev);
        }
    }

    ACLOGD(LOG_TAG,"%s: done for %d requests",__func__,n);

}

static int rilEvent_calcNextTimeout(struct timespec * tsp)
{
    struct ril_event * tev = timer_list.next;
    struct timeval now;
    struct timeval tv;

    getNow(&now);

    // Sorted list, so calc based on first node
    if (tev == &timer_list) {
        // no pending timers
        return -1;
    }
    ACLOGD(LOG_TAG,"%s: now = %ds + %dus",__func__, (int)now.tv_sec, (int)now.tv_usec);
    ACLOGD(LOG_TAG,"%s: next = %ds + %dus",__func__,
            (int)tev->timeout.tv_sec, (int)tev->timeout.tv_usec);


    if (timercmp(&tev->timeout, &now, >)) {
        // do nothing (absolute time values required !)
        tv.tv_sec =tev->timeout.tv_sec;
        tv.tv_usec=tev->timeout.tv_usec;

    } else {
        // timer already expired.
        tv.tv_sec = now.tv_sec;
        tv.tv_usec = now.tv_usec;
    }

    tsp->tv_sec=tv.tv_sec;
    tsp->tv_nsec= 1000UL * tv.tv_usec;

    return 0;
}


// Initialize an event
void rilEvent_set(rilEvent_t * ev, bool persistent, ril_event_cb func, void * param)
{
    memset(ev, 0, sizeof(rilEvent_t));
    ev->index = -1;
    ev->func = func;
    ev->param = param;
    ev->persistent= persistent;
    ev->fired=0;
}

// Add event to watch list
void rilEvent_add(rilEvent_t * ev)
{

    MUTEX_ACQUIRE();

    ev->isTimer=false;
    rilEvent_addToList(ev,&watch_table);

    MUTEX_RELEASE();
    ACLOGD(LOG_TAG,"%s: done",__func__);
}

void rilEvent_timerReAdd(rilEvent_t * ev)
{
    struct timeval now;

    MUTEX_ACQUIRE();

    rilEvent_t * list;

    do {
        if (!ev->isTimer){
            break;
        }
        if (!ev->persistent){
            break;
        }

        // add to timer list
        list = timer_list.next;
        getNow(&now);
        timeradd(&now, &ev->period, &ev->timeout);

        // keep list sorted
        while (timercmp(&list->timeout, &ev->timeout, < )
                && (list != &timer_list)) {
            list = list->next;
        }
        // list now points to the first event older than ev
        rilEvent_addToList(ev, list);
    } while (0);

    MUTEX_RELEASE();
    ACLOGD(LOG_TAG,"%s done",__func__);

}



// Add timer event
void rilEvent_timerAdd(rilEvent_t * ev, struct timeval * tv)
{
    struct timeval now;

    MUTEX_ACQUIRE();

    rilEvent_t * list;
    ev->isTimer=1;
    if (ev->persistent){
        memcpy(&ev->period,tv,sizeof(struct timeval));
    }

    if (tv != NULL) {
        // add to timer list
        list = timer_list.next;
        getNow(&now);
        timeradd(&now, tv, &ev->timeout);

        // keep list sorted
        while (timercmp(&list->timeout, &ev->timeout, < )
                && (list != &timer_list)) {
            list = list->next;
        }
        // list now points to the first event older than ev
        rilEvent_addToList(ev, list);
    }

    MUTEX_RELEASE();
    ACLOGD(LOG_TAG,"%s done",__func__);

}

// Remove event from watch or timer list
void rilEvent_del(rilEvent_t * ev)
{
    MUTEX_ACQUIRE();
    rilEvent_removeFromList(ev);
    MUTEX_RELEASE();
    FREE(ev);
}

rilEvent_t *rilEvent_create()
{
    return (rilEvent_t *)MALLOC(sizeof(rilEvent_t));
}

int rilEvent_init()
{
    pthread_t tid;
    int rc;
    pthread_attr_t attr;
    pthread_condattr_t cattr;

    ACLOGD(LOG_TAG, "%s..",__func__);

    MUTEX_INIT();
    rilEvent_initList(&watch_table);
    rilEvent_initList(&timer_list);
    rilEvent_initList(&pending_list);

    pthread_attr_init (&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    pthread_mutex_init(&s_eventMutex,NULL);

    (void)pthread_condattr_init(&cattr);
    (void)pthread_condattr_setclock(&cattr, CLOCK_MONOTONIC);
    pthread_cond_init(&s_eventCond,&cattr);

    s_isTriggered=0;

    rc=pthread_create(&tid, &attr, rilEvent_loop, NULL);
    ACLOGD(LOG_TAG, "%s done",__func__);
    return rc;
}


void *rilEvent_loop(void *param)
{
    int n;
    struct timespec *ptv;
    struct timespec timeToWait;
    int ret;

    ACLOGD(LOG_TAG, "%s..",__func__);

    do {
        if (-1 == rilEvent_calcNextTimeout(&timeToWait)) {
            // no pending timers; block indefinitely
            ACLOGD(LOG_TAG,"%s: block indefinite ",__func__);
            ptv = NULL;
        } else {
            ACLOGD(LOG_TAG,"%s: blocking until %ds + %dns",
                    __func__,(int)timeToWait.tv_sec, (int)timeToWait.tv_nsec);
            ptv = &timeToWait;
        }

        pthread_mutex_lock(&s_eventMutex );

        if (ptv==NULL){
            ret= pthread_cond_wait(&s_eventCond, &s_eventMutex);
        } else {
            ret = pthread_cond_timedwait(&s_eventCond, &s_eventMutex, ptv);
        }

        if (ret<0){
            ACLOGD(LOG_TAG,"%s: ret = %d ignore",__func__,ret);
            pthread_mutex_unlock(&s_eventMutex );
            continue;
        }

        if (ptv!=NULL){
            ACLOGD(LOG_TAG,"%s: rilEvent_processTimeouts..",__func__);
            rilEvent_processTimeouts();
        } else if (s_isTriggered==0){
            pthread_mutex_unlock(&s_eventMutex );
            continue;
        }

        s_isTriggered=0;
        pthread_mutex_unlock(&s_eventMutex );

        ACLOGD(LOG_TAG,"%s: rilEvent_processReadReadies..",__func__);
        rilEvent_processReadReadies();

        // Fire away
        ACLOGD(LOG_TAG,"%s: rilEvent_firePending..",__func__);
        rilEvent_firePending();

    } while(1);

    return param;
}


