/*
 * Copyright (C) 2005 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

//
// Miscellaneous utility functions.
//
#include "config.h"

#include <sys/syslog.h>
#include <sys/stat.h>
#include <string.h>
#include <errno.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>

#include "ril_i.h"
#include "utils_misc.h"

#include "utils_log.h"

#define RIL_FILE          1
#define RIL_EVENT_FILE    2
#define RIL_PROC_FILE     3
#define RIL_PARCEL_FILE   4


typedef struct mem
{
     struct mem* next;
     size_t      size;
     int         id;
     int         file;
     int         line;
     void        *data;
}mem_t;

typedef struct memhdr
{
    mem_t           *first;
    int             alloc_count;
    pthread_mutex_t lock;
    int             memId;
}mem_header_t;


static int mem_name2Id(const char *fileName)
{
    if (strstr(fileName,"ril.cpp") != NULL){
        return RIL_FILE;
    }
    if (strstr(fileName,"RilProc.cpp") != NULL){
        return RIL_PROC_FILE;
    }
    if (strstr(fileName,"rilEvent.cpp") != NULL){
        return RIL_EVENT_FILE;
    }
    if (strstr(fileName,"Parcel.cpp") != NULL){
        return RIL_PARCEL_FILE;
    }
    ALOGD(LOG_TAG,"%s: %s",__func__,fileName);
    return 0;
}

static void mem_attach(mem_header_t *mem_hdr, mem_t *current);
static void mem_detach(mem_header_t *mem_hdr, mem_t *current);

/** \brief mem_attach
 *   memory debug support: attach to mem info
 *
 *  \param[in] mem_hdr header of memory list
 *  \param[in] current refers to memory entity
 *
 */
void mem_attach(mem_header_t *mem_hdr, mem_t *current)
{
    if (mem_hdr->first==NULL) {
        mem_hdr->first=current;
        current->next=NULL;
        mem_hdr->alloc_count=1;
    } else {
        //attach at the start of current list
        current->next=mem_hdr->first;
        mem_hdr->first=current;
        mem_hdr->alloc_count++;
    }

    //ALOGD(LOG_TAG,"%s: length=%d",mem_hdr->alloc_count,__func__);
}

/** \brief mem_detach
 *   memory debug support: detach of mem info
 *
 *  \param[in] mem_hdr header of memory list
 *  \param[in] current refers to memory entity
 *
 */
void mem_detach(mem_header_t *mem_hdr, mem_t *current)
{
    mem_t *prev;
    mem_t *next;

    do {

        if (current == mem_hdr->first){
            mem_hdr->alloc_count--;
            mem_hdr->first = current->next;
            //ALOGD(LOG_TAG,"%s: header removed",__func__);
            break;
        }

        prev = mem_hdr->first;
        next = prev->next;

        while (next != NULL){
            if (next == current){
                prev->next = current->next;
                mem_hdr->alloc_count--;
                //ALOGD(LOG_TAG,"%s: list element removed",__func__);
                break;
            }

            prev=next;
            next=prev->next;
        }

    } while (0);
}


/** \brief  cpp_mem_init
 *   memory list initializer
 *
 *  \return head of list
 */
mem_header_t * cpp_mem_init(void){
    return mem_init();
}

/** \brief  cpp_mem_dump
 *   dump memory list to syslog
 *
 *  \param[in] mem_hdr head of list
 */
void cpp_mem_dump(mem_header_t *mem_hdr )
{
    mem_dump(mem_hdr);
}

void cpp_mem_free(mem_header_t *mem_hdr, void* data, const char *fileName, int line)
{
    mem_free(mem_hdr,data,fileName,line);
}

void *cpp_mem_malloc(mem_header_t *mem_hdr, size_t size,const char *fileName, int line)
{
    return mem_malloc(mem_hdr, size, fileName ,line);
}

void *cpp_mem_realloc(mem_header_t *mem_hdr,void *data, size_t size,const char *fileName, int line)
{
    return mem_realloc(mem_hdr, data, size, fileName,line);
}


extern "C" {


/** \brief mem_init
 *
 *  initialize mem control
 *
 * \return memory list
 */
mem_header_t * mem_init(void)
{
    mem_header_t *mem=NULL;
    while (1) {

        mem =(mem_header_t *)malloc(sizeof(mem_header_t));
        if (mem==NULL){
            break;
        }

        mem->first=NULL;
        mem->alloc_count=0;
        mem->memId=0;
        (void)pthread_mutex_init(&mem->lock,NULL);
        break;
    }
    return mem;
}


/** \brief mem_malloc
 *
 *  allocate memory instance
 *
 *  \param[in] mem_hdr header of memory element list
 *  \param[in] size size of memory block zo be allocated
 *  \param[in] fileName identifies the source code file requesting the release
 *  \param[in] line identifies the source code line where the release is requested
 *
 *  \return allocated memory object
 *
 */
void *mem_malloc(mem_header_t *mem_hdr, size_t size,  const char *fileName, int line)
{
    int idx;
    void *newData;
    mem_t *current;

    newData = malloc(size+sizeof(mem_t));
    if (newData==NULL){
        return NULL;
    }

    current=(mem_t*)newData;
    current->size=size;
    current->line=line;
    current->file=mem_name2Id(fileName);
    current->data=(void*)((char *)newData+sizeof(mem_t));

    (void)pthread_mutex_lock(&mem_hdr->lock);

    current->id= mem_hdr->memId++;
    if (mem_hdr->memId < 0){
        mem_hdr->memId=1;
    }
    mem_attach(mem_hdr,current);

    (void)pthread_mutex_unlock(&mem_hdr->lock);

    ACLOGD(LOG_TAG,"%s: %d(%d): len %d id=%d at %p ",__func__,
            current->file, line, size,current->id, current->data);
    return current->data;
}

/** \brief mem_realloc
 *
 *  reallocate mem info
 *
 *  \param[in] mem_hdr header of memory element list
 *  \param[in] data poinzter of current data element
 *  \param[in] size size of memory block zo be allocated
 *  \param[in] fileName identifies the source code file requesting the release
 *  \param[in] line identifies the source code line where the release is requested
 *  \return allocated memory object
 *
 */
void *mem_realloc(mem_header_t *mem_hdr, void *data, size_t size, const char *fileName, int line)
{
     mem_t *current=NULL;
     void *newData=NULL;
     int doRealloc=0;

     int originalSize;

     if (data==NULL){
         return mem_malloc(mem_hdr,size,fileName,line);
     }

     pthread_mutex_lock(&mem_hdr->lock);
     current=mem_hdr->first;
     while (current) {
         if (current->data == data){

             if (size==0){
                 mem_detach(mem_hdr,current);
                 free(current);
                 newData=NULL;
                 break;
             }

             if (current->size >= size){
                 current->size = size;
                 newData = data;
                 break;
             }

             originalSize=current->size;
             mem_detach(mem_hdr,current);
             doRealloc=1;
             break;
         }
         current=current->next;
     }
     pthread_mutex_unlock(&mem_hdr->lock);

     if (doRealloc){
         newData = mem_malloc(mem_hdr,size,fileName,line);
         if (newData){
             memset(newData,0,size);
             memcpy(newData,data,originalSize);
         }
         ALOGD(LOG_TAG,"%s: release %d ",__func__,current->id);
         free(current);
     }
     return newData;
}

/** \brief mem_free
 *
 *   release mem info
 *
 *   called during unloading of the lib
 *   nothing to be done, as all allocated resources are released before
 *
 *  \param[in] mem_hdr header of memory object list
 *  \param[in] data data element to be released
 *  \param[in] fileName identifies the source code file requesting the release
 *  \param[in] line identifies the source code line where the release is requested
 *
 */
void mem_free(mem_header_t *mem_hdr, void* data, const char *fileName, int line)
{
    mem_t *current=NULL;

    (void)pthread_mutex_lock(&mem_hdr->lock);

    current  = mem_hdr->first;
    while (current != NULL){
        if (current->data == data){
            mem_detach(mem_hdr,current);
            ACLOGD(LOG_TAG,"%s[%d] at %d(%d)",__func__,current->id,mem_name2Id(fileName),line);
            free(current);
            break;
        }
        current = current->next;
    }

    (void)pthread_mutex_unlock(&mem_hdr->lock);

    ACLOGD(LOG_TAG,"%s done",__func__);
}

/** \brief mem_dump
 *  dump mem control info to syslog
 *
 *  \param[in] mem_hdr header of memory object list
 */
void mem_dump(mem_header_t *mem_hdr)
{
    mem_t *current;

    ACLOGN(LOG_TAG,"%s ..",__func__);

    (void)pthread_mutex_lock(&mem_hdr->lock);

    current=mem_hdr->first;
    while (current!=NULL){
        ACLOGN(LOG_TAG,"%s[%d]: %d(%d), address %p, size %d",__func__,
                        current->id,current->file, current->line, current->data, current->size);
        current=current->next;
    }

    (void)pthread_mutex_unlock(&mem_hdr->lock);
}

} //extern C

/** \brief countArgv
 * Count the number of args in an argument vector.
 * Don't count the final NULL.
 *
 *  \param[in] argv  argument list
 *
 *  \return number of arguments
 */
int countArgv(const char* const argv[])
{
    int count = 0;

    while (argv[count] != NULL)
        count++;

    return count;
}

#include <stdio.h>

/** \brief getFileType
 *
 *   Get a file's type.
 *   \param[in] fileName  file name
 *
 * \return file type
 */
FileType getFileType(const char* fileName)
{
    struct stat sb;

    if (stat(fileName, &sb) < 0) {
        if (errno == ENOENT || errno == ENOTDIR)
            return kFileTypeNonexistent;
        else {
            fprintf(stderr, "getFileType got errno=%d on '%s'\n",
                errno, fileName);
            return kFileTypeUnknown;
        }
    } else {
        if (S_ISREG(sb.st_mode))
            return kFileTypeRegular;
        else if (S_ISDIR(sb.st_mode))
            return kFileTypeDirectory;
        else if (S_ISCHR(sb.st_mode))
            return kFileTypeCharDev;
        else if (S_ISBLK(sb.st_mode))
            return kFileTypeBlockDev;
        else if (S_ISFIFO(sb.st_mode))
            return kFileTypeFifo;
#ifdef HAVE_SYMLINKS
        else if (S_ISLNK(sb.st_mode))
            return kFileTypeSymlink;
        else if (S_ISSOCK(sb.st_mode))
            return kFileTypeSocket;
#endif
        else
            return kFileTypeUnknown;
    }
}

/** \brief getFileModDate
 *
 *   Get a file's modification data.
 *   \param[in] fileName  file name
 *
 * \return file type
 */
time_t getFileModDate(const char* fileName)
{
    struct stat sb;

    if (stat(fileName, &sb) < 0)
        return (time_t) -1;

    return sb.st_mtime;
}

/** \brief roundUpPower2
 * Round up to the next highest power of 2.
 * Found on http://graphics.stanford.edu/~seander/bithacks.html.
 *
 *   \param[in] val  input value
 *
 *   \return rounded result
 */
unsigned int roundUpPower2(unsigned int val)
{
    val--;
    val |= val >> 1;
    val |= val >> 2;
    val |= val >> 4;
    val |= val >> 8;
    val |= val >> 16;
    val++;

    return val;
}


