/*
 * Copyright (C) 2005 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

//
// Handy utility functions and portability code.
//
#ifndef _LIBS_UTILS_MISC_H
#define _LIBS_UTILS_MISC_H

#include <pthread.h>
#include <sys/time.h>

#include "ril_i.h"

#ifdef MEM_CTL
#undef MEM_CTL
#endif

#ifdef __cplusplus
extern "C" {
#endif

extern void mem_dump(mem_header_t *mem_hdr);
extern void mem_free(mem_header_t *mem_hdr, void* data,const char *fileName, int line);
extern void *mem_malloc(mem_header_t *mem_hdr, size_t size, const char *fileName, int line);
extern void *mem_realloc(mem_header_t *mem_hdr,void *data, size_t size,const char *fileName, int line);

extern mem_header_t *mem_init(void);

#ifdef __cplusplus
}
#endif


extern void cpp_mem_dump(mem_header_t *mem_hdr);
extern void cpp_mem_free(mem_header_t *mem_hdr, void* data,const char *fileName, int line);
extern void *cpp_mem_malloc(mem_header_t *mem_hdr, size_t size,const char *fileName, int line);
extern void *cpp_mem_realloc(mem_header_t *mem_hdr,void *data, size_t size,const char *fileName, int line);
extern mem_header_t *cpp_mem_init(void);

#ifdef __cplusplus

    #ifdef MEM_CTL
        #define MALLOC(a) cpp_mem_malloc(mem_hdr, a, (__FILE__) ,(__LINE__))
        #define FREE(a)   cpp_mem_free(mem_hdr,a, (__FILE__) ,(__LINE__))
        #define MEM_DUMP  cpp_mem_dump(mem_hdr)
        #define REALLOC(a,b) cpp_mem_realloc(mem_hdr,a,b, (__FILE__) ,(__LINE__))


    #elif defined MEM_TRACE

    /* efficient memory debug instrumentation using livetime and oobcheck of libolcutils library */

        #include <olcutils/alloc.h>

        #define MALLOC( size ) cul_malloc( size )
        #define FREE( ptr )  cul_free( ptr )
        #define MALLOC_EXT( type, size ) cul_malloc( size )
        #define REALLOC( ptr, size ) cul_realloc( ptr, size )
        #define MEM_DUMP  while (0){}

    #else
        #define MALLOC(a) malloc(a)
        #define FREE(a) free(a)
        #define REALLOC(a,b) realloc(a,b)
        #define MEM_DUMP  while (0){}
    #endif
#else
    #ifdef MEM_CTL
        #define MALLOC(a) mem_malloc(mem_hdr, a, (__FILE__) ,(__LINE__))
        #define FREE(a)   mem_free(mem_hdr,a,(__FILE__),(__LINE__))
        #define MEM_DUMP  mem_dump(mem_hdr)
        #define REALLOC(a,b) mem_realloc(mem_hdr,a,b,(__FILE__),(__LINE__))
    #else
        #define MALLOC(a) malloc(a)
        #define FREE(a) free(a)
        #define REALLOC(a,b) realloc(a,b)
        #define MEM_DUMP  while (0){}
    #endif

#endif




/* get #of elements in a static array */
#ifndef NELEM
# define NELEM(x) ((int) (sizeof(x) / sizeof((x)[0])))
#endif



/*
 * Count up the number of arguments in "argv".  The count does not include
 * the final NULL entry.
 */
int countArgv(const char* const argv[]);

/*
 * Some utility functions for working with files.  These could be made
 * part of a "File" class.
 */
typedef enum FileType {
    kFileTypeUnknown = 0,
    kFileTypeNonexistent,       // i.e. ENOENT
    kFileTypeRegular,
    kFileTypeDirectory,
    kFileTypeCharDev,
    kFileTypeBlockDev,
    kFileTypeFifo,
    kFileTypeSymlink,
    kFileTypeSocket,
} FileType;
/* get the file's type; follows symlinks */
FileType getFileType(const char* fileName);
/* get the file's modification date; returns -1 w/errno set on failure */
time_t getFileModDate(const char* fileName);

/*
 * Round up to the nearest power of 2.  Handy for hash tables.
 */
unsigned int roundUpPower2(unsigned int val);

void strreverse(char* begin, char* end);
void k_itoa(int value, char* str, int base);
char* itoa(int val, int base);


#endif // _LIBS_UTILS_MISC_H
