/*
    intercom - Event based Interprocess Communication for Dummies
    Copyright 2016 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef EVENTS_H
#define EVENTS_H

#include <pthread.h>
#include <olcutils/clist.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \file events.h
    \brief event handling

    \addtogroup event_api Events
    @{
 */

#define ICOM_EVT_SERVER_DATA  0x1                          /*!< data package from server socket handler */
#define ICOM_EVT_SERVER_CON   0x2                          /*!< server socket handler connected */
#define ICOM_EVT_SERVER_DIS   0x3                          /*!< server socket handler disconnected */
#define ICOM_EVT_CLIENT_DATA  0x4                          /*!< data package from client socket handler */
#define ICOM_EVT_CLIENT_CON   0x5                          /*!< client socket handler connected */
#define ICOM_EVT_CLIENT_DIS   0x6                          /*!< client socket handler disconnected */
#define ICOM_EVT_MAX          0x7                          /*!< highest socket library event id in use */

/*! event type declaration */
typedef struct  {
  t_clist                     node;                         /*!< for circular buffer queueing, must be first */
  int                         type;                         /*!< event type for dereferencing p_source */
  char*                       p_data;                       /*!< payload data */
  int                         max_data_size;                /*!< capacity of p_data */
  int                         data_len;                     /*!< used data length */
  void*                       p_source;                     /*!< reference to event source (server or client) */
  void*                       p_user_ctx;                   /*!< reference to user data context */
  void*                       p_server_state;               /*!< reference to library's global context */
} t_icom_evt;

/*! event callback handler type declaration */
typedef int (*t_evt_cb)( t_icom_evt* p_evt );


/*!
 * Event handler state  object. Two interconnected doubly  linked circular lists
 * are used to store and handle all incorporated events for a given handler, the
 * pool and  the ready list.  New events are unlinked  from the pool  and linked
 * into the  ready list. Consumed  events are unlinked  from the ready  list and
 * linked at to the  pool. This allows to widely decouple  an event producer and
 * the correpsonding consumer. Furthermore the processing sequence can be easily
 * altered to introduce a specific priority scheme.
 */
typedef struct {
  t_clist                     pool;                         /*!< event pool, takes initially all events */
  t_clist                     ready_list;                   /*!< read list where for unprocessed events */
  int                         pool_size;                    /*!< number of maximum cached events */
  pthread_mutex_t             mutex;                        /*!< mutex for access protection */
  pthread_cond_t              signal;                       /*!< signal indicating that a new event is ready */
  t_icom_evt*                 p_evt;                        /*!< used for tempory storare in producer */
  pthread_t                   handler;                      /*!< event handler thread */
  t_evt_cb                    p_evt_cb;                     /*!< callback function to invoke with new events */
} t_icom_events;


/*!
 * destructor for event object
 *
 * The destructor shall be never directly invoked outside form \ref icom_release_event_pool.
 *
 * \param p pointer to event object
 */
void icom_release_evt( t_icom_evt* p );


/*!
 * constructor for new event object
 *
 * The constructor shall be never directly invoked outside form \ref icom_create_event_pool.
 *
 * \param max_data_size size of payload data
 * \return pointer to event object or NULL in case of error
 */
t_icom_evt* icom_create_evt( const int max_data_size );


/*!
 * destructor for event pool
 *
 * \param p pointer to event pool object to be released
 */
void icom_release_event_pool( t_icom_events* p );


/*!
 * constructor for new event pool
 *
 * \param max_evt_data_size maximum size of payload data
 * \param pool_size total number of maintained elements (pool + read list)
 * \return pointer to event pool object or NULL in case of error
 */
t_icom_events* icom_create_event_pool( const int max_evt_data_size, const int pool_size );


/*!
 * cancels events handler and releases all associated data
 *
 * \param p pointer to event pool object to be released
 */
void kill_icom_event_handler( t_icom_events* p );


/*!
 * event handler, refrenced internally only
 *
 * \param pCtx pointer to associated t_icom_events instance
 * \return pointer to events structure (not used)
 */
void* icom_event_handler( void* pCtx );


/*!
 * create new event handler and associated data
 *
 * creates the  actual event  handler thread and  allocates and  initializes all
 * internally used data (event pool, mutex, etc.)
 *
 * \param max_evt_data_size maximum size of payload data
 * \param pool_size total number of maintained elements (pool + read list)
 * \param p_evt_cb callback function to be invoked whenever an event is consumed
 * \return pointer to event pool object or NULL in case of error
 */
t_icom_events* icom_create_event_handler(
  const int max_evt_data_size,
  const int pool_size,
  const t_evt_cb p_evt_cb );

/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef EVENTS_H */
