/*
    intercom - Event based Interprocess Communication for Dummies
    Copyright 2016 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <getopt.h>
#include <assert.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <stdint.h>

#include <olcutils/alloc.h>
#include <intercom/client.h>
#include <intercom/log.h>

#include <utils.h>
#include <config.h>

/*
 * releases unix domain or TCP client socket
 */
static void icom_release_sock_client( void* arg )
{
  t_icom_sock_client* p = (t_icom_sock_client *)arg;
  t_icom_events* p_events = p->p_events;

  assert( p != NULL );

  /* unlink parent reference in order to avoid double freeing (race condition) */
  if( p->p_client_state ) {
    pthread_mutex_lock( & p->p_client_state->mutex );
    p->p_client_state->p_sock_client = NULL;
    pthread_mutex_unlock( & p->p_client_state->mutex );
  }

  if( p_events )
  {
    /* unlink space disconnection event for processing out of pool */
    pthread_mutex_lock( & p_events->mutex );

    if( p->p_evt == NULL ) {
      if ( !clist_is_empty( & p_events->pool ) ) {
        p->p_evt = (t_icom_evt*)clist_remove_head( & p_events->pool );
      }
      else {
        icom_error("event queue overflow error, overwriting existing events\n");
        p->p_evt = (t_icom_evt*)clist_remove_head( & p_events->ready_list );
      }
    }

    p->p_evt->type = ICOM_EVT_CLIENT_DIS;
    p->p_evt->p_source = p;
    p->p_evt->p_user_ctx = p->p_user_ctx;
    memset( p->p_evt->p_data, 0, p->p_evt->max_data_size );
    clist_insert_tail( & p_events->ready_list, & p->p_evt->node);
    p->p_evt = NULL;
    pthread_cond_signal( & p_events->signal );

    pthread_mutex_unlock( & p_events->mutex );
  }

  usleep( 100 ); /* hack to ensure event handler invocation */

  if( p->fd >= 0 ) {
    icom_message("--- close port: %d\n", p->fd );
    close( p->fd );
    p->fd = -1;
  }

  cul_free( p->p_data_chunk );
  icom_release_data_stream( p->p_data_stream );
  cul_free( p );
}


/*
 * creates and binds unix domain or TCP client socket
 */
static t_icom_sock_client*  icom_create_sock_client(
  t_icom_client_state* p_client_state,
  const int sock_family,
  const char* addr,
  const uint16_t port,
  const int bind_retries
  )
{
  t_icom_sock_client* p;
  int bind_retry_cnt = bind_retries;
  const int max_data_size =  get_evt_max_data_size( p_client_state->p_events );

#if defined(SO_NOSIGPIPE)
  int y =1;
#endif

  p = (t_icom_sock_client *) cul_malloc( sizeof(t_icom_sock_client) );
  if( p == NULL ) {
    icom_error( "could not allocate socket client state data error\n");
    return NULL;
  }
  memset( p, 0, sizeof( t_icom_sock_client ) );

  p->p_data_stream = icom_create_data_stream( MAX_MESSAGE_SPLITS, max_data_size,
                                              MESSAGE_DELIM, strlen( MESSAGE_DELIM ) );
  if( p->p_data_stream == NULL ) {
    cul_free( p );
    return NULL;
  }

  p->p_data_chunk = cul_malloc( max_data_size );
  if( p->p_data_chunk == NULL ) {
    icom_release_data_stream( p->p_data_stream );
    cul_free( p );
    return NULL;
  }

  switch( sock_family )
  {
  case PF_UNIX:
    p->sock_family = PF_UNIX;
    p->fd = socket(PF_UNIX, SOCK_STREAM, 0);
    if(p->fd < 0) {
      icom_error( "could not create new unix domain client socket error %d\n", errno);
      icom_release_sock_client( p );
      return( NULL );
    }

    /* start with clean uds_address struture */
    memset( & p->addr.uds, 0, sizeof(struct sockaddr_un));
    p->addr.uds.sun_family = AF_UNIX;
    strncpy( p->addr.uds.sun_path, addr, sizeof( p->addr.uds.sun_path ) );
    p->addr_len = sizeof(struct sockaddr_un);
    break;


  case AF_INET:
    p->fd = socket(AF_INET, SOCK_STREAM, 0);
    if(p->fd < 0) {
      icom_error( "could not create new tcp client socket error %d\n", errno );
      icom_release_sock_client( p );
      return( NULL );
    }

    /* start with clean uds_address struture */
    memset( & p->addr.inet, 0, sizeof(struct sockaddr_in));
    p->addr.inet.sin_family = AF_INET;
    p->addr.inet.sin_addr.s_addr = inet_addr( addr );
    p->addr.inet.sin_port = htons( port );
    p->addr_len = sizeof(struct sockaddr_in);
    break;

  } /* switch( sock_family ) */

#if defined(SO_NOSIGPIPE)
    setsockopt( p->fd, SOL_SOCKET, SO_NOSIGPIPE, &y, sizeof(int));
#endif


  while( connect( p->fd, (struct sockaddr *) & p->addr.uds, p->addr_len ) && bind_retry_cnt > 0 )  {
    switch( sock_family ) {
    case PF_UNIX:
      icom_error( "%s, %d: could not bind unix client socket error %d, try again ... \n", __func__, __LINE__, errno ); break;
    case AF_INET:
      icom_error( "%s, %d: could not bind network client socket error %d, try again ... \n", __func__, __LINE__, errno ); break;
    }
    --bind_retry_cnt;
    usleep( 100000 );
  }

  if( ! bind_retry_cnt ) {
    icom_error( "%s, %d: socket bind error, give up\n", __func__, __LINE__ );
    icom_release_sock_client( p );
    return( NULL );
  }

  p->p_client_state = p_client_state;
  return p;
}

static void* icom_client_handler( void* pCtx )
{
  t_icom_sock_client* p = (t_icom_sock_client *)pCtx;
  t_icom_events* p_events = p->p_events;
  t_icom_data_stream_msg msg;
  const int max_data_size =  get_evt_max_data_size( p_events );
  int received;

  pthread_cleanup_push( icom_release_sock_client, p );

  icom_message("client is connected server %d!\n", p->fd);

  /* fist event is connection event */
  p->p_evt =  create_new_event( p_events );
  p->p_evt->type = ICOM_EVT_CLIENT_CON;
  p->p_evt->p_source = p;
  p->p_evt->p_user_ctx = p->p_user_ctx;
  memset( p->p_evt->p_data, 0, max_data_size );
  send_event( p_events, p->p_evt );

  /* prepare data event container */
  p->p_evt =  create_new_event( p_events );
  p->p_evt->type = ICOM_EVT_CLIENT_DATA;
  p->p_evt->p_source = p;
  p->p_evt->p_user_ctx = p->p_user_ctx;
  msg.data = p->p_evt->p_data;
  msg.len = max_data_size;

  while( 1 )  {

    received = recv( p->fd, p->p_data_chunk, max_data_size, 0 );
    if( received > 0 )
    {
// #define TEST
#ifdef TEST
      icom_message( "received message: %s\n", p->p_evt->p_data );
#endif /* #ifdef TEST */
      icom_push_data_segment( p->p_data_stream, p->p_data_chunk, received );
      while( icom_get_next_data_stream_msg( p->p_data_stream, & msg ) ) {
        /* got complete message, insert newly created event in ready list */
        msg.data[MIN( msg.len, max_data_size-1) ] = '\0';
        p->p_evt->data_len = msg.len;
        send_event( p_events, p->p_evt );

        /* prepare data event container */
        p->p_evt =  create_new_event( p_events );
        p->p_evt->type = ICOM_EVT_CLIENT_DATA;
        p->p_evt->p_source = p;
        p->p_evt->p_user_ctx = p->p_user_ctx;
        msg.data = p->p_evt->p_data;
        msg.len = max_data_size;
      }
    }
    else
    {
      icom_error( "socket stream broke!\n");
      break;
    }
  } /* forever */

  pthread_cleanup_pop( 1 );
  return p;
}

static void icom_kill_sock_client_handler( t_icom_sock_client* p )
{
  icom_message("kill socket client ...\n");
  pthread_cancel( p->handler );
}


static t_icom_sock_client* icom_create_sock_client_handler(
  t_icom_client_state* p_client_state,
  const int sock_family,
  const char* addr,
  const uint16_t port,
  t_icom_events* p_events,
  void* p_user_ctx,
  const int retries
  )
{
  int retcode;

  t_icom_sock_client* p = (t_icom_sock_client *) icom_create_sock_client( p_client_state, sock_family, addr, port, retries );
  if( p ) {
    p->p_events = p_events;
    p->p_user_ctx = p_user_ctx;

    retcode = pthread_create( &p->handler, NULL, icom_client_handler, p );
    if( ! retcode )
      retcode = pthread_detach( p->handler );
    if( retcode ) {
      icom_error( "creation of client handler thread failed with error %d\n", retcode );
      icom_release_sock_client( p );
      p = NULL;
    }
  }

  return p;
}


int icom_client_reply( t_icom_evt* p_evt, const char* data, const int len )
{
  if( send_with_delim( ((struct s_sock_client *)(p_evt->p_source))->fd, data, len, MSG_NOSIGNAL ) == len )
    return 0;
  else
    return -1;
}


int icom_client_request( t_icom_client_state* p, const char* data, const int len )
{
  if( send_with_delim( p->p_sock_client->fd, data, len, MSG_NOSIGNAL ) == len )
    return 0;
  else
    return -1;
}


void icom_kill_client_handler( t_icom_client_state* p )
{
  if( p == NULL )
    return;

  pthread_mutex_lock( & p->mutex );
  if( p->p_sock_client )
    icom_kill_sock_client_handler( p->p_sock_client );
  pthread_mutex_unlock( & p->mutex );

  /* wait for the thread to be completely released */
  while( p->p_sock_client && p->p_sock_client->fd >= 0 )
    usleep( 100000 );

  if( p->p_events_needs_to_released )
  {
    kill_icom_event_handler( p->p_events );
    p->p_events = NULL;
  }

  pthread_mutex_destroy( &p->mutex );
  cul_free( p );
}


t_icom_client_state* icom_create_client_handler_with_event_pool(
  const t_icom_addr_decl* p_addr_decl,
  t_icom_events* p_events,
  void* p_user_ctx,
  const int retries
  )
{
  t_icom_client_state* p = (t_icom_client_state *) cul_malloc( sizeof( t_icom_client_state ) );
  if( p == NULL ) {
    icom_error( "out of memory error occured when creating \n" );
  }

  if( pthread_mutex_init( &p->mutex, 0 ) != 0 ) {
    icom_error( "%s: could initialize mutex error\n", __func__ );
    cul_free( p );
    return NULL;
  }

  p->p_events = p_events;
  p->p_events_needs_to_released = false;

  p->p_sock_client = icom_create_sock_client_handler(
    p,
    p_addr_decl->sock_family,
    p_addr_decl->address,
    p_addr_decl->port,
    p_events,
    p_user_ctx,
    retries );

  if( p->p_sock_client == NULL ) {
    icom_error( "error while creating client socket handler!\n" );
    cul_free( p );
    return NULL;
  }

  return p;
}


t_icom_client_state* icom_create_client_handler(
  const t_icom_addr_decl* p_addr_decl,
  const int max_evt_data_size,
  const int pool_size,
  const t_evt_cb p_evt_cb,
  void* p_user_ctx,
  const int retries
  )
{
  t_icom_client_state* p;
  t_icom_events* p_events;

  p_events = icom_create_event_handler( max_evt_data_size, pool_size, p_evt_cb );
  if( p_events == NULL ) {
    icom_error( "out of memory error when allocating event pool!\n" );
    return NULL;
  }

  p = icom_create_client_handler_with_event_pool( p_addr_decl, p_events, p_user_ctx, retries );
  if( p == NULL ) {
    icom_error( "creation of client handlers failed error!\n" );
    icom_release_event_pool( p_events );
    return NULL;
  }

  p->p_events_needs_to_released = true;

  return p;
}



int icom_client_handler_request( t_icom_client_conn_handler *p, const char* data, const int len )
{
  int retcode = 0;

  pthread_mutex_lock( & p->mutex );
  if( p->connection_state == ICOM_CLIENT_CON_CONNECTED ) {
    retcode = icom_client_request( p->p_client_state, data, len );
  } else  {
    icom_error( "%s: send error because not connected to server!\n", __func__ );
  }
  pthread_mutex_unlock( & p->mutex );

  return retcode;
}



static void icom_release_client_conn_handler( void* arg )
{
  t_icom_client_conn_handler* p = (t_icom_client_conn_handler *)arg;

  assert( p != NULL );

  icom_kill_client_handler( p->p_client_state );

  pthread_mutex_destroy( &p->mutex );
  pthread_cond_destroy( & p->signal );
  cul_free( p );
}


static int icom_client_evt_proxy( t_icom_evt* p_evt )
{
  int retcode = 0;
  t_icom_client_conn_handler* p = (t_icom_client_conn_handler *) ( p_evt->p_user_ctx );

  if( p->connection_state == ICOM_CLIENT_CON_KILLED ) {
    return retcode;
  }

  pthread_mutex_lock( & p->mutex );
  switch( p_evt->type ) {
  case ICOM_EVT_CLIENT_CON:
  case ICOM_EVT_CLIENT_DATA:
    p->connection_state = ICOM_CLIENT_CON_CONNECTED;
    break;
  case ICOM_EVT_CLIENT_DIS:
    p->connection_state = ICOM_CLIENT_CON_DISCONNECTED;
    break;
  }

  if( p->connection_state == ICOM_CLIENT_CON_DISCONNECTED )  {
    icom_message("--- disconnected from server, try to reconnect ...!\n" );
    pthread_cond_signal( & p->signal );
  }
  pthread_mutex_unlock( & p->mutex );

  /* proxy exclusively data events */
  if( p_evt->type == ICOM_EVT_CLIENT_DATA ) {
    p_evt->p_user_ctx = p->p_user_ctx;
    retcode =  p->p_evt_cb( p_evt );
  }

  return retcode;
}

static void* icom_client_connection_handler( void* pCtx )
{
  t_icom_client_conn_handler* p = (t_icom_client_conn_handler *)pCtx;

  assert( p != NULL );

  pthread_cleanup_push( icom_release_client_conn_handler, p );

  while( 1 ) {
    pthread_mutex_lock( & p->mutex );
    p->p_client_state = icom_create_client_handler_with_event_pool( & p->addr_decl, p->p_events, p, 10 );
    pthread_mutex_unlock( & p->mutex );

    if( p->p_client_state == NULL ) {
      icom_error("%s: could not create connection, try again ...\n", __func__ );
      sleep( 1 );
    } else {
      // conditional wait
      pthread_mutex_lock( & p->mutex );
      do {
        pthread_cond_wait( & p->signal, & p->mutex );
      } while( p->connection_state != ICOM_CLIENT_CON_DISCONNECTED ); /* spurios wakeups */
      pthread_mutex_unlock( & p->mutex );

      pthread_mutex_lock( & p->mutex );
      icom_kill_client_handler( p->p_client_state );
      pthread_mutex_unlock( & p->mutex );
    }
  }

  pthread_cleanup_pop( 1 );
  return p;
}

void icom_kill_client_connection_handler( t_icom_client_conn_handler* p )
{
  if( p == NULL )
    return;

  icom_message("kill client connection handler ...\n");

  pthread_mutex_lock( & p->mutex );
  p->connection_state = ICOM_CLIENT_CON_KILLED;
  pthread_cancel( p->handler );
  pthread_mutex_unlock( & p->mutex );

  sleep( 1 );

  kill_icom_event_handler( p->p_events );
}

t_icom_client_conn_handler* icom_create_client_connection_handler(
  const t_icom_addr_decl* p_addr_decl,
  const int max_evt_data_size,
  const int pool_size,
  const t_evt_cb p_evt_cb,
  void* p_user_ctx
  )
{
  int retcode;
  t_icom_client_conn_handler* p = (t_icom_client_conn_handler *) cul_malloc(
    sizeof( t_icom_client_conn_handler ) );
  if( p == NULL ) {
    icom_error( "out of memory error occured when creating connection handler\n" );
    return NULL;
  }

  memset( p, 0, sizeof(t_icom_client_conn_handler) );

  if( pthread_mutex_init( &p->mutex, 0 ) != 0 ) {
    icom_error( "could initialize connection list mutex error\n");
    cul_free( p );
    return NULL;
  }

  if( pthread_cond_init( & p->signal, 0 ) != 0 ) {
    icom_error( "could not initialize client connection handler conditional error!\n" );
    pthread_mutex_destroy( &p->mutex );
    cul_free( p );
    return NULL;
  }

  memcpy( & p->addr_decl, p_addr_decl, sizeof( t_icom_addr_decl ) );
  p->p_evt_cb = p_evt_cb;
  p->p_user_ctx = p_user_ctx;

  p->p_events = icom_create_event_handler( max_evt_data_size, pool_size, icom_client_evt_proxy );
  if( p->p_events == NULL ) {
    icom_error( "out of memory error when allocating event pool!\n" );
    pthread_cond_destroy( & p->signal );
    pthread_mutex_destroy( &p->mutex );
    cul_free( p );
    return NULL;
  }

  retcode = pthread_create( &p->handler, NULL, icom_client_connection_handler, p );
  if( ! retcode )
    retcode = pthread_detach( p->handler );
  if( retcode ) {
    icom_error( "creation of client handler thread failed with error %d\n", retcode );
    pthread_mutex_destroy( &p->mutex );
    cul_free( p );
    p = NULL;
  }

  return p;
}
