/*
    intercom - Event based Interprocess Communication for Dummies
    Copyright 2016 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <getopt.h>
#include <assert.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <stdint.h>

#include <olcutils/alloc.h>
#include <olcutils/refstring.h>
#include <intercom/server.h>
#include <intercom/log.h>

#include <utils.h>
#include <config.h>

static void icom_release_connection_list( t_icom_connections* p )
{
  if( p ) {
    pthread_mutex_lock( & p->mutex );
    slist_free( p->p_lst_listeners );
    pthread_mutex_unlock( & p->mutex );

    pthread_mutex_destroy( &p->mutex );
    cul_free( p );
  }
}

static t_icom_connections* icom_create_connection_list( void )
{
  t_icom_connections* p = (t_icom_connections *) cul_malloc( sizeof( t_icom_connections ) );
  if( p ) {
    if( pthread_mutex_init( &p->mutex, 0 ) != 0 ) {
      icom_error( "could initialize connection list mutex error\n");
      cul_free( p );
      return NULL;
    }

    p->p_lst_listeners = slist_alloc();
    if( p->p_lst_listeners == NULL ) {
      icom_error( "could initialize connection list error\n");
      pthread_mutex_destroy( &p->mutex );
      cul_free( p );
      return NULL;
    }
  }
  return p;
}



/**
 * releases socket listener state data
 */
static void icom_release_sock_listener( t_icom_sock_listener* p )
{
  if( ! p )
    return;

  if( p->p_connections ) {
    slist_t* l;

    pthread_mutex_lock( & p->p_connections->mutex );
    if( p->p_connections ) {
      l = p->p_connections->p_lst_listeners;
      if( l ) {
        p->p_connections->p_lst_listeners = clone_slist_without_elem( l, p );
        slist_free( l );
      }
    }
    pthread_mutex_unlock( & p->p_connections->mutex );
  }

  if( p->fd >= 0 )
    close( p->fd );
  pthread_mutex_destroy( &p->mutex );
  slist_free( p->server_list );
  cul_free( p );
}



/**
 * creates and binds unix domain or TCP socket
 */
static t_icom_sock_listener*  icom_create_sock_listener(
  const int sock_family,
  const char* addr,
  const uint16_t port,
  t_icom_events* p_events,
  t_icom_connections* p_connections )
{
  t_icom_sock_listener* p;
  int y;
  int bind_retry_cnt = 30;

  p = (t_icom_sock_listener *) cul_malloc( sizeof(t_icom_sock_listener) );
  if( p == NULL ) {
    icom_error( "could not allocate socket listener state data error\n");
    return NULL;
  }

  p->p_events = p_events;
  p->p_connections = p_connections;

  p->server_list = slist_alloc();
  if( p->server_list == NULL ) {
    icom_error( "could not allocate server list data error\n");
    cul_free( p );
    return NULL;
  }

  if( pthread_mutex_init( &p->mutex, 0 ) != 0 ) {
    icom_error( "could initialize socket listener mutex error\n");
    slist_free( p->server_list );
    cul_free( p );
    return NULL;
  }

  switch( sock_family )
  {
  case PF_UNIX:
    p->sock_family = PF_UNIX;
    p->fd = socket(PF_UNIX, SOCK_STREAM, 0);
    if(p->fd < 0) {
      icom_error( "could not create new unix domain socket error\n");
      icom_release_sock_listener( p );
      return( NULL );
    }

    /* start with clean uds_address struture */
    memset( & p->addr.uds, 0, sizeof(struct sockaddr_un));
    p->addr.uds.sun_family = AF_UNIX;
    cstr_strlcpy( p->addr.uds.sun_path, addr, sizeof( p->addr.uds.sun_path ) );
    p->addr_len = sizeof(struct sockaddr_un);
    unlink( addr );
    break;


  case AF_INET:
    p->fd = socket(AF_INET, SOCK_STREAM, 0);
    if(p->fd < 0) {
      icom_error( "could not create new tcp socket error\n");
      icom_release_sock_listener( p );
      return( NULL );
    }

    /* start with clean uds_address struture */
    memset( & p->addr.inet, 0, sizeof(struct sockaddr_in));
    p->addr.inet.sin_family = AF_INET;
    p->addr.inet.sin_addr.s_addr = inet_addr( addr );
    p->addr.inet.sin_port = htons( port );
    p->addr_len = sizeof(struct sockaddr_in);
    y = 1;
    setsockopt( p->fd, SOL_SOCKET, SO_REUSEADDR, &y, sizeof(int));
    break;
  } /* switch( sock_family ) */


  while( bind( p->fd, (struct sockaddr *) & p->addr.uds, p->addr_len ) && bind_retry_cnt > 0 )  {
    switch( sock_family ) {
    case PF_UNIX:
      icom_error( "could not bind unix socket error %d, try again ... \n", errno ); break;
    case AF_INET:
      icom_error( "could not bind network socket error %d, try again ... \n", errno ); break;
    }
    --bind_retry_cnt;
    sleep( 3 );
  }

  if( ! bind_retry_cnt ) {
    icom_error( "socket bind error, give up\n");
    icom_release_sock_listener( p );
    return( NULL );
  }

  if( listen( p->fd, SOCK_LISTEN_BACKLOG ) )  {
    icom_error( "error %d while listening to control socket!\n", errno );
    icom_release_sock_listener( p );
    return( NULL );
  }


  if( p_connections ) {
    pthread_mutex_lock( & p_connections->mutex );
    p_connections->p_lst_listeners = slist_prepend( p_connections->p_lst_listeners, p );
    pthread_mutex_unlock( & p_connections->mutex );
  }

  return p;
}


static void icom_release_sock_server( void* arg )
{
  t_icom_sock_server* p = (t_icom_sock_server *)arg;
  t_icom_sock_listener* l = p->listener;
  t_icom_server_state* p_server_state = (t_icom_server_state * ) ( l->p_server_state );
  t_icom_events* p_events = l->p_events;
  slist_t* new_server_list;
  int all_events_processed;

  assert( p != NULL );

  /* poll event ready list until its has been fully emptied by event handler */
  while( true ) {
    pthread_mutex_lock( & p_events->mutex );
    all_events_processed = clist_is_empty( & p_events->ready_list );
    pthread_mutex_unlock( & p_events->mutex );
    if( all_events_processed )
      break;
    usleep( 100000 );
  }

  /* unlink space disconnection event for processing out of pool */
  pthread_mutex_lock( & p_events->mutex );

  if( p->p_evt == NULL ) {
    if ( !clist_is_empty( & p_events->pool ) ) {
      p->p_evt = (t_icom_evt*)clist_remove_head( & p_events->pool );
    }
    else {
      icom_error("event queue overflow error, overwriting existing events\n");
      p->p_evt = (t_icom_evt*)clist_remove_head( & p_events->ready_list );
    }
  }

  p->p_evt->type = ICOM_EVT_SERVER_DIS;
  p->p_evt->p_source = p;
  p->p_evt->p_user_ctx = l->p_user_ctx;
  memset( p->p_evt->p_data, 0, p->p_evt->max_data_size );
  clist_insert_tail( & p_events->ready_list, & p->p_evt->node);
  p->p_evt = NULL;
  pthread_cond_signal( & p_events->signal );

  pthread_mutex_unlock( & p_events->mutex );


  pthread_mutex_lock( & p_server_state->mutex );

  new_server_list =  clone_slist_without_elem( l->server_list, p );
  if( new_server_list ) {
    slist_free( l->server_list );
    l->server_list =  new_server_list;
  }
  else {
    icom_error( "out of memory occur during clean up resulting in memory leak!\n" );
  }

  if( p->fd >= 0 ) {
    icom_message("--- close port: %d\n", p->fd );
    close( p->fd );
    p->fd = -1;
  }

  cul_free( p->p_data_chunk );
  icom_release_data_stream( p->p_data_stream );
  cul_free( p );

  pthread_mutex_unlock( & p_server_state->mutex );
}

static void* icom_server_sock_handler( void* pCtx )
{
  t_icom_sock_server* p = (t_icom_sock_server *)pCtx;
  t_icom_sock_listener* l = p->listener;
  t_icom_events* p_events = l->p_events;
  t_icom_data_stream_msg msg;
  const int max_data_size = get_evt_max_data_size( l->p_events );;
  int received;

  pthread_cleanup_push( icom_release_sock_server, p );

  icom_message("client is connected to control socket %d!\n", p->fd);

  /* fist event is connection event */
  p->p_evt =  create_new_event( p_events );
  p->p_evt->type = ICOM_EVT_SERVER_CON;
  p->p_evt->p_source = p;
  p->p_evt->p_user_ctx = l->p_user_ctx;
  memset( p->p_evt->p_data, 0, max_data_size );
  send_event( p_events, p->p_evt );

  /* prepare data event container */
  p->p_evt =  create_new_event( p_events );
  p->p_evt->type = ICOM_EVT_SERVER_DATA;
  p->p_evt->p_source = p;
  p->p_evt->p_user_ctx = l->p_user_ctx;
  msg.data = p->p_evt->p_data;
  msg.len = max_data_size;

  while ( 1 )  {

    received = recv( p->fd, p->p_data_chunk, max_data_size, 0 );
    if( received > 0 )
    {
// #define TEST
#ifdef TEST
      icom_message( "received message: %10s\n", p->p_data_chunk );
#endif /* #ifdef TEST */

      icom_push_data_segment( p->p_data_stream, p->p_data_chunk, received );
      while( icom_get_next_data_stream_msg( p->p_data_stream, & msg ) ) {

        /* do not process empty messages */
        if( msg.len == 0 )
          continue;

        /* got complete message, insert newly created event in ready list */
        msg.data[MIN( msg.len, max_data_size-1) ] = '\0';
        p->p_evt->data_len = msg.len;
        send_event( p_events, p->p_evt );

        /* prepare data event container */
        p->p_evt =  create_new_event( p_events );
        p->p_evt->type = ICOM_EVT_SERVER_DATA;
        p->p_evt->p_source = p;
        p->p_evt->p_user_ctx = l->p_user_ctx;
        msg.data = p->p_evt->p_data;
        msg.len = max_data_size;
      }
    }
    else
    {
      icom_error( "socket stream broke!\n");
      break;
    }
  } /* forever */

  pthread_cleanup_pop( 1 );
  return p;
}

static void icom_kill_sock_server( t_icom_sock_server* p )
{
  icom_message("kill socket server ...\n");
  pthread_cancel( p->handler );
}


static t_icom_sock_server* icom_create_sock_server( t_icom_sock_listener* listener, int fd )
{
  int retcode;
  t_icom_sock_server* p = (t_icom_sock_server *) cul_malloc( sizeof(t_icom_sock_server) );
  if( p ) {
    const int max_data_size =  get_evt_max_data_size( listener->p_events );

    p->listener = listener;
    p->fd = fd;
    p->p_data_stream = icom_create_data_stream( MAX_MESSAGE_SPLITS, max_data_size,
                                                MESSAGE_DELIM, strlen( MESSAGE_DELIM ) );
    if( p->p_data_stream == NULL ) {
      cul_free( p );
      return NULL;
    }

    p->p_data_chunk = cul_malloc( max_data_size );
    if( p->p_data_chunk == NULL ) {
      icom_release_data_stream( p->p_data_stream );
      cul_free( p );
      return NULL;
    }

    retcode = pthread_create( &p->handler, NULL, icom_server_sock_handler, p );
    if( ! retcode )
      retcode = pthread_detach( p->handler );
    if( retcode ) {
      icom_error( "creation of server handler thread failed with error %d\n", retcode );
      icom_release_sock_server( p );
      p = NULL;
    }
  }

  return p;
}


static void* icom_accept_handler( void* pCtx )
{
  t_icom_sock_listener* p = (t_icom_sock_listener*)pCtx;

  /* incoming event loop */
  while( 1 ) {
    int connect_fd;

    icom_message("start listening to control port ...\n");
    connect_fd = accept( p->fd,  (struct sockaddr *) & p->addr, & p->addr_len );
    if( connect_fd > 0 )
    {
      if( slist_cnt( p->server_list ) < p->max_connections ) {
        t_icom_sock_server* server = icom_create_sock_server( p, connect_fd );
        if( server == NULL ) {
          icom_error("out of memory when creating socket server error\n");
        } else {
          icom_message("start socket handler ...\n");

          pthread_mutex_lock( & p->mutex );
          p->server_list = slist_prepend( p->server_list, server );
          pthread_mutex_unlock( & p->mutex );
        }
      }
      else {
        icom_error("error %d, (no more connections allowed?), close request ...\n", errno);
        close( connect_fd );
        usleep( 500000 );
      }
    }
  } /* while( 1 ) */

  return p;
}



static void* l_kill_server(void *pCtx)
{
  t_icom_sock_server* p = pCtx;
  icom_kill_sock_server( p );
  return NULL;
}


static void icom_kill_accept_handler( t_icom_sock_listener* p )
{
  int list_not_empty = 1;
  int max_loop_iterations = 100;

  if( p ) {

    /* kill accept handler thread */
    pthread_mutex_lock( & p->mutex );
    pthread_cancel( p->handler );
    pthread_mutex_unlock( & p->mutex );

    /* kill all active servers */
    while( list_not_empty  &&  max_loop_iterations-- > 0 ) {
      if( ! slist_empty( p->server_list ) ) {
        l_kill_server( slist_first( p->server_list ) );
      } else {
        list_not_empty = 0;
      }
    }

    if( max_loop_iterations == 0 )
    {
      icom_error("%s: kill_accept_handler hung, forced quit!\n", __func__ );
    }

    /* release socket listener object */
    icom_release_sock_listener( p );
  }
}


static t_icom_sock_listener* icom_create_accept_handler(
  const int sock_family,
  const char* addr,
  const uint16_t port,
  const int max_connections,
  t_icom_events* p_events,
  t_icom_connections* p_connections,
  void* p_user_ctx,
  t_icom_server_state* p_server_state )
{
  t_icom_sock_listener* p;
  int retcode;

  p = icom_create_sock_listener( sock_family, addr, port, p_events, p_connections );
  if( p == NULL ) {
    icom_error( "could not create socket listener error\n" );
    return NULL;
  }
  p->p_server_state = (t_icom_server_state *) p_server_state;

  p->max_connections = max_connections;
  p->p_user_ctx = p_user_ctx;
  retcode = pthread_create( &p->handler, NULL, icom_accept_handler, p );
  if( ! retcode )
    retcode = pthread_detach( p->handler );
  if( retcode ) {
    icom_error( "creation of accept handler thread failed with error %d\n", retcode );
    icom_release_sock_listener( p );
    p = NULL;
  }

  return p;
}


int icom_reply_to_sender( t_icom_evt* p_evt, const char* data, const int len )
{
  if( send_with_delim( ((struct s_sock_server *)(p_evt->p_source))->fd, data, len, MSG_NOSIGNAL ) == len )
    return 0;
  else
    return -1;
}


int icom_reply_to_address( t_icom_evt* p_evt, const char* data, const int len )
{
  t_icom_sock_listener* p_sock_listener = ((struct s_sock_server *)(p_evt->p_source))->listener;
  int errors = 0;

  pthread_mutex_lock( & p_sock_listener->mutex );
  slist_t* l  = p_sock_listener->server_list;
  while( l && l->data ) {
    t_icom_sock_server* s = (t_icom_sock_server *) ( l->data );
    if( send_with_delim( s->fd, data, len, MSG_NOSIGNAL ) != len )
      --errors;
    l = l->next;
  }
  pthread_mutex_unlock( & p_sock_listener->mutex );

  return errors;
}


int icom_broadcast_to_all( t_icom_server_state* p_server_state, const char* data, const int len )
{
  t_icom_connections* p_connections;
  slist_t* server_list;
  int errors = 0;

  pthread_mutex_lock( & p_server_state->mutex );

  p_connections = p_server_state->p_connections;
  if( p_connections ) {
    pthread_mutex_lock( & p_connections->mutex );
    slist_t* cl  = p_connections->p_lst_listeners;
    while( cl && cl->data ) {
      t_icom_sock_listener* sl = (t_icom_sock_listener *) ( cl->data );
      pthread_mutex_lock( & sl->mutex );
      server_list = sl->server_list;
      while( server_list && server_list->data ) {
        t_icom_sock_server* s = (t_icom_sock_server *) ( server_list->data );
        if( send_with_delim( s->fd, data, len, MSG_NOSIGNAL ) != len )
          --errors;
        server_list = server_list->next;
      }
      pthread_mutex_unlock( & sl->mutex );
      cl = cl->next;
    }
    pthread_mutex_unlock( & p_connections->mutex );
  }

  pthread_mutex_unlock( & p_server_state->mutex );

  return errors;
}


int icom_reply_to_all_connections( t_icom_evt* p_evt, const char* data, const int len )
{
  t_icom_server_state* p_server_state = (t_icom_server_state *)( p_evt->p_server_state );
  return icom_broadcast_to_all( p_server_state, data, len );
}


void icom_kill_server_handlers( t_icom_server_state* p )
{
  slist_t* p_lst_listeners;
  slist_t* l;

  if( p == NULL )
    return;

  pthread_mutex_lock( & p->p_connections->mutex );
  p_lst_listeners = p->p_connections->p_lst_listeners;
  // enforce explicit releasing of accept handler list
  p->p_connections->p_lst_listeners = NULL;
  pthread_mutex_unlock( & p->p_connections->mutex );

  l = p_lst_listeners;
  while( l && l->data ) {
    icom_kill_accept_handler( l->data );
    l = l->next;
  }

  slist_free( p_lst_listeners );
  icom_release_connection_list( p->p_connections );

  if( p->p_events_needs_to_released )
    kill_icom_event_handler( p->p_events );

  pthread_mutex_destroy( &p->mutex );
  cul_free( p );
}


t_icom_server_state* icom_create_server_handlers_with_event_pool(
  const t_icom_server_decl decl_table[],
  const int decl_table_len,
  t_icom_events* p_events,
  void* p_user_ctx
  )
{
  t_icom_server_state* p = (t_icom_server_state *) cul_malloc( sizeof( t_icom_server_state ) );
  t_icom_evt* p_evt_iter;
  int i;

  if( p == NULL ) {
    icom_error( "out of memory error when allocating server state!\n" );
    return NULL;
  }

  p->p_events = p_events;
  p->p_events_needs_to_released = false;

  /* inject back reference to main server context. No event must be consumed yet! */
  pthread_mutex_lock ( & p_events->mutex );
  for (i = 0;  i < p_events->pool_size; i++ ) {
    p_evt_iter = (t_icom_evt *)clist_remove_head( & p_events->pool );
    p_evt_iter->p_server_state = p;
    clist_insert_tail( & p_events->pool, (t_clist *) p_evt_iter );
  }
  pthread_mutex_unlock ( & p_events->mutex );

  if( pthread_mutex_init( &p->mutex, 0 ) != 0 ) {
    icom_error( "%s,%d: could initialize socket listener mutex error\n", __func__, __LINE__ );
    cul_free( p );
    return NULL;
  }

  p->p_connections = icom_create_connection_list();
  if( p->p_connections == NULL ) {
    icom_error( "could not create connection list error\n" );
    cul_free( p );
    return NULL;
  }

  for( i=0; i < decl_table_len; ++i ) {
    t_icom_sock_listener* p_sock_lst;

    p_sock_lst = icom_create_accept_handler( decl_table[i].addr.sock_family,
                                             decl_table[i].addr.address,
                                             decl_table[i].addr.port,
                                             decl_table[i].max_connections,
                                             p->p_events,
                                             p->p_connections,
                                             p_user_ctx,
                                             p );
    if( p_sock_lst == NULL ) {
      switch( decl_table[i].addr.sock_family ) {
      case AF_INET:
        icom_error( "could not create network socket handler for port %d\n", decl_table[i].addr.port );
        break;
      case PF_UNIX:
        icom_error( "could not create domain socket handler for file %s\n", decl_table[i].addr.address );
        break;
      }
    }
  }

  return p;
}


t_icom_server_state* icom_create_server_handlers(
  const t_icom_server_decl decl_table[],
  const int decl_table_len,
  const int max_evt_data_size,
  const int pool_size,
  const t_evt_cb p_evt_cb,
  void* p_user_ctx
  )
{
  t_icom_server_state* p;
  t_icom_events* p_events;
  int retcode;

  p_events = icom_create_event_pool( max_evt_data_size, pool_size );
  if( p_events == NULL ) {
    icom_error( "out of memory error when allocating event pool!\n" );
    return NULL;
  }

  p = icom_create_server_handlers_with_event_pool( decl_table, decl_table_len, p_events, p_user_ctx );
  if( p == NULL ) {
    icom_error( "creation of server handlers failed error!\n" );
    icom_release_event_pool( p_events );
    return NULL;
  }

  /* start event handler */
  p_events->p_evt_cb = p_evt_cb;
  retcode = pthread_create( &p_events->handler, NULL, icom_event_handler, p_events );
  if( ! retcode ) {
    retcode = pthread_detach( p_events->handler );
  }

  if( retcode ) {
    icom_error( "creation of event handler thread failed with error %d\n", retcode );
    icom_release_event_pool( p_events );
    return NULL;
  }

  p->p_events_needs_to_released = true;

  return p;
}
