/*
    intercom - Event based Interprocess Communication for Dummies
    Copyright 2016 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <intercom/data_stream_splitter.h>
#include <olcutils/alloc.h>
#include <olcutils/memtrace.h>
#include <intercom/log.h>
#include <intercom/revision.h>


#define ICOM_TST_MAX_STREAM_SEGMENTS        5
#define ICOM_TST_MAX_STEAM_DATA_SIZE        10
#define ICOM_TST_STREAM_DELIM               "\r\n"

void icom_print_msg( t_icom_data_stream_msg *p )
{
  int i, c = 1;
  printf("reveived message:\n\t");

  for( i=0; i < p->len; ++i, ++c ) {
    printf( "%u, ", (unsigned char) (p->data[i]) );
    if( !(c++ % 10 ) )
      printf("\n\t");
  }
  printf("\n");
}


int icom_test_unfragmented( t_icom_data_stream* p )
{
  const int msg_data_len = ICOM_TST_MAX_STEAM_DATA_SIZE;
  const int delim_len = strlen(ICOM_TST_STREAM_DELIM);
  char msg_data[msg_data_len];
  t_icom_data_stream_msg msg = { msg_data, msg_data_len };

  char data1[] = { 0, 1, 2, '\r', '\n' };
  char data2[] = { 3, 4, 5, '\r', '\n' };

  int t;

  for( t = 0; t < 30; ++t )
  {
    printf("%s, %d: Running test nr %d ...\n", __func__, __LINE__, t );
    printf("------------------------------\n");
    printf( "pushing 0, 1, 2 + delim ...\n" );
    icom_push_data_segment( p, data1, sizeof( data1 ) );
    // icom_print_stream_stats( p );

    printf( "read message ...\n" );
    assert( icom_get_next_data_stream_msg( p, & msg ) );
    icom_print_msg( & msg );
    assert( memcmp( msg.data, data1, sizeof( data1 ) - delim_len ) == 0 );

    icom_push_data_segment( p, data2, sizeof( data2 ) );

    assert( icom_get_next_data_stream_msg( p, & msg ) );
    icom_print_msg( & msg );
    assert( memcmp( msg.data, data2, sizeof( data2 ) - delim_len ) == 0 );
    printf("\n\n");
  }

  return 0;
}


int icom_test_fragmented_by_two( t_icom_data_stream* p )
{
  const int msg_data_len = ICOM_TST_MAX_STEAM_DATA_SIZE;
  char msg_data[msg_data_len];
  t_icom_data_stream_msg msg = { msg_data, msg_data_len };

  char data1[] = { 0, 1, 2, 3, 4, 5, 6 };
  char data2[] = { 7, 8, 9, '\r', '\n' };
  char check[] = { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 };

  int t;

  for( t = 0; t < 30; ++t )
  {
    printf("%s, %d: Running test nr %d ...\n", __func__, __LINE__, t );
    printf("------------------------------\n");
    printf( "pushing 0, 1, 2 + delim ...\n" );
    icom_push_data_segment( p, data1, sizeof( data1 ) );
    // icom_print_stream_stats( p );

    printf( "read message, first try...\n" );
    assert( ! icom_get_next_data_stream_msg( p, & msg ) );

    icom_push_data_segment( p, data2, sizeof( data2 ) );

    assert( icom_data_stream_bytes( p ) == sizeof( data1) + sizeof( data2) );

    printf( "read message, second try...\n" );
    assert( icom_get_next_data_stream_msg( p, & msg ) );
    icom_print_msg( & msg );
    assert( memcmp( msg.data, check, sizeof( check ) ) == 0 );
    printf("\n\n");
  }

  return 0;
}

int icom_test_fragmented_by_any( t_icom_data_stream* p )
{
  const int msg_data_len = ICOM_TST_MAX_STEAM_DATA_SIZE;
  char msg_data[msg_data_len];
  t_icom_data_stream_msg msg = { msg_data, msg_data_len };

  char data[ICOM_TST_MAX_STEAM_DATA_SIZE];
  int  size;
  int  next_byte_to_push = 0;
  int  next_byte_to_pull = 0;

  int i, t;

  for( t = 0; t < 100; ++t )
  {
    printf("%s, %d: Running test nr %d ...\n", __func__, __LINE__, t );
    printf("------------------------------\n");

    size= (rand() % (ICOM_TST_MAX_STEAM_DATA_SIZE-1)) + 1;
    for( i=0; i<size; ++i ) {

      /* omit LF */
      if( next_byte_to_push == 10 )
        ++next_byte_to_push;

      data[i] = next_byte_to_push++;
      if( next_byte_to_push > 255 )
        next_byte_to_push = 0;
    }

    printf( "pushing %d bytes ...\n", size );
    icom_push_data_segment( p, data, size );
    // icom_print_stream_stats( p );

    printf( "read message, first try...\n" );
    assert( ! icom_get_next_data_stream_msg( p, & msg ) );

    icom_push_data_segment( p, ICOM_TST_STREAM_DELIM, strlen(ICOM_TST_STREAM_DELIM) );

    printf( "read message, second try...\n" );
    assert( icom_get_next_data_stream_msg( p, & msg ) );
    icom_print_msg( & msg );

    for( i=0; i < msg.len; ++i ) {

      /* omit LF */
      if( next_byte_to_pull == 10 )
        ++next_byte_to_pull;

      assert( (unsigned char)(msg.data[i]) == next_byte_to_pull++ );
      if( next_byte_to_pull > 255 )
        next_byte_to_pull = 0;
    }
    printf("\n\n");
  }

  return 0;
}

int icom_test_multiple_pushes( t_icom_data_stream* p )
{
  const int msg_data_len = ICOM_TST_MAX_STEAM_DATA_SIZE;
  char msg_data[msg_data_len];
  t_icom_data_stream_msg msg = { msg_data, msg_data_len };

  char data[ICOM_TST_MAX_STEAM_DATA_SIZE];
  int  size;
  int  next_byte_to_push = 0;
  int  next_byte_to_pull = 0;
  int  prev_byte_to_push;

  int i, t, k, pushes;

  for( t = 0; t < 250; ++t )
  {
    printf("%s, %d: Running test nr %d ...\n", __func__, __LINE__, t );
    printf("------------------------------\n");

    pushes = (rand() % (ICOM_TST_MAX_STREAM_SEGMENTS-1)) + 1;
    printf("%s, %d: try to push %d segments\n", __func__, __LINE__, pushes );

    for( k=0; k < pushes; ++k )
    {
      prev_byte_to_push = next_byte_to_push;
      size = (rand() % (ICOM_TST_MAX_STEAM_DATA_SIZE-5)) + 1;
      for( i=0; i<size; ++i ) {

        /* omit line feed */
        if( next_byte_to_push == 10 )
          ++next_byte_to_push;

        data[i] = next_byte_to_push++;
        if( next_byte_to_push > 255 )
          next_byte_to_push = 0;
      }

      if( ICOM_TST_MAX_STEAM_DATA_SIZE - icom_data_stream_bytes( p ) > size + strlen(ICOM_TST_STREAM_DELIM) ) {
        printf( "pushing %d bytes ...\n", size );
        icom_push_data_segment( p, data, size );
      } else {
        printf( "%s, %d: capacity limit reached, push less segmets in this test.\n",
                __func__, __LINE__ );

        next_byte_to_push = prev_byte_to_push;
        break;
      }
    }

    icom_push_data_segment( p, ICOM_TST_STREAM_DELIM, strlen(ICOM_TST_STREAM_DELIM) );
    icom_print_stream_stats( p );

    printf( "read message, second try...\n" );
    assert( icom_get_next_data_stream_msg( p, & msg ) );
    icom_print_msg( & msg );

    for( i=0; i < msg.len; ++i ) {

      /* omit line feed */
      if( next_byte_to_pull == 10 )
        ++next_byte_to_pull;

      assert( (unsigned char)(msg.data[i]) == next_byte_to_pull++ );
      if( next_byte_to_pull > 255 )
        next_byte_to_pull = 0;
    }
    printf("\n\n");
  }

  return 0;
}


int test_data_stream_splitter()
{
  t_icom_data_stream* p_data_stream;

  printf("allocat ...\n");
  p_data_stream = icom_create_data_stream( ICOM_TST_MAX_STREAM_SEGMENTS,
                                           ICOM_TST_MAX_STEAM_DATA_SIZE,
                                           ICOM_TST_STREAM_DELIM,
                                           strlen(ICOM_TST_STREAM_DELIM) );
  if( p_data_stream == NULL ) {
    printf("%s,%d: data stream context could not be initialized error!\n", __func__, __LINE__ );
    return -1;
  }

  assert( icom_test_unfragmented( p_data_stream ) == 0 );
  assert( icom_test_fragmented_by_two( p_data_stream ) == 0 );
  assert( icom_test_fragmented_by_any( p_data_stream ) == 0 );
  assert( icom_test_multiple_pushes( p_data_stream ) == 0 );

  printf("free ...\n");
  icom_release_data_stream( p_data_stream );

  printf( "%s: successful!\n", __func__ );
  return 0;
}

int main()
{
  int result = 0;
  cul_allocstat_t allocstat;
  FILE* free_log_fp = NULL;

  icom_log_init();
  printf( "Starting transport layer test for libintercom revision %s ... \n\n",
          g_icomlib_revision );

  memtrace_enable();

  result = test_data_stream_splitter();

  memtrace_disable();

  allocstat = get_allocstat();
  printf("\n\n");
  printf("Memory Allocation Statistics in cutillib functions\n");
  printf("--------------------------------------------------\n");
  printf("       number of open allocations: %ld\n", allocstat.nr_allocs );
  printf("     number of failed allocations: %ld\n", allocstat.nr_allocs_failed );
  printf("  still allocated memory in bytes: %ld\n", allocstat.mem_allocated );
  printf("maximum allocated memory in bytes: %ld\n", allocstat.max_allocated );
  printf("\n");

  if( allocstat.nr_allocs != 0 || allocstat.mem_allocated != 0 )
  {
    fprintf( stderr, "ATTENTION: Some memory is leaking out, configure with memcheck option and check for root cause!\n\n");
    result = -1;
  }

  memtrace_print_log( stdout );

  free_log_fp = fopen("/tmp/freelog.txt", "w" );
  if( free_log_fp != NULL )
  {
    memtrace_print_free_log( free_log_fp );
    fclose( free_log_fp );
  }

  icom_log_release();

  return result;
}
