/* Copyright (c) 2016-2017, The Linux Foundation. All rights reserved.

 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/version.h>
#include <linux/module.h>
#include <linux/clk.h>
#include <linux/types.h>
#include <linux/platform_device.h>
#include <linux/kernel.h>
#include <linux/debugfs.h>
#include <linux/init.h>
#include <linux/platform_data/qcom_crypto_device.h>
#include <linux/msm-bus.h>

MODULE_LICENSE("GPLv2");

struct test_device {
	uint32_t bus_scale_handle;
	void *bus_scale_table;
	struct clk *ce_core_src_clk;	/* Handle to CE src clk*/
	struct clk *ce_core_clk;		/* Handle to CE clk */
	struct clk *ce_clk;				/* Handle to CE clk */
	struct clk *ce_bus_clk;			/* Handle to CE AXI clk*/
	struct clk *xo_clk;				/* Handle to XO clk*/
};

struct test_device test_dev;
void enable_crypto_clk(void);
void disable_crypto_clk(void);

void pdev_set_bus(bool high_bw_req)
{
	int ret = 0;
	if (high_bw_req) {
		enable_crypto_clk();
		ret = msm_bus_scale_client_update_request(test_dev.bus_scale_handle, 1);
		if (ret) {
			pr_err("%s Unable to set to high bandwidth\n", __func__);
			disable_crypto_clk();
			goto clk_err;
		}
	} else {
		ret = msm_bus_scale_client_update_request(test_dev.bus_scale_handle, 0);
		if (ret) {
			pr_err("%s Unable to set to low bandwidth\n", __func__);
			goto clk_err;
		}
		disable_crypto_clk();
	}
	clk_err:
	return;
}

void enable_crypto_clk(void)
{
	int rc;
	if (test_dev.ce_core_src_clk) {
		rc = clk_prepare_enable(test_dev.ce_core_src_clk);
		if (rc) {
			pr_err("Unable to enable/prepare CE core src clk\n");
			return ;
		}
	}

	if (test_dev.ce_core_clk) {
		rc = clk_prepare_enable(test_dev.ce_core_clk);
		if (rc) {
			pr_err("Unable to enable/prepare CE core clk\n");
			goto exit_disable_core_src_clk;
		}
	}

	if (test_dev.ce_clk) {
		rc = clk_prepare_enable(test_dev.ce_clk);
		if (rc) {
			pr_err("Unable to enable/prepare CE clk\n");
			goto exit_disable_core_clk;
		}
	}

	if (test_dev.ce_bus_clk) {
		rc = clk_prepare_enable(test_dev.ce_bus_clk);
		if (rc) {
			pr_err("Unable to enable/prepare CE bus c clk\n");
			goto exit_disable_ce_clk;
		}
	}
	return;

exit_disable_ce_bus_clk:
	if (test_dev.ce_bus_clk)
		clk_disable_unprepare(test_dev.ce_bus_clk);
exit_disable_ce_clk:
	if (test_dev.ce_clk)
		clk_disable_unprepare(test_dev.ce_clk);
exit_disable_core_clk:
	if (test_dev.ce_core_clk)
		clk_disable_unprepare(test_dev.ce_core_clk);
exit_disable_core_src_clk:
	if (test_dev.ce_core_src_clk)
		clk_disable_unprepare(test_dev.ce_core_src_clk);
	return;
}

void disable_crypto_clk(void)
{
	if (test_dev.ce_bus_clk)
		clk_disable_unprepare(test_dev.ce_bus_clk);
	if (test_dev.ce_clk)
		clk_disable_unprepare(test_dev.ce_clk);
	if (test_dev.ce_core_clk)
		clk_disable_unprepare(test_dev.ce_core_clk);
	if (test_dev.ce_core_src_clk)
		clk_disable_unprepare(test_dev.ce_core_src_clk);
}

EXPORT_SYMBOL(pdev_set_bus);

static int pdev_test_probe(struct platform_device *pdev)
{
	pr_debug("we got in test match probe\n");

	test_dev.bus_scale_table = (struct msm_bus_scale_pdata *) msm_bus_cl_get_pdata(pdev);
	if (!test_dev.bus_scale_table)
		pr_warn("bus_scale_table is NULL\n");

	test_dev.bus_scale_handle = msm_bus_scale_register_client((struct msm_bus_scale_pdata *) test_dev.bus_scale_table);
	if (!test_dev.bus_scale_handle) {
		pr_err("%s not able to get bus scale\n",
		__func__);
		return -ENOMEM;
	}

	test_dev.ce_core_src_clk = clk_get (&pdev->dev, "core_clk_src");
	if (!test_dev.ce_core_src_clk) {
		pr_err("unable to get the clock for core_clk_src\n");
		return -1;
	} else {
		if (clk_set_rate (test_dev.ce_core_src_clk,100000000)) {
			pr_err("unable_set clk rate");
			return -1;
		}
	}

	test_dev.ce_core_clk = clk_get (&pdev->dev, "core_clk");
	if (!test_dev.ce_core_clk) {
		pr_err("unable to get the clock for core_clk_src\n");
		return -1;
	}

	test_dev.ce_clk = clk_get (&pdev->dev, "iface_clk");
	if (!test_dev.ce_clk) {
		pr_err("unable to get the clock for core_clk_src\n");
		return -1;
	}

	test_dev.ce_bus_clk = clk_get (&pdev->dev, "bus_clk");
	if (!test_dev.ce_bus_clk) {
		pr_err("unable to get the clock for core_clk_src\n");
		return -1;
	}
	return 0;
}

static int pdev_test_remove(struct platform_device *pdev)
{
	if (test_dev.bus_scale_handle != 0)
		msm_bus_scale_unregister_client(test_dev.bus_scale_handle);
	test_dev.bus_scale_handle = 0;
	return 0;
}

static struct of_device_id test_match[] = {
	{ .compatible = "qcom,tztest",
	},
	{}
};

static struct platform_driver pdev_test_dev = {
	.probe = pdev_test_probe,
	.remove = pdev_test_remove,
	.driver = {
		.name = "tztest",
		.of_match_table = test_match,
		.owner = THIS_MODULE,
	},
};

static int pdev_test_init(void)
{
	return platform_driver_register(&pdev_test_dev);
}

static void pdev_test_exit(void)
{
	platform_driver_unregister(&pdev_test_dev);
	return;
}

module_init(pdev_test_init);
module_exit(pdev_test_exit);
