/*
 * vtz2 Valeo TrustZone driver2 for Qualcomm MDM9607 SoC
 *
 *   Copyright (C) 2016-2021 Valeo peiker Telematik GmbH
 *
 *   Authors: Mohamed Ahmed Hassan <mohamed.ahmed-hassan@valeo.com>
 *            Simon Gleissner <simon.gleissner@valeo.com>
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   version 2 as published by the Free Software Foundation.
 *
 *   This program is licensed "as is" without any warranty of any kind,
 *   whether express or implied.
 */

#include <linux/random.h>	/* prandom_bytes() */
#include <asm/barrier.h> /*smp_mb*/ /*included here and in vtz2_main*/
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/device.h> /*dev_dbg*/

#include <vtz2_helper.h>

/*********************************************/
/* EWOULDBLOCK & mutex locking inside ioctrl */
/*********************************************/

/* atomic counter helper */

void vtz_short_tz_duration_count_inc(void)
{
	atomic_inc(&vtz_drvdata->inside_short_tz_call_counter);
	smp_mb__after_atomic();
}

void vtz_short_tz_duration_count_dec(void)
{
	atomic_dec(&vtz_drvdata->inside_short_tz_call_counter);
	smp_mb__after_atomic();
}

int vtz_short_tz_duration_running(void)
{
	return (atomic_read(&vtz_drvdata->inside_short_tz_call_counter) != 0);
}

void vtz_long_tz_duration_count_inc(void)
{
	atomic_inc(&vtz_drvdata->inside_long_tz_call_counter);
	smp_mb__after_atomic();
}

void vtz_long_tz_duration_count_dec(void)
{
	atomic_dec(&vtz_drvdata->inside_long_tz_call_counter);
	smp_mb__after_atomic();
}

int vtz_long_tz_duration_running(void)
{
	return (atomic_read(&vtz_drvdata->inside_long_tz_call_counter) != 0);
}

/* locking api */

int vtz_enter_critical_short_tz_duration(void)
{
	vtz_short_tz_duration_count_inc();		/* announce short tz duration call */
							/* now long tz duration calls will not block us */
	if(vtz_long_tz_duration_running()) {		/* query long tz duration call */
		vtz_short_tz_duration_count_dec();	/* cleanup */
		return -EWOULDBLOCK;			/* long tz duration call active */
	}

	mutex_lock(&vtz_drvdata->inside_tz_call_mutex);	/* short call already announced, we can wait */
	return 0;					/* success */
}

void vtz_leave_critical_short_tz_duration(void)
{
	vtz_short_tz_duration_count_dec();
	mutex_unlock(&vtz_drvdata->inside_tz_call_mutex);
}

void vtz_enter_critical_long_tz_duration(void)
{
	vtz_long_tz_duration_count_inc();				/* announce long tz duration call, now short tz duration calls will not be blocked */
	mutex_lock(&vtz_drvdata->inside_tz_call_mutex);			/* wait for exclusive tz access */

	while(vtz_short_tz_duration_running()) {			/* are short tz duration calls active? we might need several tries to aquire exclusive access */
		mutex_unlock(&vtz_drvdata->inside_tz_call_mutex);	/* we must not block for a long time now, so try again later, */
		schedule();						/* let other processes take mutex first */
		mutex_lock(&vtz_drvdata->inside_tz_call_mutex);		/* wait again for exclusive tz access */
	}
}

void vtz_leave_critical_long_tz_duration(void)
{
	vtz_long_tz_duration_count_dec();
	mutex_unlock(&vtz_drvdata->inside_tz_call_mutex);
}

/****************************/
/* error ring buffer helper */
/****************************/

static void increment_head(void) //must be called only while holding log_buffer_mutex
{

	(vtz_drvdata->vtz_ring_buffer).head = ((vtz_drvdata->vtz_ring_buffer).head + 1) & (BUFFER_SIZE - 1);
	((vtz_drvdata->vtz_ring_buffer).count)++;
}

static void increment_tail(void) //must be called only while holding log_buffer_mutex
{
	(vtz_drvdata->vtz_ring_buffer).tail = ((vtz_drvdata->vtz_ring_buffer).tail + 1) & (BUFFER_SIZE - 1);
	((vtz_drvdata->vtz_ring_buffer).count)--;
}

/****************************/
/* error ring buffer functions */
/****************************/

static void vtz_buffer_write(int rc, vtz_err_type_t error_code) //Static function as it doesn't need to be used outside this translation unit
{
	uint32_t head_index, count;
	pr_debug(PFX "Writing an exception to vtz2 ring buffer\n");

	mutex_lock(&vtz_drvdata->log_buffer_mutex);
	head_index = (vtz_drvdata->vtz_ring_buffer).head;
	count = (vtz_drvdata->vtz_ring_buffer).count;

	if (count == BUFFER_SIZE) //buffer is full, overwrite
		increment_tail(); // push the tail by one to overwrite the first entry

	((vtz_drvdata->vtz_ring_buffer).error_array[head_index]).error_type = error_code;
	((vtz_drvdata->vtz_ring_buffer).error_array[head_index]).error_code = rc;
	increment_head();
	mutex_unlock(&vtz_drvdata->log_buffer_mutex);
}

int vtz_buffer_read(vtz_error *error_return)
{
	uint32_t tail_index, count;
	pr_debug(PFX "Reading an exception from vtz2 ring buffer\n");

	if (!error_return)
		return -EFAULT;

	mutex_lock(&vtz_drvdata->log_buffer_mutex);
	tail_index = (vtz_drvdata->vtz_ring_buffer).tail;
	count = (vtz_drvdata->vtz_ring_buffer).count;

	if (count == 0) { //Buffer is empty
		mutex_unlock(&vtz_drvdata->log_buffer_mutex);
		pr_debug(PFX "Failed to read from vtz2 ring buffer, buffer empty.\n");
		return -ENODATA;
	}

	error_return->error_type = ((vtz_drvdata->vtz_ring_buffer).error_array[tail_index]).error_type;
	error_return->error_code = ((vtz_drvdata->vtz_ring_buffer).error_array[tail_index]).error_code;
	increment_tail();
	mutex_unlock(&vtz_drvdata->log_buffer_mutex);
	return 0;
}

void vtz_buffer_clear(void)
{
	pr_debug (PFX "Clearing vtz2 ring buffer\n");

	mutex_lock(&vtz_drvdata->log_buffer_mutex);
	(vtz_drvdata->vtz_ring_buffer).tail = 0;
	(vtz_drvdata->vtz_ring_buffer).head = 0;
	(vtz_drvdata->vtz_ring_buffer).count = 0;
	mutex_unlock(&vtz_drvdata->log_buffer_mutex);
}

uint32_t vtz_buffer_count(void)
{
	uint32_t count;
	pr_debug(PFX "Reading vtz2 ring buffer elements count\n");

	mutex_lock(&vtz_drvdata->log_buffer_mutex);
	count = (vtz_drvdata->vtz_ring_buffer).count;
	mutex_unlock(&vtz_drvdata->log_buffer_mutex);

	return count;
}

/*********************/
/* error exit helper */
/*********************/

int vtz_throw_exception(int rc, vtz_mem_internal_t *tz_memory_cleanup, vtz_err_type_t error_code, const char* const funcname, const int line)
{
	vtz_buffer_write(rc,error_code);
	wake_up(&vtz_drvdata->log_event); //Wake up processes waiting on the log_event wait queue (polling processes)

	vtz_mem_free(tz_memory_cleanup);				/* NULL check is done inside in vtz_mem_free */
	pr_err(PFX "exception %d in %s() at line %d\n", rc, funcname, line);
	return rc;
}


/************************************/
/* transaction random number helper */
/************************************/

void vtz_next_transaction(uint64_t* transaction_ptr)
{									/* for now, use the kernel PRNG */
	prandom_bytes(transaction_ptr, sizeof(uint64_t));		/* pseudo random numbers, do not consume entrophy */
}
