/*
 * vtz2 Valeo TrustZone driver2 for Qualcomm MDM9607 SoC
 *
 *   Copyright (C) 2016-2021 Valeo peiker Telematik GmbH
 *
 *   Authors: Mohamed Ahmed Hassan <mohamed.ahmed-hassan@valeo.com>
 *            Simon Gleissner <simon.gleissner@valeo.com>
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   version 2 as published by the Free Software Foundation.
 *
 *   This program is licensed "as is" without any warranty of any kind,
 *   whether express or implied.
 */
#include <linux/errno.h>
#include <linux/fs.h>		 /*file ops*/
#include <linux/wait.h>		 /*Wait Queues*/
#include <linux/poll.h>		/* poll_wait()*/

#include <vtz2_log.h>
#include <vtz_user.h>
#include <vtz2_main.h>
#include <vtz2_mem.h>
#include <vtz2_helper.h>
#include <vtz2_scm.h>

/**********************************/
/* ioctl functions for VTZ2 Logging */
/**********************************/
static long vtz_log_read(vtz_error __user * error_return)
{
	long rc;
	vtz_error error_return_kernel;

	if(TZ_MEM_IOCTL_INIT(&error_return_kernel, error_return))/* Null and size compatibility check */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_FROM_USR_ERR,__func__, __LINE__);

	rc = vtz_buffer_read(&error_return_kernel);

	if (TZ_MEM_IOCTL_EXIT(&error_return_kernel, error_return))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_TO_USR_ERR,__func__, __LINE__);

	return rc;
}

static long vtz_log_count (uint32_t* elements_count)
{
	uint32_t elements_count_kernel;

	if(TZ_MEM_IOCTL_INIT(&elements_count_kernel, elements_count))/* Null and size compatibility check */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_FROM_USR_ERR,__func__, __LINE__);

	elements_count_kernel = vtz_buffer_count();

	if (TZ_MEM_IOCTL_EXIT(&elements_count_kernel, elements_count))	/* copy ioctl argument back into user space */
		return vtz_throw_exception(-EFAULT, NULL, MEM_COPY_TO_USR_ERR,__func__, __LINE__);

	return 0;
}

static void vtz_log_clear(void)
{
	vtz_buffer_clear();
}

/*********************/
/* file i/o vdlog */
/*********************/

static int vtz_log_file_open(struct inode *inode, struct file *file)
{
	struct vtz_drvdata_t *private_driver_data = vtz_drvdata;

	if (!private_driver_data)
		return -ENODATA;

	if (inode->i_rdev != private_driver_data->dev_vdlog.dev)
		return -ENXIO;

	if (file->private_data)		/* device already open? */
		return -EBADF;

	file->private_data = (void*) &private_driver_data->dev_vdlog;

	return nonseekable_open(inode, file);
}

static int vtz_log_file_release(struct inode *inode, struct file *file)
{
	file->private_data = NULL;
	return 0;
}


static long vtz_log_file_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct vtz_device_t *private_device_data = (struct vtz_device_t*) file->private_data;
	long rc;

	if(!private_device_data)
		return -ENODATA;

	switch (cmd) {
	case VTZ_IOCTL_LOG_READ:
		rc = vtz_log_read((vtz_error __user*)arg);
		break;
	case VTZ_IOCTL_LOG_GET_COUNT:
		rc = vtz_log_count((uint32_t __user*)arg);
		break;
	case VTZ_IOCTl_LOG_CLEAR:
		vtz_log_clear();
		rc = 0; //buffer clear method can't fail
		break;
	default:
		rc = -ENOTTY;
		break;
	}
	return rc;

}

static unsigned int vtz_log_file_poll (struct file * file, struct poll_table_struct * poll_table )
{
	unsigned int mask = 0;
	poll_wait(file, &vtz_drvdata->log_event, poll_table);

	if (vtz_drvdata->vtz_ring_buffer.count != 0) //Check if there is data in the buffer to be read
		mask |= (POLLIN | POLLRDNORM);

	return mask;
}

const struct file_operations vtz_fops_log = {
	.owner=THIS_MODULE,
	.unlocked_ioctl	= vtz_log_file_ioctl,
	.open		= vtz_log_file_open,
	.release	= vtz_log_file_release,
	.poll = vtz_log_file_poll,
	.llseek=no_llseek,
};