/*
 * Watchdog driver for Qualcomm MSM9640 & MSM9628 SoC
 *
 *   Copyright (C) 2017, 2018 peiker acustic GmbH & Co. KG
 *
 *   Authors: Simon Gleissner <simon.gleissner@valeo.com>
 *            Kishore Jagadeesha <kishore.jagadeesha@valeo.com>
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   version 2 as published by the Free Software Foundation.
 *
 *   This program is licensed "as is" without any warranty of any kind,
 *   whether express or implied.
 *
 * Based on omap_wdt.c, the watchdog driver for the TI OMAP 16xx & 24xx/34xx
 *
 *   Copyright (c) 2003 MontaVista Software, Inc.,
 *                      George G. Davis <gdavis@mvista.com>
 *   Copyright (c) 2004 Texas Instruments
 *   Copyright (c) 2005 David Brownell
 *   Copyright (c) 2000 Oleg Drokin <green@crimea.edu>
 *
 * Based on softdog.c, the software watchdog device
 *
 *   Copyright (c) 1995/1996/1998 Alan Cox <alan@lxorguk.ukuu.org.uk>
 *   Copyright (c) 1996 Angelo Haritsis <ah@doc.ic.ac.uk>
 *   Copyright (c) 2001/2002 Joel Becker <joel.becker@oracle.com>
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/types.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/fs.h>
#include <linux/reboot.h>
#include <linux/init.h>
#include <linux/uaccess.h>
#include <linux/kernel.h>
#include <linux/pm_runtime.h>
#include <linux/io.h>
#include <linux/ioport.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/cdev.h>
#include <linux/delay.h>
#include <asm/cacheflush.h>
#include <linux/qpnp/power-on.h>

/* module name */
#define DRV_NAME "mdm_wdt"
#define PFX DRV_NAME ": "

/* allowed timer values */
#define TIMER_MARGIN_MAX	16	/* real hw bark maximum is at 16 secs - 1 tic */
#define TIMER_MARGIN_DEFAULT	15	/* 15 secs */
#define TIMER_MARGIN_MIN	1	/*  1 sec  */
#define TIMER_BARK_DEADLINE	1	/*  1 sec  */
#define TIMER_FREQUENCY		32768	/*  timer freq in hz */

/* hardware register offsets */
#define MDM_WDOG_RESET		0x04
#define MDM_WDOG_CTRL		0x08
#define MDM_WDOG_STATUS		0x0C
#define MDM_WDOG_BARK_TIME	0x10
#define MDM_WDOG_BITE_TIME	0x14

/* hardware register values */
/* MDM_WDOG_RESET */
#define MDM_WDOG_RESET_PET	0x01
/* MDM_WDOG_CTRL */
#define MDM_WDOG_CTRL_ENABLE	0x01
#define MDM_WDOG_CTRL_DISABLE	0x00

#define CLOSE_UNEXPECTED 0
#define CLOSE_EXPECTED 42

/* module parameter */
static unsigned timer_margin = TIMER_MARGIN_DEFAULT;
module_param(timer_margin, uint, 0);
MODULE_PARM_DESC(timer_margin,
		 "initial watchdog timeout (in seconds: 1..."
		__MODULE_STRING(TIMER_MARGIN_MAX) ")");

static int nowayout = WATCHDOG_NOWAYOUT;
module_param(nowayout, int, 0);
MODULE_PARM_DESC(nowayout,
		 "Watchdog cannot be stopped once started (default="
		 __MODULE_STRING(WATCHDOG_NOWAYOUT) ")");

/* module variables */
static struct platform_device *mdm_wdt_pdev = NULL;
static spinlock_t wdt_lock;
static char expect_close;
static char watchdog_barked;

/* private watchdog device struct */
struct mdm_wdt_dev {
	void __iomem			*base;          /* physical */
	struct device			*dev;
	volatile ulong			mdm_wdt_users;
	struct mdm_wdt_dev __percpu	**wdog_cpu_dd;
	struct resource 		*mem;
	struct miscdevice		mdm_wdt_miscdev;
	int				bark_irq;
	bool				irq_ppi;
};

/* bark routine, called from IRQ handler */
static irqreturn_t mdm_wdt_bark_handler(int irq, void *dev_id)
{
	struct mdm_wdt_dev* wdev = (struct mdm_wdt_dev*)dev_id;

	dev_emerg(wdev->dev, "watchdog barked - flush cpu caches,"
				"call panic and wait for watchdog reset!\n");
	qpnp_pon_set_restart_reason(PON_RESTART_REASON_WATCHDOG);
        watchdog_barked = 1;
	flush_cache_all();
	panic("watchdog bark received!");
	return IRQ_HANDLED;
}

static irqreturn_t mdm_wdt_ppi_bark_handler(int irq, void *dev_id)
{
	struct mdm_wdt_dev *wdev = *(struct mdm_wdt_dev **)(dev_id);
	return mdm_wdt_bark_handler(irq, wdev);
}

/* low-level hardware access routines */
static void mdm_wdt_ping(struct mdm_wdt_dev *wdev)
{
	void __iomem *base = wdev->base;
	spin_lock(&wdt_lock);

	/* pet watchdog */
	__raw_writel(MDM_WDOG_RESET_PET, base + MDM_WDOG_RESET);
	mb();

	spin_unlock(&wdt_lock);
}

static void mdm_wdt_enable(struct mdm_wdt_dev *wdev)
{
	void __iomem *base = wdev->base;

	spin_lock(&wdt_lock);

	/* enable watchdog & clock, pet watchdog */
	__raw_writel(MDM_WDOG_CTRL_ENABLE,	base + MDM_WDOG_CTRL);
	__raw_writel(MDM_WDOG_RESET_PET,	base + MDM_WDOG_RESET);
	mb();

	spin_unlock(&wdt_lock);
}

static void mdm_wdt_disable(struct mdm_wdt_dev *wdev)
{
	void __iomem *base = wdev->base;

	spin_lock(&wdt_lock);

	/* disable watchdog & clock */
	__raw_writel(MDM_WDOG_RESET_PET,	base + MDM_WDOG_RESET);
	__raw_writel(MDM_WDOG_CTRL_DISABLE,	base + MDM_WDOG_CTRL);
	mb();

	spin_unlock(&wdt_lock);
}

static void mdm_wdt_set_timeout(struct mdm_wdt_dev *wdev)
{
	void __iomem *base = wdev->base;
	u32 bark_margin = 2;	/* timer_margin=0: no bark */
	u32 bite_margin = 1;	/* timer_margin=0: fast bite */

	if(timer_margin) {
		bark_margin = timer_margin * TIMER_FREQUENCY - (timer_margin == TIMER_MARGIN_MAX); /* respect maximum hw value */
		bite_margin = (timer_margin + TIMER_BARK_DEADLINE) * TIMER_FREQUENCY;   /* has higher maximum hw value */
	}

	spin_lock(&wdt_lock);

	/* set watchdog timeout */
	__raw_writel(bark_margin, base + MDM_WDOG_BARK_TIME);
	__raw_writel(bite_margin, base + MDM_WDOG_BITE_TIME);
	mb();

	spin_unlock(&wdt_lock);
}

static int mdm_wdt_get_bark_timeout(struct mdm_wdt_dev *wdev)
{
	void __iomem *base = wdev->base;
	int bark_timeout = -1;	/* default: watchdog is inactive */

	spin_lock(&wdt_lock);

	/* get programmed watchdog bark timeout, if enabled */
	if((__raw_readl(base + MDM_WDOG_CTRL) & MDM_WDOG_CTRL_ENABLE) != 0){
		bark_timeout = (__raw_readl(base + MDM_WDOG_BARK_TIME) & 0xfffff) / TIMER_FREQUENCY;
	}
	mb();

	spin_unlock(&wdt_lock);

	return bark_timeout;
}

static int mdm_wdt_test_active_timeout(struct mdm_wdt_dev *wdev)
{
	void __iomem *base = wdev->base;
	int active_timeout = -1;	/* default: watchdog is inactive */

	spin_lock(&wdt_lock);

	/* get watchdog timeout, if enabled */
	if((__raw_readl(base + MDM_WDOG_CTRL) & MDM_WDOG_CTRL_ENABLE) != 0){
		active_timeout = (__raw_readl(base + MDM_WDOG_BITE_TIME) & 0xfffff) / TIMER_FREQUENCY;
	}
	mb();

	spin_unlock(&wdt_lock);

	return active_timeout;
}

static int mdm_wdt_time_until_reset(struct mdm_wdt_dev *wdev)
{
	void __iomem *base = wdev->base;
	int bark_value;
	int bite_count = -1;
	int time_left = -1;	/* default: watchdog is inactive */
	unsigned int status_raw = 0xFFFFFFFFU;

	spin_lock(&wdt_lock);

	if((__raw_readl(base + MDM_WDOG_CTRL) & MDM_WDOG_CTRL_ENABLE) != 0){
                bark_value = __raw_readl(base + MDM_WDOG_BARK_TIME) & 0xfffff;
		status_raw = __raw_readl(base + MDM_WDOG_STATUS);
		bite_count = (status_raw >> 2) & 0xfffff;
		time_left = (bite_count < bark_value) ? ((bark_value - bite_count) / TIMER_FREQUENCY) : 0;
	}
	mb();

	spin_unlock(&wdt_lock);
	return time_left;
}

/* check changed timeout value */
static void mdm_wdt_adjust_timeout(unsigned new_timeout)
{
	if (new_timeout < TIMER_MARGIN_MIN)
		new_timeout = TIMER_MARGIN_MIN;
	if (new_timeout > TIMER_MARGIN_MAX)
		new_timeout = TIMER_MARGIN_MAX;
	timer_margin = new_timeout;
}

/* do fast hardware watchdog reset now */
void mdm_wdt_reset_now(void)
{
        struct mdm_wdt_dev *wdev = (mdm_wdt_pdev) ? \
					platform_get_drvdata(mdm_wdt_pdev) : NULL;

	if(wdev) {
        	dev_emerg(wdev->dev, "Force immediate reset\n");
		if(watchdog_barked!=0)    /* detect previous watchdog bark */
			qpnp_pon_set_restart_reason(PON_RESTART_REASON_WATCHDOG);
		timer_margin = 0;
		mdm_wdt_set_timeout(wdev);
		mdm_wdt_ping(wdev);
		mdelay(50);  /* block for at least 1/32768 s until reset */
		dev_crit(wdev->dev, "Immediate watchdog bite failed\n");
	}
}

/* file i/o */
static int mdm_wdt_open(struct inode *inode, struct file *file)
{
        struct mdm_wdt_dev *wdev = (mdm_wdt_pdev) ? \
					platform_get_drvdata(mdm_wdt_pdev) : NULL;

	if (wdev && test_and_set_bit(1, &wdev->mdm_wdt_users))
		return -EBUSY;

	file->private_data = (void *) wdev;

	pm_runtime_get_sync(wdev->dev);
	mdm_wdt_set_timeout(wdev);
	mdm_wdt_enable(wdev);
	pm_runtime_put_sync(wdev->dev);

	return nonseekable_open(inode, file);
}

static int mdm_wdt_release(struct inode *inode, struct file *file)
{
	struct mdm_wdt_dev *wdev = file->private_data;

	pm_runtime_get_sync(wdev->dev);

	if (expect_close == CLOSE_EXPECTED) {
		mdm_wdt_disable(wdev);
	} else {
		if (!nowayout) {
			dev_crit(wdev->dev,
				"Unexpected close, you have %ds left for reinitialization until watchdog reset!\n",
				 timer_margin);
		} else {
			dev_crit(wdev->dev,
				"'nowayout' is set, you have %ds left for reinitialization until watchdog reset!\n",
				 timer_margin);
		}
		mdm_wdt_ping(wdev);
	}

	pm_runtime_put_sync(wdev->dev);

	wdev->mdm_wdt_users = 0;
	expect_close = CLOSE_UNEXPECTED;

	return 0;
}

static ssize_t mdm_wdt_write(struct file *file, const char __user *data,
		size_t len, loff_t *ppos)
{
	struct mdm_wdt_dev *wdev = file->private_data;

	if (len) {
		if (!nowayout) {
			size_t i;
			expect_close = CLOSE_UNEXPECTED;	/* In case it was set long ago */

			for (i = 0; i != len; i++) {
				char c;

				if (get_user(c, data + i))
					return -EFAULT;
				if (c == 'V')	/* magic close character */
					expect_close = CLOSE_EXPECTED;
			}
		}
		pm_runtime_get_sync(wdev->dev);
		mdm_wdt_ping(wdev);
		pm_runtime_put_sync(wdev->dev);
	}
	return len;
}

static long mdm_wdt_ioctl(struct file *file, unsigned int cmd,
						unsigned long arg)
{
	struct mdm_wdt_dev *wdev = file->private_data;
	int new_margin;
	int options;
	int ret;
	static const struct watchdog_info ident = {
		.identity = "Mdm Watchdog",
		.options = WDIOF_SETTIMEOUT |
		           WDIOF_MAGICCLOSE |
		           WDIOF_KEEPALIVEPING,
		.firmware_version = 0,
	};

	switch (cmd) {
	case WDIOC_GETSUPPORT:
		return copy_to_user((struct watchdog_info __user *)arg, &ident,
				sizeof(ident)) ? -EFAULT : 0;
	case WDIOC_GETSTATUS:
	case WDIOC_GETBOOTSTATUS:
		return put_user(0, (int __user *)arg);
	case WDIOC_SETOPTIONS:
		if (copy_from_user(&options, (int __user *)arg, sizeof(options)))
			return -EFAULT;
		ret = -EINVAL;
		if (options & WDIOS_DISABLECARD) {
			if (!nowayout) {
				pm_runtime_get_sync(wdev->dev);
				mdm_wdt_disable(wdev);
				pm_runtime_put_sync(wdev->dev);
				ret = 0;
			} else {
				pm_runtime_get_sync(wdev->dev);
				mdm_wdt_ping(wdev);
				pm_runtime_put_sync(wdev->dev);
				dev_crit(wdev->dev,
					"'nowayout' is set, not stopping watchdog!\n");
				ret = -EFAULT;
			}
		}
		if (options & WDIOS_ENABLECARD) {
			pm_runtime_get_sync(wdev->dev);
			mdm_wdt_set_timeout(wdev);
			mdm_wdt_enable(wdev);
			pm_runtime_put_sync(wdev->dev);
			ret = 0; /* overwrite previous error */
		}
		return ret;
	case WDIOC_KEEPALIVE:
		pm_runtime_get_sync(wdev->dev);
		mdm_wdt_ping(wdev);
		pm_runtime_put_sync(wdev->dev);
		return 0;
	case WDIOC_SETTIMEOUT:
		if (get_user(new_margin, (int __user *)arg))
			return -EFAULT;
		pm_runtime_get_sync(wdev->dev);
		mdm_wdt_disable(wdev);
		mdm_wdt_adjust_timeout(new_margin);
		mdm_wdt_set_timeout(wdev);
		mdm_wdt_enable(wdev);
		pm_runtime_put_sync(wdev->dev);
		return put_user(timer_margin, (int __user *)arg);
	case WDIOC_GETTIMEOUT:
		ret = mdm_wdt_get_bark_timeout(wdev);
		return put_user(ret, (int __user *)arg);
	case WDIOC_GETTIMELEFT:
		ret = mdm_wdt_time_until_reset(wdev);
		return put_user(ret , (int __user *)arg);
	default:
		return -ENOTTY;
	}
}

static const struct file_operations mdm_wdt_fops = {
	.owner = THIS_MODULE,
	.write = &mdm_wdt_write,
	.unlocked_ioctl = &mdm_wdt_ioctl,
	.open = &mdm_wdt_open,
	.release = &mdm_wdt_release,
	.llseek = no_llseek,
};

/* module control */
static int mdm_wdt_probe(struct platform_device *pdev)
{
	struct resource *res;
	struct mdm_wdt_dev *wdev;
	struct device *dev = &pdev->dev;
	int ret;
	int active_timeout;

	spin_lock_init(&wdt_lock);
	if(mdm_wdt_pdev){
		dev_err(&pdev->dev, "mdm_wdt_probe(): device already initialized\n");
		return -EBUSY;
	}

	wdev = devm_kzalloc(dev, sizeof(*wdev), GFP_KERNEL);
	if(!wdev){
		ret = -ENOMEM;
		dev_err(dev,
			"mdm_wdt_probe(): kzalloc() failed\n");
		goto err;
	}

	wdev->mdm_wdt_users = 0;
	wdev->dev = dev;

	/* get static register mappings resource */
	res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "wdt-base");
	if (!res) {
		ret = -ENOENT;
		dev_err(dev, "mdm_wdt_probe(): memory resource info not available\n");
		goto err;
	}

	wdev->bark_irq = platform_get_irq(pdev, 0);

	wdev->mem = devm_request_mem_region(dev, res->start, resource_size(res), pdev->name);
	if (!wdev->mem) {
		ret = -EBUSY;
		dev_err(dev, "mdm_wdt_probe(): request_mem_region() failed\n");
		goto err;
	}

	wdev->base = devm_ioremap(dev, res->start, resource_size(res));
	if (!wdev->base) {
		ret = -ENOMEM;
		dev_err(dev, "mdm_wdt_probe(): devm_ioremap() failed\n");
		goto err;
	}

	wdev->irq_ppi = (irq_is_percpu(wdev->bark_irq) != 0);

	/* irq request for watchdog bark */
	if (wdev->bark_irq >= 0) {
		if (wdev->irq_ppi) {
			wdev->wdog_cpu_dd = alloc_percpu(struct mdm_wdt_dev *);
			if (!wdev->wdog_cpu_dd) {
				wdev->bark_irq = -1;
				wdev->irq_ppi = 0;
			} else {
				*raw_cpu_ptr(wdev->wdog_cpu_dd) = wdev;
				ret = request_percpu_irq(wdev->bark_irq,
					mdm_wdt_ppi_bark_handler,
					DRV_NAME "_bark",
					wdev->wdog_cpu_dd);
				if (ret) {
					dev_err(dev, "failed to request bark irq, disable bark interrupt!\n");
					free_percpu(wdev->wdog_cpu_dd);
					wdev->bark_irq = -1;
					wdev->irq_ppi = 0;
				}
			}
		} else {
			ret = devm_request_irq(	dev,
					wdev->bark_irq,
					mdm_wdt_bark_handler,
					IRQF_TRIGGER_RISING,
					DRV_NAME "_bark",
					wdev);
			if (ret) {
				dev_err(dev, "mdm_wdt_probe(): devm_request_irq() failed"
					" with %d, disable bark interrupt!\n", ret);
				wdev->bark_irq = -1;
			}
		}
	} else {
		dev_warn(&pdev->dev,
			"mdm_wdt_probe(): no bark interrupt configured\n");
	}

	/* initialize the watchdog */
	platform_set_drvdata(pdev, wdev);

	pm_runtime_enable(wdev->dev);
	pm_runtime_get_sync(wdev->dev);

	active_timeout = mdm_wdt_test_active_timeout(wdev);

	if (active_timeout < 0) {	// watchdog previously not running
		mdm_wdt_disable(wdev);
		mdm_wdt_adjust_timeout(timer_margin);
	} else {
		mdm_wdt_disable(wdev);
		timer_margin = active_timeout;
		mdm_wdt_adjust_timeout(timer_margin);
		mdm_wdt_set_timeout(wdev);
		mdm_wdt_enable(wdev);
	}

	wdev->mdm_wdt_miscdev.parent = &pdev->dev;
	wdev->mdm_wdt_miscdev.minor = WATCHDOG_MINOR;
	wdev->mdm_wdt_miscdev.name = "watchdog";
	wdev->mdm_wdt_miscdev.fops = &mdm_wdt_fops;

	ret = misc_register(&(wdev->mdm_wdt_miscdev));
	if (ret) {
		dev_err(dev, "mdm_wdt_probe(): misc_register() failed\n");
		goto err_misc;
	}

	if (wdev->irq_ppi)
		enable_percpu_irq(wdev->bark_irq, IRQ_TYPE_NONE);

	dev_info(dev, "initialized, %s, timeout %d sec\n",
		 (active_timeout < 0) ? "inactive" : "active",
		 timer_margin);

	pm_runtime_put_sync(wdev->dev);
	mdm_wdt_pdev = pdev;

	return 0;

err_misc:
	pm_runtime_put_sync(wdev->dev);
	platform_set_drvdata(pdev, NULL);
	if (wdev->bark_irq >= 0) {
		if (wdev->irq_ppi) {
			free_percpu_irq(wdev->bark_irq, wdev->wdog_cpu_dd);
			free_percpu(wdev->wdog_cpu_dd);
		}
	}
err:
	return ret;
}

static void mdm_wdt_shutdown(struct platform_device *pdev)
{
	struct mdm_wdt_dev *wdev = platform_get_drvdata(pdev);

	if (wdev->mdm_wdt_users) {
		pm_runtime_get_sync(wdev->dev);
		mdm_wdt_disable(wdev);
		pm_runtime_put_sync(wdev->dev);
	}
}

static int mdm_wdt_remove(struct platform_device *pdev)
{
	struct mdm_wdt_dev *wdev = platform_get_drvdata(pdev);

	if (wdev->bark_irq >= 0) {
		if (wdev->irq_ppi) {
			free_percpu_irq(wdev->bark_irq, wdev->wdog_cpu_dd);
			free_percpu(wdev->wdog_cpu_dd);
		} 
	}

	misc_deregister(&(wdev->mdm_wdt_miscdev));
	platform_set_drvdata(pdev, NULL);
	mdm_wdt_pdev = NULL;

	return 0;
}

static int __maybe_unused mdm_wdt_suspend(struct device *dev)
{
	struct mdm_wdt_dev *wdev = dev_get_drvdata(dev);

	if (wdev->mdm_wdt_users)
		mdm_wdt_disable(wdev);
	return 0;
}

static int __maybe_unused mdm_wdt_resume(struct device *dev)
{
	struct mdm_wdt_dev *wdev = dev_get_drvdata(dev);

	if (wdev->mdm_wdt_users)
		mdm_wdt_enable(wdev);
	return 0;
}

static struct of_device_id mdm_wdt_match_table[] = {
	{ .compatible = "qcom,msm-watchdog" },
	{}
};

static const struct platform_device_id mdm_wdt_platform_id[] = {
	{ .name = DRV_NAME },
	{}
};

static const struct dev_pm_ops mdm_wdt_dev_pm_ops = {
	.suspend_noirq = mdm_wdt_suspend,
	.resume_noirq = mdm_wdt_resume,
};

static struct platform_driver mdm_wdt_driver = {
	.probe		= mdm_wdt_probe,
	.remove		= mdm_wdt_remove,
	.shutdown	= mdm_wdt_shutdown,
	.driver		= {
		.name	= DRV_NAME,
		.owner	= THIS_MODULE,
		.pm	= &mdm_wdt_dev_pm_ops,
		.of_match_table = mdm_wdt_match_table,
	},
};

MODULE_DEVICE_TABLE(platform, mdm_wdt_platform_id);

module_platform_driver(mdm_wdt_driver);

MODULE_AUTHOR("Simon Gleissner <simon.gleissner@valeo.com>");
MODULE_DESCRIPTION("LTENAD Qcom MDM Watchdog Device Driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS_MISCDEV(WATCHDOG_MINOR);
