/* //device/system/rild/rild.c
**
** Copyright 2006, The Android Open Source Project
**
** Licensed under the Apache License, Version 2.0 (the "License");
** you may not use this file except in compliance with the License.
** You may obtain a copy of the License at
**
**     http://www.apache.org/licenses/LICENSE-2.0
**
** Unless required by applicable law or agreed to in writing, software
** distributed under the License is distributed on an "AS IS" BASIS,
** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
** See the License for the specific language governing permissions and
** limitations under the License.
*/

#include <config.h>


#include <stdio.h>
#include <stdlib.h>
#include <dlfcn.h>
#include <string.h>
#include <stdint.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <dlfcn.h>
#include <signal.h>

#include <ril/ril.h>

#include <sys/types.h>
#include <sys/stat.h>

#if defined (SA515M)
#include <systemd/sd-daemon.h>
#endif

#define MAX_LIB_ARGS        16

#include "utils_log.h"

#if defined (USE_DLTLOG)
DLT_DECLARE_CONTEXT(RILD);
DLT_DECLARE_CONTEXT(VRIL);
DLT_DECLARE_CONTEXT(LRIL);
#endif


int loglevel=LOG_NOTICE;
int timestamp=0;

static void usage(const char *argv0)
{
    fprintf(stderr, "Usage: %s -l <ril impl library> [ -u (=support of local socket) ] -a <inet address> -p <inet port> -- <args for impl library>\n", argv0);
    exit(-1);
}


extern int RIL_register (const RIL_RadioFunctions *callbacks,int localSupport, int inetSupport,
                          char *inetAddr, int inetPort);


extern void RIL_onRequestComplete(RIL_Token t, RIL_Errno e,
                           void *response, size_t responselen);

extern void RIL_onUnsolicitedResponse(int unsolResponse, const void *data, size_t datalen);

extern void RIL_requestTimedCallback (RIL_TimedCallback callback,
                               void *param, const struct timeval *relativeTime);


static struct RIL_Env s_rilEnv = {
    RIL_onRequestComplete,
    RIL_onUnsolicitedResponse,
    RIL_requestTimedCallback
};


extern void RIL_startEventLoop(char *inetAddr, int inetPort);

static int shutDown=0;

static void catch_function(int signo) {
    int loglevel=LOG_NOTICE;
    ACLOGN(SYSLOGTAG, "%s: Interactive attention signal caught; exit",__FUNCTION__);
    shutDown=1;
}

static int make_argv(char * args, char ** argv)
{
    // Note: reserve argv[0]
    int count = 1;
    char * tok;
    char * s = args;

    while ((tok = strtok(s, " \0"))) {
        argv[count] = tok;
        s = NULL;
        count++;
    }
    return count;
}


#define PROPERTY_VALUE_MAX 92

int main(int argc, char **argv)
{
#ifdef USE_DLTLOG
/* register for DLT logs for RILD/APP contexts */
    DLT_REGISTER_APP("RIL", "RIL Logs");
    DLT_REGISTER_CONTEXT(RILD, "RILD", "RIL daemon logs");
    DLT_REGISTER_CONTEXT(VRIL, "VRIL", "lib valeo RIL");
    DLT_REGISTER_CONTEXT(LRIL, "LRIL", "lib RIL");
    DLT_LOG(RILD, DLT_LOG_INFO,DLT_STRING("starting RIL daemon"));
#endif
    const char * rilLibPath = NULL;
    char **rilArgv;
    void *dlHandle;
    const RIL_RadioFunctions *(*rilInit)(const struct RIL_Env *, int, char **);
    const RIL_RadioFunctions *funcs;
    char libPath[PROPERTY_VALUE_MAX];
    unsigned char hasLibArgs = 0;

    char inetAddr[16];
    int inetPort=0;
    int inetSupport=0;
    int localSupport=0;
    int libargNumber;
    int libArgSize;
    int libArgIdx;
    int stringPos;

    char* libArgs=NULL;
    int i;

    strcpy(inetAddr,"0.0.0.0");

    ALOGOPEN(SYSLOGTAG);

    umask(S_IRGRP | S_IWGRP | S_IXGRP | S_IROTH | S_IWOTH | S_IXOTH);
    for (i = 1; i < argc ;) {
        if (0 == strcmp(argv[i], "-l") && (argc - i > 1)) {
            rilLibPath = argv[i + 1];
            i += 2;
        } else if (0 == strcmp(argv[i], "-p") && (argc - i > 1)) {
            inetSupport=1;
            inetPort = atoi(argv[i + 1]);
            i += 2;
        } else if (0 == strcmp(argv[i], "-a") && (argc - i > 1)) {
            inetSupport=1;
            strcpy(inetAddr,argv[i + 1]);
            i += 2;
        } else if (0 == strcmp(argv[i], "-u") && (argc -i >= 1)) {
            localSupport=1;
            i += 1;
        } else if (0 == strcmp(argv[i], "-d") && (argc -i >= 1)) {
            loglevel=atoi(argv[i + 1]);
            if (loglevel<LOG_EMERG){
                loglevel=LOG_EMERG;
            }
            if (loglevel>LOG_DEBUG+1){
                loglevel=LOG_DEBUG+1;
            }
            i += 2;
        } else if (0 == strcmp(argv[i], "--")) {
            i++;
            hasLibArgs = 1;
            break;
        } else {
            usage(argv[0]);
        }
    }


    if (signal(SIGINT, catch_function) == SIG_ERR) {
        ACLOGE(SYSLOGTAG,"%s: An error occurred while setting a signal handler for SIGINT",__FUNCTION__);
        exit(-1);
    }
    if (signal(SIGHUP, catch_function) == SIG_ERR) {
        ACLOGE(SYSLOGTAG,"%s: An error occurred while setting a signal handler for SIGHUP",__FUNCTION__);
        exit(-1);
    }
    if (signal(SIGTERM, catch_function) == SIG_ERR) {
        ACLOGE(SYSLOGTAG,"%s: An error occurred while setting a signal handler  for SIGTERM",__FUNCTION__);
        exit(-1);
    }

    if (rilLibPath == NULL) {
        ACLOGE(SYSLOGTAG,"%s: library path not set",__FUNCTION__);
        exit(-1);
    }

    ACLOGN(SYSLOGTAG,"%s open lib '%s' ..\n",__FUNCTION__, rilLibPath);
    dlHandle = dlopen(rilLibPath, RTLD_NOW);

    if (dlHandle == NULL) {
        printf("dlopen failed: %s \n", dlerror());
        ACLOGE(SYSLOGTAG,"%s: dlopen failed: %s \n", __FUNCTION__, dlerror());
        exit(-1);
    }

    localSupport=0; // local socket disabled; listen on inet socket only
    RIL_startEventLoop(inetAddr, inetPort);

    ACLOGD(SYSLOGTAG,"%s: check for RIL_Init symbol..",__FUNCTION__);
    ACLOGD(SYSLOGTAG,"%s: provide '%s' to vendorlib",__FUNCTION__, libArgs);
    rilInit = (const RIL_RadioFunctions *(*)(const struct RIL_Env *, int, char **))dlsym(dlHandle, "RIL_Init");

    if (rilInit == NULL) {
        ACLOGE(SYSLOGTAG,"%s: RIL_Init not defined or exported in %s", __FUNCTION__, rilLibPath);
        exit(-1);
    }

    ACLOGN(SYSLOGTAG,"%s: exec RIL_Init",__FUNCTION__);

     if (hasLibArgs) {
        rilArgv = argv + i - 1;
        argc = argc -i + 1;
    } else {
        static char * newArgv[MAX_LIB_ARGS];
        static char args[PROPERTY_VALUE_MAX];
        rilArgv = newArgv;
        argc = make_argv(args, rilArgv);
    }
    rilArgv[0] = argv[0];

    funcs = rilInit(&s_rilEnv, argc, rilArgv);
    ACLOGN(SYSLOGTAG,"%s: exec RIL_Register ..",__FUNCTION__);
    shutDown=RIL_register(funcs, localSupport, inetSupport, inetAddr, inetPort);
    ACLOGN(SYSLOGTAG,"%s: wait for shutdown",__FUNCTION__);
    
#if defined (SA515M)
    ACLOGW(SYSLOGTAG,"%s: Send systemd notify",__FUNCTION__);
    sd_notify(0, "READY=1");
#endif

    while (shutDown == 0) {
        // sleep(UINT32_MAX) seems to return immediately on bionic
        sleep(3);

        if ((funcs == NULL) || (funcs->getVersion() == NULL)){
            shutDown=1;
        }
    }

    ACLOGN(SYSLOGTAG,"%s: shutdown..",__FUNCTION__);
    rilInit(NULL, 0, NULL);
    dlclose(dlHandle);

    ACLOGN(SYSLOGTAG,"%s: done!!",__FUNCTION__);
    ALOGCLOSE();

#ifdef USE_DLTLOG
    /* unregister for DLT logs for RILD/VRIL and LRIL contexts */
    DLT_LOG(RILD, DLT_LOG_DEBUG, DLT_STRING(__FUNCTION__) ,DLT_STRING(": RIL daemon finished"));
    DLT_UNREGISTER_CONTEXT(RILD);
    DLT_UNREGISTER_CONTEXT(VRIL);
    DLT_UNREGISTER_CONTEXT(LRIL);
    DLT_UNREGISTER_APP();
#endif
}
