/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <unistd.h>
#include <event_handler.h>
#include <ssc.h>
#include <ssc_events.h>
#include <ssc_control.h>
#include <ssc_tasks.h>
#include <uv_utils.h>
#include <mediatype.h>
#include <log.h>


/*! \file event_handler.c
    \brief event handler main callback function

    \addtogroup event_handler
    @{
 */

static int reschedule_if_required( ssc_tasks_t* p_ssc_tasks, ssc_evt_t* p_evt )
{
  ssc_proc_handle_t* p_proc;
  int error = 0, reschedule_required = 0;

  p_proc = ssc_get_task( p_ssc_tasks, p_evt->id );
  if( p_proc ) {
    if( p_evt->rescheduled_cnt == 0 ) {
      ssc_message("%s,%d: task id '%s' already open, exec_context: %d, close it first and defer processing ...\n",
                  __func__, __LINE__, p_evt->id, p_proc->exec_context );
    }
    if( p_proc->exec_context == ssc_exec_background_thread ) {
      /* log only every 3rd message to reduce log load */
      if( p_evt->rescheduled_cnt > 0 ) {
        ssc_message("%s,%d: task id '%s' is rescheduled %d times.\n",
                    __func__, __LINE__, p_evt->id, p_evt->rescheduled_cnt );
      }
      ++p_evt->rescheduled_cnt;
      reschedule_required = 1;
    }

    /* callback decides whether to immediately or delayed released */
    error = ssc_close_task( p_ssc_tasks, p_evt->id );
    if( error ) {
      ssc_error("%s,%d: error while closing task '%s' occured!\n",
                __func__, __LINE__, p_evt->id );
    }
  }

  return reschedule_required;
}


/*!
 * handle given event
 *
 * \params p pointer to command & control structure
 * \param p_evt event (request) to process
 * \return 1 when event could not be handle and needs to be rescheduled
 */
static int handle_evt( ssc_control_t* p, ssc_evt_t* p_evt )
{
  ssc_t* p_ssc = p->p_ssc;
  ssc_tasks_t*  p_ssc_tasks  = p_ssc->p_ssc_tasks;
  ssc_events_t* p_ssc_events = p_ssc->p_ssc_events;
  ssc_proc_handle_t* p_proc;
  // ssc_events_t* p_ssc_events = p_ssc->p_ssc_events;
  int reschedule = 0;
  int error = 0, state = 0;

  ssc_message("%s,%d: process event: %s, id: %s \n",
              __func__, __LINE__, ssc_event_name[ p_evt->type ], p_evt->id );

  /* one event loop to rule them all, say good by to memory races;-) */
  switch( p_evt->type )
  {
    /* background thread processing events */
  case ssc_proc_idle:
    break;

  case ssc_proc_open:
    (void) ssc_reply_to_all( p->p_tcp_cc_client_list,
                             "new state=openidle, id=%s, device=%s, filename=%s\n",
                             p_evt->id, p_evt->devicename, p_evt->filename );
    break;

  case ssc_proc_close:
    (void) ssc_reply_to_all( p->p_tcp_cc_client_list,
                             "new state=close, id=%s, device=%s, filename=%s\n",
                             p_evt->id, p_evt->devicename, p_evt->filename );
    break;

  case ssc_proc_init:
    (void) ssc_reply_to_all( p->p_tcp_cc_client_list,
                             "new state=initialized, id=%s, device=%s, filename=%s\n",
                             p_evt->id, p_evt->devicename, p_evt->filename );
    break;

  case ssc_proc_preparing:
    (void) ssc_reply_to_all( p->p_tcp_cc_client_list,
                             "new state=preparing, id=%s, device=%s, filename=%s\n",
                             p_evt->id, p_evt->devicename, p_evt->filename );
    break;

  case ssc_proc_running:
    (void) ssc_reply_to_all( p->p_tcp_cc_client_list,
                             "new state=running, id=%s, device=%s, filename=%s\n",
                             p_evt->id, p_evt->devicename, p_evt->filename );
    break;

  case ssc_proc_paused:
    (void) ssc_reply_to_all( p->p_tcp_cc_client_list,
                             "new state=paused, id=%s, device=%s, filename=%s\n",
                             p_evt->id, p_evt->devicename, p_evt->filename );
    break;

  case ssc_proc_stopping:
    (void) ssc_reply_to_all( p->p_tcp_cc_client_list,
                             "new state=stopping, id=%s, device=%s, filename=%s\n",
                             p_evt->id, p_evt->devicename, p_evt->filename );
    break;

  case ssc_proc_terminating:
    (void) ssc_reply_to_all( p->p_tcp_cc_client_list,
                             "new state=terminating, id=%s, device=%s, filename=%s\n",
                             p_evt->id, p_evt->devicename, p_evt->filename );
    break;

  case ssc_proc_terminated:
    p_proc = ssc_get_task( p_ssc_tasks, p_evt->id );
    if( p_proc ) {
      ssc_message("%s,%d: '%s' is terminated, release its memory.\n",
                  __func__, __LINE__, p_evt->id );

      /* callback decides whether to immediately or delayed released */
      error = p_proc->p_proc_release_cb( p_proc );
      if( error ) {
        ssc_error("%s,%d: error while closing task '%s' occured!\n",
                  __func__, __LINE__, p_evt->id );
      }
    }
    (void) ssc_reply_to_all( p->p_tcp_cc_client_list,
                             "new state=terminated, id=%s, device=%s, filename=%s\n",
                             p_evt->id, p_evt->devicename, p_evt->filename );

    /* now wake up event handler after process has been removed from task list */
    uv_async_send( & p_ssc_events->signal );
    break;

    /* external command requests */
  case ssc_proc_openidle_request:

    reschedule = reschedule_if_required( p_ssc_tasks, p_evt );
    if( !reschedule ) {
      /* no background processing takes place so we can immedially reopen */
      error = ssc_create_open_idle_task( p_ssc_tasks,
                                         p_evt->id,
                                         p_evt->devicename,
                                         p_evt->direction,
                                         (void *)p_ssc );
      if( error ) {
        ssc_message("%s,%d: could not open-idle id: '%s' error\n",
                    __func__, __LINE__, p_evt->id );

        (void)ssc_reply_to_sender( p_evt->client,
                                   "error when opening id %s, device: %s occured!\n",
                                   p_evt->id, p_evt->devicename );
      }
    }
    ssc_message("%s,%d: do open idle done\n", __func__, __LINE__ );
    break;

  case ssc_proc_close_request:
    error = ssc_close_task( p_ssc_tasks, p_evt->id );
    if( error ) {
      ssc_message("%s,%d: could not close id: '%s' error\n",
                  __func__, __LINE__, p_evt->id );

      (void)ssc_reply_to_sender( p_evt->client,
                                 "error when closing id %s, device: %s occured!\n",
                                 p_evt->id, p_evt->devicename );
    }
    break;

  case ssc_proc_close_all_request:
    error = ssc_close_all_tasks( p_ssc_tasks );
    break;

  case ssc_proc_playback_request:
    reschedule = reschedule_if_required( p_ssc_tasks, p_evt );
    if( !reschedule ) {
      /* no background processing takes place so we can immediately reopen */

      switch( ssc_media_type( p_evt->filename ) ) {
      case ssc_media_wav:
        error = ssc_create_wav_playback_task( p_ssc_tasks,
                                              p_evt->id,
                                              p_evt->devicename,
                                              p_evt->filename,
                                              p_evt->repeat_flag,
                                              (void *)p_ssc );
        break;

#if HAVE_LIBMAD
      case ssc_media_mp3:
        error = ssc_create_mp3_playback_task( p_ssc_tasks,
                                              p_evt->id,
                                              p_evt->devicename,
                                              p_evt->filename,
                                              p_evt->repeat_flag,
                                              (void *)p_ssc );
        break;
#endif

      default:
      case ssc_media_invalid:
        ssc_error("%s,%d: media type for filename %s is not supported error!\n",
                  __func__, __LINE__, p_evt->filename );
        break;
      }

      if( error ) {
        ssc_message("%s,%d: could not create tasks id: %s error\n",
                    __func__, __LINE__, p_evt->id );
        (void)ssc_reply_to_sender( p_evt->client,
                                   "could not create task id: %s, device: %s error %d!\n",
                                   p_evt->id, p_evt->devicename, error );
      } else {
        ssc_message("%s,%d: playback task '%s' successfully created!\n",
                    __func__, __LINE__, p_evt->id );
      }
    }
    break;

  case ssc_proc_record_request:
    reschedule = reschedule_if_required( p_ssc_tasks, p_evt );
    if( !reschedule ) {
      error = ssc_create_wav_record_task( p_ssc_tasks,
                                          p_evt->id,
                                          p_evt->devicename,
                                          p_evt->filename,
                                          p_evt->channels,
                                          p_evt->sample_rate,
                                          p_evt->duration,
                                          (void *)p_ssc );

      if( error ) {
        ssc_message("%s,%d: could not create tasks id: %s error\n",
                    __func__, __LINE__, p_evt->id );
        (void)ssc_reply_to_sender( p_evt->client,
                                   "could not create task id: %s, device: %s error %d!\n",
                                   p_evt->id, p_evt->devicename, error );
      } else {
        ssc_message("%s,%d: recording task '%s' successfully created!\n",
                    __func__, __LINE__, p_evt->id );
      }
    }
    break;

  case ssc_proc_pause_request:
    p_proc = ssc_get_task( p_ssc_tasks, p_evt->id );
    if( p_proc ) {
      ssc_message("%s,%d: '%s' pause playback\n",
                  __func__, __LINE__, p_evt->id );

      pthread_mutex_lock( & p_proc->mutex );
      state = p_proc->pause;
      p_proc->pause = 1;
      pthread_mutex_unlock( & p_proc->mutex );
    }
    if( state == 0 ) {
      (void) ssc_reply_to_all( p->p_tcp_cc_client_list,
                               "new state=paused, id=%s, device=%s, filename=%s\n",
                               p_evt->id, p_evt->devicename, p_evt->filename );
    }
    break;

  case ssc_proc_resume_request:
    p_proc = ssc_get_task( p_ssc_tasks, p_evt->id );
    if( p_proc ) {
      ssc_message("%s,%d: '%s' resume playback\n",
                  __func__, __LINE__, p_evt->id );

      pthread_mutex_lock( & p_proc->mutex );
      state = p_proc->pause;
      p_proc->pause = 0;
      pthread_mutex_unlock( & p_proc->mutex );
    }
    if( state == 1 ) {
      (void) ssc_reply_to_all( p->p_tcp_cc_client_list,
                               "new state=running, id=%s, device=%s, filename=%s\n",
                               p_evt->id, p_evt->devicename, p_evt->filename );
    }
    break;

  case ssc_proc_term_app_request:
    error = ssc_close_all_tasks( p_ssc_tasks );
    p->kill_signal = 1;
    break;

  default:
    break;
  }

  return reschedule;
}

void ssc_event_handler( uv_async_t* handle )
{
  ssc_control_t* p = (ssc_control_t *)uv_loop_get_data( handle->loop );
  ssc_events_t*  p_ssc_events = (ssc_events_t *)p->p_ssc_events;
  ssc_evt_t* p_evt;
  int pending_events = get_pending_events_cnt( p_ssc_events );

  pthread_mutex_lock( & p_ssc_events->evt_handling_mutex );
  while( pending_events-- > 0 )
  {
    p_evt = get_event_from_ready_list( p_ssc_events );
    if( p_evt == NULL) {
      /* This condition should actually never happen! */
      break;
    }

    if( handle_evt( p, p_evt ) ) {
      /* event could not processed yet, reschedule it */
      put_event_to_ready_list( p_ssc_events, p_evt );
    } else {
      /* revent has been processed, put the data blob back to the pool */
      put_event_to_pool( p_ssc_events, p_evt );
    }
  }
  pthread_mutex_unlock( & p_ssc_events->evt_handling_mutex );
}

/*! @} */
