/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <mediatype.h>
#include <string.h>
#include <ctype.h>
#include <utils.h>


/*! \file mediatype.c
    \brief utility functions for playback mediatypes

    \addtogroup tasks
    @{
 */

typedef struct {
  ssc_media_t      type;
  const char*      ext;
} ssc_media_ext_t;


static ssc_media_ext_t media_types[] = {
  { ssc_media_invalid, "not supported" },
  { ssc_media_wav, "wav" },
#if HAVE_LIBMAD
  { ssc_media_mp3, "mp3" }
#endif
};


ssc_media_t ssc_media_type( const char* filename )
{
  size_t filename_len;
  size_t ext_len;
  char ext[4];
  ssc_media_t found_medium = ssc_media_invalid;
  size_t last_idx, first_idx, i;

  if( filename == NULL ) {
    return ssc_media_invalid;
  }

  filename_len = strlen(filename);
  if( filename_len < 5 ) {
    return ssc_media_invalid;
  }

  /* find position of separator char */
  last_idx = filename_len - 1;
  for( first_idx = last_idx - 1; first_idx > 0; --first_idx ) {
    if( filename[first_idx] == '.' ) {
      break;
    }
  }

  /* first file extension char is one character to the right */
  ++first_idx;
  ext_len = last_idx - first_idx + 1;

  if( !(first_idx > 1) || !(first_idx < last_idx) ) {
    /* separator char '.' not found */
    return ssc_media_invalid;
  }

  if( !(ext_len < sizeof(ext)) ) {
    /* file extension too big */
    // printf("%s,%d: file extension too big\n", __func__, __LINE__ );
    return ssc_media_invalid;
  }

  for( i=0; i < ext_len; ++i ) {
    ext[i]= tolower( filename[first_idx + i] );
  }
  ext[ext_len]='\0';


  for( i=0; i < sizeof(media_types) / sizeof(ssc_media_ext_t); ++i ) {
    if( strcmp( media_types[i].ext, ext ) == 0 ) {
      found_medium = media_types[i].type;
    }
  }

  return found_medium;
}


int ssc_media_type_test( void )
{
  int error = 0;

  if( ssc_media_type( "testfile.WAV" ) != ssc_media_wav ) {
    printf("%s,%d: incorrect media type detected error!\n", __func__, __LINE__ );
    error = -1;
  }

#if HAVE_LIBMAD
  if( ssc_media_type( "testfile.mp3" ) != ssc_media_mp3 ) {
    printf("%s,%d: incorrect media type detected error!\n", __func__, __LINE__ );
    error = -1;
  }
#endif

  if( ssc_media_type( "testfile.ogg" ) != ssc_media_invalid ) {
    printf("%s,%d: incorrect media type detected error!\n", __func__, __LINE__ );
    error = -1;
  }

#if HAVE_LIBMAD
  if( ssc_media_type( "testfile.ogg.mp3" ) != ssc_media_mp3 ) {
    printf("%s,%d: incorrect media type detected error!\n", __func__, __LINE__ );
    error = -1;
  }
#endif

  if( ssc_media_type( "testfile.mpeg" ) != ssc_media_invalid ) {
    printf("%s,%d: incorrect media type detected error!\n", __func__, __LINE__ );
    error = -1;
  }

  return error;
}

/*! @} */
