/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <config.h>
#if HAVE_LIBMAD

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <mp3_cb.h>
#include <common.h>
#include <olcutils/alloc.h>
#include <utils.h>
#include <log.h>

/*! \file mp3_cb.c
    \brief call backs to read and write audio files in MP3 format

    \addtogroup mp3
    @{
 */

static enum mad_flow input(void *data, struct mad_stream *stream)
{
  mp3_proc_t* p = (mp3_proc_t *)data;
  enum mad_flow retcode = MAD_FLOW_CONTINUE;

  pthread_mutex_lock( & p->mutex );
  if( p->state == mp3_state_not_initialized ) {
    /* clear distinctive state that decoder thread is
       not running, Now it needs to be cleaned up */
    p->state = mp3_state_init;
  }

  if( p->state == mp3_state_terminating ) {
    retcode = MAD_FLOW_STOP;
  }
  else {
    if ( p->state == mp3_state_init ) {
      ssc_message("%s,%d: initial stream provision, addr: %p, size: %lu\n",
                  __func__, __LINE__, p->fdm, p->stat.st_size );
      mad_stream_buffer( stream, p->fdm, p->stat.st_size );
      p->state = mp3_state_filling;
      retcode = MAD_FLOW_CONTINUE;
    } else {
      if( p->repeat ) {
        ssc_message("%s,%d: repeating stream provision, addr: %p, size: %lu\n",
                    __func__, __LINE__, p->fdm, p->stat.st_size );
        mad_stream_buffer( stream, p->fdm, p->stat.st_size );
        retcode = MAD_FLOW_CONTINUE;
      } else {
        if( p->state == mp3_state_playing ) {
          ssc_message("%s,%d: no more data, decoding will terminate\n",
                      __func__, __LINE__ );
          p->state = mp3_state_eof_reached;
        }
        retcode = MAD_FLOW_STOP;
      }
    }
  }

  pthread_mutex_unlock( & p->mutex );

  return retcode;
}

static inline signed int scale(mad_fixed_t sample)
{
  /* round */
  sample += (1L << (MAD_F_FRACBITS - 16));

  /* clip */
  if (sample >= MAD_F_ONE) {
    sample = MAD_F_ONE - 1;
  }
  else if (sample < -MAD_F_ONE) {
    sample = -MAD_F_ONE;
  }

  /* quantize */
  return sample >> (MAD_F_FRACBITS + 1 - 16);
}

static enum mad_flow output( void *data,
                             struct mad_header const *header,
                             struct mad_pcm *pcm )
{
  mp3_proc_t* p = (mp3_proc_t *)data;
  ssc_ahl_config_t* p_config = & p->config;
  t_ssc_mp3_state mp3_playback_state;
  size_t queue_filled = 0, queue_size = 0;
  unsigned int nchannels, nsamples;
  unsigned long frames_decoded;
  mad_fixed_t const *left_ch, *right_ch;
  int result = 0;
  enum mad_flow retcode = MAD_FLOW_CONTINUE;

  nchannels = pcm->channels;
  nsamples  = pcm->length;
  left_ch   = pcm->samples[0];
  right_ch  = pcm->samples[1];

  pthread_mutex_lock( & p->mutex );

  if( p_config->channels != nchannels ||
      p_config->sample_rate != pcm->samplerate ) {
    p->config_update = 1;
    ssc_message( "%s, %d: got configuration update for stream %s, "
                 "channels: %u, rate: %u\n",
                 __func__, __LINE__,
                 p->filename, nchannels, pcm->samplerate );
  }

  p_config->channels = nchannels;
  p_config->sample_rate = pcm->samplerate;
  p_config->bits_per_sample = 16;

  /* check streaming state */
  if( p->state == mp3_state_terminating ) {
      ssc_message( "%s, %d: stop decoding thread\n", __func__, __LINE__ );
      retcode = MAD_FLOW_STOP;
      result = -1;
  }
  if( p->p_queue ) {
    queue_filled = p->p_queue->filled;
    queue_size = p->p_queue->size;
  }
  frames_decoded = (p->frames_decoded)++;

  pthread_mutex_unlock( & p->mutex );


  if( ! result )
  {
    if( frames_decoded == 0 ) {

      pthread_mutex_lock( & p->mutex );
      queue_size = (size_t)g_ssc_mp3_decoding_queue_len *
        (size_t)(pcm->samplerate) * (size_t)nchannels / 1000;
      p->p_queue = sample_queue_init( queue_size );
      if( p->p_queue ) {
        ssc_message("%s,%d: allocated sample queue with capacity: %d!\n",
                  __func__, __LINE__, queue_size );
      }
      else {
        ssc_error("%s,%d: could not allocate sample queue error!\n",
                  __func__, __LINE__ );
        result = -1;
        retcode = MAD_FLOW_STOP;
      }
      pthread_mutex_unlock( & p->mutex );

      if( ! result ) {
        ssc_message( "%s,%d: start playback thread for mp3 file playback\n",
                     __func__, __LINE__ );
        ssc_message( "%s,%d: sample rate: %d, number channels: %d, bitrate: %d\n",
                     __func__, __LINE__,
                     pcm->samplerate, nchannels, header->bitrate );
        ssc_message( "%s,%d: mpeg layer: %d, channel mode: %d, mode ext.: %d, flags: %d\n",
                     __func__, __LINE__,
                     header->layer, header->mode, header->mode_extension, header->flags ) ;
        ssc_message( "%s,%d: frame playing time: %lf, tot. playing time: %lf\n",
                     __func__, __LINE__,
                     (double)(header->duration.seconds) +
                     (double)(header->duration.fraction)/(double)MAD_TIMER_RESOLUTION,
                     (double)(p->stat.st_size) / (double)pcm->samplerate );
      }
    }
  }

  if( ! result )
  {
    /* no more space left in queue, wait until playback has consumed enough samples */
    while( queue_size - queue_filled < nsamples * nchannels ) {
      // ssc_message( "%s, %d: decoding buffer is filled, wait for playback.\n", __func__, __LINE__ );

      /* select for poll frequency two times of playback frame rate */
      usleep( g_ssc_period_duration * 1000 / 2 );
      pthread_mutex_lock( & p->mutex );
      queue_filled = p->p_queue->filled;
      mp3_playback_state = p->state;
      pthread_mutex_unlock( & p->mutex );

      if( mp3_playback_state == mp3_state_terminating ) {
        ssc_message( "%s, %d: stop decoding thread\n", __func__, __LINE__ );
        retcode = MAD_FLOW_STOP;
        result = -1;
        break;
      }
    }
  }

  /* feed the decoded samples in the queue */
  if( ! result )
  {
    pthread_mutex_lock( & p->mutex );
    while (nsamples--) {
      signed int sample;

      /* output sample(s) in 16-bit signed little-endian PCM */

      sample = scale(*left_ch++);
      sample_queue_push_sample( p->p_queue, (int16_t)sample );

      if (nchannels == 2) {
        sample = scale(*right_ch++);
        sample_queue_push_sample( p->p_queue, (int16_t)sample );
      }
    }
    pthread_mutex_unlock( & p->mutex );
  }

  return retcode;
}

static enum mad_flow error(void *data,
                           struct mad_stream *stream,
                           struct mad_frame *frame)
{
  mp3_proc_t* p = (mp3_proc_t *)data;

  ssc_error( "%s, %d: decoding error 0x%04x (%s) at byte offset %u\n",
             __func__, __LINE__,
             stream->error, mad_stream_errorstr(stream),
             stream->this_frame - (unsigned char *)(p->fdm) );

  return MAD_FLOW_CONTINUE;
}



int mp3_release_cb( void* p_arg )
{
  mp3_proc_t* p = (mp3_proc_t *)p_arg;
  int error = 0;
  t_ssc_mp3_state state;
  const int max_term_wait_cycles = 10;
  int term_wait_cycle = 0;

  pthread_mutex_lock( & p->mutex );
  state = p->state;
  pthread_mutex_unlock( & p->mutex );


  if( state != mp3_state_not_initialized ) {
    /* decoder state is running */

    /* so enable its termination flag */
    ssc_message( "%s, %d: terminate mp3 decoding thread ...\n", __func__, __LINE__ );
    pthread_mutex_lock( & p->mutex );
    if( p->state != mp3_state_terminated ) {
      p->state = mp3_state_terminating;
    }
    pthread_mutex_unlock( & p->mutex );


    /* and wait for its termination */
    do {
      pthread_mutex_lock( & p->mutex );
      state = p->state;
      pthread_mutex_unlock( & p->mutex );

      if( !(term_wait_cycle < max_term_wait_cycles) ) {
        ssc_error( "%s,%d: Fatal error, mp3 decoder %s thread did not terminate!\n",
                   __func__, __LINE__, p->filename );
        error = -1;
        break;
      }
      if( term_wait_cycle ) {
        ssc_message( "%s,%d: waited for mp3 decoder %s thread termination %d ms!\n",
                     __func__, __LINE__, p->filename, term_wait_cycle * 100  );
        usleep( 100000 );
      }

      ++term_wait_cycle;
    } while( state != mp3_state_terminated );
  }


  if( term_wait_cycle < max_term_wait_cycles ) {
    ssc_message( "%s,%d: mp3 decoder thread %s terminated!\n",
                 __func__, __LINE__, p->filename );
  }

  if( munmap( p->fdm, p->stat.st_size ) == -1 ) {
      ssc_message("%s,%d could not unmap file %s\n",
                  __func__, __LINE__, p->filename );
  }
  close( p->fd );

  pthread_mutex_destroy( &p->mutex );

  if( p->p_queue ) {
    sample_queue_release( p->p_queue );
  }

  mad_decoder_finish( &p->decoder );

  cul_free( p );

  return error;
}


static void* mp3_decoder_handler( void* arg )
{
  mp3_proc_t* p = (mp3_proc_t *)arg;
  t_ssc_mp3_state state;
  const int max_term_wait_cycles = 20;
  int term_wait_cycle = 0;
  int retcode;

  retcode = mad_decoder_run( &p->decoder, MAD_DECODER_MODE_SYNC );
  if( retcode ) {
    ssc_error( "%s,%d: could not start mp3 decoder thread error: %d\n",
               __func__, __LINE__, retcode );
  }

  do {
    pthread_mutex_lock( & p->mutex );
    state = p->state;
    pthread_mutex_unlock( & p->mutex );

    if( !(term_wait_cycle < max_term_wait_cycles) ) {
      ssc_error( "%s,%d: mp3 decoding queue has not been emptied error!\n",
                 __func__, __LINE__ );
      break;
    }
    if( term_wait_cycle ) {
      ssc_message( "%s,%d: waited for emptying decoding queue for %d ms!\n",
                   __func__, __LINE__, term_wait_cycle * 100  );
      usleep( 100000 );
    }

    ++term_wait_cycle;
  } while( state != mp3_state_terminating && state != mp3_state_terminated );


  pthread_mutex_lock( & p->mutex );
  p->state = mp3_state_terminated;
  pthread_mutex_unlock( & p->mutex );

  ssc_message( "%s,%d: mp3 decoder %s stopped!\n",
               __func__, __LINE__, p->filename );

  return NULL;
}



void* mp3_init_cb( const char* filename,
                   const ssc_atl_dir dir,
                   const int repeat,
                   ssc_ahl_config_t** config_handle )
{
  mp3_proc_t* p;
  int retcode = 0;
  struct sched_param sched_param;
  sched_param.sched_priority = sched_get_priority_max( SCHED_FIFO ) - 1;

  p = cul_malloc( sizeof( mp3_proc_t ) );
  if( p == NULL ) {
    ssc_error("%s,%d: out of memory error!\n", __func__, __LINE__ );
    return NULL;
  }

  memset( p, 0, sizeof( mp3_proc_t ) );
  ssc_strlcpy( p->filename, filename, SSC_MAX_PATH );
  p->repeat = repeat;

  /* This is actually audio hardware dependend, but since the audio period
     is created initially here, we initialize these global conf data here
     as well */
  p->config.max_buf_periods = g_ssc_max_buf_periods;
  p->config.period_duration = g_ssc_period_duration;
  *config_handle = & p->config;

  if( ! retcode ) {
    if (stat( p->filename, &p->stat) == -1 ) {
      ssc_error( "%s,%d: could not get file stats of %s for playback error!\n",
                 __func__, __LINE__, p->filename );
      retcode = -2;
    } else if( p->stat.st_size == 0 ) {
      ssc_error( "%s,%d: size of file %s is 0 error\n",
                 __func__, __LINE__, p->filename );
      retcode = -2;
    }
  }

  if( ! retcode ) {
    p->fd = open( p->filename, O_RDONLY );
    if( p->fd < 0 ) {
      ssc_error( "%s,%d: could not open %s for playback error!\n",
                 __func__, __LINE__, p->filename );
      retcode = -2;
    }
  }

  if( ! retcode ) {
    p->fdm = mmap( 0, p->stat.st_size, PROT_READ, MAP_SHARED, p->fd, 0 );
    if( p->fdm == MAP_FAILED ) {
      ssc_error( "%s,%d: could not create memory map for %s error!\n",
                 __func__, __LINE__, p->filename );
      close( p->fd );
      retcode = -3;
    }
  }

  if( pthread_mutex_init( &p->mutex, 0 ) != 0 ) {
    ssc_error( "%s, %d: could initialize mp3 queue mutex error\n", __func__, __LINE__ );

    close( p->fd );
    if( munmap( p->fdm, p->stat.st_size) == -1 ) {ssc_message("%s,%d could not unmap file %s\n",
                                                              __func__, __LINE__, p->filename );
    }
    sample_queue_release( p->p_queue );
    cul_free( p );
    retcode = -4;
  }

  if( ! retcode ) {
    mad_decoder_init(&p->decoder, p,
                     input, 0 /* header */, 0 /* filter */, output,
                     error, 0 /* message */);
  }

  if( ! retcode ) {
    retcode = pthread_create( &p->handler, NULL, mp3_decoder_handler, p );
    if( ! retcode ) {
      retcode = pthread_detach( p->handler );
    }
    if( retcode ) {
      ssc_error( "%s,%d: creation of decoder handler thread failed with error %d!\n",
                 __func__, retcode );
      close( p->fd );
      if( munmap( p->fdm, p->stat.st_size ) == -1 ) {
        ssc_message("%s,%d could not unmap file %s\n",
                    __func__, __LINE__, p->filename );
      }
    } else {
      int prioerr = pthread_setschedparam( p->handler, SCHED_FIFO, &sched_param );
      if( prioerr ) {
        ssc_error( "%s,%d: could not assign SCHED_FIFO priority error: %d!\n",
                   __func__, __LINE__, prioerr );
      }
    }
  }

  if( retcode ) {
    cul_free( p );
    p = NULL;
  }

  return p;
}


/* for playback */
size_t mp3_producer_cb( void* p_arg, ssc_atl_period_t* p_period )
{
  mp3_proc_t*       p = (mp3_proc_t *)p_arg;
  ssc_ahl_config_t* p_config = & p->config;
  size_t queue_filled, queue_size;
  size_t samples, samples_requested, return_val = 0;
  int16_t sample, *p_samples = (int16_t *) p_period->buf.buf;
  int idx = 0;

  pthread_mutex_lock( & p->mutex );

  if( p->config_update ) {
    /* decoder is running and has at least once called its output cb
       where we the confugration data for the mp3 stream has been set */
    p_period->type = ssc_atl_config_type;
    p_period->p_config = p_config;
    p->config_update = 0;
    p->stream_configured = 1;
    /* configuration data for channels, sample_rate, etc.
       has been done in mp3 decoder output cb already */
    return_val = sizeof( ssc_atl_period_t );
  }
  else if( ! p->stream_configured ) {
    /* nothing decoded yet, so we do not know the sound configuration */
    p_period->type = ssc_atl_data_type;
    return_val = sizeof( ssc_atl_period_t );
  }
  else
  {
    /* stream has been configured, send back audio frames */

    p_period->type = ssc_atl_data_type;

    samples_requested =
      p_config->channels * p_config->sample_rate * p_config->period_duration / 1000;

    /* check streaming state */
    queue_filled = p->p_queue->filled;
    queue_size = p->p_queue->size;
    samples = MIN( queue_filled, samples_requested );

    if( p->state == mp3_state_filling  &&  queue_size - queue_filled < 2000 ) {
      /* when less then 2000 samples capacity left in buffer, start playback */
      p->state = mp3_state_playing;
      ssc_message( "%s,%d: start playing %s", __func__, __LINE__, p->filename );
    }

    if( p->state == mp3_state_playing  &&  samples_requested > queue_filled ) {
      ssc_error( "%s,%d: mp3 decoder for %s delivered only %lu instead of %lu samples, eof or real time problem!",
                 __func__, __LINE__, p->filename, queue_filled, samples_requested );
    }

    /* check whether we have reached end of file condition */
    if( p->state == mp3_state_eof_reached && queue_filled == 0 ) {
      /* end of file reached and playback queue is empty */
      ssc_message( "%s,%d: playback queue empty for %s", __func__, __LINE__, p->filename );
      p->state = mp3_state_terminating;
    }

    /* write samples to playback process */
    if( p->state == mp3_state_playing  ||  p->state == mp3_state_eof_reached ) {
      for( idx=0; idx < samples; ++idx ) {
        sample = sample_queue_pop_sample( p->p_queue );
        p_samples[idx] = sample;
      }
    }
    else if ( p->state == mp3_state_terminating ) {
      ssc_message( "%s,%d: stop playing", __func__, __LINE__ );
      samples = 0;
    }
    else {
      samples = samples_requested;
      memset( p_period->buf.buf, 0, p_period->buf.buf_size );
    }

    /* return frames */
    return_val = samples / p_config->channels;
  }

  ++( p->invocation_cnt );

  pthread_mutex_unlock( & p->mutex );

  return return_val;
}

#endif /* #if HAVE_LIBMAD */

/*! @} */
