/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef MP3_CB_H
#define MP3_CB_H

#include <config.h>
#if HAVE_LIBMAD

#include <ssc_atl.h>
#include <sample_queue.h>
#include <mad.h>

#ifdef __cplusplus
extern "C" {
#endif


/*! \file mp3_cb.h
    \brief call backs to read and write audio files in MP3 format

    \addtogroup mp3
    @{
 */


/*!
 * length of decoding sample queue in milliseconds
 */
#define MP3_DECODING_QUEUE_LEN_MS ( 500 )


/*!
 * MP3 decoding state
 */
typedef enum {
  mp3_state_not_initialized, /*!< decoder thread not running */
  mp3_state_init,            /*!< decoder running, but no samples received yet, stream format still unknown */
  mp3_state_filling,         /*!< decoded samples are received, filling queue with decoded samples */
  mp3_state_playing,         /*!< sample queue is filled, start playback */
  mp3_state_eof_reached,     /*!< end of file reached, play remaining samples */
  mp3_state_terminating,     /*!< decoding thread termination process */
  mp3_state_terminated,      /*!< proc handler has been terminated */
} t_ssc_mp3_state;



/*! internal data context object for reading and decoring audio data from mp3 files */
typedef struct {
  char                        filename[SSC_MAX_PATH];        /*!< filename with mp3 extension */
  ssc_ahl_config_t            config;                        /*!< audio stream configuration data, owned and manged here */
  int                         config_update;                 /*! indicate audio configuration update received from decoder */
  int                         stream_configured;             /*! indicates that audio stream has been configured */
  int                         repeat;                        /*!< continuously repeat when set to true */
  size_t                      bytes_per_period;              /*!< number of bytes per audio period */
  size_t                      invocation_cnt;               /*!< counts number of r/w callback invocation */

  pthread_t                   handler;                      /*!< mp3 decoder thread */
  t_ssc_mp3_state             state;                        /*!< streaming state */
  int                         fd;                           /*!< mp3 file descriptor */
  struct stat                 stat;                         /*!< file stat, currently only required for length */
  void*                       fdm;                          /*!< memory mapped file descriptor */
  struct mad_decoder          decoder;                      /*!< MP3 decoder context */
  pthread_mutex_t             mutex;                        /*!< mutex for access protection */

  t_sample_queue*             p_queue;
  unsigned long               frames_decoded;               /*!< number of decoded frames */
} mp3_proc_t;


/*!
 * initialization of of reader context data for audio files in mp3 format
 *
 * \param filename fully qualified filename, extension .mp3 must be used
 * \param dir not used here
 * \param repeat continuously repeat when set to true
 * \param config_handle returned pointer (handle) to audio stream configuration data
 * \return pointer to mp3 reader context data or NULL in case of error
 */
void*     mp3_init_cb( const char* filename,
                       const ssc_atl_dir dir,
                       const int repeat,
                       ssc_ahl_config_t** config_handle );


/*!
 * release of of reader context data for audio files in mp3 format
 *
 * \param p pointer to wav reader context data or NULL in case of error
 * \return 0 in case of success, otherwise negative error code
 */
int       mp3_release_cb( void* );


/*!
 * producer of audio data read out from audio file in mp3 format
 *
 * \param p pointer to wav reader context data
 * \param p_period pointer to generated audio period data
 * \return number of audio frames read
 */
size_t    mp3_producer_cb( void*, ssc_atl_period_t* );


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #if HAVE_LIBMAD */
#endif /* #ifndef MP3_CB_H */
