/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <olcutils/alloc.h>
#include <sample_queue.h>
#include <log.h>

/*! \file sample_queue.c
    \brief mp3 decoding

    \addtogroup mp3
    @{
 */

t_sample_queue* sample_queue_init( const size_t number_of_samples )
{
  t_sample_queue* p;

  p = (t_sample_queue*) cul_malloc( sizeof(t_sample_queue) );
  if( ! p ) {
    ssc_error( "%s, %d: out of memory error!\n", __func__, __LINE__ );
    return NULL;
  }

  memset( p, 0, sizeof(t_sample_queue) );

  p->buf = cul_malloc( sizeof( int16_t) * number_of_samples );
  if( p->buf == NULL ) {
    ssc_error( "%s, %d: out of memory error!\n", __func__, __LINE__ );
    cul_free( p );
    return NULL;
  }

  p->size = number_of_samples;

  return p;
}

void sample_queue_release( t_sample_queue* p )
{
  if( p ) {
    if( p->buf ) {
      cul_free( p->buf );
    }
    cul_free( p );
  }
}

void sample_queue_push_sample( t_sample_queue* p, const int16_t s )
{
  size_t idx = p->write_idx;

  if( !( p->filled < p->size )) {
    fprintf( stderr, "overflow of sample queue error!\n");
    exit(-1);
  }

  p->buf[idx++] = s;
  if( !(idx < p->size) ) {
    idx = 0;
  }
  p->write_idx = idx;

  if( p->filled < p->size )
    ++(p->filled);
}

int16_t sample_queue_pop_sample( t_sample_queue* p )
{
  int16_t s = 0;
  size_t idx = p->read_idx;

  if( p->filled > 0 ) {
    s = p->buf[idx++];
    if( !(idx < p->size) ) {
      idx = 0;
    }
    p->read_idx = idx;
    --(p->filled);
  }

  return s;
}

/*! @} */
