/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef SAMPLE_QUEUE_
#define SAMPLE_QUEUE_

#include <pthread.h>
#include <stddef.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \file sample_queue.h
    \brief mp3 decoding

    \addtogroup mp3
    @{
 */

/*!
 * sample queue type, used to connect MP3 decoding with WAV playback
 */
typedef struct {
  int16_t*          buf;         /*!< interleaving sample storage (left, right, left right) for stereo, otherwise mono data */
  size_t            read_idx;    /*!< read next sample from this index */
  size_t            write_idx;   /*!< write next sample to this index */
  size_t            filled;      /*!< number of stored overall samples (left+right) stored in buffer */
  size_t            size;        /*!< buffer capacity in samples (left and right) */
} t_sample_queue;


/*!
 * allocate mutex protected sample queue
 *
 * \param number_of_samples capacity in samples (sum of left and right in case of stereo)
 * \return pointer to sample queue structure or NULL in case of error
 */
t_sample_queue* sample_queue_init( const size_t number_of_samples );


/*!
 * release sample queue data structure
 *
 * \param p pointer to allocated sample queue structure
 */
void sample_queue_release( t_sample_queue* p );


/*!
 * push one sample (left or right) to queue
 *
 * ATTENTION: This function will be usually invoked from a loop. Don't forget
 *            to lock/unlock the qeue data structure outside this loop.
 *
 * \param p pointer to sample queue data structure
 * \param s sample value to push to the queue
 */
void sample_queue_push_sample( t_sample_queue* p, const int16_t s );

/*!
 * read one sample (left or right) from the queue
 *
 * ATTENTION: This function will be usually invoked from a loop. Don't forget
 *            to lock/unlock the qeue data structure outside this loop.
 *
 * \param p pointer to sample queue data structure
 * \return sample value to read from the queue
 */
int16_t sample_queue_pop_sample( t_sample_queue* p );


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef SAMPLE_QUEUE_ */
