/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <getopt.h>
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>

#include <olcutils/alloc.h>
#include <olcutils/memtrace.h>
#include <olcutils/revision.h>

#include <ssc.h>
#include <event_handler.h>
#include <ssc_config.h>
#include <utils.h>
#include <revision.h>
#include <log.h>
#include <config.h>


/*! \file ssc.c
    \brief global data context

    \addtogroup control
    @{
 */


static void release_ssc( ssc_t* p )
{
  if( p->p_ssc_events ) {
    ssc_release_events( p->p_ssc_events );
  }

  if( p->p_ssc_control ) {
    release_control( p->p_ssc_control );
  }

  if( p->p_ssc_tasks ) {
    release_ssc_tasks( p->p_ssc_tasks );
  }

  cul_free( p );
}


static ssc_t* alloc_ssc( void )
{
  ssc_t* p = cul_malloc( sizeof( ssc_t ) );

  if( p == NULL ) {
      ssc_error("%s,%d: out of memory error!\n", __func__, __LINE__ );
      return NULL;
  }

  memset( p, 0, sizeof( ssc_t ) );

  p->p_ssc_control = alloc_control();
  if( p->p_ssc_control == NULL ) {
    ssc_error("%s,%d: out of memory error!\n", __func__, __LINE__ );
    release_ssc( p );
    return NULL;
  }

  p->p_ssc_events = ssc_alloc_events( p->p_ssc_control->loop,
                                      (void*)p,
                                      ssc_event_handler,
                                      SSC_MAX_BUF_MSGS );
  if( p->p_ssc_events == NULL ) {
    ssc_error("%s,%d: out of memory error!\n", __func__, __LINE__ );
    release_ssc( p );
    return NULL;
  }

  p->p_ssc_tasks = init_ssc_tasks();
  if( p->p_ssc_tasks == NULL ) {
    ssc_error("%s,%d: out of memory error!\n", __func__, __LINE__ );
    release_ssc( p );
    return NULL;
  }

  /* add reference for event context into control context */
  p->p_ssc_control->p_ssc_events = p->p_ssc_events;

  /* add reference for global context into control context */
  p->p_ssc_control->p_ssc = p;

  return p;
}


int sscd2(void)
{
  cul_allocstat_t allocstat;
  int result = 0, i;
  ssc_t* p;

  memtrace_enable();

  ssc_log_init();
  ssc_message("Simple Sound Controller Daemon started\n" );
  ssc_message("sscd revision: %s\n", g_ssc_revision );
  ssc_message("libcutils revision: %s\n", g_cutillib_revision );
#ifdef HAVE_LIBMAD
  ssc_message("libmad present, mp3 playback is supported\n" );
#else
  ssc_message("libmad NOT present, mp3 playback is NOT supported\n" );
#endif

  ssc_init_config();

  p = alloc_ssc();
  if( p ) {
    /* run main I/O event loop */
    result = uv_run( p->p_ssc_control->loop, UV_RUN_DEFAULT );

    printf("\tclose and release background processes ...\n" );
    release_ssc( p );
  }

  memtrace_disable();

  allocstat = get_allocstat();
  printf("\n\n");
  printf("Memory Allocation Statistics in cutillib functions\n");
  printf("--------------------------------------------------\n");
  printf("       number of open allocations: %ld\n", allocstat.nr_allocs );
  printf("     number of failed allocations: %ld\n", allocstat.nr_allocs_failed );
  printf("  still allocated memory in bytes: %ld\n", allocstat.mem_allocated );
  printf("maximum allocated memory in bytes: %ld\n", allocstat.max_allocated );
  printf("\n");

  if( allocstat.nr_allocs != 0 || allocstat.mem_allocated != 0 )
  {
    fprintf( stderr, "ATTENTION: Some memory is leaking out, configure with memcheck option and check for root cause!\n\n");

    if( allocstat.alloc_trace.addr_table_head > 0 ) {
      fprintf( stderr, "Non-freed Memory Allocations\n");
      fprintf( stderr, "----------------------------\n");
      for( i=0; i<allocstat.alloc_trace.addr_table_head; ++i ) {
        fprintf( stderr, "idx: %.3d, function: %-20.20s, file: %-10.10s, line: %.4d, size: %.4ld, addr: %.5lx\n",
                 i,
                 allocstat.alloc_trace.addr_table[i].dbg.func,
                 allocstat.alloc_trace.addr_table[i].dbg.file,
                 allocstat.alloc_trace.addr_table[i].dbg.line,
                 (unsigned long) (allocstat.alloc_trace.addr_table[i].size),
                 (unsigned long) (allocstat.alloc_trace.addr_table[i].addr) );
      }
      fprintf( stderr, "\n" );
    }
    result = -1;
  }

  // memtrace_print_log( stdout );

  ssc_log_release();
  return result;
}

/*! @} */
