/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef SSC_AHL_H
#define SSC_AHL_H

#include <stdint.h>
#include <config.h>
#include <ssc_config.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \file ssc_ahl.h
    \brief audio hardware abstraction layer

    \addtogroup ahl
    @{
 */

/*! fundamental configuration parameters for all digital audio devices */
typedef struct {
  uint16_t          channels;                     /*!< number of channels */
  uint32_t          sample_rate;                  /*!< sample rate */
  uint16_t          bits_per_sample;              /*!< resolution of one sample in bits */
  uint16_t          max_buf_periods;              /*!< number of periods to be buffered in hw */
  uint16_t          period_duration;              /*!< duration of one period in ms, period_size = period_duration * rate / 1000 */
  uint16_t          duration_in_seconds;          /*!< maximum recording duration */
} ssc_ahl_config_t;


/*! base audio abstraction device type */
typedef struct {
  char              devname[SSC_MAX_ADDR_LEN];    /*!< full qualified audio device file name (audio hw) */
  int               direction;                    /*!< read (playback) or write (capture) */
  ssc_ahl_config_t* p_config;                     /*!< basic audio configuration, owned by transport layer  */
  int               configured;                   /*!< set to 1 when configured, Qualcomm does only allow to do this once */
  void*             data;                         /*!< implementation data */
} ssc_ahl_dev_t;


/*! synonymous to alsa's SND_PCM_STREAM_PLAYBACK */
#define SSC_AHL_PCM_STREAM_PLAYBACK 0

/*! synonymous to alsa's SND_PCM_STREAM_CAPTURE */
#define SSC_AHL_PCM_STREAM_CAPTURE  1

/*!
 * open an abstract audio device
 *
 * \param devname fully qualified audio device filename
 * \param ::SSC_AHL_PCM_STREAM_PLAYBACK for sending or
 *        ::SSC_AHL_PCM_STREAM_CAPTURE for receiving
 * \param set_default_configuration configure stream for 16kHz rate (used for openidle only)
 * \return pointer to abstract audio device or NULL in case of error
 */
ssc_ahl_dev_t* ssc_ahl_open( const char* devname,
                             const int direction,
                             const int set_default_configuration );


/*!
 * close an abstract audio device
 *
 * \param p pointer to abstract audio device to release
 * \return 0 in case of success or negative error code
 */
int ssc_ahl_close( ssc_ahl_dev_t* p );


/*!
 * configure audio hardware with given parameters
 *
 * \param p pointer to abstract audio device to configure
 * \param c pointer to confiration block
 * \return 0 in case of success or negative error code
 */
int ssc_ahl_config( ssc_ahl_dev_t* p, ssc_ahl_config_t* c );


/*!
 * read audio data from hardware
 *
 * \param p pointer to abstract audio device to read from
 * \param p_buf pointer to audio data
 * \param frames number of frames to be read
 * \return number of frames read or negative error code
 */
ssize_t ssc_ahl_pcm_read(  ssc_ahl_dev_t* p, void* p_buf, const ssize_t frames );


/*!
 * write audio data to hardware
 *
 * \param p pointer to abstract audio device to read from
 * \param p_buf pointer to audio data
 * \param frames number of frames to be written
 * \return number of frames written or negative error code
 */
ssize_t ssc_ahl_pcm_write( ssc_ahl_dev_t* p , const void* p_buf, const ssize_t frames );


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef SSC_AHL_H */
