/*
  Simple Sound Controller 2
  Copyright 2023 Otto Linnemann

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, see
  <http://www.gnu.org/licenses/>.
*/

#include <config.h>
#ifdef USE_ALSA_API

#include <string.h>
#include <values.h> /* for MAXLONG */
#include <ssc_ahl.h>
#include <olcutils/alloc.h>
#include <asoundlib.h>
#include <ssc_config.h>
#include <utils.h>
#include <log.h>


/*! \file ssc_alsa.c
  \brief implementation of alsa wrapper for audio hardware abstr. layer

  \addtogroup ahl
  @{
*/


/* --- private implementation --- */

/*! alsa device parameters */
typedef struct {
  ssc_ahl_dev_t*        p_ahl_dev;                        /*!< backrefence to ahl */
  snd_pcm_t*            p_pcm_handle;                     /*!< alsa pcm descriptor */
  snd_pcm_uframes_t     buf_frames;                       /*!< hw buffer size in frames */
  size_t                buf_bytes;                        /*!< hw buffer size in bytes */
  snd_pcm_uframes_t     period_frames;                    /*!< frames per period */
  size_t                period_bytes;                     /*!< bytes per period */
  size_t                bytes_per_frame;                  /*!< bytes per frames */
  snd_pcm_sframes_t     tot_frame_cnt;                    /*!< tot. read or written frames */
} ssc_alsa_t;


/*!
 * configures format of pcm data stream
 *
 * \param pcm_handle alsa pcm handle to configure
 * \param low_latency: when true starts auto play immediately
 * \param bits bits per sample (16)
 * \param channels number of channels (1 mono, 2 stereo, anything else not supported)
 * \param rate (sample rate)
 * \param max_buf_periods  number of periods to be buffered by hw
 * \param period_duration of one period in ms, period_size = period_duration * rate / 1000;
 * \return 0 in case of success, negative value in case of error
 */
static int alsa_set_params(
  ssc_alsa_t *p_ssc_alsa,
  const int low_latency,
  const int bits,
  const int channels,
  const int rate,
  const int max_buf_periods,
  const int period_duration )
{
  snd_pcm_t *pcm_handle = p_ssc_alsa->p_pcm_handle;
  snd_pcm_hw_params_t *hwparams=NULL;
  snd_pcm_sw_params_t *swparams=NULL;
  int dir;
  unsigned int exact_uvalue;
  unsigned long exact_ulvalue;

  int periods = max_buf_periods;
  int periodsize;
  snd_pcm_uframes_t buffersize, start_threshold_periods;
  int err;
  int format;

  snd_pcm_access_t access = (g_ssc_alsa_access_mode == alsa_access_mode_memory_mapped) ?
    SND_PCM_ACCESS_MMAP_INTERLEAVED : SND_PCM_ACCESS_RW_INTERLEAVED;


  /* Allocate the snd_pcm_hw_params_t structure on the stack. */
  snd_pcm_hw_params_alloca(&hwparams);

  /* Init hwparams with full configuration space */
  if (snd_pcm_hw_params_any(pcm_handle, hwparams) < 0) {
    ssc_error("%s,%d: cannot configure alsa pcm device error!\n",
              __func__, __LINE__ );
    return -1;
  }

  if (snd_pcm_hw_params_set_access(pcm_handle, hwparams, access ) < 0) {
    ssc_error("%s,%d: error setting hw param access!\n",
              __func__, __LINE__ );
    /* some embedded alsa devices do not support this parameter but need
       nevertheless to be configured with the following parameters to
       stay operational. We do not exit the function for this reason in
       case error here */
    /* return -1; */
  }
  /* Set sample format */
  format=SND_PCM_FORMAT_S16;
  if (snd_pcm_hw_params_set_format(pcm_handle, hwparams, format) < 0) {
    ssc_error("%s,%d: error setting sample format!\n",
              __func__, __LINE__ );
    return -1;
  }
  /* Set number of channels */
  if (snd_pcm_hw_params_set_channels(pcm_handle, hwparams, channels) < 0) {
    ssc_error("%s,%d: error setting channels!\n",
              __func__, __LINE__ );
    return -1;
  }
  /* Set sample rate. If the exact rate is not supported */
  /* by the hardware, use nearest possible rate.         */
  exact_uvalue=rate;
  dir=0;
  if ((err=snd_pcm_hw_params_set_rate_near(pcm_handle, hwparams, &exact_uvalue, &dir))<0){
    ssc_error("%s,%d: error setting rate to %i:%s!\n",
              __func__, __LINE__, rate, snd_strerror(err) );
    return -1;
  }
  if (dir != 0) {
    ssc_message("%s,%d: the rate %d Hz is not supported by your hardware! "
                "==> Using %d Hz instead.\n",
                __func__, __LINE__, rate, exact_uvalue);
  }
  /* choose greater period size when rate is high */
  periodsize=period_duration * rate / 1000;

  /* Set buffer size (in frames). The resulting latency is given by */
  /* latency = periodsize * periods / (rate * bytes_per_frame)     */
  /* set period size */
  exact_ulvalue=periodsize;
  dir=0;
  if (snd_pcm_hw_params_set_period_size_near(pcm_handle, hwparams, &exact_ulvalue, &dir) < 0) {
    ssc_error("%s,%d: error setting period size!\n",
              __func__, __LINE__ );
    return -1;
  }
  if (dir != 0) {
    ssc_message("%s,%d: The period size %d is not supported by your hardware! "
                "==> Using %d instead.\n",
                __func__, __LINE__,
                periodsize, (int)exact_ulvalue );
  }
  ssc_message("%s,%d: periodsize:%d using %d\n",
              __func__, __LINE__, periodsize, (int)exact_ulvalue );
  periodsize=exact_ulvalue;
  /* Set number of periods. Periods used to be called fragments. */
  exact_uvalue=periods;
  dir=0;
  if (snd_pcm_hw_params_set_periods_near(pcm_handle, hwparams, &exact_uvalue, &dir) < 0) {
    ssc_error("%s,%d: error setting periods!\n",
              __func__, __LINE__ );
    return -1;
  }
  ssc_error("%s,%d: period:%d using %d.\n",
            __func__, __LINE__, periods, exact_uvalue);
  if (dir != 0) {
    ssc_message("%s,%d: the number of periods %d is not supported by your hardware! "
                "==> Using %d instead.\n",
                __func__, __LINE__, periods, exact_uvalue);
  }
  periods = (int)exact_uvalue;

  /* Apply HW parameter settings to */
  /* PCM device and prepare device  */
  if ((err=snd_pcm_hw_params(pcm_handle, hwparams)) < 0) {
    ssc_error("%s,%d: error setting HW params:%s!\n",
              __func__, __LINE__, snd_strerror(err) );
    return -1;
  }

  if (snd_pcm_hw_params_get_buffer_size(hwparams, &buffersize)<0){
    buffersize=0;
    ssc_message("%s,%d: could not obtain hw buffer size!\n",
                __func__, __LINE__ );
  }

  /*prepare sw params */
  if( low_latency ) {
    start_threshold_periods = 1;
  } else {
    start_threshold_periods = periods;
  }

  snd_pcm_sw_params_alloca(&swparams);
  snd_pcm_sw_params_current(pcm_handle, swparams);
  ssc_message("%s,%d: periodsize=%i, buffersize=%i\n",
              __func__, __LINE__, (int) periodsize,(int)buffersize );

  p_ssc_alsa->bytes_per_frame = channels * 2;
  p_ssc_alsa->buf_bytes = buffersize * p_ssc_alsa->bytes_per_frame;
  p_ssc_alsa->period_frames = periodsize;
  p_ssc_alsa->period_bytes = periodsize * p_ssc_alsa->bytes_per_frame;
  p_ssc_alsa->buf_frames = buffersize;

  if((err=snd_pcm_sw_params_get_start_threshold(swparams, &exact_ulvalue)) < 0 ){
    ssc_error("%s,%d: error getting start threshold:%s!\n",
              __func__, __LINE__, snd_strerror(err) );
  } else {
    ssc_message("%s,%d: the default start threshold is:%lu.\n",
                __func__, __LINE__, exact_ulvalue );
  }

  if ((err=snd_pcm_sw_params_set_start_threshold( pcm_handle, swparams, start_threshold_periods * periodsize ))<0){
    ssc_error("%s,%d: error setting start threshold:%s!\n",
              __func__, __LINE__, snd_strerror(err) );
  }

  if ((err=snd_pcm_sw_params_get_start_threshold( swparams, &exact_ulvalue))<0){
    ssc_error("%s,%d: error getting start threshold:%s\n",
              __func__, __LINE__, snd_strerror(err) );
  } else {
    ssc_message("%s,%d: the adjusted start threshold is:%lu\n",
                __func__, __LINE__, exact_ulvalue);
  }

  /* changed stop threshold to maxlong which - hopefully - should disable complete filling
     of alsa fifo buffers (periods) on MDM9615 with respect to minimal latency.
     2013-02-19 @peiker/ol */
  if ((err=snd_pcm_sw_params_set_stop_threshold(pcm_handle, swparams, MAXLONG))<0){
    ssc_error("%s,%d: error setting stop threshold:%s!\n",
              __func__, __LINE__, snd_strerror(err) );
  }
  if ((err=snd_pcm_sw_params(pcm_handle, swparams))<0){
    ssc_error("%s,%d: error setting SW params:%s!\n",
              __func__, __LINE__, snd_strerror(err) );
    return -1;
  }

  return 0;
}



/* --- public API --- */

int ssc_ahl_close( ssc_ahl_dev_t* p )
{
  ssc_alsa_t* p_ssc_alsa = ( ssc_alsa_t * ) p->data;

  if( p_ssc_alsa ) {
    if( p_ssc_alsa->p_pcm_handle ) {
      snd_pcm_close( p_ssc_alsa->p_pcm_handle );
    }

    cul_free( p_ssc_alsa );
  }

  cul_free( p );

  return 0;
}

ssc_ahl_dev_t* ssc_ahl_open( const char* device_name,
                             const int direction,
                             const int set_default_configuration )
{
  ssc_ahl_dev_t* p;
  ssc_alsa_t* p_ssc_alsa;
  snd_pcm_stream_t dir = (direction == SSC_AHL_PCM_STREAM_PLAYBACK) ?
    SND_PCM_STREAM_PLAYBACK :
    SND_PCM_STREAM_CAPTURE;
  int error;

  p = (ssc_ahl_dev_t *) cul_malloc( sizeof( ssc_ahl_dev_t ) );
  if( p == NULL ) {
    ssc_error("%s,%d: out of memory error!\n", __func__, __LINE__ );
    return NULL;
  }

  memset( p, 0, sizeof( ssc_ahl_dev_t ) );
  p->direction = direction;

  p->data = (ssc_alsa_t *) cul_malloc( sizeof( ssc_alsa_t ) );
  if( p->data == NULL ) {
    ssc_error("%s,%d: out of memory error!\n", __func__, __LINE__ );
    cul_free( p );
    return NULL;
  }

  memset( p->data, 0, sizeof( ssc_alsa_t ) );
  ssc_strlcpy( p->devname, device_name, SSC_MAX_DEVNAME_LEN );
  p_ssc_alsa = ( ssc_alsa_t * ) p->data;
  p_ssc_alsa->p_ahl_dev = p;

  error = snd_pcm_open( & p_ssc_alsa->p_pcm_handle, device_name, dir, 0 );
  if( error < 0) {
    ssc_error( "%s,%d: could not open alsa pcm device %s error: '%s'\n",
               __func__, __LINE__, device_name, snd_strerror(error) );
  }

  if( set_default_configuration )
  {
    /*
     * Only do  the configuration  for the 'openidle'  request. On  Qualcomm MDM
     * SoC's obviously a crash inside the alsa library itself occurs when we try
     * to  reconfigure a  pcm handle.  So in  case we  always trigger  a default
     * configuration here  we will later on  experience a crash when  the stream
     * needs to be configured  with actual audio setup in use.  When we open the
     * handle to just start streaming telephony audio voice data inside the dsp,
     * Qualcomm stated in various support cases that we actual need to configure
     * and start  the stream with  a sample  rate of 16kHz.  So we only  do this
     * then.
     */
    if( ! error ) {
      error = alsa_set_params(
        p_ssc_alsa,
        ((dir==SND_PCM_STREAM_PLAYBACK) ? 0 : 1), /* legacy setup, don't touch it! */
        16,
        1,
        16000,
        g_ssc_max_buf_periods,
        g_ssc_period_duration
        );

      p->configured = 1;

      if( error ) {
        ssc_error("%s,%d: could not set parameters for device %s, "
                  " it might not work as intended!\n",
                  __func__, __LINE__, device_name );
      }

      /* ignore error in asla_set_params, some actions always fail */
      error = 0;
    }

    if( ! error ) {
      error = snd_pcm_prepare( p_ssc_alsa->p_pcm_handle );
      if( error ) {
        ssc_error( "%s,%d: could not prepare alsa pcm device %s error\n",
                   __func__, __LINE__, device_name );
      }
    }

    if( ! error ) {
      error = snd_pcm_start( p_ssc_alsa->p_pcm_handle );
      if( error ) {
        ssc_error( "%s,%d: could not start alsa pcm device %s error\n",
                   __func__, __LINE__, device_name );
      }
    }
  }

  if( error < 0) {
    ssc_ahl_close( p );
    p = NULL;
  }

  return p;
}


/*! only when AHL_ALLOW_RECONFIGURATION is defined
  allow to reconfigure abstract audio device. Otherwise throw an error. */
// #define AHL_ALLOW_RECONFIGURATION

int ssc_ahl_config( ssc_ahl_dev_t* p, ssc_ahl_config_t* p_ssc_ahl_config )
{
  ssc_alsa_t* p_ssc_alsa = ( ssc_alsa_t * ) p->data;
  int error = 0;

  p->p_config = p_ssc_ahl_config;

  ssc_message("%s,%d: setup alsa with -> bits_per_sample: %u"
              ", channels: %u, sample_rate: %u, max_buf_periods: %u, period_duration: %u\n",
              __func__, __LINE__,
              p_ssc_ahl_config->bits_per_sample,
              p_ssc_ahl_config->channels,
              p_ssc_ahl_config->sample_rate,
              p_ssc_ahl_config->max_buf_periods,
              p_ssc_ahl_config->period_duration );

#ifdef AHL_ALLOW_RECONFIGURATION
  /* reconfiguration crashes on Qualcomm MDM SoC's, don't do it!
     refer to ssc_ahl_open() for more detailed explanation. */
  if( ! error ) {
    /* drop frames in buffer and stop playback or recording in driver */
    error = snd_pcm_drop( p_ssc_alsa->p_pcm_handle );
    if( error ) {
      ssc_error( "%s,%d: could not stop alsa pcm device %s error\n",
                 __func__, __LINE__, p->devname );
    }
  }
#else
  /* no reconfiguration supported, required for Qualcomm MDM SoC's */
  if( p->configured ) {
    ssc_error("%s,%d: device %s is already configured, can be done only once!\n",
              __func__, __LINE__, p->devname );
    error = 1;
  }
#endif

  if( ! error ) {
    error = alsa_set_params(
      p_ssc_alsa,
      0 /* low_latency */,
      p_ssc_ahl_config->bits_per_sample,
      p_ssc_ahl_config->channels,
      p_ssc_ahl_config->sample_rate,
      p_ssc_ahl_config->max_buf_periods,
      p_ssc_ahl_config->period_duration );

    if( error ) {
      ssc_error( "%s,%d: could configure pcm device %s error\n",
                 __func__, __LINE__, p->devname );
    } else {
      p->configured = 1;
    }
  }

  if( ! error ) {
    error = snd_pcm_prepare( p_ssc_alsa->p_pcm_handle );
    if( error ) {
      ssc_error( "%s,%d: could not prepare alsa pcm device %s error\n",
                 __func__, __LINE__, p->devname );
    }
  }

  if( ! error ) {
    error = snd_pcm_start( p_ssc_alsa->p_pcm_handle );
    if( error ) {
      ssc_error( "%s,%d: could not start alsa pcm device %s error\n",
                 __func__, __LINE__, p->devname );
    }
  }

  return error;
}



/*
 * Underrun and suspend recovery
 * taken from http://www.alsa-project.org/alsa-doc/alsa-lib/_2test_2pcm_8c-example.html
 */
static int xrun_recovery( snd_pcm_t *handle, int err )
{
  const int verbose = 1;

  if (verbose)
    ssc_message("%s,%d: err: %s\n", __func__, __LINE__,
                (err == -EPIPE) ? "epipe" : ((err == -ESTRPIPE) ? "estrpipe" : "unrecoverable"));
  if (err == -EPIPE) {    /* under-run */
    err = snd_pcm_prepare(handle);
    if (err < 0)
      ssc_error("%s,%d: can't recovery from underrun, prepare failed: %s\n",
                __func__, __LINE__, snd_strerror(err));
    return 0;
  } else if (err == -ESTRPIPE) {
    while ((err = snd_pcm_resume(handle)) == -EAGAIN)
      sleep(1);       /* wait until the suspend flag is released */
    if (err < 0) {
      err = snd_pcm_prepare(handle);
      if (err < 0)
        ssc_error("%s,%d: can't recovery from suspend, prepare failed: %s\n",
                  __func__, __LINE__, snd_strerror(err));
    }
    return 0;
  }
  return err;
}


ssize_t ssc_ahl_pcm_read( ssc_ahl_dev_t* p, void* p_buf, const ssize_t frames )
{
  ssc_alsa_t* p_ssc_alsa = ( ssc_alsa_t * ) p->data;
  snd_pcm_t * pcm_handle = p_ssc_alsa->p_pcm_handle;
  ssize_t frames_to_read = frames, frames_read;
  snd_pcm_sframes_t (*ssc_pcm_read)( snd_pcm_t *pcm,
                                     void *buffer,
                                     snd_pcm_uframes_t size );
  int alsa_error = 0;

  if(g_ssc_alsa_access_mode == alsa_access_mode_memory_mapped) {
    ssc_pcm_read = snd_pcm_mmap_readi;
  } else {
    ssc_pcm_read = snd_pcm_readi;
  }

  while( frames_to_read > 0 )
  {
    frames_read = (*ssc_pcm_read)( pcm_handle, p_buf, frames_to_read );
    if( frames_read == -EAGAIN ) {
      continue;
    } else if( frames_read < 0 ) {
      if( xrun_recovery( pcm_handle, frames_read ) < 0 ) {
        ssc_error("%s,%d: pcm data read error in frame %ld!\n",
                  __func__, __LINE__, p_ssc_alsa->tot_frame_cnt );
        alsa_error = -1;
        break;
      }
      ssc_error("%s,%d: pcm frame skipped %ld!\n",
                __func__, __LINE__, p_ssc_alsa->tot_frame_cnt );
      break; /* skip one period */
    } else {
      frames_to_read -= frames_read;
      p_ssc_alsa->tot_frame_cnt += frames_read;
    }
  }

  if( alsa_error ) {
    return alsa_error;
  } else {
    return frames - frames_to_read;
  }
}


ssize_t ssc_ahl_pcm_write( ssc_ahl_dev_t* p, const void* p_buf, const ssize_t frames )
{
  ssc_alsa_t* p_ssc_alsa = ( ssc_alsa_t * ) p->data;
  snd_pcm_t * pcm_handle = p_ssc_alsa->p_pcm_handle;
  ssize_t frames_to_write = frames, frames_written;
  snd_pcm_sframes_t (*ssc_pcm_write)(  snd_pcm_t* pcm,
                                       const void* buffer,
                                       snd_pcm_uframes_t size );
  int alsa_error = 0;

  if(g_ssc_alsa_access_mode == alsa_access_mode_memory_mapped) {
    ssc_pcm_write = snd_pcm_mmap_writei;
  } else {
    ssc_pcm_write = snd_pcm_writei;
  }

  while( frames_to_write > 0 ) {
    frames_written = (ssize_t)
      (*ssc_pcm_write)( pcm_handle, p_buf, frames_to_write );

    if( frames_written == -EAGAIN ) {
      continue;
    } else if( frames_written < 0 ) {
      if( xrun_recovery( pcm_handle, frames_written ) < 0 ) {
        ssc_error("%s, %d: pcm data write error in frame %ld!\n",
                  __func__, __LINE__, p_ssc_alsa->tot_frame_cnt );
        alsa_error = frames_written;
        break;
      }
      ssc_error("%s, %d: pcm frame skipped %ld!\n",
                __func__, __LINE__, p_ssc_alsa->tot_frame_cnt );
      break; /* skip one period */
    } else {
      frames_to_write -= frames_written;
      p_ssc_alsa->tot_frame_cnt += frames_written;
    }
  }

  if( alsa_error ) {
    return alsa_error;
  } else {
    return frames - frames_to_write;
  }
}

/*! @} */

#endif /* #ifdef USE_ALSA_API */
