/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <string.h>
#include <ssc_atl.h>
#include <olcutils/alloc.h>
#include <log.h>


/*! \file ssc_atl.c
    \brief abstract transport layer

    \addtogroup atl
    @{
 */


void release_ssc_atl_period( ssc_atl_period_t* p )
{
  if( p->buf.buf ) {
    cul_free( p->buf.buf );
  }

  cul_free( p );
}


static void adjust_atl_buf_config( ssc_atl_period_t* p,
                                   const ssc_ahl_config_t* p_config )
{
  p->period_size = p_config->sample_rate * p_config->period_duration / 1000;
  p->buf.buf_size = p->period_size * p_config->channels * p_config->bits_per_sample / 8;
}


ssc_atl_period_t* init_ssc_atl_period( ssc_ahl_config_t* p_config )
{
  ssc_atl_period_t* p;

  p = cul_malloc( sizeof( ssc_atl_period_t ) );
  if( p == NULL ) {
    ssc_error( "%s,%d: out of memory error!\n", __func__, __LINE__ );
    return NULL;
  }

  memset( p, 0, sizeof( ssc_atl_period_t ) );

  p->p_config = p_config;
  adjust_atl_buf_config( p, p->p_config );

  p->buf.buf = cul_malloc( p->buf.buf_size );
  if( p->buf.buf == NULL ) {
    ssc_error( "%s,%d: out of memory error!\n", __func__, __LINE__ );
    release_ssc_atl_period( p->buf.buf );
    return NULL;
  }

  return p;
}

int resize_ssc_atl_period( ssc_atl_period_t* p , ssc_ahl_config_t* p_config )
{
  int error = 0;

  p->p_config = p_config;
  adjust_atl_buf_config( p, p->p_config );

  cul_free( p->buf.buf );
  p->buf.buf = cul_malloc( p->buf.buf_size );
  if( p->buf.buf == NULL ) {
    ssc_error( "%s,%d: out of memory error!\n", __func__, __LINE__ );
    release_ssc_atl_period( p->buf.buf );
    p->buf.buf_size = 0;
    error = -1;
  }

  return error;
}

/*! @} */
