/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef SSC_ATL_H
#define SSC_ATL_H

#include <ssc_ahl.h>

#ifdef __cplusplus
extern "C" {
#endif


/*! \file ssc_atl.h
    \brief abstract transport layer

    \addtogroup atl
    @{
 */

/*! audio frame data, always specified as interleaved samples */
typedef struct {
  void*                   buf;                    /*!< pointer to sample buffer for one 'alsa' period of data */
  size_t                  buf_size;               /*!< bufsize */
} ssc_atl_buf_t;


/*!
 * type of provided audio data or (re)configuration or audio frames
 */
typedef enum {
  ssc_atl_config_type,                            /*!< configuration type specifier */
  ssc_atl_data_type,                              /*!< audio frame type specifier */
} ssc_atl_period_type;


/*!
 * audio event type, encapsulates (re)configuration or frame buffer
 */
typedef struct {
  ssc_atl_period_type      type;                   /*!< type of the provided data, configuration or samples */
  ssc_ahl_config_t*        p_config;               /*!< pointer to configuration data block, initialized by .ssc_player_open(), ssc_recoreder_open() */
  ssc_atl_buf_t            buf;                    /*!< audio sample data buffer */
  size_t                   period_size;            /*!< buffer size in terms of number of audio frames per period */
  int                      error_code;             /*!< error code */
  char                     error_msg[SSC_MAX_MSG_LEN]; /*!< error message */
} ssc_atl_period_t;

/*!
 * typedef which specifies whether we read or write to storage device
 */
typedef enum {
  ssc_atl_read,                                    /*!< read sample values for playback */
  ssc_atl_write,                                   /*!< write sample values for recording */
} ssc_atl_dir;


/* --- public API for initialization, releasing and reconfiguration --- */

/*!
 * initialize memory area to hold one period of audio frames
 *
 * \param p_config borrowed pointer to audio configuration data, managed from caller
 * \return pointer to period buffer or NULL in case of error
 */
ssc_atl_period_t* init_ssc_atl_period( ssc_ahl_config_t* p_config );


/*!
 * release audio period memory area
 *
 * \param p pointer to period memory area to be released
 */
void release_ssc_atl_period( ssc_atl_period_t* p );


/*!
 * change audio period buffer size and configuration
 *
 * \param p pointer to period memory area to be resized
 * \param p_config borrowed pointer to audio configuration data, managed from caller
 * \return 0 in case of success or negative error code
 */
int resize_ssc_atl_period( ssc_atl_period_t* p , ssc_ahl_config_t* p_config );



/* forward declaration */
struct ssc_proc_handle_s;


/* --- abstract interface, implemented elsewhere --- */

/*!
 * abstract data source/sink init function ( implemented in wav/mp3 callbacks )
 *
 * \param filename of audio file to be played back or recorded
 * \param dir read (playback) or write (record)
 * \param repeat continous repeat playback from beginning
 * \param handle to configuration data owned and managed by transport layer
 * \return pointer to transport layer instance or NULL in case of error
 */
typedef void*   (* ssc_atl_init_cb_t)( const char* filename,
                                       const ssc_atl_dir dir,
                                       const int repeat,
                                       ssc_ahl_config_t** config_handle );


/*!
 * abstract data provider release function ( implemented in wav/mp3 callbacks )
 *
 * \param p pointer to transport layer instance to be released
 * \return 0 in case of success ortherwise negative error code
 */
typedef int     (* ssc_atl_release_cb_t)( void * p );


/*!
 * abstract data producer used for playback ( implemented in wav/mp3 callbacks )
 *
 * \param p pointer to atl instance data
 * \param p_perod pointer to atl audio period buffer
 * \return number of frames produced (read from storage)
 */
typedef size_t  (* ssc_atl_producer_cb_t)( void * p, ssc_atl_period_t* p_period );


/*!
 * abstract data consumer used for recording ( implemented in wav/mp3 callbacks )
 *
 * \param p pointer to atl instance data
 * \param p_period pointer to atl audio period buffer
 * \return number of frames consumed (written to storage)
 */
typedef size_t  (* ssc_atl_consumer_cb_t)( void *p, ssc_atl_period_t* p_period );


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef SSC_ATL_H */
