/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <limits.h>
#include <unistd.h>
#include <sys/types.h>
#include <pwd.h>
#include <ssc_config.h>
#include <utils.h>
#include <log.h>
#include <olcutils/alloc.h>
#include <olcutils/cfg_string.h>


/*! \file ssc_config.c
    \brief server configuration

    \addtogroup config
    @{
 */


char g_ssc_server_ip_address[SSC_MAX_ADDR_LEN] = { "127.0.0.1" };
int  g_ssc_server_ip_port = 5042;

char g_ssc_server_socket_filename[SSC_MAX_ADDR_LEN] = "/tmp/ssc.sock";

int  g_ssc_server_max_connections = 10;

int g_ssc_max_buf_periods = 4;
int g_ssc_period_duration = 100;
int g_ssc_playback_trailing_silence_frames = 0;
t_ssc_alsa_access_mode g_ssc_alsa_access_mode = alsa_access_mode_memory_mapped;
int g_ssc_mp3_decoding_queue_len = 500;


static void* free_string_val( void* p )
{
  string_release( (string_t*)p );
  return NULL;
}

static int string2int( const string_t* s, int* p_i, const int minval, const int maxval )
{
  char tmp_cstring[30];
  char* end = 0;
  long l;
  int retcode = 0;

  string_tmp_cstring_from( s, tmp_cstring, sizeof( tmp_cstring ) );
  l = strtol( tmp_cstring, &end, 10 );
  if( l == LONG_MIN || l == LONG_MAX ) {
    retcode = -1;
  } else  {
    if( l >= minval && l <=maxval ) {
      *p_i = (int)l;
    } else {
      retcode = -2;
    }
  }

  return retcode;
}

int ssc_init_config(void)
{
  struct passwd* pw = getpwuid(getuid());
  char local_conf_path[SSC_MAX_PATH];
  char global_conf_path1[SSC_MAX_PATH], global_conf_path2[SSC_MAX_PATH];
  char* pUsedConfFileName;
  char* p_conf_data;
  int conf_data_len;
  FILE* fp;

  int retcode = 0;

  p_conf_data = (char *) cul_malloc( SSC_MAX_CONF_SIZE );
  if( ! p_conf_data ) {
    ssc_error("%s: out of memory error!\n", __func__ );
    return -1;
  }

  memset( p_conf_data, 0, SSC_MAX_CONF_SIZE );


  /* intialize path to local and global conf files */
  snprintf( local_conf_path, sizeof(local_conf_path), "%s/.ssc.rc", pw ? pw->pw_dir : "." );
  snprintf( global_conf_path1, sizeof(global_conf_path1), "/etc/ssc.rc" );
  snprintf( global_conf_path2, sizeof(global_conf_path2), "/usr/local/etc/ssc.rc" );

  fp = fopen(local_conf_path, "r" );
  pUsedConfFileName = local_conf_path;
  if( fp ==  NULL )
  {
    fp = fopen(global_conf_path1, "r" );
    pUsedConfFileName = global_conf_path1;

    if( fp == NULL )
    {
      fp = fopen(global_conf_path2, "r" );
      pUsedConfFileName = global_conf_path2;
    }
  }

  if( fp == NULL )
  {
    ssc_error( "%s: configuration data neither at %s nor at %s or %s given!\n",
               __func__, local_conf_path, global_conf_path1, global_conf_path2 );
    cul_free( p_conf_data );
    return -1;
  } else {
    ssc_message( "%s, %d: use configuration file %s\n", __func__, __LINE__, pUsedConfFileName );
  }

  conf_data_len = fread( p_conf_data, 1, SSC_MAX_CONF_SIZE-1 /*-1 for null term. ensurance! */, fp );
  if( conf_data_len > 0 ) {
    string_t* s = string_new_from( p_conf_data );
    hm_t* params;
    hm_leaf_node_t* ln;

    params = cfgstring_parse( s );

    ln = hm_find( params, cstring_hash( "server-ip-address" ) );
    if( ln ) {
      string_tmp_cstring_from( ln->val, g_ssc_server_ip_address, sizeof( g_ssc_server_ip_address ) );
      ssc_message("%s: overwrite default IP address with %s\n", __func__, g_ssc_server_ip_address );
    }

    ln = hm_find( params, cstring_hash( "server-ip-port" ) );
    if( ln ) {
      if( ! string2int( ln->val, & g_ssc_server_ip_port, 0, 65535 ) ) {
        ssc_message("%s: overwrite default IP port with %d\n", __func__, g_ssc_server_ip_port );
      } else {
        ssc_error("%s: could not parse IP port argument error!\n", __func__ );
      }
    }

    ln = hm_find( params, cstring_hash( "server-socket-filename" ) );
    if( ln ) {
      string_tmp_cstring_from( ln->val, g_ssc_server_socket_filename,
                               sizeof( g_ssc_server_socket_filename ) );
      ssc_message("%s: overwrite default UDS filename with %s\n", __func__, g_ssc_server_socket_filename );
    }

    ln = hm_find( params, cstring_hash( "server-max-connections" ) );
    if( ln ) {
      if( ! string2int( ln->val, & g_ssc_server_max_connections, 0, 20 ) ) {
        ssc_message("%s: overwrite default nr of max. connections %d\n",
                    __func__, g_ssc_server_max_connections );
      } else {
        ssc_error("%s: could not parse nr of max. socket connections error!\n", __func__ );
      }
    }

    ln = hm_find( params, cstring_hash( "max-buf-periods" ) );
    if( ln ) {
      if( ! string2int( ln->val, & g_ssc_max_buf_periods, 4, 10 ) ) {
        ssc_message("%s: overwrite default number of buffered periods with %d\n",
                    __func__, g_ssc_max_buf_periods );
      } else {
        ssc_error("%s: could not parse number of buffered periods error!\n", __func__ );
      }
    }

    ln = hm_find( params, cstring_hash( "period-duration" ) );
    if( ln ) {
      if( ! string2int( ln->val, & g_ssc_period_duration, 50, 200 ) ) {
        ssc_message("%s: overwrite default period duration with %d\n",
                    __func__, g_ssc_period_duration );
      } else {
        ssc_error("%s: could not parse number of buffered periods error!\n", __func__ );
      }
    }

    ln = hm_find( params, cstring_hash( "playback-trailing-silence-frames" ) );
    if( ln ) {
      if( ! string2int( ln->val, & g_ssc_playback_trailing_silence_frames, 0, 480000 ) ) {
        ssc_message("%s: overwrite default trailing silence frames with %d\n",
                    __func__, g_ssc_playback_trailing_silence_frames );
      } else {
        ssc_error("%s: could not parse trailing silence frames error!\n", __func__ );
      }
    }

    ln = hm_find( params, cstring_hash( "alsa-access-mode" ) );
    if( ln ) {
      uint32_t hashval = string_hash( ln->val );
      if( hashval == cstring_hash( "memory-mapped" ) ) {
        ssc_message("%s: default alsa access mode with: memory mapped\n", __func__ );
        g_ssc_alsa_access_mode = alsa_access_mode_memory_mapped;
      } else if( hashval == cstring_hash( "direct" ) ) {
        ssc_message("%s: default alsa access mode with: direct\n", __func__ );
        g_ssc_alsa_access_mode = alsa_access_mode_direct;
      } else {
        ssc_error("%s: could not parse alsa access mode error!\n", __func__ );
      }
    }

    ln = hm_find( params, cstring_hash( "mp3-decoding-queue-len" ) );
    if( ln ) {
      if( ! string2int( ln->val, & g_ssc_mp3_decoding_queue_len, 300, 10000 ) ) {
        ssc_message("%s: overwrite default mp3 decoding queue len with %d\n",
                    __func__, g_ssc_mp3_decoding_queue_len );
      } else {
        ssc_error("%s: could not parse mp3 decoding queue len error!\n", __func__ );
      }
    }

    string_release( s );
    hm_free_deep( params, 0, free_string_val );

  } else {
    ssc_error( "%s: could not read configuration data error!\n" );
  }

  cul_free( p_conf_data );
  fclose( fp );
  return retcode;
}

/*! @} */
