/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef SSC_CONFIG_H
#define SSC_CONFIG_H

#include <olcutils/refstring.h>
#include <olcutils/hashmap.h>
#include <common.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \file ssc_config.h
    \brief server configuration

    \addtogroup config
    @{
 */


/*! maximum length for server addresses according to posix standards */
#define SSC_MAX_ADDR_LEN      108


/*! maximum size of configuration file */
#define SSC_MAX_CONF_SIZE     4096


/*!
 * served IP address
 */
extern char g_ssc_server_ip_address[SSC_MAX_ADDR_LEN];


/*!
 * served IP port
 */
extern int  g_ssc_server_ip_port;


/*!
 * served socket file or empty string when no socket file shall be served
 */
extern char g_ssc_server_socket_filename[SSC_MAX_ADDR_LEN];

/*!
 * maximum allowed connections via Unix Domain Sockets and/or TCP
 * or zero for disabling UDS communication
 */
extern int  g_ssc_server_max_connections;


/*!
 * maximum number of alsa periods in hw ring buffer
 */
extern int g_ssc_max_buf_periods;


/*!
 * duration of one alsa period in milliseconds
 */
extern int g_ssc_period_duration;


/*!
 * silence frames to be played back to emty DSP jitter buffer
 * This is sometimes required due to address popping sounds
 */
extern int g_ssc_playback_trailing_silence_frames;


/*! alsa access mode type */
typedef enum {
  alsa_access_mode_direct,         /*!< direct access mode, used for pulse audio */
  alsa_access_mode_memory_mapped,  /*!< memory mapped address mode, QTI */
} t_ssc_alsa_access_mode;

/*!
 * Alsa  provides two  different access  modes for  reading/writing audio
 * samples. By  default the direct  access mode  is mostly used  which is
 * mapped to  the function snd_pcm_readi, snd_pcm_writei.  Sometimes only
 * memory mapped access is supported e.g.  in the case of Qualcomm SoC's.
 * Then  the functions  snd_pcm_mmmap_readi,  snd_pcm_mm_writei shall  be
 * invoked instead. Memory mapped access is not supported by pulse audio.
 * Specify one of the following options:
 */
extern t_ssc_alsa_access_mode g_ssc_alsa_access_mode;


/*!
 * Length  of the  mp3 decoding  queue in  milliseconds. Increasing  this
 * valeu might help with some real time constraint problems but this also
 * increases the latency for initial  playback. Be aware that files which
 * are below this value will not be properly played back.
 */
extern int g_ssc_mp3_decoding_queue_len;



/*!
 * initialize configuration data
 */
int ssc_init_config(void);


/*! @} */

#ifdef __cplusplus
}
#endif


#endif /* #ifndef SSC_CONFIG_H */
