/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <ssc_control.h>
#include <ssc.h>
#include <ssc_ahl.h>
#include <ssc_config.h>
#include <ssc_events.h>
#include <ssc_tasks.h>
#include <mediatype.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <log.h>
#include <olcutils/alloc.h>
#include <olcutils/slist.h>
#include <tcp_msg.h>
#include <utils.h>
#include <uv_utils.h>
#include <revision.h>
#include <config.h>


/*! \file  ssc_control.c
    \brief command and control interface

    \addtogroup control
    @{
 */


/*! timer interval for generating the status log message */
#define MSG_INTERVAL    5


/*!
 * context data for every client connection
 */
typedef struct {
  t_icom_data_stream*   p_data_stream;        /*!< pointer to data stream splitter instance */
  int                   terminate_connection; /*!< terminate connection when true */
} client_context_t;



/* --- begin of control commands --- */

static void* ssc_server_op_openidle( uv_stream_t* client, string_t* args )
{
  ssc_control_t* p = (ssc_control_t *)uv_loop_get_data( client->loop );
  ssc_events_t* p_ssc_events = (ssc_events_t *) p->p_ssc_events;
  const int max_args = 20;
  char* argv[max_args];
  int argc = 0, i;
  int optindex, optchar, errors = 0;
  const struct option long_options[] =
    {
      { "id", required_argument, NULL, 'i' },
      { "device", required_argument, NULL, 'D' },
      { "dir", required_argument, NULL, 'd' },
      { "help", no_argument, NULL, 'h' },
      { 0, 0, 0, 0 }
    };
  char identifier[SSC_MAX_ID_LEN] = { '\0' };
  char devicename[SSC_MAX_DEVNAME_LEN] = { '\0' };
  int dir = -1;

  if( ! args ) {
    (void)ssc_reply_to_sender( client, "no argumets specified error!\n" );
    return NULL;
  }

  argc = string2arglist( args, argv, max_args );
  optind = opterr = 0;
  while( ( optchar = getopt_long( argc, argv, "i:D:d:h", long_options, &optindex ) ) != -1 )
  {
    switch( optchar )
    {
    case 'i':
      ssc_strlcpy( identifier, optarg, sizeof(identifier) );
      break;

    case 'D':
      ssc_strlcpy( devicename, optarg, sizeof(devicename) );
      break;

    case 'd':
      if( strncmp( optarg, "playback", 4 ) == 0 )
        dir = (int)SSC_AHL_PCM_STREAM_PLAYBACK;
      else if( strncmp( optarg, "capture", 4 ) == 0 )
        dir = (int)SSC_AHL_PCM_STREAM_CAPTURE;
      break;

    case 'h':
      (void)ssc_reply_to_sender( client,
                                 "openidle -i <process identifier> -D <alsa dev>\n"
                                 "     keeps specified device file open\n"
                                 "     -i process identifier\n"
                                 "     -D alsa device filen name\n"
                                 "     --dir 'playback' or 'capture'\n" );
      errors = -1;
      break;

    default:
      (void)ssc_reply_to_sender( client,
                                 "input argument %c not supported error!\n", optopt );
      errors = -1;
      break;
    }
  }

  if( ! errors ) {
    if( *devicename == '\0' ) {
      (void)ssc_reply_to_sender( client,
                                 "name of alsa device not specified error!\n" );
      errors = -1;
    }

    if( *identifier == '\0' ) {
      (void)ssc_reply_to_sender( client,
                                 "verb identifier not specified error!\n" );
      errors = -1;
    }

    if( dir < 0 ) {
      (void)ssc_reply_to_sender( client,
                                 "playback direction not correctly specified (capture or playback)!\n" );
      errors = -1;
    }
  }

  if( ! errors ) {
    (void)send_event_request(
      p_ssc_events,
      client,
      ssc_proc_openidle_request,
      identifier,
      NULL /*file name not required here */,
      devicename,
      0 /* channels, not specified here */,
      0 /* sample_rate not specified here */,
      0 /* duration not specifed here */,
      dir /* direction not required here */,
      0 /* repeat_flag not required here */
      );
  }

  /* release argument list */
  for(i=0; i<argc; ++i )
    if( argv[i] )
      cul_free( argv[i] );

  return NULL;
}


static void* ssc_server_op_close( uv_stream_t* client, string_t* args )
{
  ssc_control_t* p = (ssc_control_t *)uv_loop_get_data( client->loop );
  ssc_events_t* p_ssc_events = (ssc_events_t *) p->p_ssc_events;
  const int max_args = 20;
  char* argv[max_args];
  int argc = 0, i;
  int optindex, optchar, errors = 0;
  const struct option long_options[] =
    {
      { "id", required_argument, NULL, 'i' },
      { "all", no_argument, NULL, 'a' },
      { "help", no_argument, NULL, 'h' },
      { 0, 0, 0, 0 }
    };
  char identifier[SSC_MAX_ID_LEN] = { '\0' };
  int close_all_flag = 0;

  if( ! args ) {
    ssc_reply_to_sender( client, "no argumets specified error!\n" );
    return NULL;
  }

  argc = string2arglist( args, argv, max_args );
  optind = opterr = 0;
  while( ( optchar = getopt_long( argc, argv, "i:ha", long_options, &optindex ) ) != -1 )
  {
    switch( optchar )
    {
    case 'i':
      ssc_strlcpy( identifier, optarg, sizeof(identifier) );
      break;
    case 'a':
      close_all_flag = 1;
      break;
    case 'h':
      ssc_reply_to_sender(
        client,
        "close -i <process identifier>\n"
        "     close playback for given process identifier (verb)\n"
        "     -i process identifier\n"
        "     -a close all process instances (verbs)\n"
        );
      errors = -1;
      break;
    default:
      ssc_reply_to_sender( client,
                           "input argument %c not supported error!\n", optopt );
      errors = -1;
      break;
    }
  }

  if( ! errors ) {
    if( *identifier == '\0' && !close_all_flag ) {
      ssc_reply_to_sender( client,
                           "neither verb identifier nor --all specified error!\n" );
      errors = -1;
    }
  }

  if( ! errors ) {
    if( close_all_flag ) {
      (void)send_event_request(
        p_ssc_events,
        client,
        ssc_proc_close_all_request,
        NULL, /* not required here */
        NULL /*file name not required here */,
        NULL /* devicename not required here */,
        0 /* channels, not specified here */,
        0 /* sample_rate not specified here */,
        0 /* duration not specifed here */,
        0 /* direction not required here */,
        0 /* repeat_flag not required here */
        );
    } else {
      (void)send_event_request(
        p_ssc_events,
        client,
        ssc_proc_close_request,
        identifier,
        NULL /*file name not required here */,
        NULL /* devicename not required here */,
        0 /* channels, not specified here */,
        0 /* sample_rate not specified here */,
        0 /* duration not specifed here */,
        0 /* direction not required here */,
        0 /* repeat_flag not required here */
        );
    }
  }

  /* release argument list */
  for(i=0; i<argc; ++i )
    if( argv[i] )
      cul_free( argv[i] );

  return NULL;
}


static void* ssc_server_op_play( uv_stream_t* client,  string_t* args )
{
  ssc_control_t* p = (ssc_control_t *)uv_loop_get_data( client->loop );
  ssc_events_t* p_ssc_events = (ssc_events_t *) p->p_ssc_events;
  const int max_args = 20;
  char* argv[max_args];
  int argc = 0, i;
  int optindex, optchar, errors = 0;
  const struct option long_options[] =
    {
      { "id", required_argument, NULL, 'i' },
      { "device", required_argument, NULL, 'D' },
      { "file", required_argument, NULL, 'f' },
      { "repeat", no_argument, NULL, 'r' },
      { "help", no_argument, NULL, 'h' },
      { 0, 0, 0, 0 }
    };
  char filename[256] = { '\0' };
  char identifier[SSC_MAX_ID_LEN] = { '\0' };
  char devicename[SSC_MAX_DEVNAME_LEN] = { '\0' };
  int  repeat_flag = 0;

  if( ! args ) {
    ssc_reply_to_sender( client, "no argumets specified error!\n" );
    return NULL;
  }

  argc = string2arglist( args, argv, max_args );
  optind = opterr = 0;
  while( ( optchar = getopt_long( argc, argv, "i:D:f:hr", long_options, &optindex ) ) != -1 )
  {
    switch( optchar )
    {
    case 'f':
      ssc_strlcpy( filename, optarg, sizeof(filename) );
      break;

    case 'i':
      ssc_strlcpy( identifier, optarg, sizeof(identifier) );
      break;

    case 'D':
      ssc_strlcpy( devicename, optarg, sizeof(devicename) );
      break;

    case 'h':
      ssc_reply_to_sender(
        client,
        "play -i <process identifier> -D <alsa dev> -f <wav/mp3 file> [-r]\n"
        "     plays sound file via specified device\n"
        "     -i process identifier\n"
        "     -D alsa device filen name\n"
        "     -f wav or mp3 file to play back\n"
        "     -r repeat from beginning when specified\n"
        );
      errors = -1;
      break;

    case 'r':
      repeat_flag = 1;
      break;

    default:
      ssc_reply_to_sender( client,
                                 "input argument %c not supported error!\n", optopt );
      errors = -1;
      break;
    }
  }

  if( ! errors ) {
    if( *filename == '\0' ) {
      ssc_reply_to_sender( client, "filename not specified error!\n" );
      errors = -1;
    }

    if( *devicename == '\0' ) {
      ssc_reply_to_sender( client, "name of alsa device not specified error!\n" );
      errors = -1;
    }

    if( *identifier == '\0' ) {
      ssc_reply_to_sender( client, "verb identifier not specified error!\n" );
      errors = -1;
    }
  }

  if( ! errors ) {
    if( ssc_media_type( filename ) == ssc_media_invalid ) {
      ssc_reply_to_sender( client, "media type is not supported error!\n" );
      errors = -1;
    }
  }

  if( ! errors ) {
    (void)send_event_request(
      p_ssc_events,
      client,
      ssc_proc_playback_request,
      identifier,
      filename,
      devicename,
      0 /* channels, not specified here */,
      0 /* sample_rate not specified here */,
      0 /* duration not specifed here */,
      0 /* direction not required here */,
      repeat_flag
      );
  }

  /* release argument list */
  for(i=0; i<argc; ++i )
    if( argv[i] )
      cul_free( argv[i] );

  return NULL;
}

static void* ssc_server_op_record( uv_stream_t* client,  string_t* args )
{
  ssc_control_t* p = (ssc_control_t *)uv_loop_get_data( client->loop );
  ssc_events_t* p_ssc_events = (ssc_events_t *) p->p_ssc_events;
  const int max_args = 20;
  char* argv[max_args];
  int argc = 0, i;
  int optindex, optchar, errors = 0;
  const struct option long_options[] =
    {
      { "id", required_argument, NULL, 'i' },
      { "device", required_argument, NULL, 'D' },
      { "file", required_argument, NULL, 'f' },
      { "channels", required_argument, NULL, 'c' },
      { "rate", required_argument, NULL, 'r' },
      { "duration", required_argument, NULL, 'd' },
      { "help", no_argument, NULL, 'h' },
      { 0, 0, 0, 0 }
    };
  char tmp_str[256] = { '\0' };
  char filename[256] = { '\0' };
  char identifier[SSC_MAX_ID_LEN] = { '\0' };
  char devicename[SSC_MAX_DEVNAME_LEN] = { '\0' };
  int  channels = 1, sample_rate = 16000, duration = 10;

  if( ! args ) {
    ssc_reply_to_sender( client, "no argumets specified error!\n" );
    return NULL;
  }

  argc = string2arglist( args, argv, max_args );
  optind = opterr = 0;
  while( ( optchar = getopt_long( argc, argv, "i:D:f:c:r:d:h", long_options, &optindex ) ) != -1 )
  {
    switch( optchar )
    {
    case 'f':
      ssc_strlcpy( filename, optarg, sizeof(filename) );
      break;

    case 'i':
      ssc_strlcpy( identifier, optarg, sizeof(identifier) );
      break;

    case 'D':
      ssc_strlcpy( devicename, optarg, sizeof(devicename) );
      break;

    case 'c':
      ssc_strlcpy( tmp_str, optarg, sizeof(tmp_str) );
      channels = atoi( tmp_str );
      break;

    case 'r':
      ssc_strlcpy( tmp_str, optarg, sizeof(tmp_str) );
      sample_rate = atoi( tmp_str );
      break;

    case 'd':
      ssc_strlcpy( tmp_str, optarg, sizeof(tmp_str) );
      duration = atoi( tmp_str );
      break;

    case 'h':
      ssc_reply_to_sender(
        client,
        "record -i <process identifier> -D <alsa dev> -f <wav file> [-r]\n"
        "       records MS wav sound file via specified device\n"
        "       -i process identifier\n"
        "       -D alsa device filen name\n"
        "       -f wav file to pay back\n"
        "       -c number of channels (default = 1)\n"
        "       -r sample rate (default = 16000)\n"
        "       -d max duration in secods (default 10s)\n"
        );
      errors = -1;
      break;

    default:
      ssc_reply_to_sender( client,
                           "input argument %c not supported error!\n", optopt );
      errors = -1;
      break;
    }
  }

  if( ! errors ) {
    if( *filename == '\0' ) {
      ssc_reply_to_sender( client, "filename not specified error!\n" );
      errors = -1;
    }

    if( *devicename == '\0' ) {
      ssc_reply_to_sender( client, "name of alsa device not specified error!\n" );
      errors = -1;
    }

    if( *identifier == '\0' ) {
      ssc_reply_to_sender( client, "verb identifier not specified error!\n" );
      errors = -1;
    }
  }

  if( ! errors ) {
    if( ssc_media_type( filename ) != ssc_media_wav ) {
      ssc_reply_to_sender( client,
                           "wrong format error, "
                           "recording supports ms wav format!\n" );
      errors = -1;
    }
  }

  if( ! errors ) {
    (void)send_event_request(
      p_ssc_events,
      client,
      ssc_proc_record_request,
      identifier,
      filename,
      devicename,
      channels,
      sample_rate,
      duration,
      0 /* direction not required here */,
      0 /* repeat_flag not required here */
      );
  }

  /* release argument list */
  for(i=0; i<argc; ++i )
    if( argv[i] )
      cul_free( argv[i] );

  return NULL;
}

static void* ssc_server_op_pause( uv_stream_t* client, string_t* args )
{
  ssc_control_t* p = (ssc_control_t *)uv_loop_get_data( client->loop );
  ssc_events_t* p_ssc_events = (ssc_events_t *) p->p_ssc_events;
  const int max_args = 20;
  char* argv[max_args];
  int argc = 0, i;
  int optindex, optchar, errors = 0;
  const struct option long_options[] =
    {
      { "id", required_argument, NULL, 'i' },
      { "help", no_argument, NULL, 'h' },
      { 0, 0, 0, 0 }
    };
  char identifier[SSC_MAX_ID_LEN] = { '\0' };

  if( ! args ) {
    ssc_reply_to_sender( client, "no argumets specified error!\n" );
    return NULL;
  }

  argc = string2arglist( args, argv, max_args );
  optind = opterr = 0;
  while( ( optchar = getopt_long( argc, argv, "i:h", long_options, &optindex ) ) != -1 )
  {
    switch( optchar )
    {
    case 'i':
      ssc_strlcpy( identifier, optarg, sizeof(identifier) );
      break;
    case 'h':
      ssc_reply_to_sender(
        client,
        "pause -i <process identifier>\n"
        "     pause playback for given process identifier (verb)\n"
        "     -i process identifier\n"
        );
      errors = -1;
      break;
    default:
      ssc_reply_to_sender( client,
                                 "input argument %c not supported error!\n", optopt );
      errors = -1;
      break;
    }
  }

  if( ! errors ) {
    if( *identifier == '\0' ) {
      ssc_reply_to_sender( client, "verb identifier not specified error!\n" );
      errors = -1;
    }
  }

  if( ! errors ) {
    (void)send_event_request(
      p_ssc_events,
      client,
      ssc_proc_pause_request,
      identifier,
      NULL /*file name not required here */,
      NULL /* devicename not required here */,
      0 /* channels, not specified here */,
      0 /* sample_rate not specified here */,
      0 /* duration not specifed here */,
      0 /* direction not required here */,
      0 /* repeat_flag not required here */
      );
  }

  /* release argument list */
  for(i=0; i<argc; ++i )
    if( argv[i] )
      cul_free( argv[i] );

  return NULL;
}

static void* ssc_server_op_resume( uv_stream_t* client, string_t* args )
{
  ssc_control_t* p = (ssc_control_t *)uv_loop_get_data( client->loop );
  ssc_events_t* p_ssc_events = (ssc_events_t *) p->p_ssc_events;
  const int max_args = 20;
  char* argv[max_args];
  int argc = 0, i;
  int optindex, optchar, errors = 0;
  const struct option long_options[] =
    {
      { "id", required_argument, NULL, 'i' },
      { "help", no_argument, NULL, 'h' },
      { 0, 0, 0, 0 }
    };
  char identifier[SSC_MAX_ID_LEN] = { '\0' };

  if( ! args ) {
    ssc_reply_to_sender( client, "no argumets specified error!\n" );
    return NULL;
  }

  argc = string2arglist( args, argv, max_args );
  optind = opterr = 0;
  while( ( optchar = getopt_long( argc, argv, "i:h", long_options, &optindex ) ) != -1 )
  {
    switch( optchar )
    {
    case 'i':
      ssc_strlcpy( identifier, optarg, sizeof(identifier) );
      break;
    case 'h':
      ssc_reply_to_sender(
        client,
        "resume -i <process identifier>\n"
        "     resume playback for given process identifier (verb)\n"
        "     -i process identifier\n"
        );
      errors = -1;
      break;
    default:
      ssc_reply_to_sender( client,
                                 "input argument %c not supported error!\n", optopt );
      errors = -1;
      break;
    }
  }

  if( ! errors ) {
    if( *identifier == '\0' ) {
      ssc_reply_to_sender( client, "verb identifier not specified error!\n" );
      errors = -1;
    }
  }

  if( ! errors ) {
    (void)send_event_request(
      p_ssc_events,
      client,
      ssc_proc_resume_request,
      identifier,
      NULL /*file name not required here */,
      NULL /* devicename not required here */,
      0 /* channels, not specified here */,
      0 /* sample_rate not specified here */,
      0 /* duration not specifed here */,
      0 /* direction not required here */,
      0 /* repeat_flag not required here */
      );
  }

  /* release argument list */
  for(i=0; i<argc; ++i )
    if( argv[i] )
      cul_free( argv[i] );

  return NULL;
}

static void* ssc_server_op_help( uv_stream_t* client, string_t* args )
{
  (void)args;
  ssc_reply_to_sender(
    client,
    "Simple Sounce Controller Daemon 2\n\n"
    "The current implementation reacts to the following commands:\n"
    "\topenidle  : keeps an alsa pcm device open e.g. to activate dsp\n"
    "\tclose     : release background process and close pcm device\n"
    "\tplay      : start playback background process\n"
    "\trecord    : start recording background process\n"
    "\tpause     : pause playback/recording\n"
    "\tresume    : resume playback/recording where it was paused\n"
    "\tlistverbs : list currently active process id's (verbs)\n"
    "\tsync-request : pongs back request for sync purposes\n"
    "\thelp      : this help test\n"
    "\tquit      : terminate active connection\n"
    "\tterminate : gracefully terminate daemon (for debugging purposes)\n\n"
    "Each command supports various options, use --help for more information.\n"
    );
  return NULL;
}


static void* ssc_server_sync_request( uv_stream_t* client, string_t* args )
{
  const int max_args = 20;
  char* argv[max_args];
  int argc = 0, i;
  char sync_response[80];

  if( ! args ) {
    (void)ssc_reply_to_sender( client, "no argumets specified error!\n" );
    return NULL;
  }

  /* ATTENTION: We need to block here until all openidle requests
     are processed */
  // TO BE DONE

  argc = string2arglist( args, argv, max_args );
  if( argc > 1 ) {
    snprintf( sync_response, sizeof( sync_response ), "sync-response %s\n", argv[1] );
    (void)ssc_reply_to_sender( client, sync_response );
  } else {
    (void)ssc_reply_to_sender( client, "could not parse sync request arguments!\n"  );
  }

  /* release argument list */
  for(i=0; i<argc; ++i )
    if( argv[i] )
      cul_free( argv[i] );

  return NULL;
}

/*!
 * Unfortunately there is no extension pointer in available for hm iteration
 * but thanks to single threaded event loop this not a problem.
 */
static uv_stream_t* lambda_p_kv_send_task_client;
static void* lambda_p_kv_send_task( void* key, void* val )
{
  hm_kv_t* kv = (hm_kv_t *)val;
  ssc_proc_handle_t* p_proc = ( ssc_proc_handle_t * )kv->val;
  char* filename;

  if( strlen( p_proc->filename ) > 0 ) {
    filename = p_proc->filename;
  } else {
    filename = "undefined";
  }

  (void)ssc_reply_to_sender( lambda_p_kv_send_task_client,
                             "id=%.30s, state=%s, device=%.30s, filename=%.128s\n",
                             p_proc->id,
                             ssc_event_name[p_proc->state],
                             p_proc->p_dev->devname,
                             filename );

  return NULL;
}

static void* ssc_server_op_listverbs( uv_stream_t* client, string_t* args )
{
  ssc_control_t* p = (ssc_control_t *)uv_loop_get_data( client->loop );
  ssc_t* p_ssc = p->p_ssc;
  ssc_tasks_t* p_tasks = p_ssc->p_ssc_tasks;
  const int max_args = 20;
  char* argv[max_args];
  int argc = 0, i;
  int optindex, optchar, errors = 0;
  const struct option long_options[] =
    {
      { "help", no_argument, NULL, 'h' },
      { 0, 0, 0, 0 }
    };

  if( ! args ) {
    ssc_reply_to_sender( client, "no argumets specified error!\n" );
    return NULL;
  }

  argc = string2arglist( args, argv, max_args );
  optind = opterr = 0;
  while( ( optchar = getopt_long( argc, argv, "h", long_options, &optindex ) ) != -1 )
  {
    switch( optchar )
    {
    case 'h':
      ssc_reply_to_sender(
        client,
        "listverbs\n"
        "     list actively served identifiers (verbs)\n"
        );
      errors = -1;
      break;
    default:
      ssc_reply_to_sender( client,
                           "input argument %c not supported error!\n", optopt );
      errors = -1;
      break;
    }
  }

  if( ! errors ) {
    lambda_p_kv_send_task_client = client;
    hm_doseq( p_tasks->p_task_hm, 0, lambda_p_kv_send_task );
  }

  /* release argument list */
  for(i=0; i<argc; ++i )
    if( argv[i] )
      cul_free( argv[i] );

  return NULL;
}

/* forward declaration */
static void cc_release_client_context(  client_context_t* p );

static void cc_uv_close_cb( struct uv_handle_s* client )
{
  cul_free( client );
}

static void cc_terminate_connection( uv_stream_t *client )
{
  ssc_control_t* p = (ssc_control_t *)uv_loop_get_data( client->loop );

  slist_t* l = p->p_tcp_cc_client_list;
  ssc_message( "%s,%d: close connection (%p)", __func__, __LINE__, client );
  p->p_tcp_cc_client_list = clone_slist_without_elem( l, client );
  if( p->p_tcp_cc_client_list == NULL ) {
    ssc_error( "%s,%d: out of memory error, exit program!",
               __func__, __LINE__ );
    exit( -1 );
  }
  slist_free( l );
  cc_release_client_context( (client_context_t *)(client->data) );
  uv_close((uv_handle_t*) client, cc_uv_close_cb );
  ssc_message( "%s,%d: --- got %d active c&c connections ---",
               __func__, __LINE__, slist_cnt( p->p_tcp_cc_client_list ) );

}

static void* ssc_server_op_quit( uv_stream_t* client, string_t* args )
{
  client_context_t* p_context = (client_context_t *) client->data;
  p_context->terminate_connection = 1;

  return NULL;
}

static void* ssc_server_op_terminate( uv_stream_t* client, string_t* args )
{
  ssc_control_t* p = (ssc_control_t *)uv_loop_get_data( client->loop );

  (void) send_event_request( p->p_ssc_events,
                             client,
                             ssc_proc_term_app_request,
                             NULL /* id not required */,
                             NULL /* filename not required */,
                             NULL /* devicename not required */,
                             0 /* channels not required */,
                             0 /* sample_rate not required */,
                             0 /* duration not required */,
                             0 /* direction not required */,
                             0 /* repeat_flag not required */ );

  return NULL;
}


/* --- end of control commands --- */

const static t_pair_tok_fcnt tok_fcnt_parser_table[] = {
  PAIR_TOK_FCNT( "openidle", ssc_server_op_openidle ),
  PAIR_TOK_FCNT( "close", ssc_server_op_close ),
  PAIR_TOK_FCNT( "play", ssc_server_op_play ),
  PAIR_TOK_FCNT( "record", ssc_server_op_record ),
  PAIR_TOK_FCNT( "pause", ssc_server_op_pause ),
  PAIR_TOK_FCNT( "resume", ssc_server_op_resume ),
  PAIR_TOK_FCNT( "help", ssc_server_op_help ),
  PAIR_TOK_FCNT( "sync-request", ssc_server_sync_request ),
  PAIR_TOK_FCNT( "listverbs", ssc_server_op_listverbs ),
  PAIR_TOK_FCNT( "quit", ssc_server_op_quit ),
  PAIR_TOK_FCNT( "terminate", ssc_server_op_terminate )
};


/*!
 * release client connection context object
 *
 * \param p pointer to context
 */
static void cc_release_client_context(  client_context_t* p )
{
  if( p ) {
    icom_release_data_stream( p->p_data_stream );
    cul_free( p );
  }
}

/*!
 * create a new client connection context object
 *
 * \return pointer to new context or NULL in case of error
 */
static client_context_t* cc_init_client_context( void )
{
  client_context_t* p = cul_malloc( sizeof( client_context_t ) );

  if( p ) {
    memset( p, 0, sizeof( client_context_t ) );
    p->p_data_stream = icom_create_data_stream( MAX_MESSAGE_SPLITS, MAX_MSG_DATA_SIZE,
                                                MESSAGE_DELIM, strlen( MESSAGE_DELIM ) );
    if( p->p_data_stream == NULL ) {
      cul_free( p );
      p = NULL;
    }
  }

  return p;
}


static void* free_cc_tcp_client( void* c )
{
  uv_tcp_t* client = (uv_tcp_t *)c;
  client_context_t* p_context = (client_context_t *) client->data;

  cc_release_client_context( p_context );
  cul_free( client );
  return NULL;
}


static uv_tcp_t* init_cc_tcp_client( void )
{
  uv_tcp_t *client = (uv_tcp_t*) cul_malloc( sizeof(uv_tcp_t) );
  if( client == NULL ) {
    ssc_error( "%s,%d: out of memory error!", __func__, __LINE__ );
    return NULL;
  }

  client->data = cc_init_client_context();
  if( client->data == NULL ) {
    ssc_error("%s,%d: could not initialize data stream splitter error!",
              __func__, __LINE__ );
    cul_free( client );
    return NULL;
  }

  return client;
}


static void signal_handler( uv_signal_t *handle, int signum )
{
  ssc_control_t* p = (ssc_control_t*)uv_loop_get_data( handle->loop );

  ssc_error( "%s, %d: termination handler invoked, got signal: %d!", __func__, __LINE__, signum );

  (void) send_event_request( p->p_ssc_events,
                             NULL /* client */,
                             ssc_proc_term_app_request,
                             NULL /* id not required */,
                             NULL /* filename not required */,
                             NULL /* devicename not required */,
                             0 /* channels not required */,
                             0 /* sample_rate not required */,
                             0 /* duration not required */,
                             0 /* direction not required */,
                             0 /* repeat_flag not required */ );
}


static void timer_cb( uv_timer_t* handle )
{
  ssc_control_t* p = (ssc_control_t*)uv_loop_get_data( handle->loop );
  ssc_message( "%s,%d: alive, ssc-revision: %s\n",
               __func__, __LINE__, g_ssc_revision );

  if( p->kill_signal ) {
    ssc_t* p_ssc = ( ssc_t *)p->p_ssc;
    ssc_tasks_t* p_ssc_tasks = p_ssc->p_ssc_tasks;
    int tasks = p_ssc_tasks->task_cnt;

    if( tasks > 0 ) {
      ssc_message( "%s,%d: application is termating, waiting for %d tasks to terminate ...\n",
                   __func__, __LINE__, tasks );
    } else {
      ssc_message( "%s,%d: got license to kill, terminate now!\n",
                   __func__, __LINE__, tasks );
      uv_stop( handle->loop );
    }
  }
}


/*
 * function pointer to server action operation
 *
 * \param p_evt pointer to event
 * \param args pointer argument list, elements are of type string_t*
 * \return server instance
 */
typedef void* (*cc_op_t)( uv_stream_t* p_stream, string_t* args );


/*
 * parse and handle requested command
 *
 * \param client pointer to uv_stream where all instance data can be derived from
 * \param p_msg  pointer to the received ascii text message
 * \param len    message length
 */
static int cc_command_handler( uv_stream_t *client, const char* p_msg, const int len )
{
  (void)len;
  ssc_control_t* p = (ssc_control_t *)uv_loop_get_data( client->loop );
  ssc_events_t*  p_ssc_events = (ssc_events_t *) p->p_ssc_events;
  cc_op_t        p_server_op;
  string_t      *s_tok, *s_cmd;
  slist_t       *lst_arg_cmd;
  int            error = 0;
  char           error_msg[256];
  int            error_msg_len;

  if( get_pending_events_cnt( p_ssc_events ) > p_ssc_events->pool_size * 8 / 10 ) {
    /* reject new event requests when event queue is in danger to overflow! */
    error_msg_len = snprintf( error_msg, sizeof( error_msg ),
                              "number of pending events exceeds 80%% of buffer capacity,"
                              " request '%.30s' is denied error!\n",
                              p_msg );
    (void)send_to_client( client, error_msg, (size_t)error_msg_len );
    ssc_error( "%s,%d: %s", __func__, __LINE__, error_msg );
    return -1;
  }

  s_tok = string_trim( string_new_from( p_msg ) );
  lst_arg_cmd = string_split( s_tok, " \t", 2 );
  if( slist_cnt( lst_arg_cmd ) >= 2 ) {
    s_cmd = (string_t *) slist_first( slist_rest( lst_arg_cmd ) );
  } else {
    s_cmd = (string_t *) slist_first( lst_arg_cmd );
  }

  if( s_cmd ) {
    p_server_op = (cc_op_t) hm_find_val_for_key( p->p_cmd_hm, s_cmd );
    if( p_server_op ) {
      (void) (*p_server_op)( client, s_tok );
    }
    else {
      char wrong_cmd[30];
      string_tmp_cstring_from( s_cmd, wrong_cmd, sizeof(wrong_cmd) );
      error_msg_len = snprintf( error_msg, sizeof(error_msg), "command %.30s not recognized error!\n", wrong_cmd );
      (void) send_to_client( client, error_msg, (size_t)error_msg_len );
      ssc_error( "%s,%d: %s", __func__, __LINE__, error_msg );
      error = -1;
    }
  }

  slist_free_deep( lst_arg_cmd, (lambda_t)string_release );
  string_release( s_tok );

  return error;
}


static void cc_client_read( uv_stream_t *client, ssize_t nread, const uv_buf_t *buf )
{
  ssc_control_t* p = (ssc_control_t *)uv_loop_get_data( client->loop );
  client_context_t* p_context = (client_context_t *) client->data;
  int term_connection = 0;
  char msg_buf[MAX_MSG_DATA_SIZE];
  t_icom_data_stream_msg msg;

  msg.data = msg_buf;
  msg.len = sizeof(msg_buf);

  ssc_message( "%s,%d: --- entering command & control reader (%p), nread=%d ---",
               __func__, __LINE__, client, nread );

  if (nread < 0) {
    if (nread != UV_EOF)
      ssc_error( "%s,%d: command & control read error %s", __func__, __LINE__, uv_err_name(nread) );
    else
      ssc_message( "%s,%d: command & control client disconnected", __func__, __LINE__ );

    term_connection = 1;
  }

  if( p->kill_signal ) {
    ssc_error( "%s,%d: application is shutting down, no more requests are processed!",
               __func__, __LINE__ );
    term_connection = 1;
  }

  if( term_connection ) {
    cc_terminate_connection( client );
    cul_free( buf->base );
    return;
  }

  /* read and process data */
  ssc_message( "%s,%d: command & control server: %d bytes received", __func__, __LINE__, nread );
  icom_push_data_segment( p_context->p_data_stream, buf->base, nread );
  while( icom_get_next_data_stream_msg( p_context->p_data_stream, & msg ) ) {

    /* do not process empty messages */
    if( msg.len == 0 )
      continue;

    /* got complete message, ensure that it is null terminated */
    msg.data[MIN( msg.len, MAX_MSG_DATA_SIZE-1)] = '\0';
    ssc_message( "%s,%d: --%d--> %s", __func__, __LINE__, msg.len, msg.data );

    /* handle command */
    (void)cc_command_handler( client, msg.data, msg.len );

#if( 0 )
    /* echo request  */
    (void) send_to_client( client, msg.data, msg.len );
    (void) send_to_client( client, "\n", 1 );
#endif
  }

  /* request to quit the tcp connection */
  if( p_context->terminate_connection ) {
    cc_terminate_connection( client );
  }

  /* release data */
  cul_free( buf->base );
}

/*!
 * callback which is invoked when new tcp or unix domain socket has been established
 *
 * \param server pointer to associated libuv server context
 * \param wstatus status code, negative when error occured
 */
static void on_new_cc_connection( uv_stream_t *server, int wstatus )
{
  ssc_control_t* p = (ssc_control_t *)uv_loop_get_data( server->loop );
  struct sockaddr_storage peername;
  char peername_str[128];
  int peername_len = sizeof(struct sockaddr_storage), acc_status;
  char resp_buf[80];
  int resp_buf_len = 0;

  ssc_message( "%s,%d: got new command & control connection request!", __func__, __LINE__ );

  if( wstatus < 0 ) {
    ssc_error( "%s,%d: new command & control connection error: %s", __func__, __LINE__, uv_strerror(wstatus) );
    return;
  }

  uv_tcp_t *client = init_cc_tcp_client();
  if( client == NULL ) {
    return;
  }

  uv_tcp_init( server->loop, client );

  acc_status = uv_accept( server, (uv_stream_t*)client );
  if( acc_status ) {
    ssc_error( "%s,%d: accept command & control connection error %s!",
               __func__, __LINE__, uv_strerror(acc_status) );
  }

  if( ! acc_status ) {
    if( slist_cnt( p->p_tcp_cc_client_list ) >= g_ssc_server_max_connections ) {
      ssc_error( "%s,%d: reject command & control connection request, only %d connections allowed!",
                 __func__, __LINE__, g_ssc_server_max_connections );
      acc_status = -1;
    }
  }

  if( ! acc_status ) {
    acc_status = uv_tcp_getpeername( (uv_tcp_t*)client, (struct sockaddr*)&peername, &peername_len );
    if( acc_status ) {
      ssc_error( "%s,%d: could not retrieve c&c peer name, err: %d, close connection!",
                 __func__, __LINE__, acc_status );
      ssc_error( "%s,%d: error message: %s",
                 __func__, __LINE__, uv_strerror( acc_status ) );
    }
  }

  if( ! acc_status ) {
    acc_status = uv_ip4_name( (struct sockaddr_in*)&peername, peername_str, sizeof(peername_str) );
    if( acc_status ) {
      ssc_error( "%s,%d: could not convert c&c peer address to string, err: %d, close connection!",
                 __func__, __LINE__, acc_status );
      ssc_error( "%s,%d: error message: %s",
                 __func__, __LINE__, uv_strerror( acc_status ) );
    }
  }

  ssc_message(  "%s,%d: --- connection request by c&c peer: %s ---",
                __func__, __LINE__, peername_str );


  if( ! acc_status )
  {
    ssc_message( "%s,%d: command & control connection request accepted (%p)",
                 __func__, __LINE__, client );
    uv_read_start( (uv_stream_t*) client, alloc_buffer, cc_client_read );
    p->p_tcp_cc_client_list = slist_prepend( p->p_tcp_cc_client_list, client );
    if( p->p_tcp_cc_client_list == NULL ) {
      ssc_error( "%s,%d: out of memory error, end program now!", __func__, __LINE__ );
      exit( -1 );
    }

    ssc_error( "%s,%d: --- got %d active connections ---",
               __func__, __LINE__, slist_cnt( p->p_tcp_cc_client_list ) );

    resp_buf_len = snprintf( resp_buf, sizeof(resp_buf),
                             "Simple Sound Controller 2, enter command or help!\n" );
    (void) send_to_client( (uv_stream_t*)client, resp_buf, (size_t)resp_buf_len );
  }
  else
  {
    uv_close((uv_handle_t*) client, (void (*)(struct uv_handle_s *)) free_cc_tcp_client );
  }
}

void release_control( ssc_control_t* p )
{
  if( p->p_tcp_cc_client_list ) {
    slist_free_deep( p->p_tcp_cc_client_list, free_cc_tcp_client );
  }

  if( p->p_cmd_hm ) {
    hm_free_with_keys( p->p_cmd_hm );
  }

  uv_timer_stop( &p->timer_req );

  cul_free( p );
}

ssc_control_t* alloc_control( void )
{
  ssc_control_t* p = cul_malloc( sizeof( ssc_control_t ) );

  if( p == NULL ) {
      ssc_error("%s,%d: out of memory error!\n", __func__, __LINE__ );
      return NULL;
  }

  memset( p, 0, sizeof( ssc_control_t ) );

  p->p_cmd_hm = ssc_init_cmd_hm( tok_fcnt_parser_table,
                                 sizeof( tok_fcnt_parser_table ) / sizeof(t_pair_tok_fcnt) );
  if( ! p->p_cmd_hm ) {
    ssc_error( "%s: hash initialization error!\n", __func__ );
    release_control( p );
    return NULL;
  }

  p->p_tcp_cc_client_list = slist_alloc();
  if( p->p_tcp_cc_client_list == NULL ) {
    ssc_error("%s,%d: out of memory error!\n", __func__, __LINE__ );
    release_control( p );
    return NULL;
  }

  p->loop = uv_default_loop();
  uv_loop_set_data( p->loop, p );

  uv_signal_init( p->loop, &p->sigint );
  uv_signal_start( &p->sigint, signal_handler, SIGINT );

  uv_signal_init( p->loop, &p->sigterm );
  uv_signal_start( &p->sigterm, signal_handler, SIGTERM );

  uv_signal_init( p->loop, &p->sighup );
  uv_signal_start( &p->sighup, signal_handler, SIGHUP );

  uv_timer_init( p->loop, &p->timer_req );
  uv_timer_start( &p->timer_req, timer_cb, MSG_INTERVAL * 1000, MSG_INTERVAL * 1000  );

  uv_tcp_init( p->loop, &p->tcp);

  uv_ip4_addr( g_ssc_server_ip_address, g_ssc_server_ip_port,  &p->addr );

  uv_tcp_bind( &p->tcp, (const struct sockaddr*)&p->addr, 0);
  int r = uv_listen( (uv_stream_t*) &p->tcp, 1, on_new_cc_connection );
  if( r ) {
    ssc_error( "%s, %d: listen error %s!", __func__, __LINE__, uv_strerror(r) );
    release_control( p );
    return NULL;
  }

  unlink( g_ssc_server_socket_filename );
  uv_pipe_init( p->loop, &p->pipe, 0 );
  r = uv_pipe_bind( &p->pipe, g_ssc_server_socket_filename );
  if( r != 0 ) {
    ssc_error( "%s, %d: uv_pipe_bind returned error %d: %s!", __func__, __LINE__, r, uv_strerror(r) );
  } else {

    if( chmod(g_ssc_server_socket_filename, S_IRUSR | S_IWUSR | S_IRGRP  | S_IWGRP ) != 0 ) {
      ssc_error("%s,%d: Could not change file attr. for %s, error: %d\n",
                __func__, __LINE__, g_ssc_server_socket_filename, errno );
    }

    r = uv_listen( (uv_stream_t*) &p->pipe, 1, on_new_cc_connection );
    if( r ) {
      ssc_error( "%s, %d: listen error %s!", __func__, __LINE__, uv_strerror(r) );
    }
  }

  return p;
}

/*! @} */
