/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef SSC_CONTROL_H
#define SSC_CONTROL_H

#include <olcutils/hashmap.h>
#include <olcutils/slist.h>
#include <uv.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \file  ssc_control.h
    \brief command and control interface

    \addtogroup control
    @{
 */

/*! maximum allowed size for one message */
#define MAX_MSG_DATA_SIZE 256

/*! maximum message fragmentation in TCP layer */
#define MAX_MESSAGE_SPLITS 10

/*! delimiter string to separate messages in tcp data stream */
#define MESSAGE_DELIM "\n"


/*!
 * context data for the command & control interface
 */
typedef struct {
  void*               p_ssc;                      /*!< pointer to global context */
  void*               p_ssc_events;               /*!< pointer to ssc event queue */
  uv_loop_t*          loop;                       /*!< libuv's event loop */
  uv_tcp_t            tcp;                        /*!< associated tcp server handle */
  uv_pipe_t           pipe;                       /*!< pipe for unix domain socket connection */
  struct sockaddr_in  addr;                       /*!< bound address and port */
  slist_t*            p_tcp_cc_client_list;       /*!< pointer to list of active command & control clients */
  hm_t*               p_cmd_hm;                   /*!< command hash map */

  uv_signal_t         sigterm;                    /*!< termination signal */
  uv_signal_t         sigint;                     /*!< interrupt signal */
  uv_signal_t         sighup;                     /*!< hang up signal */

  uv_timer_t          timer_req;                  /*!< cyclic timer cb */
  int                 kill_signal;                /*!< wenn true (1) exit application */
} ssc_control_t;


/*!
 * release of the command & control interface data context
 *
 * \param p pointer to the command & control interface data context to be released
 */
void release_control( ssc_control_t* p );


/*!
 *  allocate and initialize the command & control interface
 *
 * \return pointer to the command & control interface data context
 */
ssc_control_t* alloc_control( void );


/*! @} */

#ifdef __cplusplus
}
#endif


#endif /* #ifndef SSC_CONTROL_H */
