/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <string.h>
#include <time.h>
#include <uv.h>
#include <ssc_events.h>
#include <ssc_control.h>
#include <utils.h>
#include <log.h>


/*! \file ssc_events.c
    \brief event type definitions

    \addtogroup event_handler
    @{
 */


/*!
 * char pointer array with stringified event name
 */
const char* const ssc_event_name[] = { SSC_EVT_NAMES };


static ssc_evt_t* ssc_alloc_event( void )
{
  ssc_evt_t* p = cul_malloc( sizeof(ssc_evt_t) );

  if( p == NULL ) {
    ssc_error("%s,%d: out of memory error!\n", __func__, __LINE__ );
  } else {
    memset( p, 0, sizeof( ssc_evt_t ) );
  }

  return p;
}


static void ssc_release_event( ssc_evt_t* p )
{
  cul_free( p );
}


void ssc_release_events( ssc_events_t* p )
{
  if( p )
  {
    ssc_evt_t* p_evt;

    while( !clist_is_empty( & p->pool ) ) {
      p_evt = (ssc_evt_t *)clist_remove_head( & p->pool );
      ssc_release_event( p_evt );
    }

    while( !clist_is_empty( & p->ready_list ) ) {
      p_evt = (ssc_evt_t *)clist_remove_head( & p->ready_list );
      ssc_release_event( p_evt );
    }

    pthread_mutex_destroy( & p->mutex );
    pthread_mutex_destroy( & p->evt_handling_mutex );
    cul_free( p );
  }
}


ssc_events_t* ssc_alloc_events( uv_loop_t* loop,
                                void* p_ssc,
                                event_received_cb event_cb,
                                const int pool_size )
{
  int i;
  ssc_events_t* p = cul_malloc( sizeof(ssc_events_t) );

  if( p == NULL )
    return p;

  memset( p, 0, sizeof( ssc_events_t ) );
  p->p_ssc = p_ssc;

  if( pthread_mutex_init( &p->mutex, 0 ) != 0 ) {
    ssc_error( "%s,%d: could initialize event pool mutex error\n",
               __func__, __LINE__ );
    cul_free( p );
    return NULL;
  }

  if( pthread_mutex_init( &p->evt_handling_mutex, 0 ) != 0 ) {
    ssc_error( "%s,%d: could initialize event handling mutex error\n",
               __func__, __LINE__ );
    pthread_mutex_destroy( & p->mutex );
    cul_free( p );
    return NULL;
  }

  /* initialize pool an ready ring buffers */
  p->pool_size = pool_size;
  clist_init( & p->pool );
  clist_init( & p->ready_list );
  for (i = 0; i < pool_size; i++)
  {
    ssc_evt_t* p_evt = ssc_alloc_event();
    if( p_evt ) {
      clist_insert_tail( & p->pool, (t_clist *) p_evt );
    }
    else {
      ssc_release_events( p );
      return NULL;
    }
  }

  uv_async_init( loop, & p->signal, event_cb );

  return p;
}

ssc_evt_t* get_event_from_pool( ssc_events_t* p )
{
  ssc_evt_t* p_evt;

  /* unlink space disconnection event for processing out of pool */
  pthread_mutex_lock( & p->mutex );

  if ( !clist_is_empty( & p->pool ) ) {
    p_evt = (ssc_evt_t*)clist_remove_head( & p->pool );
  }
  else {
    ssc_error("%s,%d: event queue overflow error, overwriting existing events\n",
              __func__, __LINE__ );
    p_evt = (ssc_evt_t*)clist_remove_head( & p->ready_list );
  }

  pthread_mutex_unlock( & p->mutex );

  return p_evt;
}

void put_event_to_ready_list( ssc_events_t* p, ssc_evt_t* p_evt )
{
  p_evt->ts = time( NULL );

  pthread_mutex_lock( & p->mutex );
  clist_insert_tail( & p->ready_list, & p_evt->node );
  ++p->pending_events_cnt;
  pthread_mutex_unlock( & p->mutex );

  /* generate data  available signal for main event loop  only in case
  the event has  not already been rescheduled. In the  latter case the
  processing of the termination event  will take care for retriggering
  the event  handler. Otherwise we create  a lot of load  by litteraly
  invoking the event handler in every libuv event loop! */

  if( p_evt->rescheduled_cnt == 0 ||
      ( p_evt->type != ssc_proc_playback_request &&
        p_evt->type != ssc_proc_record_request ) ) {
    uv_async_send( & p->signal );
  }
}

ssc_evt_t* get_event_from_ready_list( ssc_events_t* p )
{
  ssc_evt_t* p_evt;

  pthread_mutex_lock( & p->mutex );
  if( clist_is_empty( & p->ready_list ) ) {
    p_evt = NULL;
  } else {
    p_evt = (ssc_evt_t*)clist_remove_head( & p->ready_list );
    --p->pending_events_cnt;
  }
  pthread_mutex_unlock( & p->mutex );

  return p_evt;
}

void put_event_to_pool( ssc_events_t* p, ssc_evt_t* p_evt )
{
  pthread_mutex_lock( & p->mutex );
  clist_insert_tail( & p->pool, & p_evt->node);
  pthread_mutex_unlock( & p->mutex );
}

int get_pending_events_cnt( ssc_events_t* p)
{
  int cnt;

  pthread_mutex_lock( & p->mutex );
  cnt = p->pending_events_cnt;
  pthread_mutex_unlock( & p->mutex );

  return cnt;
}

/*!
 * generates and sends a new event request to event queue (ready list)
 *
 * \param p pointer to event handler state object
 * \param client pointer to associated tcp client where the request was received from
 * \param event type, make shure that the specifier name ends with '_request'
 * \param id identifier of the associated background process or foreground handle
 * \param filename in case of recorder or player the used filename, otherwise unused
 * \param devicename name of the audio (alsa) device for playback or capture
 * \param channels in case of recording the number of channels to be recoreded
 * \param sample_rate in case of recording the sample rate used, otherwise unused
 * \param duration in case of recording the maximum duration of the recorded file
 * \param direction, use SSC_AHL_PCM_STREAM_PLAYBACK or SSC_AHL_PCM_STREAM_CAPTURE
 * \param repeat_flag playback is repeated when set to 1
 * \return 0 in case of success, currently no error codes are implemented
 */
int send_event_request(
  ssc_events_t*     p,
  uv_stream_t*      client,
  const ssc_event_t type,
  const char*       id,
  const char*       filename,
  const char*       devicename,
  const uint16_t    channels,
  const uint32_t    sample_rate,
  const int32_t     duration,
  const int16_t     direction,
  const int16_t     repeat_flag
  )
{
  ssc_evt_t* p_evt;

  p_evt = get_event_from_pool( p );

  p_evt->client = client;
  p_evt->type = type;
  p_evt->rescheduled_cnt = 0;

  if( id ) {
    ssc_strlcpy( p_evt->id, id, SSC_MAX_ID_LEN );
  } else {
    ssc_strlcpy( p_evt->id, "undefined", SSC_MAX_ID_LEN );
  }

  if( filename ) {
    ssc_strlcpy( p_evt->filename, filename, SSC_MAX_PATH );
  } else {
    ssc_strlcpy( p_evt->filename, "undefined", SSC_MAX_PATH );
  }

  if( devicename ) {
    ssc_strlcpy( p_evt->devicename, devicename, SSC_MAX_DEVNAME_LEN );
  } else {
    ssc_strlcpy( p_evt->devicename, "undefined", SSC_MAX_DEVNAME_LEN );
  }

  p_evt->channels = channels;
  p_evt->sample_rate = sample_rate;
  p_evt->duration = duration;
  p_evt->direction = direction;
  p_evt->repeat_flag = repeat_flag;

  put_event_to_ready_list( p, p_evt );

  return 0;

}

/*! @} */
