/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef SSC_EVENTS_H
#define SSC_EVENTS_H

#include <uv.h>
#include <pthread.h>
#include <olcutils/clist.h>
#include <olcutils/alloc.h>
#include <common.h>

#ifdef __cplusplus
extern "C" {
#endif


/*! \file ssc_events.h
    \brief event type definitions

    \addtogroup event_handler
    @{
 */

/*!
 * definition audio processing event identifiers
 */
#define SSC_EVT_NAMES \
  EVT(ssc_proc_idle)                                /*!< not initialized yet */ \
  EVT(ssc_proc_open)                                /*!< device descriptor open */ \
  EVT(ssc_proc_close)                               /*!< device descriptor closed */ \
  EVT(ssc_proc_init)                                /*!< handler is initialized but not running */ \
  EVT(ssc_proc_preparing)                           /*!< handler is not running yet but helper threads e.g. for mp3 are started and must be properly closed */ \
  EVT(ssc_proc_running)                             /*!< proc handler is running */ \
  EVT(ssc_proc_paused)                              /*!< proc handler is paused state */ \
  EVT(ssc_proc_stopping)                            /*!< proc handler is stopped */ \
  EVT(ssc_proc_terminating)                         /*!< proc handler is in termination process */ \
  EVT(ssc_proc_terminated)                          /*!< proc handler has been terminated */ \
  \
  EVT(ssc_proc_openidle_request)                    /*!< request to open a device for external processing e.g. telephony voice audio */ \
  EVT(ssc_proc_close_request)                       /*!< request to close a device for external processing or a playback or recording instance */ \
  EVT(ssc_proc_close_all_request)                   /*!< request to close all devices and playback and recording processes */ \
  EVT(ssc_proc_playback_request)                    /*!< request to start a playback process */ \
  EVT(ssc_proc_record_request)                      /*!< request to start a recording process */ \
  EVT(ssc_proc_pause_request)                       /*!< request to pause a process (playback or recording) */ \
  EVT(ssc_proc_resume_request)                      /*!< request to resume a process (playback or recording) */ \
  EVT(ssc_proc_term_app_request)                    /*!< request to terminate application */ \

#define EVT(x) x,
enum ssc_event_enum { SSC_EVT_NAMES TOP };

#undef EVT
#define EVT(x) #x,


/*!
 * char pointer array with stringified event name
 */
extern const char* const ssc_event_name[];


/*!
 * definition audio processing event enumeration type
 */
typedef enum ssc_event_enum ssc_event_t;


/*!
 * event defintion
 */
typedef struct {
  t_clist           node;                           /*!< for circular buffer queueing, must be first */
  void*             p_ssc_proc_handle;              /*!< pointer to processing handle */
  uv_stream_t*      client;                         /*!< pointer to client request */
  ssc_event_t       type;                           /*!< event type, mostly states */
  char              id[SSC_MAX_ID_LEN];             /*!< external process identifier */
  int               rescheduled_cnt;                /*!< number of resceduling this evt */
  time_t            ts;                             /*!< time stamp when queuing to ready list */

  char              filename[SSC_MAX_PATH];         /*!< playback or recorded file */
  char              devicename[SSC_MAX_DEVNAME_LEN];/*!< hardware device file */
  uint16_t          channels;                       /*!< number of channels */
  uint32_t          sample_rate;                    /*!< sample rate */
  int32_t           duration;                       /*!< max duration e.g. for recording */
  int16_t           direction;                      /*!< stream direction */
  int16_t           repeat_flag;                    /*!< playback repeat flag */

  int               errcode;                        /*!< error code */
  char              errmsg[SSC_MAX_MSG_LEN];        /*!< error msg. */
} ssc_evt_t;


/*!
 * Event handler state  object. Two interconnected doubly  linked circular lists
 * are used to store and handle all incorporated events for a given handler, the
 * pool and  the ready list.  New events are unlinked  from the pool  and linked
 * into the  ready list. Consumed  events are unlinked  from the ready  list and
 * linked at to the  pool. This allows to widely decouple  an event producer and
 * the correpsonding consumer. Furthermore the processing sequence can be easily
 * altered to introduce a specific priority scheme.
 */
typedef struct {
  t_clist           pool;                           /*!< event pool, takes initially all events */
  t_clist           ready_list;                     /*!< read list where for unprocessed events */
  void*             p_ssc;                          /*!< reference to global context */
  int               pending_events_cnt;             /*!< number pending events in ready list */
  int               pool_size;                      /*!< number of maximum cached events */
  pthread_mutex_t   mutex;                          /*!< mutex for access protection */
  pthread_mutex_t   evt_handling_mutex;             /*!< mutex which is active when event handler is processing all events */
  uv_async_t        signal;                         /*!< uv aync handle for signaling that data is available */
} ssc_events_t;



/*!
 * call back function which is invoked when event ready list has been filled
 *
 * This is the callback prototype which invokes the main event handler. It calls
 * the function ssc_event_handler().
 *
 * \param handle pointer to libuv data handle
 */
typedef void (* event_received_cb )( uv_async_t* handle );


/*!
 *  allocate main event queues, pool, ready list, mutexes, etc.
 *
 * \param loop pointer to libuv's event loop data context
 * \param p_ssc pointer to main ssc object
 * \param even_cb pointer to ssc_event_handler()
 * \param pool_size number of elements in event pool
 */
ssc_events_t* ssc_alloc_events( uv_loop_t* loop,
                                void* p_ssc,
                                event_received_cb event_cb,
                                const int pool_size );


/*!
 * release event data context
 *
 * \param p pointer to event data context to be released
 */
void ssc_release_events( ssc_events_t* p );


/*!
 * unlink an event object from event pool
 *
 * This function will always  return an event object. When the  pool is empty an
 * event is removed from the ready list which shall of course never happen!
 *
 * \return pointer to event object
 */
ssc_evt_t* get_event_from_pool( ssc_events_t* );


/*!
 * insert a given event object to the tail of the ready list
 *
 * \param p pointer to the event queue data context
 * \param p_evt pointer to the event object to be inserted into the ready list
 */
void put_event_to_ready_list( ssc_events_t* p, ssc_evt_t* p_evt );


/*!
 * retrieve next event from the head of the ready list
 *
 * \param p pointer to the event queue data context
 * \return point to the retrieved event or NULL in case the ready list is empty
 */
ssc_evt_t* get_event_from_ready_list( ssc_events_t* p );


/*!
 * insert event back to pool
 *
 * \param p pointer to the event queue data context
 * \param p_evt pointer to the event object to be inserted into the pool
 */
void put_event_to_pool( ssc_events_t* p, ssc_evt_t* p_evt );


/*!
 * get number of pending events in ready list
 *
 * \param p pointer to the event queue data context
 * \return number of elements retrievable in ready list
 */
int get_pending_events_cnt( ssc_events_t* p );


/*!
 * creates an event request
 *
 * We distinguish  between request events  that are  related to commands  of the
 * command and control interface and events that are related to state changes of
 * background processes. This function is used to generate request events.
 *
 * \param p pointer to the event queue data context
 * \param client pointer to tcp/uds client context where the event was received from
 * \param type event type
 * \param id unique processing id
 * \param filename fully qualified filename of audiofile to playback from or record to
 * \param devicename fully qualified filename of the audio device to be used
 * \param channels number of channels to record
 * \param sample_rate sample rate used for recording
 * \param duration maximum record duration in seconds
 * \param direction SSC_AHL_PCM_STREAM_PLAYBACK or SSC_AHL_PCM_STREAM_CAPTURE.
 *        Only required for ::ssc_proc_openidle_request
 * \param repeat_flag indicates continuous playback when not zero
 */
int send_event_request(
  ssc_events_t*     p,
  uv_stream_t*      client,
  const ssc_event_t type,
  const char*       id,
  const char*       filename,
  const char*       devicename,
  const uint16_t    channels,
  const uint32_t    sample_rate,
  const int32_t     duration,
  const int16_t     direction,
  const int16_t     repeat_flag
  );

/*! @} */

#ifdef __cplusplus
}
#endif


#endif /* #ifndef EVENTS_H */
