/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <string.h>
#include <unistd.h>
#include <ssc_proc.h>
#include <ssc.h>
#include <ssc_tasks.h>
#include <olcutils/alloc.h>
#include <utils.h>
#include <log.h>

/*! \file ssc_proc.c
    \brief proccesing tasks (playback/record)

    \addtogroup proc
    @{
 */

static ssc_proc_handle_t* ssc_init_proc_handle( const char* id, ssc_t* p_ssc )
{
  ssc_proc_handle_t* p = (ssc_proc_handle_t *) cul_malloc( sizeof(ssc_proc_handle_t) );

  if( p == NULL ) {
    ssc_error("%s,%d: out of memory error!\n", __func__, __LINE__ );
    return NULL;
  }

  memset( p, 0, sizeof(ssc_proc_handle_t) );

  ssc_strlcpy( p->id, id, SSC_MAX_ID_LEN );
  p->p_ssc = p_ssc;

  return p;
}


void ssc_proc_fire_event( ssc_proc_handle_t* p, ssc_event_t type )
{
  ssc_t* p_ssc = (ssc_t *)p->p_ssc;
  ssc_events_t* p_ssc_events = p_ssc->p_ssc_events;
  ssc_evt_t* p_evt;

  p_evt = get_event_from_pool( p_ssc_events );

  p_evt->type = type;
  p_evt->p_ssc_proc_handle = p;

  ssc_strlcpy( p_evt->id, p->id, SSC_MAX_ID_LEN );

  if( strlen( p->filename ) > 0 ) {
    ssc_strlcpy (p_evt->filename, p->filename, SSC_MAX_PATH );
  } else {
    ssc_strlcpy (p_evt->filename, "undefined", SSC_MAX_PATH );
  }

  if( p->p_dev && strlen( p->p_dev->devname ) > 0 ) {
    ssc_strlcpy (p_evt->devicename, p->p_dev->devname, SSC_MAX_DEVNAME_LEN );
  } else {
    ssc_strlcpy (p_evt->devicename, "undefined", SSC_MAX_DEVNAME_LEN );
  }

  pthread_mutex_lock( & p->mutex );
  p->state = type;
  pthread_mutex_unlock( & p->mutex );

  ssc_message("%s,%d: ~~ id: %s, type: %s, filename: %s, devname: %s\n",
              __func__, __LINE__,
              p_evt->id, ssc_event_name[type],
              p_evt->filename, p_evt->devicename );

  put_event_to_ready_list( p_ssc_events, p_evt );
}


/* --- openidle --- */

int open_idle_release_cb( struct ssc_proc_handle_s * p )
{
  ssc_t* p_ssc = (ssc_t *)p->p_ssc;
  ssc_tasks_t* p_ssc_tasks = p_ssc->p_ssc_tasks;
  char id[SSC_MAX_ID_LEN];
  int error;

  ssc_proc_fire_event( p, ssc_proc_close );

  ssc_strlcpy( id, p->id, SSC_MAX_ID_LEN );

  error = ssc_ahl_close( p->p_dev );
  cul_free( p );

  ssc_remove_task( p_ssc_tasks, id );

  return error;
}

ssc_proc_handle_t* ssc_open_idle( const char* id,
                                  const char* device_name,
                                  const int   dir,
                                  void* p_ssc )
{
  ssc_proc_handle_t* p = ssc_init_proc_handle( id, p_ssc );
  if( p == NULL ) {
    return NULL;
  }

  p->p_dev = ssc_ahl_open( device_name, dir, 1 /* use default config */ );
  if( p->p_dev == NULL ) {
    ssc_error( "%s,%d: could not open alsa device %s error!\n",
               __func__, __LINE__, device_name );
    cul_free( p );
    return NULL;
  }

  p->p_proc_release_cb = open_idle_release_cb;
  p->exec_context = ssc_exec_evt_loop;

  ssc_proc_fire_event( p, ssc_proc_open );

  return p;
}


/* --- release --- */

int ssc_proc_release( ssc_proc_handle_t* p )
{
  int error = 0;

  if( p->exec_context == ssc_exec_evt_loop ) {
    /* When memory was allocated and initialized in the evt loop
       which is in example the case for the openidle verb,
       we can directly call the release call back here */
    error = p->p_proc_release_cb( p );
  } else  {
    /* Otherwise when the background process still running trigger termination first */
    pthread_mutex_lock( & p->mutex );
    p->terminate = 1;
    pthread_mutex_unlock( & p->mutex );
  }

  return error;
}


/* --- playback --- */


/*!
 * memory release function for playback handler
 *
 * It must be in its termination state and the associated call back
 * handler must only be invoked from the event handler out of the
 * event loop context. Otherwise we will face terrible race conditions!
 */
static int ssc_player_release_cb( struct ssc_proc_handle_s * p )
{
  ssc_t* p_ssc = (ssc_t *)p->p_ssc;
  ssc_tasks_t* p_ssc_tasks = p_ssc->p_ssc_tasks;
  char id[SSC_MAX_ID_LEN];
  int error;

  if( p->state != ssc_proc_terminated ) {
    ssc_error( "%s,%d: FATAL ERROR, release method inoked for player task which "
               " is not terminated!\n", __func__, __LINE__ );
    return -1;
  }

  // remember task id
  ssc_strlcpy( id, p->id, SSC_MAX_ID_LEN );

  error = p->p_atl_release_cb( p->p_atl );
  if( error ) {
    ssc_error( "%s,%d: error %d in transport layer release call back function!\n",
               __func__, __LINE__, error );
  }

  error = ssc_ahl_close( p->p_dev );
  if( error ) {
    ssc_error( "%s,%d: error %d when closing abstract audio interface!\n",
               __func__, __LINE__, error );
  }

  release_ssc_atl_period( p-> p_atl_period );
  ssc_remove_task( p_ssc_tasks, id );
  cul_free( p );

  return error;
}


/*!
 * audio player thread function which is executed in background
 *
 * \params arg pointer to associated data context of type ssc_proc_handle_t
 * \return not used here
 */
static void* ssc_proc_player_handler( void* arg )
{
  ssc_proc_handle_t* p = (ssc_proc_handle_t *)arg;
  ssc_t* p_ssc = (ssc_t *)p->p_ssc;
  ssc_events_t* p_ssc_events = p_ssc->p_ssc_events;
  ssc_atl_period_t* p_atl_period = (ssc_atl_period_t *)p->p_atl_period;
  ssc_ahl_config_t* p_ahl_config;
  size_t  frames_read, frames_to_write;
  ssize_t  frames_written, tot_frames_to_write;
  const int max_wait_for_configure_loops = 30; /* 30 times 100ms */
  int wait_for_configure_count_down = max_wait_for_configure_loops;
  int configured = 0;
  int error = 0;

  int terminate = 0;
  int pause = 0;

  ssc_proc_fire_event( p, ssc_proc_init );

  while( !terminate && !error )
  {
    /* fetch outsite commands */
    pthread_mutex_lock( & p->mutex );
    terminate = p->terminate;
    pause = p->pause;
    pthread_mutex_unlock( & p->mutex );

    /* we are allowed to pause only when playback has been configured */
    if( ! configured ) {
      pause = 0;
    }

    /* handle termination request */
    if( terminate ) {
      ssc_message( "%s,%d: id %s: got termination request, end playback now!\n",
                   __func__, __LINE__, p->id );
      break;
    }

    /* silence (pause) / playback switch */
    if( pause ) {
      /* null all samples */
      memset( p_atl_period->buf.buf, 0, p_atl_period->buf.buf_size );
    } else {
      /*
       * --- read audio config or data block from transport layer ---
       */
      frames_read = p->p_atl_producer_cb( p->p_atl, p_atl_period );
      p_ahl_config = (ssc_ahl_config_t *) p_atl_period->p_config;
    }

    switch( p_atl_period->type )
    {
    case ssc_atl_config_type:
      /* process transport layer configuration data */
      ssc_proc_fire_event( p, ssc_proc_preparing );
      error = ssc_ahl_config( p->p_dev, p_ahl_config );
      if( error ) {
        ssc_error( "%s,%d: id: %s, audio hardware configuration error occured!\n",
                   __func__, __LINE__, p->id );
        break;
      }

      error = resize_ssc_atl_period( p_atl_period, p_ahl_config );
      if( error ) {
        ssc_error( "%s,%d: id: %s, error during frame size reconfiguration occured!\n",
                   __func__, __LINE__, p->id );
        break;
      }

      ssc_proc_fire_event( p, ssc_proc_running );
      configured = 1;
      break;

    case ssc_atl_data_type:
      /* process transport layer audio data */
      if( configured ) {

        if( frames_read == 0 ) {
          ssc_message( "%s,%d: id %s: no more data, end playback now!\n",
                       __func__, __LINE__, p->id );
          terminate = 1;
          break;
        }

        if( frames_read < p_atl_period->period_size ) {
          ssc_message( "%s,%d: id %s: no full period of only %ld frames received!\n",
                       __func__, __LINE__, p->id, frames_read );
        }

        /*
         * write data to audio hardware
         */
        frames_written = ssc_ahl_pcm_write( p->p_dev,
                                            p_atl_period->buf.buf,
                                            frames_read );
        if( frames_written != frames_read ) {
          ssc_error( "%s,%d: id: %s, error %d during writing to audio hw occured"
                     ", stop processing!\n",
                     __func__, __LINE__, p->id, frames_written );
          error = -1;
        }

#ifdef USE_QTI_AUDIO
        /* Unfortunately  the qti audio  driver kind  of blocks until  all audio
        frames  of typically  100ms length  are  processed. During  this time  a
        background decoder  thread e.g.  for MP3  data is not  invoked and  as a
        consequence  the decoder  is not  executed in  realtime anymore.  The so
        called Qualcomm  Audio Sample Application  which is supposed by  them as
        foundation for  all audio  software including  the qti  audio adaptation
        layer as  used in this  project is in  fact absolutely terrible  in this
        sense. The following sleep operation  hands over some processing time to
        the decoder thread.  For single channel encoded data with  a sample rate
        of 16 kHz this  works, for stereo files and higher  sample rates it does
        not! */
        usleep( 5000 );
#endif /* #ifdef USE_QTI_AUDIO */
      } else {

        /* we need to wait for configuration data */
        if( wait_for_configure_count_down-- > 0 ) {
          usleep( 100000 );
          continue;
        }

        ssc_error( "%s,%d: id: %s, got no configuration frame after %d attemps error!\n",
                   __func__, __LINE__, p->id, max_wait_for_configure_loops );
        error = -1;
      }
      break;
    }
  }

  /* main processing loop terminated, fire notification event */
  ssc_proc_fire_event( p, ssc_proc_terminating );


  /* play trailing silence in order to ensure proper buffer cleaning when configured */
  if( g_ssc_playback_trailing_silence_frames > 0  &&  ! error )
  {
    memset( p_atl_period->buf.buf, 0, p_atl_period->buf.buf_size );
    tot_frames_to_write = g_ssc_playback_trailing_silence_frames;

    while( tot_frames_to_write > 0 )
    {
      frames_to_write = MIN( tot_frames_to_write,
                             p_atl_period->period_size );

      frames_written = ssc_ahl_pcm_write( p->p_dev,
                                          p_atl_period->buf.buf,
                                          frames_to_write );
      if( frames_written < 0 ) {
        ssc_error( "%s,%d: id: %s, error %d during writing selince to audio hw occured!\n",
                   __func__, __LINE__, p->id, frames_written );
        break;
      }

      if( frames_written <= 0 ) {
        break;
      }

      tot_frames_to_write -= frames_written;
    }
  }

  pthread_mutex_lock( & p_ssc_events->evt_handling_mutex );
  ssc_proc_fire_event( p, ssc_proc_terminated );
  pthread_mutex_unlock( & p_ssc_events->evt_handling_mutex );
  /*
   * leave thread, no access to p from here on anymore!
   */

  return NULL;
}


/*!
 * audio recorder thread function which is executed in background
 *
 * \params arg pointer to associated data context of type ssc_proc_handle_t
 * \return not used here
 */
static void* ssc_proc_recorder_handler( void* arg )
{
  ssc_proc_handle_t* p = (ssc_proc_handle_t *)arg;
  ssc_t* p_ssc = (ssc_t *)p->p_ssc;
  ssc_events_t* p_ssc_events = p_ssc->p_ssc_events;
  ssc_atl_period_t* p_atl_period = p->p_atl_period;
  ssc_ahl_config_t*  p_config = p_atl_period->p_config;
  size_t max_record_frames = p_config->duration_in_seconds *
    p_config->sample_rate;
  ssize_t frames_to_read = p->p_atl_period->period_size, frames_read;
  ssize_t frames_written, tot_frames_read = 0L;
  int terminate = 0;
  int pause = 0;
  int error = 0;

  ssc_proc_fire_event( p, ssc_proc_init );

  while( !terminate && !error ) {
    /* fetch outsite commands */
    pthread_mutex_lock( & p->mutex );
    terminate = p->terminate;
    pause = p->pause;
    pthread_mutex_unlock( & p->mutex );

    /*
     * read data from audio hardware
     */
    frames_read = ssc_ahl_pcm_read( p->p_dev,
                                    p_atl_period->buf.buf,
                                    frames_to_read );

    if( tot_frames_read == 0 ) {
      ssc_proc_fire_event( p, ssc_proc_running );
    }

    if( frames_read != frames_to_read ) {
      ssc_error( "%s,%d: id: %s, error %d during writing to audio hw occured!\n"
                 ", stop processing!\n",
                 __func__, __LINE__, p->id, frames_read );
      error = -1;
    }

    if( ! error && ! pause ) {
      tot_frames_read += frames_read;
      frames_written = p->p_atl_consumer_cb( p->p_atl, p_atl_period );
      if( frames_written != frames_to_read ) {
        ssc_error("%s,%d: id %s, error %d during writing to file %s occured!\n"
                  ", stop processing!\n",
                  __func__, __LINE__, p->id, frames_written, p->filename );
        error = 1;
      }

      if( tot_frames_read >= max_record_frames ) {
        ssc_message("%s,%d: max record time for %s exceeded, top recording now!\n",
                    __func__, __LINE__, p->id );
        terminate = 1;
      }
    }
  }

  ssc_proc_fire_event( p, ssc_proc_terminating );

  pthread_mutex_lock( & p_ssc_events->evt_handling_mutex );
  ssc_proc_fire_event( p, ssc_proc_terminated );
  pthread_mutex_unlock( & p_ssc_events->evt_handling_mutex );
  /*
   * leave thread, no access to p from here on anymore!
   */

  return NULL;
}


ssc_proc_handle_t* ssc_player_open( const char* id,
                                    const char* device_name,
                                    const char* filename,
                                    const int repeat,
                                    ssc_atl_init_cb_t atl_init_cb,
                                    ssc_atl_release_cb_t atl_release_cb,
                                    ssc_atl_producer_cb_t atl_producer_cb,
                                    void* p_ssc )
{
  ssc_proc_handle_t* p;
  ssc_ahl_config_t*  p_config;
  struct sched_param sched_param;
  int retcode = 0;

  sched_param.sched_priority = sched_get_priority_max( SCHED_FIFO );

  p = ssc_init_proc_handle( id, p_ssc );
  if( p == NULL ) {
    return NULL;
  }

  p->p_dev = ssc_ahl_open( device_name, SSC_AHL_PCM_STREAM_PLAYBACK, 0 /* no config */ );
  if( p->p_dev == NULL ) {
    ssc_error( "%s,%d: could not open alsa device %s error!\n",
               __func__, __LINE__, device_name );
    cul_free( p );
    return NULL;
  }

  p->repeat = repeat;
  p->exec_context = ssc_exec_background_thread;
  p->p_atl_producer_cb = atl_producer_cb;
  p->p_atl_release_cb = atl_release_cb;
  p->p_proc_release_cb = ssc_player_release_cb;

  /* initialize sample source that is the WAV or MP3 sample reader/decoder */
  p->p_atl = atl_init_cb( filename, ssc_atl_read, p->repeat, &p_config );
  if( p->p_atl == NULL ) {
    ssc_error( "%s,%d: could not initialize transport layer for %s error!\n",
               __func__, __LINE__, device_name );
    (void) ssc_ahl_close( p->p_dev );
    cul_free( p );
    return NULL;
  }

  ssc_strlcpy( p->filename, filename, SSC_MAX_PATH );

  if( pthread_mutex_init( &p->mutex, 0 ) != 0 ) {
    ssc_error( "%s,%d: could playback mutex error\n", __func__, __LINE__ );
    (void) p->p_atl_release_cb( p->p_atl );
    (void) ssc_ahl_close( p->p_dev );
    cul_free( p );
    return NULL;
  }

  p->p_atl_period = init_ssc_atl_period( p_config );
  if( p->p_atl_period == NULL ) {
    ssc_error( "%s,%d: out of memory error\n", __func__, __LINE__ );
    (void) p->p_atl_release_cb( p->p_atl );
    (void) ssc_ahl_close( p->p_dev );
    pthread_mutex_destroy( &p->mutex );
    cul_free( p );
    return NULL;
  }

  if( ! retcode ) {
    retcode = pthread_create( &p->handler, NULL, ssc_proc_player_handler, p );
    if( ! retcode ) {
      retcode = pthread_detach( p->handler );
    }
    if( retcode ) {
      ssc_error( "%s,%d: FATAL: creation of decoder handler thread failed with error %d\n",
                 __func__, __LINE__, retcode );
      (void) p->p_atl_release_cb( p->p_atl );
      (void) ssc_ahl_close( p->p_dev );
      pthread_mutex_destroy( &p->mutex );
      cul_free( p );
      return NULL;
    } else {
      int prioerr = pthread_setschedparam( p->handler, SCHED_FIFO, &sched_param );
      if( prioerr ) {
        ssc_error( "%s,%d: could not assign SCHED_FIFO priority error: %d!\n",
                   __func__, __LINE__, prioerr );
#ifdef USE_QTI_AUDIO
        ssc_error( "%s,%d: increase of sched priority is strictly required with qti audio, terminate playback!\n",
                   __func__, __LINE__ );
        p->terminate = 1;
#endif /* #ifdef USE_QTI_AUDIO */
      }
    }
  }

  return p;
}

ssc_proc_handle_t* ssc_recorder_open( const char* id,
                                      const char* device_name,
                                      const char* filename,
                                      const int16_t channels,
                                      const uint32_t sample_rate,
                                      const int32_t  duration,
                                      ssc_atl_init_cb_t atl_init_cb,
                                      ssc_atl_release_cb_t atl_release_cb,
                                      ssc_atl_consumer_cb_t atl_consumer_cb,
                                      void* p_ssc )
{
  ssc_proc_handle_t* p;
  ssc_ahl_config_t   config;
  ssc_ahl_config_t*  p_config = &config;
  ssc_ahl_config_t** config_handle = &p_config;
  struct sched_param sched_param;
  int retcode = 0;

  sched_param.sched_priority = sched_get_priority_max( SCHED_FIFO );

  config.channels = channels;
  config.sample_rate = sample_rate;
  config.bits_per_sample = 16;
  config.duration_in_seconds = duration;
  config.max_buf_periods = g_ssc_max_buf_periods;
  config.period_duration = g_ssc_period_duration;

  p = ssc_init_proc_handle( id, p_ssc );
  if( p == NULL ) {
    return NULL;
  }

  p->p_dev = ssc_ahl_open( device_name, SSC_AHL_PCM_STREAM_CAPTURE, 0 /* no config */ );
  if( p->p_dev == NULL ) {
    ssc_error( "%s,%d: could not open alsa device %s error!\n",
               __func__, __LINE__, device_name );
    cul_free( p );
    return NULL;
  }

  p->exec_context = ssc_exec_background_thread;
  p->p_atl_consumer_cb = atl_consumer_cb;
  p->p_atl_release_cb = atl_release_cb;
  p->p_proc_release_cb = ssc_player_release_cb;

  /* initialize sample sink that writes samples, currently only in WAV format */
  p->p_atl = atl_init_cb( filename, ssc_atl_write, 0 /* repeat */, config_handle );
  if( p->p_atl == NULL ) {
    ssc_error( "%s,%d: could not initialize transport layer for %s error!\n",
               __func__, __LINE__, device_name );
    (void) ssc_ahl_close( p->p_dev );
    cul_free( p );
    return NULL;
  }

  /* update p_config wich is owned by atl and copy filename */
  p_config = *config_handle;
  ssc_strlcpy( p->filename, filename, SSC_MAX_PATH );

  retcode = ssc_ahl_config( p->p_dev, p_config );
  if( retcode ) {
    ssc_error( "%s,%d: could not configure hardware %s error!\n",
               __func__, __LINE__, device_name );
    (void) ssc_ahl_close( p->p_dev );
    (void) p->p_atl_release_cb( p->p_atl );
    cul_free( p );
    return NULL;
  }

  if( pthread_mutex_init( &p->mutex, 0 ) != 0 ) {
    ssc_error( "%s,%d: could playback mutex error\n", __func__, __LINE__ );
    (void) p->p_atl_release_cb( p->p_atl );
    (void) ssc_ahl_close( p->p_dev );
    cul_free( p );
    return NULL;
  }

  p->p_atl_period = init_ssc_atl_period( p_config );
  if( p->p_atl_period == NULL ) {
    ssc_error( "%s,%d: out of memory error\n", __func__, __LINE__ );
    (void) p->p_atl_release_cb( p->p_atl );
    (void) ssc_ahl_close( p->p_dev );
    pthread_mutex_destroy( &p->mutex );
    cul_free( p );
    return NULL;
  }

  if( ! retcode ) {
    retcode = pthread_create( &p->handler, NULL, ssc_proc_recorder_handler, p );
    if( ! retcode ) {
      retcode = pthread_detach( p->handler );
    }
    if( retcode ) {
      ssc_error( "%s,%d: creation of decoder handler thread failed with error %d\n",
                 __func__, __LINE__, retcode );
      (void) p->p_atl_release_cb( p->p_atl );
      (void) ssc_ahl_close( p->p_dev );
      pthread_mutex_destroy( &p->mutex );
      cul_free( p );
      return NULL;
    } else {
      int prioerr = pthread_setschedparam( p->handler, SCHED_FIFO, &sched_param );
      if( prioerr ) {
        ssc_error( "%s,%d: could not assign SCHED_FIFO priority error: %d!\n",
                   __func__, __LINE__, prioerr );
#ifdef USE_QTI_AUDIO
        ssc_error( "%s,%d: increase of sched priority is strictly required with qti audio, terminate playback!\n",
                   __func__, __LINE__ );
        p->terminate = 1;
#endif /* #ifdef USE_QTI_AUDIO */
      }
    }
  }

  return p;
}

/*! @} */
