/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef SSC_PROC_H
#define SSC_PROC_H

#include <pthread.h>
#include <ssc_events.h>
#include <ssc_ahl.h>
#include <ssc_atl.h>
#include <common.h>

#ifdef __cplusplus
extern "C" {
#endif


/*! \file ssc_proc.h
    \brief proccesing tasks (playback/record)

    \addtogroup proc
    @{
 */


/*!
 * process release callback type
 *
 * This is directly implemented in proc.c
 */
typedef   int (* ssc_proc_release_cb_t)( struct ssc_proc_handle_s* );


/*!
 * processing handle type
 *
 * We need to  know which type we have e.g.  when a process identifier
 * is  reused  and we  need  to  close  the  existing one  first.  For
 * 'openidle' we can directly close. For  the rest we need to postpone
 * the new open process.
 */
typedef enum {
  ssc_exec_evt_loop,                                /*!< process is initialized and executed in the context of the evt loop. */
  ssc_exec_background_thread,                       /*!< process is (partially) initialized and executed in the context of a background thread */
} ssc_exec_context_t;


/*!
 * context data type for the processing implementation
 *
 * It  uses  callbacks  for  receiving  or  sending  of  audio  frames
 * (abstract  transport layer).  The implementation  invokes a  set of
 * audio harware  abstraction (ahl) functions  to write and  read data
 * from the audio hardware.
 */
typedef struct ssc_proc_handle_s {

  char                          id[SSC_MAX_ID_LEN]; /*!< externally specified process identifier */
  void*                         p_ssc;              /*!< pointer to global context */
  ssc_ahl_dev_t*                p_dev;              /*!< abstract audio device specifier */
  char                          filename[SSC_MAX_PATH]; /*!< played/recorded filename or eol */
  ssc_exec_context_t            exec_context;       /*!< where initialized and executed */

  void*                         p_atl;              /*!< instance data of transport layer */
  ssc_atl_producer_cb_t         p_atl_producer_cb;  /*!< sample producer from file */
  ssc_atl_consumer_cb_t         p_atl_consumer_cb;  /*!< sample producer to file */
  ssc_atl_release_cb_t          p_atl_release_cb;   /*!< to release p_source_sink */

  ssc_proc_release_cb_t         p_proc_release_cb;  /*!< invoked from single thread controller task to avoid race conditions */

  ssc_atl_period_t*             p_atl_period;       /*!< pointer to read/write audio frame */
  pthread_t                     handler;            /*!< thread handler if required */
  pthread_mutex_t               mutex;              /*!< access protection */

  int                           errcode;            /*!< error code (not implemented yet), refer to error messages in syslog instead */
  char                          errmsg[SSC_MAX_MSG_LEN]; /*!< error message as string (not implemented yet) */

  int                           repeat;             /*!< set to 1 for continous repetition */
  int                           pause;              /*!< set to 1 to pause playback or recording */
  int                           terminate;          /*!< set to 1 for termination request */
  ssc_event_t                   state;              /*!< processing state we are in */
} ssc_proc_handle_t;


/*!
 * 'releasing' of a background process or a handle object
 *
 * This function  is generally  called to  close a  process. Processes
 * which  do   not  contain  a  background   process  (currently  only
 * 'openidle') are closed directly,  the associated memory is released
 * and the  entry is  removed from  the task list.  All other  data of
 * 'real'  processes  cannot  be  released directly  as  long  as  the
 * associated background process  is still running. In  this case, the
 * associated  process context  is  signaled to  stop processing.  The
 * \ref ssc_event_handler  "event handler" releases  the corresponding
 * memory and  removes the object  from the  task list when  the event
 * ::ssc_proc_terminated  is received  for  this task  by calling  the
 * function  invoked  ssc_close_task()  which  in  turn  invokes  this
 * function.
 *
 * \param p pointer to processing context to release
 * \return 0 in case of success or negative error code
 */
int ssc_proc_release( ssc_proc_handle_t* p );


/*!
 * release handler for ssc_open_idle
 *
 * This callback handler is set when calling ssc_open_idle(). When
 * the 'openidle' handle is closed by calling ssc_proc_release()
 * this function is invoked to close the handle.
 *
 * \param p pointer to processing context to release
 * \return 0 in case of success or negative error code
 */
int open_idle_release_cb( struct ssc_proc_handle_s * p );


/*!
 * creates a 'process' (handle) which just keeps an audio device open.
 *
 * This is a special case since  no background process is associated with
 * this task. It just maintains a handle to underlying audio device which
 * is kept open.  This concept is used by Qualcomm  to signal a dedicated
 * state  e.g. a  streaming  of  telephony voice  audio  data to  another
 * processing core.
 *
 * \param id pointer to unique process identifier
 * \param device_name pointer to audio device file name
 * \param dir audio processing direction, ::SSC_AHL_PCM_STREAM_PLAYBACK or
 *        ::SSC_AHL_PCM_STREAM_CAPTURE
 * \param p_ssc pointer to global data object used for back reference.
 * \return pointer to the associated process context or NULL in err. case
 */
ssc_proc_handle_t* ssc_open_idle( const char* id,
                                  const char* device_name,
                                  const int   dir,
                                  void* p_ssc );


/*!
 * creates a 'task' for the playback of an audio data
 *
 * \param id pointer to unique process identifier
 * \param device_name pointer to audio device file name
 * \param filename filename to be played back
 * \param repeat when not equal zero, continously replay from beginning
 * \param atl_init_cb initializer callback for transport layer impl. (wav/mp3, etc).
 * \param atl_release_cb release callback function to release transport impl.
 * \param atl_producer_cb callback to transport layer for audio frame procuction
 * \param p_ssc pointer to global data object used for back reference.
 * \return pointer to the associated process context or NULL in err. case
 */
ssc_proc_handle_t* ssc_player_open( const char* id,
                                    const char* device_name,
                                    const char* filename,
                                    const int repeat,
                                    ssc_atl_init_cb_t atl_init_cb,
                                    ssc_atl_release_cb_t atl_release_cb,
                                    ssc_atl_producer_cb_t atl_producer_cb,
                                    void* p_ssc );


/*!
 * creates a 'task' for the recording of an audio data
 *
 * \param id pointer to unique process identifier
 * \param device_name pointer to audio device file name
 * \param filename filename to be recorded
 * \param channels number of channels to record, must be supported by transport layer
 * \param sample_rate sample rate to be used
 * \param duration maximum duration of the recording
 * \param atl_init_cb initializer callback for transport layer impl. (wav only).
 * \param atl_release_cb release callback function to release transport impl.
 * \param atl_consumer_cb callback to transport layer for the storage of audio frames
 * \param p_ssc pointer to global data object used for back reference.
 * \return pointer to the associated process context or NULL in err. case
 */
ssc_proc_handle_t* ssc_recorder_open( const char* id,
                                      const char* device_name,
                                      const char* filename,
                                      const int16_t channels,
                                      const uint32_t sample_rate,
                                      const int32_t  duration,
                                      ssc_atl_init_cb_t atl_init_cb,
                                      ssc_atl_release_cb_t atl_release_cb,
                                      ssc_atl_consumer_cb_t atl_consumer_cb,
                                      void* p_ssc );


/*!
 * pauses playback or recording of audio data
 *
 * \param pointer to the associated process context
 * \return 0 in case of success or negative error code
 */
int ssc_pause( ssc_proc_handle_t* p );


/*!
 * resumes playback or recording of audio data
 *
 * \param pointer to the associated process context
 * \return 0 in case of success or negative error code
 */
int ssc_resume( ssc_proc_handle_t* );


/*!
 * helper function to create and fire new processing event
 *
 * We distinguish  between request events  that are  related to commands  of the
 * command and control interface and events that are related to state changes of
 * background processes. This function is used to generate the latter ones.
 *
 * \param p pointer to given process handle for dereferencing event queue
 * \param type event type
 */
void ssc_proc_fire_event( ssc_proc_handle_t* p, ssc_event_t type );

/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef SSC_PROC_H */
