/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <config.h>
#ifdef USE_QTI_AUDIO

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdint.h>
#include <time.h>
#include <sys/poll.h>
#include <sys/ioctl.h>
#include <string.h>
#include <values.h> /* for MAXLONG */
#include <ssc_ahl.h>
#include <olcutils/alloc.h>
#include <ssc_config.h>
#include <utils.h>
#include <errno.h>
#include <log.h>

#include <sound/asound.h>
#include <sound/compress_params.h>
#include <sound/compress_offload.h>
#include "alsa_audio.h"

#define MILLION 1000000L

/*! \file ssc_qti_audio.c
    \brief implementation of wrapper to Qualcomm's substandard audio sample code

    \addtogroup ahl
    @{
 */

/* --- private implementation --- */

const int debug = 0;

/*! so called Qualcomm 'alsa'! kernel interface ;-) */
typedef struct {
  ssc_ahl_dev_t*              p_ahl_dev;                    /*!< backrefence to ahl */

  struct pcm*                 pcm;
  struct pollfd               pfd[1];
  int                         start;
  size_t                      bytes_per_frame;

} ssc_qti_audio_t;


#define strlcat g_strlcat
#define strlcpy g_strlcpy

#define ID_RIFF 0x46464952
#define ID_WAVE 0x45564157
#define ID_FMT  0x20746d66
#define ID_DATA 0x61746164

#define FORMAT_PCM 1
#define LOG_NDEBUG 1

/*!
 * 'alsa' like parameter handling taken from QTI's aplay
 *
 *  Be aware that the pcm_close()  function releases this memory using
 *  non-instrumented free.  So we must not  use memory instrumentation
 *  for this. Hopefully QTI did the memory management correctly.
 */
static int set_params( struct pcm *pcm )
{
  struct snd_pcm_hw_params* params;
  struct snd_pcm_sw_params* sparams;

  int channels;

  if(pcm->flags & PCM_MONO)
    channels = 1;
  else if(pcm->flags & PCM_QUAD)
    channels = 4;
  else if(pcm->flags & PCM_5POINT1)
    channels = 6;
  else if(pcm->flags & PCM_7POINT1)
    channels = 8;
  else
    channels = 2;

  params = (struct snd_pcm_hw_params*) calloc(1, sizeof(struct snd_pcm_hw_params));
  if (!params) {
    ssc_error( "%s,%d: failed to allocate ALSA hardware parameters!",
               __func__, __LINE__ );
    return -ENOMEM;
  }

  param_init(params);

  param_set_mask(params, SNDRV_PCM_HW_PARAM_ACCESS,
                 (pcm->flags & PCM_MMAP)? SNDRV_PCM_ACCESS_MMAP_INTERLEAVED : SNDRV_PCM_ACCESS_RW_INTERLEAVED);
  param_set_mask(params, SNDRV_PCM_HW_PARAM_FORMAT, pcm->format);
  param_set_mask(params, SNDRV_PCM_HW_PARAM_SUBFORMAT,
                 SNDRV_PCM_SUBFORMAT_STD);

  param_set_min(params, SNDRV_PCM_HW_PARAM_PERIOD_TIME, 10);
  param_set_int(params, SNDRV_PCM_HW_PARAM_SAMPLE_BITS, 16);
  param_set_int(params, SNDRV_PCM_HW_PARAM_FRAME_BITS,
                pcm->channels * 16);
  param_set_int(params, SNDRV_PCM_HW_PARAM_CHANNELS,
                pcm->channels);
  param_set_int(params, SNDRV_PCM_HW_PARAM_RATE, pcm->rate);
  param_set_hw_refine(pcm, params);

  if (param_set_hw_params(pcm, params)) {
    ssc_error("%s,%d: cannot set hw params error!\n", __func__, __LINE__ );
    return -errno;
  }
  if (debug)
    param_dump(params);

  pcm->buffer_size = pcm_buffer_size(params);
  pcm->period_size = pcm_period_size(params);
  pcm->period_cnt = pcm->buffer_size/pcm->period_size;
  if (1) {
    ssc_message( "%s,%d: period_cnt = %d\n",  __func__, __LINE__, pcm->period_cnt);
    ssc_message( "%s,%d: period_size = %d\n", __func__, __LINE__, pcm->period_size);
    ssc_message( "%s,%d: buffer_size = %d\n", __func__, __LINE__, pcm->buffer_size);
  }
  sparams = (struct snd_pcm_sw_params*) calloc(1, sizeof(struct snd_pcm_sw_params));
  if (!sparams) {
    ssc_error("%s,%d: failed to allocate ALSA software parameters!\n",
              __func__, __LINE__ );
    return -ENOMEM;
  }

  // Get the current software parameters
  sparams->tstamp_mode = SNDRV_PCM_TSTAMP_NONE;
  sparams->period_step = 1;

  sparams->avail_min = pcm->period_size/(channels * 2) ;
  sparams->start_threshold =  pcm->period_size/(channels * 2) ;
  sparams->stop_threshold =  pcm->buffer_size ;
  sparams->xfer_align =  pcm->period_size/(channels * 2) ; /* needed for old kernels */

  sparams->silence_size = 0;
  sparams->silence_threshold = 0;

  if (param_set_sw_params(pcm, sparams)) {
    ssc_error( "%s,%d: cannot set sw params", __func__, __LINE__ );
    return -errno;
  }
  if(1) {
    ssc_message( "%s,%d: sparams->avail_min= %lu\n", __func__, __LINE__, sparams->avail_min);
    ssc_message( "%s,%d: sparams->start_threshold= %lu\n", __func__, __LINE__, sparams->start_threshold);
    ssc_message( "%s,%d: sparams->stop_threshold= %lu\n", __func__, __LINE__, sparams->stop_threshold);
    ssc_message( "%s,%d: sparams->xfer_align= %lu\n", __func__, __LINE__, sparams->xfer_align);
    ssc_message( "%s,%d: sparams->boundary= %lu\n", __func__, __LINE__, sparams->boundary);
  }

  return 0;
}



/* --- public API --- */

int ssc_ahl_close( ssc_ahl_dev_t* p )
{
  ssc_qti_audio_t* p_qti = ( ssc_qti_audio_t * ) p->data;
  struct pcm*      pcm = p_qti->pcm;

  if( p_qti->pcm > 0 ) {
    pcm_close( pcm );
  }

  if( p_qti ) {
    cul_free( p_qti );
  }

  cul_free( p );

  return 0;
}


ssc_ahl_dev_t* ssc_ahl_open( const char* device_name,
                             const int direction,
                             const int set_default_configuration )
{
  ssc_ahl_dev_t* p;
  ssc_qti_audio_t* p_qti;
  unsigned flags = 0;
  int error = 0;

  p = (ssc_ahl_dev_t *) cul_malloc( sizeof( ssc_ahl_dev_t ) );
  if( p == NULL ) {
    ssc_error("%s,%d: out of memory error!\n", __func__, __LINE__ );
    return NULL;
  }

  memset( p, 0, sizeof( ssc_ahl_dev_t ) );
  ssc_strlcpy( p->devname, device_name, SSC_MAX_DEVNAME_LEN );
  p->direction = direction;

  p->data = (ssc_qti_audio_t *) cul_malloc( sizeof( ssc_qti_audio_t ) );
  if( p->data == NULL ) {
    ssc_error("%s,%d: out of memory error!\n", __func__, __LINE__ );
    cul_free( p );
    return NULL;
  }

  memset( p->data, 0, sizeof( ssc_qti_audio_t ) );
  p_qti = ( ssc_qti_audio_t * ) p->data;
  p_qti->p_ahl_dev = p;

  /* open depends on stream configuration so do it in ssc_ahl_config later on */
  if( set_default_configuration )
  {
    /* except for the 'openidle' (telephony voice) case */
    if( direction == SSC_AHL_PCM_STREAM_PLAYBACK ) {
      flags |= PCM_OUT;
    } else {
      flags |= PCM_IN;
    }

    /* initial device configuration by opening it */
    p_qti->pcm = pcm_open( flags, p->devname );
    if( p_qti->pcm < 0 ) {
      ssc_error("%s,%d: could not open device %s error: %d!\n",
                __func__, __LINE__, p->devname, p_qti->pcm );
      error = -1;
    }

    /* check device state */
    if( ! error ) {
      if( !pcm_ready( p_qti->pcm ) ) {
        ssc_error("%s,%d: pcm device %s not ready error (bad fd)!\n",
                  __func__, __LINE__, p->devname );
        error = -EBADFD;
      }
    }

    /* Set the default parameters for telephony audio. These value are
       taken from Qualcomm's infamous sample applications aplay/arec */
    if( ! error ) {
      p_qti->pcm->channels = 2;
      p_qti->pcm->rate     = 44100;
      p_qti->pcm->flags    = flags;
      p_qti->pcm->format   = SNDRV_PCM_FORMAT_S16_LE;

      if( set_params( p_qti->pcm ) ) {
        ssc_error( "%s,%d: failed to set params for device %s error: %d!\n",
                   __func__, __LINE__, p->devname, errno );
        error = -errno;
      }
    }

    /* prepare sound device */
    if( ! error ) {
      if( pcm_prepare( p_qti->pcm ) ) {
        ssc_error( "%s,%d: failed in pcm_prepare for device %s error: %d!\n",
                   __func__, __LINE__, p->devname, errno );
        error = -errno;
      }
    }

    /* start streaming */
    if( ! error ) {
      if ( ioctl( p_qti->pcm->fd, SNDRV_PCM_IOCTL_START ) ) {
        ssc_error("%s,%d: hostless IOCTL_START error no %d\n",
                  __func__, __LINE__, errno );
        error = -errno;
      }
    }
  }

  if( error ) {
    error = ssc_ahl_close( p );
    if( error ) {
      ssc_error("%s,%d: error while closing qti audio device: %s\n",
                __func__, __LINE__, p->devname );
    }

    p = NULL;
  }

  return p;
}


int ssc_ahl_config( ssc_ahl_dev_t* p, ssc_ahl_config_t* p_ssc_ahl_config )
{
  ssc_qti_audio_t* p_qti = ( ssc_qti_audio_t * ) p->data;
  uint16_t         channels = p_ssc_ahl_config->channels;
  unsigned         flags = PCM_MMAP;
  int              error = 0;

  p->p_config = p_ssc_ahl_config;

  ssc_message("%s,%d: setup alsa with -> bits_per_sample: %u"
              ", channels: %u, sample_rate: %u, period_duration: %u\n",
              __func__, __LINE__,
              p_ssc_ahl_config->bits_per_sample,
              p_ssc_ahl_config->channels,
              p_ssc_ahl_config->sample_rate,
              p_ssc_ahl_config->max_buf_periods,
              p_ssc_ahl_config->period_duration );


  /* no reconfiguration supported, required for Qualcomm MDM SoC's */
  if( p->configured ) {
    ssc_error("%s,%d: device %s is already configured, can be done only once!\n",
              __func__, __LINE__, p->devname );
    error = -1;
  }

  /* open device */
  if( ! error ) {
    /* set of the pcm flags, taken over from Qualcomm */
    if( p->direction == SSC_AHL_PCM_STREAM_PLAYBACK ) {
      flags |= PCM_OUT;
    } else {
      flags |= PCM_IN;
    }

    /* strange but we are not responsible for that! */
    if (channels == 1)
      flags |= PCM_MONO;
    else if (channels == 4)
      flags |= PCM_QUAD;
    else if (channels == 6)
      flags |= PCM_5POINT1;
    else if (channels == 8)
      flags |= PCM_7POINT1;
    else
      flags |= PCM_STEREO;

    /* initial device configuration by opening it */
    p_qti->pcm = pcm_open( flags, p->devname );
    if( p_qti->pcm < 0 ) {
      ssc_error("%s,%d: could not open device %s error!\n",
                __func__, __LINE__, p->devname );
      error = -1;
    }
  }

  /* check device state */
  if( ! error ) {
    if( !pcm_ready( p_qti->pcm ) ) {
      ssc_error("%s,%d: pcm device %s not ready error (bad fd)!\n",
                __func__, __LINE__, p->devname );
      error = -EBADFD;
    }
  }

  /* set device params */
  if( ! error ) {
    /* set audio device parameters */
    p_qti->pcm->channels = channels;
    p_qti->pcm->rate = p_ssc_ahl_config->sample_rate;
    p_qti->pcm->flags = flags;
    p_qti->pcm->format = SNDRV_PCM_FORMAT_S16_LE;

    error = set_params( p_qti->pcm );
    if( error ) {
      ssc_error( "%s,%d: params setting failed error\n", __func__, __LINE__ );
    }
  }

  if( mmap_buffer( p_qti->pcm ) ) {
    ssc_error( "%s,%d: setup of memory map buffer failed error!\n",
               __func__, __LINE__ );
    error = -errno;
  }

  if( pcm_prepare( p_qti->pcm ) ) {
    ssc_error( "%s,%d: failed in pcm_prepare error!\n",
               __func__, __LINE__ );
    error = -errno;
  }

  if( error ) {
    ssc_error( "%s,%d: could configure pcm device %s error\n",
               __func__, __LINE__, p->devname );
  } else {
    p_qti->pfd[0].fd = p_qti->pcm->timer_fd;
    p_qti->pfd[0].events = POLLIN;
    p_qti->bytes_per_frame = p_ssc_ahl_config->channels * p_ssc_ahl_config->bits_per_sample / 8;
    p->configured = 1;
  }

  return error;
}


ssize_t ssc_ahl_pcm_read( ssc_ahl_dev_t* p, void* p_buf, const ssize_t frames )
{
  ssc_qti_audio_t* p_qti = ( ssc_qti_audio_t * ) p->data;
  struct pcm*      pcm = p_qti->pcm;
  size_t           bytes_per_frame =  p_qti->bytes_per_frame;
  size_t           bufsize = pcm->period_size; /* here bytes, incorrect QTI terminology! */
  size_t           bufsize_in_frames  = bufsize / bytes_per_frame;
  ssize_t          bytes_to_write = frames * bytes_per_frame;
  size_t           bytes_written = 0;
  ssize_t          avail;
  struct timespec  last_read_op, current_time;
  long             polling_diff_us;
  long             max_polling_diff_us = 1000L * 1000L; /* one second */
  struct pollfd*   pfd = p_qti->pfd;
  const int        nfds = 1;
  u_int8_t*        dst_addr = NULL;
  int              error = 0;

  if( ! p_qti->start ) {
    /* make sure that last_read_op is always initialized */
    error = clock_gettime( CLOCK_MONOTONIC_RAW, &last_read_op );
    if( error ) {
      last_read_op.tv_sec = 0;
      last_read_op.tv_nsec = 0L;
    }

    if( ioctl( pcm->fd, SNDRV_PCM_IOCTL_START ) ) {
      error = -errno;
      if( errno == EPIPE ) {
        ssc_error( "%s,%d: failed in SNDRV_PCM_IOCTL_START error!\n",
                   __func__, __LINE__ );
        /* we failed to make our window -- try to restart */
        pcm->running = 0;
      } else {
        ssc_error( "%s,%d: could not start audio stream, error no %d \n",
                   __func__, __LINE__, errno );
        return -errno;
      }
    } else {
      p_qti->start = 1;
    }
  }

  pfd[0].fd = pcm->fd;
  pfd[0].events = POLLIN;

  while( bytes_to_write > 0 )
  {
    if (!pcm->running) {
      if (pcm_prepare(pcm))
        return --errno;
      p_qti->start = 0;
    }

    pcm->sync_ptr->flags = SNDRV_PCM_SYNC_PTR_APPL | SNDRV_PCM_SYNC_PTR_AVAIL_MIN;
    error = sync_ptr( pcm );
    if( error == EPIPE ) {
      ssc_error( "%s,%d: failed in sync_ptr, EPIPE error!\n",
                 __func__, __LINE__ );
      /* we failed to make our window -- try to restart */
      pcm->running = 0;
        continue;
    }

    /*
     * Check for the available data in driver. If available data is
     * less than avail_min we need to wait
     */
    avail = pcm_avail( pcm );

    if (avail < 0)
      return avail;
    if (avail < pcm->sw_p->avail_min) {
      poll( pfd, nfds, TIMEOUT_INFINITE );

      clock_gettime( CLOCK_MONOTONIC_RAW, &current_time );
      polling_diff_us = MILLION * (current_time.tv_sec - last_read_op.tv_sec) +
        (current_time.tv_nsec - last_read_op.tv_nsec) / 1000L;

      if( polling_diff_us > max_polling_diff_us ) {
        ssc_error( "%s,%d: maximum polling time %ld ms is exceeded by %ld ms and no buf data available, real time error!\n",
                   __func__, __LINE__, max_polling_diff_us/1000L, polling_diff_us/1000L );
        return -1;
      } else {
        sched_yield();
        continue;
      }
    }

    /* We got enough audio dma buffer space to process, so reset time stamp */
    clock_gettime( CLOCK_MONOTONIC_RAW, &last_read_op );

    /*
     * Now that we have data size greater than avail_min available to
     * to be read we need to calcutate the buffer offset where we can
     * start reading from.
     */
    dst_addr = dst_address(pcm);

    /*
     * write audio bytes to requestor
     */
    memcpy( p_buf + bytes_written, dst_addr, MIN( bufsize, bytes_to_write ) );
    bytes_to_write -= bufsize;
    bytes_written  += bufsize;

    /*
     * Increment the application pointer with data read from kernel.
     * Update kernel with the new sync pointer.
     */
    pcm->sync_ptr->c.control.appl_ptr += bufsize_in_frames;
    pcm->sync_ptr->flags = 0;

    error = sync_ptr( pcm );
    if (error == EPIPE) {
      ssc_error( "%s,%d: failed in sync_ptr 2 error!\n",
                 __func__, __LINE__ );
      /* we failed to make our window -- try to restart */
      pcm->underruns++;
      pcm->running = 0;
      continue;
    }
  }

  /* when we reach here we assume that we have everything read from the audio hw! */
  return frames;
}


ssize_t ssc_ahl_pcm_write( ssc_ahl_dev_t* p, const void* p_buf, const ssize_t frames )
{
  ssc_qti_audio_t* p_qti = ( ssc_qti_audio_t * ) p->data;
  struct pcm*      pcm = p_qti->pcm;
  size_t           bytes_per_frame =  p_qti->bytes_per_frame;
  size_t           bufsize = pcm->period_size; /* here bytes, incorrect QTI terminology! */
  size_t           bufsize_in_frames  = bufsize / bytes_per_frame;
  ssize_t          bytes_to_write = frames * bytes_per_frame;
  size_t           bytes_written = 0;
  ssize_t          avail;
  struct timespec  last_write_op, current_time;
  long             polling_diff_us;
  long             max_polling_diff_us = 1000L * 1000L; /* one second */
  struct pollfd*   pfd = p_qti->pfd;
  const int        nfds = 1;
  u_int8_t*        dst_addr = NULL;
  int              error = 0;

  if( ! p_qti->start) {
    /* make sure that last_write_op is always initialized */
    error = clock_gettime( CLOCK_MONOTONIC_RAW, &last_write_op );
    if( error ) {
      last_write_op.tv_sec = 0;
      last_write_op.tv_nsec = 0L;
    }
  }


  while( bytes_to_write > 0 )
  {
    if ( !pcm->running ) {
      if( pcm_prepare( pcm ) ) {
        ssc_error( "%s,%d: failed in pcm_prepare for device %s with error %d!\n",
                   __func__, __LINE__, p->devname, errno );
        return -errno;
      }
      pcm->running = 1;
      p_qti->start = 0;
    }

    /* Sync the current Application pointer from the kernel (QTI impl. from aplay.c) */
    pcm->sync_ptr->flags = SNDRV_PCM_SYNC_PTR_APPL |
      SNDRV_PCM_SYNC_PTR_AVAIL_MIN; //SNDRV_PCM_SYNC_PTR_HWSYNC;
    error = sync_ptr( pcm );
    if( error == EPIPE ) {
      ssc_error( "%s,%d, failed in sync_ptr for device %s error!\n",
                 __func__, __LINE__, p->devname );
      /* we failed to make our window -- try to restart */
      pcm->underruns++;
      pcm->running = 0;
      continue;
    }

    /*
     * Check for the available buffer in driver. If available buffer is
     * less than avail_min we need to wait
     */
    avail = pcm_avail( pcm );
    if( avail < 0 ) {
      ssc_error( "%s,%d: failed in pcm_avail for device %s error!\n",
                 __func__, __LINE__, p->devname );
      return avail;
    }
    if( avail < pcm->sw_p->avail_min ) {
      poll( pfd, nfds, TIMEOUT_INFINITE );

      clock_gettime( CLOCK_MONOTONIC_RAW, &current_time );
      polling_diff_us = MILLION * (current_time.tv_sec - last_write_op.tv_sec) +
        (current_time.tv_nsec - last_write_op.tv_nsec) / 1000L;

      if( polling_diff_us > max_polling_diff_us ) {
        ssc_error( "%s,%d: maximum polling time %ld ms is exceeded by %ld ms and no buf data available, real time error!\n",
                   __func__, __LINE__, max_polling_diff_us/1000L, polling_diff_us/1000L );
        return -1;
      } else {
        continue;
      }
    }

    /* We got enough audio dma buffer space to process, so reset time stamp */
    clock_gettime( CLOCK_MONOTONIC_RAW, &last_write_op );

    /*
     * Now that we have buffer size greater than avail_min available to
     * to be written we need to calcutate the buffer offset where we can
     * start writting.
     */
    dst_addr = dst_address( pcm );


    /*
     * write audio bytes to shared memory
     */
    memset( dst_addr , 0x0, bufsize );

    memcpy( dst_addr, p_buf + bytes_written, MIN( bufsize, bytes_to_write ) );
    bytes_to_write -= bufsize;
    bytes_written  += bufsize;

    /*
     * Increment the application pointer with data written to kernel.
     * Update kernel with the new sync pointer.
     */
    pcm->sync_ptr->c.control.appl_ptr += bufsize_in_frames;
    pcm->sync_ptr->flags = 0;

    error = sync_ptr( pcm );
    if (error == EPIPE) {
      ssc_error( "%s,%d: failed in sync_ptr 2 error!\n",
                 __func__, __LINE__ );
      /* we failed to make our window -- try to restart */
      pcm->underruns++;
      pcm->running = 0;
      continue;
    }

    /*
     * If we have reached start threshold of buffer prefill,
     * its time to start the driver.
     */
    if( ! p_qti->start) {
      if( ioctl( pcm->fd, SNDRV_PCM_IOCTL_START ) ) {
        error = -errno;
        if( errno == EPIPE ) {
          ssc_error( "%s,%d: failed in SNDRV_PCM_IOCTL_START error!\n",
                     __func__, __LINE__ );
          /* we failed to make our window -- try to restart */
          pcm->underruns++;
          pcm->running = 0;
          continue;
        } else {
          ssc_error( "%s,%d: could not start audio stream, error no %d \n",
                     __func__, __LINE__, errno );
          return -errno;
        }
      } else {
        p_qti->start = 1;
      }
    }
  }

  /* when we reach here we assume that we have everything written to the audio hw! */
  return frames;
}

/*! @} */

#endif /* #ifdef USE_QTI_AUDIO */
