/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <string.h>
#include <ssc_tasks.h>
#include <wav_cb.h>
#include <mp3_cb.h>
#include <log.h>


void release_ssc_tasks( ssc_tasks_t* p )
{
  if( p->p_task_hm ) {
    hm_free_with_keys( p->p_task_hm );
  }

  cul_free( p );
}


ssc_tasks_t* init_ssc_tasks( void )
{
  ssc_tasks_t* p = cul_malloc( sizeof( ssc_tasks_t ) );

  if( p == NULL ) {
    ssc_error( "%s,%d: out of memory error!\n", __func__, __LINE__ );
    return NULL;
  }

  memset( p, 0, sizeof( ssc_tasks_t ) );

  p->p_task_hm = hm_alloc();
  if( p->p_task_hm == NULL ) {
    ssc_error( "%s,%d: out of memory error!\n", __func__, __LINE__ );
    cul_free( p );
    return NULL;
  }

  return p;
}


ssc_proc_handle_t* ssc_get_task( ssc_tasks_t* p_tasks, const char* id )
{
  ssc_proc_handle_t* p_proc = NULL;
  string_t* id_s = string_new_from( id );

  if( id_s == NULL ) {
    ssc_error( "%s,%d: out of memory error!\n", __func__, __LINE__ );
    return NULL;
  }

  p_proc = (ssc_proc_handle_t *) hm_find_val_for_key( p_tasks->p_task_hm, id_s );
  if( p_proc == NULL ) {
    ssc_message( "%s,%d: task id '%s' not found!\n", __func__, __LINE__, id );
  }

  string_release( id_s );

  return p_proc;
}


int ssc_add_task( ssc_tasks_t* p_tasks, const char* id, ssc_proc_handle_t* p_proc )
{
  string_t* id_s = string_new_from( id );
  int error = 0;

  if( id_s == NULL ) {
    ssc_error( "%s,%d: out of memory error!\n", __func__, __LINE__ );
    error = -1;
  }

  if( ! error ) {
    if( hm_assoc_with_key( p_tasks->p_task_hm, id_s, p_proc ) == NULL ) {
      ssc_error( "%s,%d: could not insert task %s error, close it now!\n",
                 __func__, __LINE__, id );
      error = -1;
    } else {
      ++(p_tasks->task_cnt);
    }
  }

  if( id_s ) {
    string_release( id_s );
  }

  return error;
}

void ssc_remove_task( ssc_tasks_t* p_tasks, const char* id )
{
  string_t* id_s = string_new_from( id );

  if( id_s == NULL ) {
    ssc_error( "%s,%d: out of memory error!\n", __func__, __LINE__ );
    return;
  }

  ssc_message("%s,%d: remove task: %s ...\n", __func__, __LINE__, id );
  (void)hm_dissoc_with_key( p_tasks->p_task_hm, id_s );
  --(p_tasks->task_cnt);

  string_release( id_s );
}


int ssc_create_open_idle_task( ssc_tasks_t*  p_tasks,
                               const char*   id,
                               const char*   device_name,
                               const int     dir,
                               void*         p_ssc )
{
  int error = 0;
  ssc_proc_handle_t* p_proc = NULL;

  p_proc = ssc_open_idle( id,
                          device_name,
                          dir,
                          p_ssc );

  if( p_proc ) {
    error = ssc_add_task( p_tasks, id, p_proc );
    if( error ) {
      ssc_error( "%s,%d: could not insert openidle task %s error, close it now!\n",
                 __func__, __LINE__, id );
      p_proc->p_proc_release_cb( p_proc );
    }
  } else {
    error = -1;
  }

  return error;
}


int ssc_create_wav_playback_task( ssc_tasks_t*  p_tasks,
                                  const char*   id,
                                  const char*   device_name,
                                  const char*   filename,
                                  const int     repeat,
                                  void* p_ssc )
{
  int error = 0;
  ssc_proc_handle_t* p_proc = NULL;

  /* For the moment we do only the WAV case */
  p_proc = ssc_player_open( id,
                            device_name,
                            filename,
                            repeat,
                            wav_init_cb,
                            wav_release_cb,
                            wav_producer_cb,
                            p_ssc );

  if( p_proc ) {
    error = ssc_add_task( p_tasks, id, p_proc );
    if( error ) {
      ssc_error( "%s,%d: FATAL: could insert playback task %s error, close it now!\n",
                 __func__, __LINE__, id );
      p_proc->p_proc_release_cb( p_proc );
    }
  } else {
    error = -1;
  }

  return error;
}

int ssc_create_wav_record_task( ssc_tasks_t*   p_tasks,
                                const char*    id,
                                const char*    device_name,
                                const char*    filename,
                                const uint16_t channels,
                                const uint32_t sample_rate,
                                const int32_t  duration,
                                void* p_ssc )
{
  int error = 0;
  ssc_proc_handle_t* p_proc = NULL;

  /* recording is implemented only for wav */
  p_proc = ssc_recorder_open( id,
                              device_name,
                              filename,
                              channels,
                              sample_rate,
                              duration,
                              wav_init_cb,
                              wav_release_cb,
                              wav_consumer_cb,
                              p_ssc );

  if( p_proc ) {
    error = ssc_add_task( p_tasks, id, p_proc );
    if( error ) {
      ssc_error( "%s,%d: FATAL: could insert record task %s error, close it now!\n",
                 __func__, __LINE__, id );
      p_proc->p_proc_release_cb( p_proc );
    }
  } else {
    error = -1;
  }

  return error;
}


#if HAVE_LIBMAD
int ssc_create_mp3_playback_task( ssc_tasks_t*  p_tasks,
                                  const char*   id,
                                  const char*   device_name,
                                  const char*   filename,
                                  const int     repeat,
                                  void* p_ssc )
{
  int error = 0;
  ssc_proc_handle_t* p_proc = NULL;

  /* For the moment we do only the WAV case */
  p_proc = ssc_player_open( id,
                            device_name,
                            filename,
                            repeat,
                            mp3_init_cb,
                            mp3_release_cb,
                            mp3_producer_cb,
                            p_ssc );

  if( p_proc ) {
    error = ssc_add_task( p_tasks, id, p_proc );
    if( error ) {
      ssc_error( "%s,%d: FATAL: could insert playback task %s error, close it now!\n",
                 __func__, __LINE__, id );
      p_proc->p_proc_release_cb( p_proc );
    }
  } else {
    error = -1;
  }

  return error;
}
#endif /* #if HAVE_LIBMAD */


int ssc_close_task( ssc_tasks_t* p_tasks, const char* id )
{
  ssc_proc_handle_t* p_proc;
  int error = 0;

  p_proc = ssc_get_task( p_tasks, id );
  if( p_proc == NULL ) {
    ssc_error( "%s,%d: could not find task id '%s' error!\n",
               __func__, __LINE__, id );
    error = -1;
  } else {
    error = ssc_proc_release( p_proc );
  }

  return error;
}


static void* lambda_p_kv_send_close( void* key, void* val )
{
  hm_kv_t* kv = (hm_kv_t *)val;
  ssc_proc_handle_t* p_proc = ( ssc_proc_handle_t * )kv->val;

  ssc_message("%s,%d: closing proc id '%s' ...\n", __func__, __LINE__, p_proc->id );
  ssc_proc_fire_event( p_proc, ssc_proc_close_request );

  return NULL;
}


int ssc_close_all_tasks( ssc_tasks_t* p_tasks )
{
  hm_doseq( p_tasks->p_task_hm, 0, lambda_p_kv_send_close );

  return 0;
}
