/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef SSC_TASKS_H
#define SSC_TASKS_H

#include <config.h>
#include <ssc_proc.h>
#include <olcutils/hashmap.h>

#ifdef __cplusplus
extern "C" {
#endif


/*! \file ssc_tasks.h
    \brief task handling

    \addtogroup tasks
    @{
 */


/*!
 * context data for task management
 */
typedef struct {
  hm_t*               p_task_hm;                  /*!< hashmap with all open tasks */
  int                 task_cnt;                   /*!< number of tasks in hm */
} ssc_tasks_t;


/*!
 * destructor for task management context data
 *
 * \param p pointer to task context data whose memory is going to be released
 */
void release_ssc_tasks( ssc_tasks_t* p );


/*!
 * constructor of task context data
 *
 * \return pointer to newly allocated and initialized task context data
 */
ssc_tasks_t* init_ssc_tasks( void  );


/*!
 * retrieve task von task hashtable
 *
 * \param p_tasks pointer to task context data where the task processing object is retrieved from
 * \param id pointer to task identifier string
 * \return pointer to the associated task processing object or NULL if not found
 */
ssc_proc_handle_t* ssc_get_task( ssc_tasks_t* p_tasks, const char* id );


/*!
 * add pair of task identifier string and task processing object to task hashtable
 *
 * \param p_tasks pointer to task context data where the task processing object is add to
 * \param id pointer to task identifier string
 * \param p_proc pointer to task processing object
 * \return 0 in case of success or negative error code
 */
int ssc_add_task( ssc_tasks_t* p_tasks, const char* id, ssc_proc_handle_t* p_proc );


/*!
 * remove pair of task identifier string and task processing object from task hashtable
 *
 * ATTENTTION: This function shall be exclusively called when the associated backgrund
 *             process has been terminated!
 *
 * \param p_tasks pointer to task context data where the task processing object to remove
 * \param id pointer to task identifier string
 */
void ssc_remove_task( ssc_tasks_t* p_tasks, const char* id );


/*!
 * creates a 'task' which just keeps an audio device open.
 *
 * This is a special case since  no background process is associated with
 * this task. It just maintains a handle to underlying audio device which
 * is kept open.  This concept is used by Qualcomm  to signal a dedicated
 * state  e.g. a  streaming  of  telephony voice  audio  data to  another
 * processing core.
 *
 * \param p_tasks pointer to tasks context
 * \param id pointer to unique process identifier
 * \param device_name pointer to audio device file name
 * \param dir audio processing direction, ::SSC_AHL_PCM_STREAM_PLAYBACK or
 *        ::SSC_AHL_PCM_STREAM_CAPTURE
 * \param p_ssc pointer to global data object used for back reference.
 * \return 0 in case of success or negative error code
 */
int ssc_create_open_idle_task(  ssc_tasks_t*  p_tasks,
                                const char*   id,
                                const char*   device_name,
                                const int     dir,
                                void*         p_ssc );


/*!
 * creates a 'task' for the playback of an audio file in MS WAV format
 *
 * \param p_tasks pointer to tasks context
 * \param id pointer to unique process identifier
 * \param device_name pointer to audio device file name
 * \param filename filename to be played back, must provide audio data in WAV format
 * \param repeat when not equal zero, continously replay from beginning
 * \param p_ssc pointer to global data object used for back reference.
 * \return 0 in case of success or negative error code
 */
int ssc_create_wav_playback_task( ssc_tasks_t*  p_tasks,
                                  const char*   id,
                                  const char*   device_name,
                                  const char*   filename,
                                  const int     repeat,
                                  void* p_ssc );


/*!
 * creates a 'task' for the recording of an audio file in MS WAV format
 *
 * \param p_tasks pointer to tasks context
 * \param id pointer to unique process identifier
 * \param device_name pointer to audio device file name
 * \param filename filename to be played back, must provide audio data in WAV format
 * \param channels number of audio channels to record, 2 = stereo, 1 = mono
 * \param sample_rate sample rate used in the recording, hw must of course support it.
 * \param p_ssc pointer to global data object used for back reference.
 * \return 0 in case of success or negative error code
 */
int ssc_create_wav_record_task( ssc_tasks_t*   p_tasks,
                                const char*    id,
                                const char*    device_name,
                                const char*    filename,
                                const uint16_t channels,
                                const uint32_t sample_rate,
                                const int32_t  duration,
                                void* p_ssc );


#if HAVE_LIBMAD
/*!
 * creates a 'task' for the playback of an audio file in MP3 format
 *
 * \param p_tasks pointer to tasks context
 * \param id pointer to unique process identifier
 * \param device_name pointer to audio device file name
 * \param filename filename to be played back, must provide audio data in MP3 format
 * \param repeat when not equal zero, continously replay from beginning
 * \param p_ssc pointer to global data object used for back reference.
 * \return 0 in case of success or negative error code
 */
int ssc_create_mp3_playback_task( ssc_tasks_t*  p_tasks,
                                  const char*   id,
                                  const char*   device_name,
                                  const char*   filename,
                                  const int     repeat,
                                  void* p_ssc );
#endif


/*!
 * 'closing' of processing backround tasks or handle object
 *
 * This function is  generally called to close a task.  Tasks that do not
 * contain a  background process  (currently only 'openidle')  are closed
 * directly, the associated  memory is released and the  entry is removed
 * from the task list. All other 'real' tasks cannot be released directly
 * as long as the background process  is still running. In this case, the
 * associated  process  context  is  signaled  to  stop  processing.  The
 * \ref  ssc_event_handler  "event  handler" releases  the  corresponding
 * memory  and removes  the  object from  the task  list  when the  event
 * ::ssc_proc_terminated is received for this task.
 *
 * \param p_tasks pointer to tasks context
 * \param id pointer to unique process identifier
 * \return 0 in case of success or negative error code
 */
int ssc_close_task( ssc_tasks_t* p_tasks, const char* id );


/*!
 * 'closing' of all tasks in task hashmap
 *
 * This   function   iterates  over   the   task   hashtable  and   calls
 * ssc_close_task() for each  element. It is called by  the 'close --all'
 * and 'terminate' commands.
 *
 * \param p_tasks pointer to tasks context
 * \return 0 in case of success or negative error code
 */
int ssc_close_all_tasks( ssc_tasks_t* p_tasks );


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef SSC_TASKS_H */
