/*
    intercom - Event based Interprocess Communication for Dummies
    Copyright 2016 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <olcutils/alloc.h>
#include <log.h>
#include <tcp_msg.h>


/*! \file tcp_msg.h
    \brief transport layer

    \addtogroup control
    @{
 */


#define INC_SEGMENT( p, x )      ( ( (x) < ( (p->segments_allocated) - 1) ) ? (x)+1 : 0 )
#define DEC_SEGMENT( p, x )      ( ( (x) > 0 ) ? (x)-1 : (p->segments_allocated) )


void icom_release_data_stream( t_icom_data_stream* p )
{
  int i;

  for( i=0; i < p->segments_allocated; ++i )
    cul_free( p->segment[i].data );

  cul_free( p );
}

t_icom_data_stream* icom_create_data_stream( const int max_segments,
                                             const int max_data_size,
                                             const char* delim_bytes,
                                             const int delim_bytes_len )
{
  t_icom_data_stream* p;
  int i;

  if( max_segments > ICOM_STREAM_MAX_SEGMENTS ) {
    ssc_error( "%s,%d nur of max allowed segments is exceeded error!\n", __func__, __LINE__ );
    return NULL;
  }

  p = cul_malloc( sizeof( t_icom_data_stream ) );
  if( p == NULL )
    return p;
  memset( p, 0, sizeof( t_icom_data_stream ) );
  p->max_data_size = max_data_size;
  memcpy( p->delim_bytes, delim_bytes, (size_t)delim_bytes_len );
  p->delim_bytes_len = delim_bytes_len;

  for( i=0; i < max_segments; ++i ) {
    p->segment[i].data = cul_malloc( (size_t)max_data_size );
    p->segment[i].size = 0;
    if( p->segment[i].data == NULL )
      break;
  }

  p->segments_allocated = i;

  if( i < max_segments ) {
    ssc_error( "%s,%d: could only allocate memory for %d segments, %d have been requested!\n",
               __func__, __LINE__, i, max_segments );
    icom_release_data_stream( p );
    return NULL;
  }

  return p;
}


void icom_push_data_segment( t_icom_data_stream* p, void* data, const int len )
{
  int size;

  if( len > p->max_data_size ) {
    ssc_error( "%s,%d: data chunk of %d bytes is too large error!\n",
               __func__, __LINE__, len );
    size = p->max_data_size;
  } else {
    size = len;
  }

  if( p->seg_filled < ICOM_STREAM_MAX_SEGMENTS ) {

    memcpy( p->segment[ p->seg_write ].data, data, (size_t)size );
    p->segment[ p->seg_write ].size = size;
    p->segment[ p->seg_write ].start_idx = 0;

    p->seg_write = INC_SEGMENT( p, p->seg_write );
    ++( p->seg_filled );
  } else {
    ssc_error( "%s,%d: data stream overflow error, last msg is lost!\n",
               __func__, __LINE__ );
  }
}


t_segment* icom_peek_data_segment( t_icom_data_stream* p )
{
  t_segment* p_seg;

  if( p->seg_filled > 0 ) {
    p_seg = & p->segment[ p->seg_read ];
  } else {
    ssc_error( "%s,%d: data stream underflow error!\n", __func__, __LINE__ );
    p_seg = NULL;
  }

  return p_seg;
}

void icom_drop_data_segment( t_icom_data_stream* p )
{
  if( p->seg_filled > 0 ) {
    p->seg_read = INC_SEGMENT( p, p->seg_read );
    --(p->seg_filled);
  }
}

int icom_data_stream_bytes( t_icom_data_stream* p )
{
  int i = p->seg_filled, bytes_cached = 0, seg = p->seg_read;
  t_segment *p_seg;

  while( i-- > 0 ) {
    p_seg = & p->segment[seg];
    bytes_cached += p_seg->size - p_seg->start_idx;
    seg = INC_SEGMENT( p, seg );
  }

  return bytes_cached;
}

int icom_get_next_data_stream_msg( t_icom_data_stream* p, t_icom_data_stream_msg *p_msg )
{
  int segs_read, seg_idx;
  int i, found = 0;
  char *p_msg_data = p_msg->data;
  int msg_len = 0;
  const int max_data_size = p->max_data_size;
  const char* delim_bytes = p->delim_bytes;
  const int delim_bytes_len = p->delim_bytes_len;

  for( segs_read = 0, seg_idx = p->seg_read;
       segs_read < p->seg_filled;
       ++segs_read, seg_idx = INC_SEGMENT( p, seg_idx ) ) {

    t_segment* ps = & p->segment[seg_idx];

    for( i = ps->start_idx; i < ps->size ;  ++i ) {
      if( i <= ps->size - delim_bytes_len  &&  !memcmp( & ps->data[i], delim_bytes, (size_t)delim_bytes_len ) ) {
        found = 1;
        p->seg_filled -= segs_read;
        p->seg_read = seg_idx;
        ps->start_idx = i + delim_bytes_len;

        if( ps->start_idx >= ps->size ) {
          --( p->seg_filled );
          p->seg_read = INC_SEGMENT( p, p->seg_read );
        }
        break;

      } else {
        if( msg_len < max_data_size ) {
          p_msg_data[msg_len++] = ps->data[i];
        } else {
          ssc_error( "%s,%d: drop too large message error!\n", __func__, __LINE__ );
          msg_len = 0;
        }
      }
    }
  }

  p_msg->len = msg_len;
  return found;
}

void icom_print_stream_stats( t_icom_data_stream *p )
{
  int i = p->seg_filled, seg = p->seg_read;
  t_segment *p_seg;

  printf("---- stream stats ----\n");
  printf("segments_allocated: %d\n", p->segments_allocated);
  printf("max_data_size: %d\n", p->max_data_size);
  printf("seg_read: %d\n", p->seg_read);
  printf("seg_write: %d\n", p->seg_write);
  printf("seg_filled: %d\n", p->seg_filled);
  printf("\nsegments:\n");

  while( i-- > 0 ) {
    p_seg = & p->segment[seg];
    printf("\tidx: %d, size: %d, start-idx: %d\n", seg, p_seg->size, p_seg->start_idx );
    seg = INC_SEGMENT( p, seg );
  }
}

/*! @} */
