/*
    intercom - Event based Interprocess Communication for Dummies
    Copyright 2016 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef TCP_MSG_H
#define TCP_MSG_H

#ifdef __cplusplus
extern "C" {
#endif

/*! \file tcp_msg.h
    \brief transport layer

    \addtogroup control
    @{
 */


/*!
 * maximum number of segmentations of a message in TCP stream
 */
#define ICOM_STREAM_MAX_SEGMENTS 10

/*!
 * maximum number of characters (octets) to separate messages
 */
#define MAX_DELIM_BYTES           3


/*!
 * message segment type of series of bytes received from socket as one unit
 */
typedef struct {
  char*                       data;                         /*!< pointer to recv data */
  int                         size;                         /*!< size of recv data chunk */
  int                         start_idx;                    /*!< chunk index to be read next */
} t_segment;

/*!
 * data stream input state context
 */
typedef struct {
  t_segment                   segment[ICOM_STREAM_MAX_SEGMENTS]; /*!< segment input array */
  int                         segments_allocated;           /*!< nr of successfully allocated segments, normally corresponds to ICOM_STREAM_MAX_SEGMENTS */
  int                         max_data_size;                /*!< maximum payload data size of one intercom message */
  int                         seg_read;                     /*!< index of the segment to read from next */
  int                         seg_write;                    /*!< index of the segment to write to next */
  int                         seg_filled;                   /*!< number of cached (filled) segmtens */
  char                        delim_bytes[MAX_DELIM_BYTES]; /*!< delimiter bytes to separate icom messages */
  int                         delim_bytes_len;              /*!< number of delimiter bytes used */
} t_icom_data_stream;

/*!
 * intercom message output type
 */
typedef struct {
  char*                       data;                         /*!< pointer to intercom message */
  int                         len;                          /*!< length of the message */
} t_icom_data_stream_msg;


/*!
 * release data_stream object
 *
 * \param p pointer to data_stream object to be released
 */
void icom_release_data_stream( t_icom_data_stream* p );


/*!
 * create data stream splitter object
 *
 * The  network  transport  layer  of  the  Unix  operating  system  exclusively
 * abstracts a logical data stream and  thus does not give  any guarantees about
 * how data is actually segmented. For a reliable communication of messages e.g.
 * over TCP  we need to introduce  delimiter characters used as  markers between
 * messages.  By default  the  newline character  is used  for  that. All  bytes
 * between these  markers need to  be combined  when messages are  received. The
 * process is  conceptualized as a  data stream  which consists out  of segments
 * which have been received from the network and delimiter bytes used to restore
 * the original IPC message.
 *
 * \param max_segments maximum number of input operations of recv
 * \param max_data_size maximum length of data received in one recv op (segment)
 * \param delim_bytes byte sequence used as delimiter between IPC messages
 * \param delim_bytes_len length of delimiter byte sequence
 * \return pointer to newly instantiated object or NULL in case of error
 */
t_icom_data_stream* icom_create_data_stream( const int max_segments,
                                             const int max_data_size,
                                             const char* delim_bytes,
                                             const int delim_bytes_len );


/*!
 * push a new data chunk received from the network to the data stream splitter
 *
 * \param p pointer to instance data
 * \param data pointer to chunk of data reveiced from the network
 * \param len number of reveived bytes
 */
void icom_push_data_segment( t_icom_data_stream* p, void* data, const int len );


/*!
 * returns number of internally cached bytes (for debugging purposes only)
 *
 * \param p pointer to instance data
 * \return number of internally cached bytes
 */
int icom_data_stream_bytes( t_icom_data_stream* p );


/*!
 * retrieve next IPC message when present
 *
 * When  the internally  cached  byte  stream includes  a  delimiter stream  the
 * associated IPC message  is pulled out from  the cache and the  a logical true
 * (1) value is returned. The function shall  be invoked in a loop unter no more
 * messages can be retrieved.
 *
 * \param p pointer to instance data
 * \param p_msg pointer to IPC output data
 * \return true (1) when IPC has been successfully retrieved, otherwise 0
 */
int icom_get_next_data_stream_msg( t_icom_data_stream* p, t_icom_data_stream_msg *p_msg );


/*!
 * prints internal state of data stream (for debugging purposes only)
 *
 * \param p pointer to instance data
 */
void icom_print_stream_stats( t_icom_data_stream *p );


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef TCP_MSG_H */
