/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <utils.h>
#include <common.h>
#include <olcutils/alloc.h>


/*! \file utils.h
    \brief utility functions

    \addtogroup utils
    @{
 */


/*! create a new list and omit element data */
slist_t* clone_slist_without_elem( const slist_t* l, const void* data )
{
  slist_t* nl = slist_alloc();

  if( nl ) {
    while( l && l->data ) {
      if( l->data != data ) {
        nl = slist_prepend( nl, l->data );
      } else {
      }
      l = l->next;
    }
  }
  return nl;
}

static string_t* cstr_cmd_tok( slist_t** tok_lst_handle, const char* tok )
{
  slist_t* l = *tok_lst_handle;
  string_t* t = string_new_from( tok );

  l = slist_prepend( l, t );
  *tok_lst_handle = l;

  return t;
}

static void* free_cstr_cmd_tok( void * p ) {
  string_release( p );
  return NULL;
}


hm_t* ssc_init_cmd_hm( const t_pair_tok_fcnt parser_table[], const int parser_table_len )
{
  slist_t* tok_list;
  hm_t* p = hm_alloc();
  int i;

  if( p ) {
    tok_list = slist_alloc();
    if( tok_list ) {

      for( i=0;  i < parser_table_len;  ++i) {
        hm_assoc_with_key( p,
                           cstr_cmd_tok( &tok_list, parser_table[i].p_cmd_tok ),
                           parser_table[i].p_fcnt );
      }

      slist_free_deep( tok_list, free_cstr_cmd_tok );
    } else {
      hm_free( p, 0 );
      p = NULL;
    }
  }

  return p;
}

/*!
 * ssc_strlcpy -- size-bounded string copying and concatenation
 *
 * for more detailed information refer to: bsd man page strlcpy
 *
 * taken from:
 *   http://stackoverflow.com/questions/2933725/my-version-of-strlcpy
 */
size_t ssc_strlcpy(char *dest, const char *src, size_t len)
{
    char *d = dest;
    char *e = dest + len; /* end of destination buffer */
    const char *s = src;

    /* Insert characters into the destination buffer
       until we reach the end of the source string
       or the end of the destination buffer, whichever
       comes first. */
    while (*s != '\0' && d < e)
        *d++ = *s++;

    /* Terminate the destination buffer, being wary of the fact
       that len might be zero. */
    if (d < e)        // If the destination buffer still has room.
        *d = 0;
    else if (len > 0) // We ran out of room, so zero out the last char
                      // (if the destination buffer has any items at all).
        d[-1] = 0;

    /* Advance to the end of the source string. */
    while (*s != '\0')
        s++;

    /* Return the number of characters
       between *src and *s,
       including *src but not including *s .
       This is the length of the source string. */
    return s - src;
}


static void* strlist2argv_reducer( void* f, void* r )
{
  char** h_argv = (char **)f;
  string_t* e = (string_t *)r;
  const int max_arg_len = SSC_MAX_MSG_LEN;

  *h_argv = (char *) cul_malloc( max_arg_len );
  if( *h_argv )
    string_tmp_cstring_from( e, *h_argv, max_arg_len );

  return --h_argv;
}

int string2arglist( string_t* args, char* argv[], const int max_args )
{
  slist_t *lst, *lst2;
  string_t* targs = string_trim( string_retain( args ) );
  int argc;

  memset( argv, 0, sizeof( char * ) * max_args );

  lst = string_split( targs, " \t", max_args );
  argc = slist_cnt(lst);
  lst2 = slist_prepend( lst, argv + argc - 1 );

  (void)slist_reduce( lst2, strlist2argv_reducer, NULL );
  slist_free_first( lst2 );
  slist_free_deep( lst, (lambda_t)string_release );
  string_release( targs );

  return argc;
}

const char *get_filename_ext(const char *filename) {
    const char *dot = strrchr(filename, '.');
    if(!dot || dot == filename) return "";
    return dot + 1;
}

/*! @} */
