/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef UTILS_H
#define UTILS_H

#include <stdlib.h>

#include <olcutils/hashmap.h>
#include <olcutils/lambda.h>


/*! \file utils.h
    \brief utility functions

    \addtogroup utils
    @{
 */


#ifndef MIN
#define MIN(x,y)  (((x)<(y)) ? (x) : (y))
#endif

#ifndef MAX
#define MAX(x,y)  (((x)>(y)) ? (x) : (y))
#endif

#ifdef __cplusplus
extern "C" {
#endif


/*! key-value pair for association cstring token to function for simple parsers */
typedef struct {
  const char*                 p_cmd_tok;                    /*!< cstring token */
  const lambda_t              p_fcnt;                       /*!< function pointer */
} t_pair_tok_fcnt;


/*! generated casted key-value pair of type t_pair_tok_fcnt */
#define PAIR_TOK_FCNT( t, f ) { (t), (lambda_t)(f) }


/*! create a new list and omit element data */
slist_t* clone_slist_without_elem( const slist_t* l, const void* data );


/*!
 * releases hash map token hash map
 */
void ssc_release_cmd_hm( hm_t* p );


/*!
 * create hash map for where a command keywork (key) is associated
 * with a processing function
 *
 * \param parser_table const input table for the association
 * \param parser_table_len number of elements in parser_table
 * \return created hash map
 */
hm_t* ssc_init_cmd_hm( const t_pair_tok_fcnt parser_table[], const int parser_table_len );


/*!
 * ssc_strlcpy -- size-bounded string copying and concatenation
 *
 * for more detailed information refer to: bsd man page strlcpy
 *
 * taken from:
 *   http://stackoverflow.com/questions/2933725/my-version-of-strlcpy
 *
 * \param dest destination where to store the copied bytes in
 * \param src source where bytes are copied from
 * \param len max length in bytes (octetts) include '\0' termination
 * \return number of copied bytes
 */
size_t ssc_strlcpy(char *dest, const char *src, size_t len);


/*!
 * splits a size delimited string to C argument vector
 *
 * \param args size delimited string with input argument list
 * \param argv
 * \param max_args
 * \return number of found arguments (argc)
 *
 */
int string2arglist( string_t* args, char* argv[], const int max_args );


/*!
 * return pointer to file name extension from null terminated c-string
 *
 * \param filename null terminated c-string with filename
 * \return pointer to file extension of pointer to an empty string if not found
 */
const char *get_filename_ext(const char *filename);


/*! like strdup but with memory debug instrumentation enabled
 *
 * \param s pointer to null terminated string to duplicate
 * \return pointer to newly allocated string
 */
char* cul_strdup( const char* s );


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef UTILS_H */
