/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <uv_utils.h>
#include <log.h>
#include <olcutils/alloc.h>


/*! \file uv_utils.c
    \brief libuv utility functions

    These are specifically libuv related helper functions.

    \addtogroup utils
    @{
 */

void free_write_req(uv_write_t *req)
{
  write_req_t *wr = (write_req_t*) req;

  cul_free(wr->buf.base);
  cul_free(wr);
}


void alloc_buffer( uv_handle_t *handle __attribute__((unused)),
                   size_t suggested_size,
                   uv_buf_t *buf )
{
  buf->base = (char*) cul_malloc(suggested_size); /* out of memory handled in libuv */
  buf->len = suggested_size;
}


void dlt_client_write( uv_write_t *req, int wstatus )
{
  static int err_cnt = 0;
  static int last_wstatus = 0;

  if ( wstatus ) {
    if( err_cnt < 3 ) {
      ssc_error( "%s,%d: write error no %d: %s",
                 __func__, __LINE__, err_cnt, uv_strerror(wstatus) );
    }

    if( last_wstatus == wstatus ) {
      ++err_cnt;
    }

    last_wstatus = wstatus;
  } else {
    err_cnt =0;
    last_wstatus = 0;
  }

  free_write_req(req);
}


int send_to_client( uv_stream_t *client, const char* buf, const size_t buf_len )
{
  write_req_t* req;
  int error =0;

  /* allocate new transmission request */
  req = (write_req_t*) cul_malloc(sizeof(write_req_t));
  if( req == NULL ) {
    ssc_error( "%s,%d: out of memory error!", __func__, __LINE__ );
    error = -1;
  }

  /* allocate payload buffer */
  if( ! error ) {
    alloc_buffer( (uv_handle_t *)req, buf_len, & req->buf );
    if( req->buf.base == NULL ) {
      ssc_error( "%s,%d: out of memory error!", __func__, __LINE__ );
      cul_free( req );
      error = -2;
    }
  }

  /* copy and send resp payload content */
  if( ! error ) {
    int retcode;

    memcpy( req->buf.base, buf, buf_len );
    retcode = uv_write( (uv_write_t*)req, client, &req->buf, 1, dlt_client_write );
    if( retcode ) {
      ssc_error( "%s,%d: got write error: %s",
                 __func__, __LINE__,  uv_err_name( retcode ) );
      error = -3;
    }
  }

  return error;
}


int send_to_client_list( const slist_t* l, const char* buf, const size_t buf_len )
{
  int error=0;

  while( l && l->data && ! error )
  {
    uv_stream_t *a_client = (uv_stream_t *)(l->data);

    error |= send_to_client( a_client, buf, buf_len );
    if( error ) {
      ssc_error( "%s,%d: send error occured!", __func__, __LINE__ );
    }

    l = l->next;
  }

  return error;
}


int ssc_reply_to_sender( uv_stream_t* client, const char* fmt, ... )
{
  va_list args;
  char reply[4096];
  int  reply_len;

  va_start( args, fmt );
  reply_len = vsnprintf( reply, sizeof(reply), fmt, args );
  va_end( args );


  if( reply_len > 0 ) {
      reply_len = send_to_client( client, reply, reply_len );
  }

  ssc_message("%s,%d: -> %s", __func__, __LINE__, reply );

  return reply_len;
}


int ssc_reply_to_all( const slist_t* l, const char* fmt, ... )
{
  va_list args;
  char reply[4096];
  int  reply_len;

  va_start( args, fmt );
  reply_len = vsnprintf( reply, sizeof(reply), fmt, args );
  va_end( args );


  if( reply_len > 0 ) {
    (void)send_to_client_list( l, reply, reply_len );
  }

  ssc_message("%s,%d: -> %s", __func__, __LINE__, reply );

  return reply_len;
}


/*! @} */
