/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef UV_UTILS_H_
#define UV_UTILS_H_

#include <uv.h>
#include <stdlib.h>
#include <olcutils/slist.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \file uv_utils.h
    \brief libuv utility functions

    These are specifically libuv related helper functions.

    \addtogroup utils
    @{
 */

/*!
 * helper type for call back invocation as proposed in the libuv documentation
 */
typedef struct {
  uv_write_t req;   /*!< write request type, refer to libuv for further information */
  uv_buf_t   buf;   /*!< uv buffer data type (data pointer and length) */
} write_req_t;

/*!
 * helper function for freeing write request data
 *
 * \param req pointer to write request data to be freed entirely
 */
void free_write_req( uv_write_t *req );


/*!
 * helper function for a write request data allocation as proposed
 * in the libuv documentation
 *
 * \param handle unused
 * \param suggested_size requested data buffer size
 * \param buf pointer to libuv buf
 *
 */
void alloc_buffer( uv_handle_t *handle __attribute__((unused)),
                   size_t suggested_size,
                   uv_buf_t *buf );

/*!
 * helper function for write request callback as prposed
 * in the libuv documentation
 *
 * \params req pointer to write request data
 * \params wstatus write status, 0 in case of success otherwise error code
 */
void dlt_client_write( uv_write_t *req, int wstatus );


/*!
 * send buffer to TCP client connection
 *
 * \param client pointer to tcp client connection (libuv)
 * \param buf pointer to send data buffer
 * \param buf_len size of data buffer
 * \return 0 in case of success, otherwise negative error code
 */
int send_to_client( uv_stream_t *client, const char* buf, const size_t buf_len );


/*!
 * like fprintf but sends data to uv stream instead of file descriptor
 *
 * \param client data stream client where to print to
 * \param format string like printf
 * \return number of printed characters or negative error code
 */
int ssc_reply_to_sender( uv_stream_t* client, const char* fmt, ... );


/*!
 * like fprintf but sends data to uv stream instead of file descriptor
 *
 * \param client data stream client where to print to
 * \param format string like printf
 * \return number of printed characters or negative error code
 */
int ssc_reply_to_all( const slist_t* l, const char* fmt, ... );


/*!
 * send buffer to list of TCP client connections
 *
 * \param pointer to client pointer connection list (libuv)
 * \param buf pointer to send data buffer
 * \param buf_len size of data buffer
 * \return 0 in case of success, otherwise negative error code
 */
int send_to_client_list( const slist_t* l, const char* buf, const size_t buf_len );


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef UV_UTILS_H_ */
