/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <string.h>
#include <wav_cb.h>
#include <common.h>
#include <olcutils/alloc.h>
#include <utils.h>
#include <log.h>


/*! \file wav_cb.c
    \brief call backs to read and write audio files in MS WAV format

    \addtogroup wav
    @{
 */

int wav_release_cb( void* p_arg )
{
  wav_proc_t* p = (wav_proc_t *)p_arg;
  int error = 0;

  if( p->fp ) {
    error = wavfp_close( p->fp );
    if( error ) {
    ssc_error( "%s,%d: error occurred when closing wav file '%s'!\n",
               __func__, __LINE__, p->filename );
    }
  }

  cul_free( p );

  return error;
}


void* wav_init_cb( const char* filename,
                   const ssc_atl_dir dir,
                   const int repeat,
                   ssc_ahl_config_t** config_handle )
{
  wav_proc_t* p;
  FORMATCHUNK* p_fmt_chunk;
  const char* mode = ( dir==ssc_atl_read ) ? "r" : "w";

  p = cul_malloc( sizeof( wav_proc_t ) );
  if( p == NULL ) {
    ssc_error("%s,%d: out of memory error!\n", __func__, __LINE__ );
    return NULL;
  }

  memset( p, 0, sizeof( wav_proc_t ) );
  ssc_strlcpy( p->filename, filename, SSC_MAX_PATH );
  p->dir = dir;

  p->fp = wavfp_open( filename, mode );
  if( p->fp == NULL ) {
    ssc_error( "%s,%d: could not open file '%s' for %s error!\n",
               __func__, __LINE__, filename,
               (dir==ssc_atl_read) ? "reading" : "writing" );
    wav_release_cb( p );
    return NULL;
  }

  if( dir==ssc_atl_read ) {
    /* playback case */
    p->repeat = repeat;

    p_fmt_chunk = & p->fp->header.formatChunk;

    p->config.channels = p_fmt_chunk->channels;
    p->config.sample_rate = p_fmt_chunk->sampleRate;
    p->config.bits_per_sample = p_fmt_chunk->bitsPerSample;
    p->config.max_buf_periods = g_ssc_max_buf_periods;
    p->config.period_duration = g_ssc_period_duration;

    *config_handle = & p->config;

    if( p_fmt_chunk->bitsPerSample != 16 ) {
      ssc_error( "%s,%d: file '%s' deviates from 16 bits per sample!\n",
                 __func__, __LINE__, filename );
      wav_release_cb( p );
      return NULL;
    }
  } else {
    /* recording case */

    /* The stream configuration comes from outsite in this case
       so we need to copy it over because atl is owner */
    ssc_ahl_config_t* p_config = & p->config;
    memcpy( p_config, *config_handle, sizeof( ssc_ahl_config_t ) );
    *config_handle = p_config;

    p->fp->header.formatChunk.channels = p_config->channels;
    p->fp->header.formatChunk.sampleRate = p_config->sample_rate;
    p->fp->header.formatChunk.bitsPerSample = 16;
    p->fp->header.formatChunk.bytesPerSec = p_config->sample_rate * p_config->channels * 2;
    p->fp->header.formatChunk.blockAlign = p_config->channels * 2;
    p->max_recorded_frames = (size_t) p_config->duration_in_seconds *
      p_config->sample_rate;

    ssc_message("%s: sample_rate: %d\n", __func__, p->fp->header.formatChunk.sampleRate );
    ssc_message("%s: channels: %d\n", __func__, p->fp->header.formatChunk.channels );
    ssc_message("%s: bits per samples %d\n", __func__, p->fp->header.formatChunk.bitsPerSample );
  }

  return (void *)p;
}

/* for playback */
size_t wav_producer_cb( void* p_arg, ssc_atl_period_t* p_period )
{
  wav_proc_t*  p = (wav_proc_t *)p_arg;
  FORMATCHUNK* p_fmt_chunk = & p->fp->header.formatChunk;
  int32_t      chunksize = p->fp->header.dataChunk.chunksize;
  char*        buf = p_period->buf.buf;
  const size_t bufsize = p_period->buf.buf_size;
  long         curr_pos;
  size_t       available_data, bytes_to_read, bytes_read = 0;
  size_t       tot_bytes_read = 0, return_val = 0;
  unsigned int bytes_per_frame;

  if( p->invocation_cnt == 0 ) {

    p_period->type = ssc_atl_config_type;
    p_period->p_config = & p->config;

    p->config.channels = p_fmt_chunk->channels;
    p->config.sample_rate = p_fmt_chunk->sampleRate;
    p->config.bits_per_sample = p_fmt_chunk->bitsPerSample;

    /* This is actually audio hardware dependend, but since the audio period
       is created initially here, we initialize these global conf data here
       as well
    */
    p->config.max_buf_periods = g_ssc_max_buf_periods;
    p->config.period_duration = g_ssc_period_duration;
    return_val = sizeof( ssc_atl_period_t );

  } else {
    p_period->type = ssc_atl_data_type;

    /* determine remaining sample data in wav file */
    curr_pos = ftell( p->fp->fp ) - sizeof(WAVHEADER);
    available_data = (size_t) ( chunksize - curr_pos );

    /* read from current position in file */
    bytes_to_read = MIN( available_data, bufsize );
    if( bytes_to_read > 0 ) {
      bytes_read =  wavfp_read( (char *)buf, 1, bytes_to_read, p->fp );
      tot_bytes_read += bytes_read;
    }

    /* in case repeat flag is enabled and more data has been requested,
       restart reading from beginning of file */
    bytes_to_read = bufsize - bytes_to_read;
    if( p->repeat && bytes_to_read > 0 ) {
      if( ! fseek( p->fp->fp, sizeof(WAVHEADER), SEEK_SET ) ) {
        bytes_read =  wavfp_read( buf + tot_bytes_read, 1, bytes_to_read, p->fp );
        tot_bytes_read += bytes_read;
      }
    }

    bytes_per_frame = p->config.bits_per_sample / 8 * p->config.channels;
    return_val = tot_bytes_read / bytes_per_frame;
  }

  ++( p->invocation_cnt );

  return return_val;
}

/* for recording */
size_t wav_consumer_cb( void* p_arg, ssc_atl_period_t* p_period )
{
  wav_proc_t*  p = (wav_proc_t *)p_arg;
  char*        buf = p_period->buf.buf;
  const size_t bufsize = p_period->buf.buf_size;
  size_t       return_val, bytes_per_frame, bytes_written;

  /* configuration does not change after record state
     which makes the situation much easier */

  bytes_per_frame = p->config.bits_per_sample / 8 * p->config.channels;
  bytes_written = wavfp_write( buf, 1, bufsize, p->fp );
  return_val = bytes_written / bytes_per_frame;
  ++( p->invocation_cnt );

  return return_val;
}

/*! @} */
