/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef WAV_CB_H
#define WAV_CB_H

#include <ssc_atl.h>
#include <wavio.h>

#ifdef __cplusplus
extern "C" {
#endif


/*! \file wav_cb.h
    \brief call backs to read and write audio files in MS WAV format

    \addtogroup wav
    @{
 */


/*! internal data context object for reading from and writing to wav files */
typedef struct {
  char              filename[SSC_MAX_PATH];     /*!< filename, must have the extension .wav */
  WAVFP*            fp;                         /*!< file handle */
  ssc_atl_dir       dir;                        /*!< direction which indicates reading or writing */
  ssc_ahl_config_t  config;                     /*!< audio stream configuration data, owned and manged here */
  int               repeat;                     /*!< continuously repeat when set to true */
  size_t            bytes_per_period;           /*!< number of bytes per audio period */
  size_t            invocation_cnt;             /*!< counts number of r/w callback invocation */
  size_t            max_recorded_frames;        /*!< maximum amount of frames to record */
} wav_proc_t;


/*!
 * initialization of of reader/writer context data for audio files in MS WAV format
 *
 * \param filename fully qualified filename, extension .wav must be used
 * \param dir read (playback) or write (record)
 * \param repeat continuously repeat when set to true
 * \param config_handle returned pointer (handle) to audio stream configuration data
 * \return pointer to wav reader/write context data or NULL in case of error
 */
void*     wav_init_cb( const char* filename,
                       const ssc_atl_dir dir,
                       const int repeat,
                       ssc_ahl_config_t** config_handle );


/*!
 * release of of reader/writer context data for audio files in MS WAV format
 *
 * \param p pointer to wav reader/write context data or NULL in case of error
 * \return 0 in case of success, otherwise negative error code
 */
int       wav_release_cb( void* p );


/*!
 * producer of audio data read out from audio file in MS WAV format
 *
 * \param p pointer to wav reader context data
 * \param p_period pointer to generated audio period data
 * \return number of audio frames read
 */
size_t    wav_producer_cb( void* p, ssc_atl_period_t* p_period );


/*!
 * consumer of audio data written to audio file in MS WAV format
 *
 * \param p pointer to wav writer context data
 * \param p_period pointer to provided audio period data
 * \return number of audio frames written
 */
size_t    wav_consumer_cb( void* p, ssc_atl_period_t* p_period );


/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef WAV_CB_H */
