/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <wavio.h>
#include <olcutils/alloc.h>
#include <log.h>


/*! \file wavio.c
    \brief processing of MS-WAV fileformat headers

    \addtogroup wav
    @{
 */


WAVFP* wavfp_open( const char *filename, const char *mode )
{
  WAVFP       *wfp;
  WAVHEADER   defaultHeader = { DEFAULT_RIFF, DEFAULT_FORMATCHUNK, DEFAULT_DATACHUNK };
  char        stdio_mode[3];

  /* create descriptor object */
  wfp = cul_malloc( sizeof( WAVFP ) );
  if( wfp == NULL )
  {
    ssc_error( "%s: out of memory error", __func__ );
    return NULL;
  }

  /* insert default setup */
  memset( wfp, 0, sizeof(WAVFP) );
  memcpy( &wfp->header, &defaultHeader, sizeof( WAVHEADER ) );


  /* setup read/write mode */
  if( mode[0] == 'r' )
  {
    strcpy( stdio_mode, "rb" );
    wfp->dir = WAVFP_READ;
  }
  else if( mode[0] == 'w' )
  {
    strcpy( stdio_mode, "wb" );
    wfp->dir = WAVFP_WRITE;
  }
  else
  {
    ssc_error( "%s: wrong mode argument error", __func__ );
    cul_free( wfp );
    return NULL;
  }


  wfp->fp = fopen( filename, stdio_mode );
  if( wfp->fp == NULL )
  {
    cul_free( wfp );
    ssc_error( "%s: file %s not found error", __func__, filename );
    return NULL;
  }

  /* read/write header */
  switch( wfp->dir )
  {
    case WAVFP_READ:
      if( fread( &wfp->header, sizeof(WAVHEADER), 1, wfp->fp ) != 1 )
      {
        fclose( wfp->fp );
        cul_free( wfp );
        ssc_error( "%s: could not read wav header in file %s", __func__, filename );
        return NULL;
      }
      break;


    case WAVFP_WRITE:
      if( fwrite( &wfp->header, sizeof(WAVHEADER), 1, wfp->fp ) != 1 )
      {
        fclose( wfp->fp );
        cul_free( wfp );
        ssc_error( "%s: could not write wav header to file %s", __func__, filename );
        return NULL;
      }
      break;
  }

  return wfp;
}


int wavfp_close( WAVFP* wfp )
{
  int retcode = 0;

  if( wfp==NULL )
    return -1;

  if(  wfp->dir == WAVFP_WRITE )
  {
    /*
     *  update wav header in case of output stream
     */
    wfp->header.riff.filesize = sizeof(WAVHEADER) + wfp->bytes_processed - 8;
    wfp->header.dataChunk.chunksize =  wfp->bytes_processed;

    retcode = fseek( wfp->fp, 0, SEEK_SET );

    if( retcode == 0 )
    {
      if( fwrite( &wfp->header, sizeof(WAVHEADER), 1, wfp->fp ) != 1 )
      {
        ssc_error( "%s: could not finalize wav header", __func__ );
        retcode = -1;
      }
    }
    else
    {
      ssc_error( "%s: seek error", __func__ );
    }
  }


  fclose( wfp->fp );
  cul_free( wfp );

  return retcode;
}


size_t  wavfp_read( void* ptr, size_t size, size_t nmemb, WAVFP* stream )
{
  size_t elems_read;

  elems_read = fread( ptr, size, nmemb, stream->fp );
  stream->bytes_processed += size * elems_read;

  return elems_read;
}


size_t  wavfp_write( void* ptr, size_t size, size_t nmemb, WAVFP* stream )
{
  size_t elems_written;

  elems_written = fwrite( ptr, size, nmemb, stream->fp );
  stream->bytes_processed += size * elems_written;

  return elems_written;
}




#ifdef _COMPILE_GEN_440HZ
int gen_440Hz(void)
{
  WAVFP*    fp;
  int16_t   frame[PA_DSP_BLOCKSIZE_8KHZ], frame_index = 0;
  int16_t   k, rate;
  int       retcode = 0;

  fp = wavfp_open("test.wav", "w" );
  if( fp == NULL )
  {
    fprintf( stderr, "could not open test.wav error, ");
    return -1;
  }


  rate = fp->header.formatChunk.sampleRate;
  for( k=0; k < rate*3; ++k )
  {
    frame[frame_index++] = 12000 * sin( 440.0 * 2*3.1415*k / rate  );
    if( frame_index>= PA_DSP_BLOCKSIZE_8KHZ )
    {
      if( wavfp_write( frame, 2*PA_DSP_BLOCKSIZE_8KHZ, 1, fp ) != 1 )
      {
        fprintf( stderr, "write error, ");
        retcode = -1;
        break;
      }
      frame_index = 0;
    }
  }


  if( wavfp_close( fp ) )
  {
    fprintf( stderr, "could not close test.wav error, ");
    retcode = -1;
  }

  return retcode;
}
#endif

/*! @} */
