/*
    Simple Sound Controller 2
    Copyright 2023 Otto Linnemann

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef _WAVIO_H_
#define _WAVIO_H_


#include <stdio.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif


/*! \file wavio.h
    \brief processing of MS-WAV fileformat headers

    \addtogroup wav
    @{
 */


/*!
 *  Resource Interchange Format Header
 *
 *  Refer to https://en.wikipedia.org/wiki/Resource_Interchange_File_Format for more detailed
 *  explanation.
 */
typedef struct
{
  int8_t    chunkID[4];                                     /*!< 4 byte identifier chunk */
  int32_t   filesize;                                       /*!< filesize */
  int8_t    riffType[4];                                    /*!< riff type */

} RIFF;


/*!
 *  Format chunk
 */
typedef struct
{
  int8_t    chunkID[4];                                     /*!< 4 byte identifier chunk */
  int32_t   cunksize;                                       /*!< lenght of the format chunk */
  int16_t   formatTag;                                      /*!< format tag */
  int16_t   channels;                                       /*!< number of channels */
  int32_t   sampleRate;                                     /*!< sample rate */
  int32_t   bytesPerSec;                                    /*!< bytes per second */
  int16_t   blockAlign;                                     /*!< channels * bits/sample / 8 */
  int16_t   bitsPerSample;                                  /*!< bits in one sample */

} FORMATCHUNK;


/*!
 *  Data chunk
 */
typedef struct
{
  int8_t    chunkID[4];                                     /*!< 4 byte identifier chunk */
  int32_t   chunksize;                                      /*!< length of the data chunk  */

} DATACHUNK;


/*!
 *  MS WAV Format header
 */
typedef struct
{
    RIFF          riff;                                     /*!< RIFF header */
    FORMATCHUNK   formatChunk;                              /*!< format chunk */
    DATACHUNK     dataChunk;                                /*!< data chunk */
} WAVHEADER;


/*!
 *  Streaming direction (read or write)
 */
typedef enum
{
    WAVFP_READ,                                             /*!< read from wav file */
    WAVFP_WRITE                                             /*!< write to wav file */
} WAVDIR;


/*!
 *  File pointer to WAV audio I/O stream
 */
typedef struct
{
  FILE        *fp;                                          /*!< FILE pointer */
  WAVHEADER   header;                                       /*!< pointer to WAV header */
  WAVDIR      dir;                                          /*!< direction of data flow */
  int32_t     bytes_processed;                              /*!< number of processed bytes */

} WAVFP;


/* === Constant defintions =============================================== */



/*!
 *  Default RIFF initializer
 */
#define DEFAULT_RIFF  { \
  { 'R', 'I', 'F', 'F' }, \
  0, \
  { 'W', 'A', 'V', 'E' }  \
}


/*!
 *  Default format initializer for PCM data
 */
#define DEFAULT_FORMATCHUNK { \
  { 'f', 'm', 't', ' ' }, \
  16,     /* chunksize */ \
  0x1,    /* PCM */ \
  1,      /* one channel */ \
  8000,   /* sample rate */ \
  16000,  /* bytes per second */ \
  2,      /* block align */ \
  16      /* bits per sample */ \
}


/*!
 *  Default data chunk initializer
 */
#define DEFAULT_DATACHUNK { \
  { 'd', 'a', 't', 'a' }, \
  0 \
}


/*!
 *  lenght of the error string
 */
#define WAV_ERR_STR_LEN   80


/* === Function prototpes ================================================ */


/*!
 *  wavfp_open()
 *
 *  Open a MS WAV format audio I/O stream
 *
 *  Function parameters
 *    - filename:     name of the file to read from or write to
 *    - mode:         operation mode. use "r" for read, "w" for write
 *
 *  Returnparameter
 *    - R:            pointer to the WAVFP descriptor or
 *                    NULL in case of error
 *
 *  Creates a file descriptor for read/write access to a binary
 *  audio data file in Microsoft WAV format. Usage is simlar to
 *  buffered standard I/O (fopen). In addition to fopen this
 *  functions reads or writes the WAV header, too.
 *
 *  @see fopen
 */
WAVFP* wavfp_open( const char *filename, const char *mode );



/*!
 *  wavfp_close()
 *
 *  Closes a MS WAV format audio I/O stream
 *
 *  Function parameters
 *    - wfp:          pointer to the WAVFP descriptor
 *
 *  Returnparameter
 *    - R:            0 in case of success or error code
 *
 *  Updates the WAV header and closes the WAVFP file descriptor
 *  This function corresponds to the C-library function fclose.
 *
 *  @see fclose
 */
int wavfp_close( WAVFP* wfp );



/*!
 *  wavfp_read()
 *
 *  reads a chunk of bytes from WAV file descriptor
 *
 *  Function parameters
 *    - ptr:          pointer to buffer where read data is stored to
 *    - size:         number of octets of one element
 *    - nmemb:        number of elements to read
 *    - stream:       WAV I/O data stream to read from
 *
 *  Returnparameter
 *    - R:            number of elements which have been successfully read
 *
 *  Reads a chunk of raw data from MS WAV audio data stream. This
 *  function corresponds to the C-library function fread
 *
 *  @see fread
 */
size_t  wavfp_read( void* ptr, size_t size, size_t nmemb, WAVFP* stream );



/*!
 *  wavfp_write()
 *
 *  writes a chunk of bytes to WAV file descriptor
 *
 *  Function parameters
 *    - ptr:          pointer to buffer where data is retrieved from
 *    - size:         number of octets of one element
 *    - nmemb:        number of elements to read
 *    - stream:       WAV I/O data stream to write to
 *
 *  Returnparameter
 *    - R:            number of elements which have been successfully read
 *
 *  Writes a chunk of raw data to MS WAV audio data stream. This
 *  function corresponds to the C-library function fwrite
 *
 *  @see fwrite
 */
size_t  wavfp_write( void* ptr, size_t size, size_t nmemb, WAVFP* stream );



/*! @} */

#ifdef __cplusplus
}
#endif


#endif /* _WAVIO_H */
