/*
 * Copyright (c) 2013-2018 The Linux Foundation. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *    * Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    * Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials provided
 *      with the distribution.
 *    * Neither the name of The Linux Foundation nor the names of its
 *      contributors may be used to endorse or promote products derived
 *      from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Changes from Qualcomm Technologies, Inc. are provided under the following license:
 * Copyright (c) Qualcomm Technologies, Inc. and/or its subsidiaries.
 * SPDX-License-Identifier: BSD-3-Clause-Clear
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include "IPACM_Iface.h"
#include "IPACM_ConntrackListener.h"
#include "IPACM_ConntrackClient.h"
#include "IPACM_Log.h"

#define LO_NAME "lo"

extern IPACM_EvtDispatcher cm_dis;
extern void ParseCTMessage(struct nf_conntrack *ct);

IPACM_ConntrackClient *IPACM_ConntrackClient::pInstance = NULL;
IPACM_ConntrackListener *CtList = NULL;

/* ================================
		 Local Function Definitions
		 =================================
*/
IPACM_ConntrackClient::IPACM_ConntrackClient()
{
	IPACMDBG("\n");

	tcp_hdl = NULL;
	udp_hdl = NULL;
	tcp_filter = NULL;
	udp_filter = NULL;
}

IPACM_ConntrackClient* IPACM_ConntrackClient::GetInstance()
{
	if(pInstance == NULL)
	{
		pInstance = new IPACM_ConntrackClient();

		pInstance->udp_filter = nfct_filter_create();
		if(pInstance->udp_filter == NULL)
		{
			IPACMERR("unable to create UDP filter\n");
			delete pInstance;
			return NULL;
		}
		IPACMDBG("Created UDP filter\n");

		pInstance->tcp_filter = nfct_filter_create();
		if(pInstance->tcp_filter == NULL)
		{
			IPACMERR("unable to create TCP filter\n");
			delete pInstance;
			return NULL;
		}
		IPACMDBG("Created TCP filter\n");
	}

	return pInstance;
}

int IPACM_ConntrackClient::IPAConntrackEventCB
(
	 enum nf_conntrack_msg_type type,
	 struct nf_conntrack *ct,
	 void *data
	 )
{
	ipacm_cmd_q_data evt_data;
	ipacm_ct_evt_data *ct_data;
	uint8_t ip_type = 0;
	uint16_t sport = 0;
	uint16_t dport = 0;
	IPACM_Config *config_instance = NULL;
	u_int8_t  protocol, tcp_state;
	protocol = nfct_get_attr_u8(ct, ATTR_REPL_L4PROTO);
	if((protocol == IPPROTO_TCP))
		tcp_state = nfct_get_attr_u8(ct, ATTR_TCP_STATE);
	IPACMDBG("Event callback called with msgtype is :%d\n",type);

	/*Avoiding processing of tcp conntracks if state is not established, if not fin_wait, if msg type is not destroy*/
	if((protocol == IPPROTO_TCP) && ((tcp_state != TCP_CONNTRACK_ESTABLISHED) && (tcp_state != TCP_CONNTRACK_FIN_WAIT) && (NFCT_T_DESTROY != type)))
	{
		IPACMDBG("unexpected conntracks recieving protocol = %d  msg_type = %d\n", protocol,  type);
		goto IGNORE;
	}

	/* Retrieve ip type */
	ip_type = nfct_get_attr_u8(ct, ATTR_REPL_L3PROTO);
	sport = nfct_get_attr_u16(ct, ATTR_ORIG_PORT_SRC);
	sport = ntohs(sport);
	dport = nfct_get_attr_u16(ct, ATTR_ORIG_PORT_DST);
	dport = ntohs(dport);

	/* Avoid processing conntrack with DNS 53 port */
	if(dport == 53 || sport == 53)
	{
		IPACMDBG("iptype: %d: sport: %d: dport: %d\n", ip_type, sport, dport);
		goto IGNORE;
	}
	IPACMDBG("iptype: %d\n", ip_type);

#ifndef CT_OPT
	if(AF_INET6 == ip_type)
	{
		config_instance = IPACM_Config::GetInstance();
		if (config_instance == NULL)
		{
			IPACMERR("Config instance creation failed");
			goto IGNORE;
		}
		if (!config_instance->IsIpv6CTEnabled()
#ifdef FEATURE_IPV6_NAT
				 && !config_instance->ipv6_nat_enable
#endif
			)
		{
			IPACMDBG("ip type AF_INET6 %d, IPv6CT %d, ipv6 NAT %d\n", ip_type, config_instance->IsIpv6CTEnabled(), config_instance->ipv6_nat_enable);
			IPACMDBG("Ignoring ipv6(%d) connections\n", ip_type);
			goto IGNORE;
		}
	}
#endif

	ct_data = (ipacm_ct_evt_data *)malloc(sizeof(ipacm_ct_evt_data));
	if(ct_data == NULL)
	{
		IPACMERR("unable to allocate memory \n");
		goto IGNORE;
	}

	ct_data->ct = ct;
	ct_data->type = type;

	evt_data.event = IPA_PROCESS_CT_MESSAGE;
	evt_data.evt_data = (void *)ct_data;

	if(AF_INET6 == ip_type)
	{
		IPACMDBG_H("sending IPA_PROCESS_CT_MESSAGE_V6\n");
		evt_data.event = IPA_PROCESS_CT_MESSAGE_V6;
	}

	if(0 != IPACM_EvtDispatcher::PostEvt(&evt_data))
	{
		IPACMERR("Error sending Conntrack message to processing thread!\n");
		free(ct_data);
		goto IGNORE;
	}

/* NFCT_CB_STOLEN means that the conntrack object is not released after the
	 callback That must be manually done later when the object is no longer needed. */
	return NFCT_CB_STOLEN;

IGNORE:
	nfct_destroy(ct);
	return NFCT_CB_STOLEN;

}

int IPACM_ConntrackClient::IPA_Conntrack_Filters_Ignore_Bridge_Addrs
(
	 struct nfct_filter *filter
)
{
	int fd;
	fd = socket(AF_INET, SOCK_DGRAM, 0);
	if(fd < 0)
	{
		PERROR("unable to open socket");
		return -1;
	}

	int ret;
	uint32_t ipv4_addr;
	struct ifreq ifr;

	/* retrieve bridge interface ipv4 address */
	memset(&ifr, 0, sizeof(struct ifreq));
	ifr.ifr_addr.sa_family = AF_INET;
	(void)strlcpy(ifr.ifr_name, IPACM_Iface::ipacmcfg->ipa_virtual_iface_name, sizeof(ifr.ifr_name));
	IPACMDBG("bridge interface name (%s)\n", ifr.ifr_name);

	if(strlen(IPACM_Iface::ipacmcfg->ipa_virtual_iface_name) >= sizeof(ifr.ifr_name))
	{
		IPACMERR("interface name overflows: len %d\n",
			strlen(IPACM_Iface::ipacmcfg->ipa_virtual_iface_name));
		close(fd);
		return -1;
	}

	ret = ioctl(fd, SIOCGIFADDR, &ifr);
	if (ret < 0)
	{
		IPACMERR("unable to retrieve (%s) interface address\n",ifr.ifr_name);
		close(fd);
		return -1;
	}
	IPACMDBG("Interface (%s) address %s\n", ifr.ifr_name, inet_ntoa(((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr));
	ipv4_addr = ntohl(((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr.s_addr);
	close(fd);

	/* ignore whatever is destined to or originates from broadcast ip address */
	struct nfct_filter_ipv4 filter_ipv4;

	filter_ipv4.addr = ipv4_addr;
	filter_ipv4.mask = 0xffffffff;

	nfct_filter_set_logic(filter,
		NFCT_FILTER_DST_IPV4,
		NFCT_FILTER_LOGIC_NEGATIVE);

	nfct_filter_add_attr(filter, NFCT_FILTER_DST_IPV4, &filter_ipv4);

	nfct_filter_set_logic(filter,
		NFCT_FILTER_SRC_IPV4,
		NFCT_FILTER_LOGIC_NEGATIVE);

	nfct_filter_add_attr(filter, NFCT_FILTER_SRC_IPV4, &filter_ipv4);

#ifdef FEATURE_VLAN_MPDN
	uint8_t testmac[IPA_MAC_ADDR_SIZE];

	memset(testmac, 0, IPA_MAC_ADDR_SIZE);
	for(int i = 0; i < IPA_MAX_NUM_BRIDGES; i++)
	{
		/* mac address !=0 i.e. bridge exists*/
		if(memcmp(IPACM_Iface::ipacmcfg->vlan_bridges[i].bridge_mac,
			testmac,
			sizeof(uint8_t) * IPA_MAC_ADDR_SIZE))
		{
			/* skip the default bridge - was handled above */
			if(strcmp(IPACM_Iface::ipacmcfg->ipa_virtual_iface_name,
				IPACM_Iface::ipacmcfg->vlan_bridges[i].bridge_name) == 0)
			{
				continue;
			}

			IPACMDBG("bridge (%s)", IPACM_Iface::ipacmcfg->vlan_bridges[i].bridge_name);
			/* ignore whatever is destined to or originates from broadcast ip address */
			struct nfct_filter_ipv4 filter_ipv4;

			filter_ipv4.addr = IPACM_Iface::ipacmcfg->vlan_bridges[i].bridge_ipv4_addr;
			filter_ipv4.mask = 0xffffffff;
			iptodot("ignore connections destined to", filter_ipv4.addr);
			nfct_filter_set_logic(filter,
				NFCT_FILTER_DST_IPV4,
				NFCT_FILTER_LOGIC_NEGATIVE);

			nfct_filter_add_attr(filter, NFCT_FILTER_DST_IPV4, &filter_ipv4);

			nfct_filter_set_logic(filter,
				NFCT_FILTER_SRC_IPV4,
				NFCT_FILTER_LOGIC_NEGATIVE);

			nfct_filter_add_attr(filter, NFCT_FILTER_SRC_IPV4, &filter_ipv4);
		}
	}
#endif
	return IPACM_SUCCESS;
}

int IPACM_ConntrackClient::IPA_Conntrack_Filters_Ignore_Local_Iface
(
	 struct nfct_filter *filter,
	 ipacm_event_iface_up *param
)
{
	struct nfct_filter_ipv4 filter_ipv4;

	filter_ipv4.addr = param->ipv4_addr;
	filter_ipv4.mask = 0xffffffff;

	/* ignore whatever is destined to local interfaces */
	IPACMDBG("Ignore connections destinated to interface %s", param->ifname);
	iptodot("with ipv4 address", param->ipv4_addr);
	nfct_filter_set_logic(filter,
												NFCT_FILTER_DST_IPV4,
												NFCT_FILTER_LOGIC_NEGATIVE);

	nfct_filter_add_attr(filter, NFCT_FILTER_DST_IPV4, &filter_ipv4);

	IPACMDBG("Ignore connections orignated from interface %s", param->ifname);
	iptodot("with ipv4 address", filter_ipv4.addr);
	nfct_filter_set_logic(filter,
												NFCT_FILTER_SRC_IPV4,
												NFCT_FILTER_LOGIC_NEGATIVE);

	nfct_filter_add_attr(filter, NFCT_FILTER_SRC_IPV4, &filter_ipv4);

	/* Retrieve broadcast address */
	/* Intialize with 255.255.255.255 */
	uint32_t bc_ip_addr = 0xFFFFFFFF;

	/* calculate broadcast address from addr and addr_mask */
	bc_ip_addr = (bc_ip_addr & (~param->addr_mask));
	bc_ip_addr = (bc_ip_addr | (param->ipv4_addr & param->addr_mask));

	/* netfitler expecting in host-byte order */
	filter_ipv4.addr = bc_ip_addr;
	filter_ipv4.mask = 0xffffffff;

	iptodot("with broadcast address", filter_ipv4.addr);
	nfct_filter_set_logic(filter,
												NFCT_FILTER_DST_IPV4,
												NFCT_FILTER_LOGIC_NEGATIVE);

	nfct_filter_add_attr(filter, NFCT_FILTER_DST_IPV4, &filter_ipv4);

	return 0;
}

void IPACM_ConntrackClient::IPA_Conntrack_Filters_Ignore_Local_Iface_v6(struct nfct_filter *filter,
	struct nfct_handle *handle, ipacm_event_iface_up *data)
{
	const struct nfct_filter_ipv6 filter_ipv6_addr =
	{
		{data->ipv6_addr[0], data->ipv6_addr[1], data->ipv6_addr[2], data->ipv6_addr[3]},
		{0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff},
	};
	IPA_Conntrack_Filters_Ipv6_Add_Src_Dst_Attr(filter, filter_ipv6_addr);

	if (handle == NULL)
	{
		return;
	}

	IPACMDBG("attaching the filter to the handle\n");
	int ret = nfct_filter_attach(nfct_fd(handle), filter);
	if (ret)
	{
		PERROR("unable to attach the filter to the handle\n");
		IPACMERR("The handle:%pK, fd:%d Error: %d\n", handle, nfct_fd(handle), ret);
		return;
	}
}

/* Function which sets up filters to ignore
		 connections to and from local interfaces */
int IPACM_ConntrackClient::IPA_Conntrack_Filters_Ignore_Local_Addrs
(
	 struct nfct_filter *filter
)
{
	struct nfct_filter_ipv4 filter_ipv4;

	/* ignore whatever is destined to or originates from broadcast ip address */
	filter_ipv4.addr = 0xffffffff;
	filter_ipv4.mask = 0xffffffff;

	nfct_filter_set_logic(filter,
												NFCT_FILTER_DST_IPV4,
												NFCT_FILTER_LOGIC_NEGATIVE);

	nfct_filter_add_attr(filter, NFCT_FILTER_DST_IPV4, &filter_ipv4);

	nfct_filter_set_logic(filter,
												NFCT_FILTER_SRC_IPV4,
												NFCT_FILTER_LOGIC_NEGATIVE);

	nfct_filter_add_attr(filter, NFCT_FILTER_SRC_IPV4, &filter_ipv4);

	return 0;
} /* IPA_Conntrack_Filters_Ignore_Local_Addrs() */

/* Function which sets up received filter to ignore connections to and from link-local, site-local, unique-local and
   multicast addresses */
void IPACM_ConntrackClient::IPA_Conntrack_Filters_Ignore_Ipv6_Addresses(struct nfct_filter *filter)
{
#ifdef FEATURE_IPV6_NAT
	IPACM_Config *config_instance = NULL;

	config_instance = IPACM_Config::GetInstance();

	if(config_instance && config_instance->ipv6_nat_enable)
		return;
#endif
	const struct nfct_filter_ipv6 filter_ipv6_private_network_addresses =
	{
		{0xfc000000, 0x0, 0x0, 0x0 },
		{0xfe000000, 0x0, 0x0, 0x0 },
	};
	IPA_Conntrack_Filters_Ipv6_Add_Src_Dst_Attr(filter, filter_ipv6_private_network_addresses);

	const struct nfct_filter_ipv6 filter_ipv6_link_local_addresses =
	{
		{0xfe800000, 0x0, 0x0, 0x0},
		{0xffc00000, 0x0, 0x0, 0x0},
	};
	IPA_Conntrack_Filters_Ipv6_Add_Src_Dst_Attr(filter, filter_ipv6_link_local_addresses);

	const struct nfct_filter_ipv6 filter_ipv6_site_local_addresses =
	{
		{0xfec00000, 0x0, 0x0, 0x0},
		{0xffc00000, 0x0, 0x0, 0x0},
	};
	IPA_Conntrack_Filters_Ipv6_Add_Src_Dst_Attr(filter, filter_ipv6_site_local_addresses);

	const struct nfct_filter_ipv6 filter_ipv6_multi_cast_addresses =
	{
		{0xff000000, 0x0, 0x0, 0x0},
		{0xff000000, 0x0, 0x0, 0x0},
	};
	IPA_Conntrack_Filters_Ipv6_Add_Src_Dst_Attr(filter, filter_ipv6_multi_cast_addresses);
}

void IPACM_ConntrackClient::IPA_Conntrack_Filters_Ipv6_Add_Src_Dst_Attr(struct nfct_filter *filter,
	const struct nfct_filter_ipv6 &attr)
{
	nfct_filter_set_logic(filter, NFCT_FILTER_DST_IPV6, NFCT_FILTER_LOGIC_NEGATIVE);
	nfct_filter_add_attr(filter, NFCT_FILTER_DST_IPV6, &attr);

	nfct_filter_set_logic(filter, NFCT_FILTER_SRC_IPV6, NFCT_FILTER_LOGIC_NEGATIVE);
	nfct_filter_add_attr(filter, NFCT_FILTER_SRC_IPV6, &attr);
}

/* Initialize TCP Filter */
int IPACM_ConntrackClient::IPA_Conntrack_TCP_Filter_Init(void)
{
	int ret = 0;
	IPACM_ConntrackClient *pClient;

	IPACMDBG("\n");

	pClient = IPACM_ConntrackClient::GetInstance();
	if(pClient == NULL)
	{
		IPACMERR("unable to get conntrack client instance\n");
		return -1;
	}

	ret = nfct_filter_set_logic(pClient->tcp_filter,
															NFCT_FILTER_L4PROTO,
															NFCT_FILTER_LOGIC_POSITIVE);
	if(ret == -1)
	{
		IPACMERR("Unable to set filter logic\n");
		return -1;
	}

	/* set protocol filters as tcp and udp */
	nfct_filter_add_attr_u32(pClient->tcp_filter, NFCT_FILTER_L4PROTO, IPPROTO_TCP);


	struct nfct_filter_proto tcp_proto_state;
	tcp_proto_state.proto = IPPROTO_TCP;
	tcp_proto_state.state = TCP_CONNTRACK_ESTABLISHED;

	ret = nfct_filter_set_logic(pClient->tcp_filter,
															NFCT_FILTER_L4PROTO_STATE,
															NFCT_FILTER_LOGIC_POSITIVE);
	if(ret == -1)
	{
		IPACMERR("unable to set filter logic\n");
		return -1;
	}
	nfct_filter_add_attr(pClient->tcp_filter,
											 NFCT_FILTER_L4PROTO_STATE,
											 &tcp_proto_state);


	tcp_proto_state.proto = IPPROTO_TCP;
	tcp_proto_state.state = TCP_CONNTRACK_FIN_WAIT;
	ret = nfct_filter_set_logic(pClient->tcp_filter,
															NFCT_FILTER_L4PROTO_STATE,
															NFCT_FILTER_LOGIC_POSITIVE);
	if(ret == -1)
	{
		IPACMERR("unable to set filter logic\n");
		return -1;
	}

	nfct_filter_add_attr(pClient->tcp_filter,
											 NFCT_FILTER_L4PROTO_STATE,
											 &tcp_proto_state);

	IPA_Conntrack_Filters_Ignore_Ipv6_Addresses(pClient->tcp_filter);

	return 0;
}


/* Initialize UDP Filter */
int IPACM_ConntrackClient::IPA_Conntrack_UDP_Filter_Init(void)
{
	int ret = 0;
	IPACM_ConntrackClient *pClient = IPACM_ConntrackClient::GetInstance();
	if(pClient == NULL)
	{
		IPACMERR("unable to get conntrack client instance\n");
		return -1;
	}

	ret = nfct_filter_set_logic(pClient->udp_filter,
															NFCT_FILTER_L4PROTO,
															NFCT_FILTER_LOGIC_POSITIVE);
	if(ret == -1)
	{
		IPACMERR("unable to set filter logic\n");
	}
	/* set protocol filters as tcp and udp */
	nfct_filter_add_attr_u32(pClient->udp_filter, NFCT_FILTER_L4PROTO, IPPROTO_UDP);

	IPA_Conntrack_Filters_Ignore_Ipv6_Addresses(pClient->udp_filter);

	return 0;
}

void* IPACM_ConntrackClient::UDPConnTimeoutUpdate(void *ptr)
{
	NatApp *nat_inst = NULL;
#ifdef IPACM_DEBUG
	IPACMDBG("\n");
#endif

	nat_inst = NatApp::GetInstance();
	if(nat_inst == NULL)
	{
		IPACMERR("unable to create nat instance\n");
	}

	uint32_t natsCount = 0;
	NatBase* nats[] = {NULL, NULL};
	NatBase* ipv6ct = Ipv6ct::GetInstance();
	if (ipv6ct != NULL)
	{
		nats[natsCount++] = ipv6ct;
	}

	if (!natsCount && nat_inst == NULL)
	{
		IPACMERR("There are no NAT instances. Exiting 'Conntrack Timeout Update' thread\n");
		return NULL;
	}

	ConntrackTimestampUtil::Init();

	while(1)
	{
		if (nat_inst != NULL)
		{
			nat_inst->UpdateUDPTimeStamp();
		}

		bool isTcpUdpTimeoutUpToDate = false;
		for (uint32_t i = 0; i < natsCount; ++i)
		{
			nats[i]->UpdateTcpUdpTimeStamps(isTcpUdpTimeoutUpToDate);
		}

		sleep(UDP_TIMEOUT_UPDATE);
	} /* end of while(1) loop */

#ifdef IPACM_DEBUG
	IPACMDBG("Returning from %s() %d\n", __FUNCTION__, __LINE__);
#endif

	return NULL;
}

/* Thread to initialize TCP Conntrack Filters*/
void* IPACM_ConntrackClient::TCPRegisterWithConnTrack(void *)
{
	int ret;
	IPACM_ConntrackClient *pClient;
	unsigned subscrips = 0;
	int buf_size = 2097152, recbuff=0, res;
	socklen_t optlen;

	IPACMDBG("\n");

	pClient = IPACM_ConntrackClient::GetInstance();
	if(pClient == NULL)
	{
		IPACMERR("unable to get conntrack client instance\n");
		return NULL;
	}

	subscrips = (NF_NETLINK_CONNTRACK_UPDATE | NF_NETLINK_CONNTRACK_DESTROY);

#if defined(CT_OPT) || defined(FEATURE_IPACM_UL_FIREWALL)
	subscrips |= NF_NETLINK_CONNTRACK_NEW;
#endif

	pClient->tcp_hdl = nfct_open(CONNTRACK, subscrips);

	if(pClient->tcp_hdl == NULL)
	{
		PERROR("nfct_open\n");
		return NULL;
	}

	/* Initialize the filter */
	ret = IPA_Conntrack_TCP_Filter_Init();
	if(ret == -1)
	{
		IPACMERR("Unable to initliaze TCP Filter\n");
		return NULL;
	}

	/* Attach the filter to net filter handler */
	ret = nfct_filter_attach(nfct_fd(pClient->tcp_hdl), pClient->tcp_filter);
	if(ret == -1)
	{
		IPACMDBG("unable to attach TCP filter\n");
		return NULL;
	}

	/* Register callback with netfilter handler */
	IPACMDBG_H("tcp handle:%pK, fd:%d\n", pClient->tcp_hdl, nfct_fd(pClient->tcp_hdl));
#ifndef CT_OPT
	nfct_callback_register(pClient->tcp_hdl,
			(nf_conntrack_msg_type)	(NFCT_T_UPDATE | NFCT_T_DESTROY | NFCT_T_NEW),
						IPAConntrackEventCB, NULL);
#else
	nfct_callback_register(pClient->tcp_hdl, (nf_conntrack_msg_type) NFCT_T_ALL, IPAConntrackEventCB, NULL);
#endif

	optlen = sizeof(recbuff);
	res = getsockopt(nfct_fd(pClient->tcp_hdl), SOL_SOCKET, SO_RCVBUF, &recbuff, &optlen);

	if(res == -1)
	{
		IPACMDBG("Error getsockopt (%d)(%s)\n", ret, strerror(errno));
	}
	else
	{
		IPACMDBG("original receive buffer size = %d\n", recbuff);
	}

	IPACMDBG("set the receive buffer to %d\n", buf_size);

	if (setsockopt(nfct_fd(pClient->tcp_hdl), SOL_SOCKET, SO_RCVBUFFORCE, &buf_size, sizeof(int)) == -1)
		IPACMERR("Error setting socket opts (%d)(%s)\n", ret, strerror(errno));

	res = getsockopt(nfct_fd(pClient->tcp_hdl), SOL_SOCKET, SO_RCVBUF, &recbuff, &optlen);

	if(res == -1)
	{
		IPACMDBG("Error getsockopt (%d)(%s)\n", ret, strerror(errno));
	}
	else
	{
		IPACMDBG("new receive buffer size = %d\n", recbuff);
	}

	/* Block to catch events from net filter connection track */
	/* nfct_catch() receives conntrack events from kernel-space, by default it
			 blocks waiting for events. */
	IPACMDBG("Waiting for events\n");

ctcatch:
	ret = nfct_catch(pClient->tcp_hdl);
	if((ret == -1) && (errno != ENOMSG) && (errno != ENOBUFS))
	{
		IPACMERR("(%d)(%d)(%s)\n", ret, errno, strerror(errno));
		return NULL;
	}
	else
	{
		IPACMDBG("ctcatch ret:%d, errno:%d\n", ret, errno);
		goto ctcatch;
	}

	IPACMDBG("Exit from tcp thread\n");

	/* destroy the filter.. this will not detach the filter */
	nfct_filter_destroy(pClient->tcp_filter);
	pClient->tcp_filter = NULL;

	/* de-register the callback */
	nfct_callback_unregister(pClient->tcp_hdl);
	/* close the handle */
	nfct_close(pClient->tcp_hdl);
	pClient->tcp_hdl = NULL;

	pthread_exit(NULL);
	return NULL;
}

/* Thread to initialize UDP Conntrack Filters*/
void* IPACM_ConntrackClient::UDPRegisterWithConnTrack(void *)
{
	int ret;
	IPACM_ConntrackClient *pClient = NULL;
	int buf_size = 2097152, recbuff=0, res;
	socklen_t optlen;

	IPACMDBG("\n");

	pClient = IPACM_ConntrackClient::GetInstance();
	if(pClient == NULL)
	{
		IPACMERR("unable to retrieve instance of conntrack client\n");
		return NULL;
	}

	pClient->udp_hdl = nfct_open(CONNTRACK,
					(NF_NETLINK_CONNTRACK_NEW | NF_NETLINK_CONNTRACK_UPDATE |
					NF_NETLINK_CONNTRACK_DESTROY));
	if(pClient->udp_hdl == NULL)
	{
		PERROR("nfct_open\n");
		return NULL;
	}

	/* Initialize Filter */
	ret = IPA_Conntrack_UDP_Filter_Init();
	if(-1 == ret)
	{
		IPACMDBG("Unable to initalize udp filters\n");
		return NULL;
	}

	/* Attach the filter to net filter handler */
	ret = nfct_filter_attach(nfct_fd(pClient->udp_hdl), pClient->udp_filter);
	if(ret == -1)
	{
		IPACMDBG("unable to attach the filter\n");
		return NULL;
	}

	/* Register callback with netfilter handler */
	IPACMDBG_H("udp handle:%pK, fd:%d\n", pClient->udp_hdl, nfct_fd(pClient->udp_hdl));
	nfct_callback_register(pClient->udp_hdl,
			(nf_conntrack_msg_type)(NFCT_T_NEW | NFCT_T_UPDATE | NFCT_T_DESTROY),
			IPAConntrackEventCB,
			NULL);

	optlen = sizeof(recbuff);

	res = getsockopt(nfct_fd(pClient->udp_hdl), SOL_SOCKET, SO_RCVBUF, &recbuff, &optlen);

	if(res == -1)
	{
		IPACMDBG("Error getsockopt (%d)(%s)\n", ret, strerror(errno));
	}
	else
	{
		IPACMDBG("original receive buffer size = %d\n", recbuff);
	}

	IPACMDBG("set the receive buffer to %d\n", buf_size);

	if (setsockopt(nfct_fd(pClient->udp_hdl), SOL_SOCKET, SO_RCVBUFFORCE, &buf_size, sizeof(int)) == -1)
		IPACMERR("Error setting socket opts (%d)(%s)\n", ret, strerror(errno));

	res = getsockopt(nfct_fd(pClient->udp_hdl), SOL_SOCKET, SO_RCVBUF, &recbuff, &optlen);

	if(res == -1)
	{
		IPACMDBG("Error getsockopt (%d)(%s)\n", ret, strerror(errno));
	}
	else
	{
		IPACMDBG("new send receive size = %d\n", recbuff);
	}

	IPACMDBG("Waiting for events\n");
	/* Block to catch events from net filter connection track */
ctcatch:
	ret = nfct_catch(pClient->udp_hdl);
	if((ret == -1) && (errno != ENOMSG) && (errno != ENOBUFS))
	{
		IPACMDBG("(%d)(%d)(%s)\n", ret, errno, strerror(errno));
		return NULL;
	}
	else
	{
		IPACMDBG("ctcatch ret:%d, errno:%d\n", ret, errno);
		goto ctcatch;
	}

	IPACMDBG("Exit from udp thread with ret: %d\n", ret);

	/* destroy the filter.. this will not detach the filter */
	nfct_filter_destroy(pClient->udp_filter);
	pClient->udp_filter = NULL;

	/* de-register the callback */
	nfct_callback_unregister(pClient->udp_hdl);
	/* close the handle */
	nfct_close(pClient->udp_hdl);
	pClient->udp_hdl = NULL;

	pthread_exit(NULL);
	return NULL;
}

void IPACM_ConntrackClient::UpdateUDPFilters(void *param, bool isWan)
{
	static bool isIgnore = false;
	int ret = 0;
	IPACM_ConntrackClient *pClient = NULL;

	pClient = IPACM_ConntrackClient::GetInstance();
	if(pClient == NULL)
	{
		IPACMERR("unable to retrieve conntrack client instance\n");
		return;
	}

	if(pClient->udp_filter == NULL)
	{
		 return;
	}

	if(!isWan)
	{
		IPA_Conntrack_Filters_Ignore_Local_Iface(pClient->udp_filter,
																		 (ipacm_event_iface_up *)param);

		if(!isIgnore)
		{
			IPA_Conntrack_Filters_Ignore_Bridge_Addrs(pClient->udp_filter);
			IPA_Conntrack_Filters_Ignore_Local_Addrs(pClient->udp_filter);
			isIgnore = true;
		}
	}

	/* Attach the filter to udp handle */
	if(pClient->udp_hdl != NULL)
	{
		IPACMDBG("attaching the filter to udp handle\n");
		ret = nfct_filter_attach(nfct_fd(pClient->udp_hdl), pClient->udp_filter);
		if(ret == -1)
		{
			PERROR("unable to attach the filter to udp handle\n");
			IPACMERR("udp handle:%pK, fd:%d Error: %d\n",pClient->udp_hdl, nfct_fd(pClient->udp_hdl), ret);
			return;
		}
	}

	return;
}

void IPACM_ConntrackClient::UpdateTCPFilters(void *param, bool isWan)
{
	static bool isIgnore = false;
	int ret = 0;
	IPACM_ConntrackClient *pClient = NULL;

	pClient = IPACM_ConntrackClient::GetInstance();
	if(pClient == NULL)
	{
		IPACMERR("unable to retrieve conntrack client instance\n");
		return;
	}

	if(pClient->tcp_filter == NULL)
		return;

	if(!isWan)
	{
		IPA_Conntrack_Filters_Ignore_Local_Iface(pClient->tcp_filter,
																	(ipacm_event_iface_up *)param);

		if(!isIgnore)
		{
			IPA_Conntrack_Filters_Ignore_Bridge_Addrs(pClient->udp_filter);
			IPA_Conntrack_Filters_Ignore_Local_Addrs(pClient->udp_filter);
			isIgnore = true;
		}
	}

	/* Attach the filter to tcp handle */
	if(pClient->tcp_hdl != NULL)
	{
		IPACMDBG("attaching the filter to tcp handle\n");
		ret = nfct_filter_attach(nfct_fd(pClient->tcp_hdl), pClient->tcp_filter);
		if(ret == -1)
		{
			PERROR("unable to attach the filter to tcp handle\n");
			IPACMERR("tcp handle:%pK, fd:%d Error: %d\n",pClient->tcp_hdl, nfct_fd(pClient->tcp_hdl), ret);
			return;
		}
	}

  return;
}

void IPACM_ConntrackClient::UpdateFilters_v6(ipacm_event_iface_up* data)
{
	IPACM_ConntrackClient* client = IPACM_ConntrackClient::GetInstance();
	if (client == NULL)
	{
		IPACMERR("unable to retrieve conntrack client instance\n");
		return;
	}

	IPA_Conntrack_Filters_Ignore_Local_Iface_v6(client->udp_filter, client->udp_hdl, data);
	IPA_Conntrack_Filters_Ignore_Local_Iface_v6(client->tcp_filter, client->tcp_hdl, data);
}

