# Event based Interprocess Communication for Dummies

## Object
Many  established solutions  for  Inter-Process Communication  (IPC) like  dBus,
Corba, COM, etc. are ways too complex  for the needs of small sized applications
especially when  implemented for the so  called embedded system domain  in the C
programming language. They either necessitate to  deal with the odds and ends of
an interface  definition language or they  come with an huge  dependency tree of
underlying libraries and  frameworks. This makes prototyping  and debugging much
more difficult, expands the memory  footprint and generally increases the effort
for  maintenance  in example  caused  by  the  requirement to  provide  security
updates. For  the given reasons  many application developers directly  deal with
the Posix Socket API  which generally means to write a lot  of boiler plate code
for every daemon process  again. The aim of this library is  to take this burden
of your shoulders. One  simple API function call and a  callback function is all
that is needed to get the IPC part  of your damon process up and running. Within
the development phase  the remote side can be simply  simulated by standard Unix
tools like  telnet, socat  or netcat which  dramatically simplifies  matters for
initial bring up.

## How to build and use the library.
### Build Dependencies
In addition  to the  commonly used  Unix System Libraries  and build  tools such
as  libc, pthreads  and  the  autobuild suite,  libintercom  depends on  another
utitily library  libcutils which is maintained  by the same author  and provides
fundamental  data structures  such  as  circular linked  lists  where the  event
management  is based  on. You  will  need to  compile and  install this  library
on  you  system  first.  The  library  libcutils  is  provided  under  the  same
license  terms  as  libintercom.  Refer to  the  [installation  instructions  on
Github](https://github.com/linneman/libcutils)  for  an  explanation how  to  do
this.

### Download and Compile
Compiling libintercom  is done in the  very same way as  for libcutils following
the guidelines of the free software foundation:

    git clone git://github.com/linneman/libintercom.git
    cd libintercom

    ./autogen # invokes autoreconf -i for the generation of the configure script

    ./configure
    make
    make install # as root

In case you download an official release as gzipped tarball you exclusively will
need the  last three commands  since the  configure script is  already delivered
with stable releases.

Pay  attention to  the fact  that  the library  is per  default installed  under
`/usr/local/lib` which is  not in the library path on  many linux distributions.
Thus you will need to put `/usr/local/lib` to your library path e.g. by defining
the enviroment variable

    export LD_LIBRARY_PATH=/usr/local/lib.

Alternatively,  you  can  edit  `/etc/ld.so.conf`  which  contains  the  default
directories  searched.  Note  that  you  may  also  need  to  update  the  cache
`/etc/ld.so.cache` by running ldconfig (as root, or with sudo).

## Using libintercom
The  following code  snippet illustrate  how to  initialize a  server background
threads which listen  to port TCP port  5000 and to the unix  domain socket file
/tmp/icom.sock for incoming connections. Each  address accepts 10 connections at
maximum.

    #include <stdio.h>
    #include <intercom/server.h>

    main() {
      t_icom_server_state* p_server_state;
      t_icom_server_decl decl_table[] = {
        { { AF_UNIX, "/tmp/icom.sock", 0 }, 10 },
        { { AF_INET, "0.0.0.0", 5000 }, 10 },
      };
      const int decl_table_len = sizeof(decl_table) / sizeof( t_icom_server_decl );


      p_server_state = icom_create_server_handlers( decl_table, decl_table_len, 256, 10, evt_cb_demo, NULL );
      if( p_server_state == NULL ) {
        fprintf( stderr, "could not create server handlers error!\n" );
        return -1;
      }

      /* do something else or for the sake of simplicity just sleep a while */
      sleep( 100 );
    }

The callback handler evt_cb_demo is invoked whenever new data has been received from
a client. Here is how a simple loopback implementation would look like:

    static int evt_cb_demo( t_icom_evt* p_evt )
    {
      char data[128];
      int errors = 0;

      snprintf( data, sizeof(data), "pong-> %s\n", p_evt->p_data );
      errors = icom_reply_to_sender( p_evt, data, strlen( data ) );

      if( errors)
        fprintf( stderr, "error while sending \"%30s\" occured!\n", data );

      return errors;
    }

In addition  to the function icom_reply_to_sender,  is possible to reply  to all
clients connected to  the same address where the message  has been received from
via the  function icom_reply_to_address or  simply to all connected  clients via
the function icom_reply_to_all_connections.

For a more detailed explanation about underlying concepts refer to the complete
[API description](http://linneman.github.io/libintercom/doc/doxygen/html/group__intercom.html).


### Integration of libintercom into your own project
The interation of the libctutils within  your own FSF compliant application is a
nobrainer. Just put the following line within your `configure.ac` file:

    PKG_CHECK_MODULES(libcutils, libcutils >= 1.0.0 )
    PKG_CHECK_MODULES(libintercom, libintercom >= 1.0.0 )

Refer to `libintercom/example` for a small sample project which provide a complete
example about a minimum version of `configure.ac` and `Makefile.am` as the input
for the autoconf and automake.

## License
This implementation code stands under the terms of the
[GNU LESSER GENERAL PUBLIC LICENSE, Version 2.1](http://www.gnu.org/licenses/old-licenses/lgpl-2.1).

March 2016, Otto Linnemann