/*
    intercom - Event based Interprocess Communication for Dummies
    Copyright 2016 Otto Linnemann

    This program is free software: you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation, either version 2.1
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General
    Public License along with this program. If not, see
    <http://www.gnu.org/licenses/>.
*/

#ifndef CLIENT_H
#define CLIENT_H

#include <intercom/events.h>
#include <intercom/common.h>
#include <intercom/data_stream_splitter.h>
#include <olcutils/slist.h>
#include <stdbool.h>

#ifdef __cplusplus
extern "C" {
#endif

/*! \file client.h
    \brief IP/unix domain socket client

    \addtogroup client_api Client
    @{
 */

struct s_icom_client_state;

/*! state data for a client socket connection */
typedef struct s_sock_client {
  t_icom_sock_addr            addr;                         /*!< socket API address specification */
  socklen_t                   addr_len;                     /*!< length of socket address */
  int                         sock_family;                  /*!< socket family, unix domain or IP */
  int                         fd;                           /*!< file descript for connection to r/w data */
  pthread_t                   handler;                      /*!< server socket handler */
  struct s_icom_client_state* p_client_state;               /*!< back reference to client state struct */
  t_icom_events*              p_events;                     /*!< reference to event object */
  t_icom_evt*                 p_evt;                        /*!< next processed event */
  t_icom_data_stream*         p_data_stream;                /*!< pointer to data stream splitter instance */
  void*                       p_data_chunk;                 /*!< pointer to temporary socket receive data chunk */
  void*                       p_user_ctx;                   /*!< reference to user data context */
} t_icom_sock_client;


/*! client side (convenient) state object, provides all information */
typedef struct s_icom_client_state {
  const t_icom_addr_decl*     p_addr_decl;                  /*!< reference to input address spec */
  t_icom_events*              p_events;                     /*!< reference to event object */
  bool                        p_events_needs_to_released;   /*!< true, when created internally */
  t_icom_sock_client*         p_sock_client;                /*!< reference to client socket connection */
  pthread_mutex_t             mutex;                        /*!< mutex for access protection */
} t_icom_client_state;


/*! connection state */
typedef enum {
  ICOM_CLIENT_CON_IDLE,                                     /*!< idle state, set after intializiation */
  ICOM_CLIENT_CON_DISCONNECTED,                             /*!< connection lost, try to reconnect */
  ICOM_CLIENT_CON_CONNECTED,                                /*!< system is connected */
  ICOM_CLIENT_CON_KILLED,                                   /*!< system is connected */
} t_icom_client_conn_state;

/*! client (re)connection handler */
typedef struct {
  pthread_t                   handler;                      /*!< connection handler */
  pthread_mutex_t             mutex;                        /*!< mutex for access protection */
  t_icom_client_state*        p_client_state;               /*!< reference to icom_client_state */
  t_icom_client_conn_state    connection_state;             /*!< connection state */
  pthread_cond_t              signal;                       /*!< signal indicating state change */
  t_icom_events*              p_events;                     /*!< reference to event object */

  t_icom_addr_decl            addr_decl;                     /*!< addr spec for the server */
  t_evt_cb p_evt_cb;                                        /*!< callback fcnt, invoked when data received */
  void* p_user_ctx;                                         /*!< reference to user context data */
} t_icom_client_conn_handler;


/*!
 * client side message send back to requesting process from call back function
 *
 * \param p_evt pointer to event object retrieved from call back function
 * \param data pointer to buffer to be transfered
 * \param len buffer len
 * \return 0 in case of success, otherwise negative error code
 */
int icom_client_reply( t_icom_evt* p_evt, const char* data, const int len );


/*!
 * send client request to connected server
 *
 * The connection has be established first by the function
 * \ref icom_create_client_handler_with_event_pool or
 * \ref icom_create_client_handler
 *
 * \param p pointer to client handler's state data
 * \param data pointer to buffer to be transfered
 * \param len buffer len
 * \return 0 in case of success, otherwise negative error code
 */
int icom_client_request( t_icom_client_state* p, const char* data, const int len );


/*! kill or working tasks and release state data of a client comm. handler
 *
 * \param p pointer to client state data to be released
 */
void icom_kill_client_handler( t_icom_client_state* p );


/*!
 * create communication handler for a client address
 *
 * This function works similar to  \ref icom_create_client_handler but
 * on  contrast to  that  it does  not  create a  new  event pool  but
 * delivers its data  to existing one instead. This  allows to forward
 * traffic from client and server connections to the same event pool.
 *
 * \param p_addr_decl pointer to address specification for the server to connect to
 * \param p_events pointer to an existing event pool, must be created externally
 * \param p_user_ctx reference to user context data
 * \param retries number of retries for connections establishment
 * \return pointer to the newly instantiated object or NULL in case of error
 */
t_icom_client_state* icom_create_client_handler_with_event_pool(
  const t_icom_addr_decl* p_addr_decl,
  t_icom_events* p_events,
  void* p_user_ctx,
  const int retries );


/*!
 * create communication handler for a client address and an event handler
 *
 * This function establishes  a client connection to  the specified IP
 * or unix  domain socket server  once. When the connection  cannot be
 * established it retries the specified amount of times. Futhermore an
 * internal  event  managing task  is  instantiated  where the  client
 * process  delivers  its data  to.  Finally  a callback  function  is
 * registered which is  invoked whenever a message from  the server is
 * received.
 *
 * The handler terminates when the connection to the server is lost e.g.
 * when the server crashes. Use the function
 * \ref icom_kill_client_connection_handler in order to automatically
 * start to reconnect in such a case.
 *
 * \param p_addr_decl pointer to address specification for the server to connect to
 * \param max_evt_data_size maximum event size in octets
 * \param pool_size number of elementes in the event pool
 * \param p_evt_cb callback function to be invoked when data is received
 * \param p_user_ctx reference to user context data
 * \param retries number of retries for connections establishment
 * \return pointer to the newly instantiated object or NULL in case of error
 */
t_icom_client_state* icom_create_client_handler(
  const t_icom_addr_decl* p_addr_decl,
  const int max_evt_data_size,
  const int pool_size,
  const t_evt_cb p_evt_cb,
  void* p_user_ctx,
  const int retries );



/*!
 * send client request to via connection handler to connected server
 *
 * The connection has be established first by the function
 * \ref icom_kill_client_connection_handler
 *
 * \param p pointer to client handler's state data
 * \param data pointer to buffer to be transfered
 * \param len buffer len
 * \return 0 in case of success, otherwise negative error code
 */
int icom_client_handler_request( t_icom_client_conn_handler *p, const char* data, const int len );


/*! kill or working tasks and release state data of a client connection handler
 *
 * \param p pointer to client state data to be released
 */
void icom_kill_client_connection_handler( t_icom_client_conn_handler* p );


/*!
 * create connection handler for a client address and an event handler
 *
 * This function establishes  a client connection to  the specified IP
 * or unix  domain socket server  once. When the connection  cannot be
 * established it retries the specified amount of times. Futhermore an
 * internal  event  managing task  is  instantiated  where the  client
 * process  delivers  its data  to.  Finally  a callback  function  is
 * registered which is  invoked whenever a message from  the server is
 * received.
 *
 * The handler  automatically reestablishes the specified  TCP or Unix
 * Domain Socket connection when the connection to the server is lost.
 * Requests cannot be  send to the server in the  connection case. The
 * function call icom_client_handler_request  returns a negative error
 * code in this case.
 *
 * \param p_addr_decl pointer to address specification for the server to connect to
 * \param max_evt_data_size maximum event size in octets
 * \param pool_size number of elementes in the event pool
 * \param p_evt_cb callback function to be invoked when data is received
 * \param p_user_ctx reference to user context data
 * \return pointer to the newly instantiated object or NULL in case of error
 */
t_icom_client_conn_handler* icom_create_client_connection_handler(
  const t_icom_addr_decl* p_addr_decl,
  const int max_evt_data_size,
  const int pool_size,
  const t_evt_cb p_evt_cb,
  void* p_user_ctx
  );

/*! @} */

#ifdef __cplusplus
}
#endif

#endif /* #ifndef CLIENT_H */
